/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>

#include "moo_subtract_nod.h"
#include "moo_fits.h"
#include "moo_pfits.h"
#include "moo_badpix.h"
#include "moo_qc.h"
#include "moo_fibres_table.h"
#include "moo_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Sky from object subtraction
  @param    object [DET] object det file
  @param    sky [DET] skt det file
  @param    filename file name of the new nod DET
  @return   object-sky [DET]

 * _Flags considered as bad :
 *
 * _Bad pixels flags_:

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the SCI list have a size
 */
/*----------------------------------------------------------------------------*/
moo_det *
moo_subtract_nod(moo_det *object, moo_det *sky, const char *filename)
{
    cpl_frame *nod_det_frame = NULL;
    moo_det *nod_det = NULL;
    cpl_propertylist *sky_header = NULL;
    cpl_ensure(object != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(sky != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    unsigned int badpix_level = MOO_BADPIX_GOOD;

    moo_try_check(moo_det_save(object, filename), " ");
    nod_det_frame = cpl_frame_new();
    cpl_frame_set_filename(nod_det_frame, filename);

    nod_det = moo_det_create(nod_det_frame);
    moo_try_check(moo_det_load(nod_det, badpix_level), " ");
    moo_try_check(moo_det_rescale_using_exptime(sky, object), " ");

    moo_try_check(moo_det_subtract(nod_det, sky), " ");

    sky_header = moo_det_get_primary_header(sky);
    if (cpl_propertylist_has(sky_header, MOONS_QC_FRAME_RAW1)) {
        const char *sky_filename =
            cpl_propertylist_get_string(sky_header, MOONS_QC_FRAME_RAW1);
        moo_qc_set_frame_raw2(nod_det->primary_header, sky_filename);
    }
    cpl_table *fibre_table = moo_det_get_fibre_table(nod_det);
    moo_try_check(moo_fibres_table_add_nod_cols(fibre_table), " ");
    cpl_table *sky_fibre_table = moo_det_get_fibre_table(sky);

    int sky_nrow = cpl_table_get_nrow(sky_fibre_table);
    int nrow = cpl_table_get_nrow(fibre_table);

    moo_try_assure(
        sky_nrow == nrow, CPL_ERROR_ILLEGAL_INPUT,
        "OBJECT fibre table rows and sky fibre table rows doesnt match");
    for (int i = 0; i < nrow; i++) {
        const char *t =
            cpl_table_get_string(sky_fibre_table, MOO_FIBRES_TABLE_TYPE, i);
        cpl_table_set_string(fibre_table, MOO_FIBRES_TABLE_TYPE_NOD, i, t);
    }
moo_try_cleanup:
    cpl_frame_delete(nod_det_frame);
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_det_delete(nod_det);
        nod_det = NULL;
    }
    return nod_det;
}
/**@}*/
