/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "math.h"
#include <cpl.h>
#include <gsl/gsl_multifit.h>
#include <gsl/gsl_statistics.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_spline.h>
#include <gsl/gsl_fit.h>
#include <gsl/gsl_sort.h>

#include "moo_badpix.h"
#include "moo_utils.h"
#include "sc_skycorr.h"
#include "hdrl_types.h"

#include <string.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_utils     Miscellaneous Utilities
 */
/*----------------------------------------------------------------------------*/

/**@{*/

static const char *const _moons_license =
    "This file is part of the MOONS Instrument Pipeline\n"
    "Copyright (C) 2015-2016 European Southern Observatory\n"
    "\n"
    "This program is free software; you can redistribute it and/or modify\n"
    "it under the terms of the GNU General Public License as published by\n"
    "the Free Software Foundation; either version 2 of the License, or\n"
    "(at your option) any later version.\n"
    "\n"
    "This program is distributed in the hope that it will be useful,\n"
    "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
    "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
    "GNU General Public License for more details.\n"
    "\n"
    "You should have received a copy of the GNU General Public License\n"
    "along with this program; if not, write to the Free Software\n"
    "Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,\n"
    "MA 02110-1301 USA.";

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the pipeline copyright and license
  @return   The copyright and license string

  The function returns a pointer to the statically allocated license string.
  This string should not be modified using the returned pointer.
 */
/*----------------------------------------------------------------------------*/
const char *
moo_get_license(void)
{
    return _moons_license;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Compute image of sigma variation
  @param    list list of images to compare to reference image
  @param    qlist list of QUAL images to ignore badpixel in computation og sigma
  @param    img reference image
  @return   the sigma map or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the size is equal to zero
 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_compute_sigma_map(cpl_imagelist *list, cpl_imagelist *qlist, cpl_image *img)
{
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(qlist != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(img != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int size = cpl_imagelist_get_size(list);
    int qsize = cpl_imagelist_get_size(qlist);
    cpl_ensure(size > 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(size == qsize, CPL_ERROR_ILLEGAL_INPUT, NULL);

    int nx = cpl_image_get_size_x(img);
    int ny = cpl_image_get_size_y(img);
    const double *med_data = cpl_image_get_data_double_const(img);
    int i, j;
    cpl_image *res = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    double *res_data = cpl_image_get_data_double(res);

    for (i = 0; i < nx * ny; i++) {
        double med = med_data[i];

        double val = 0;
        int nbcomb = 0;
        for (j = 0; j < size; j++) {
            const cpl_image *im = cpl_imagelist_get_const(list, j);
            const cpl_image *qim = cpl_imagelist_get_const(qlist, j);
            const double *im_data = cpl_image_get_data_double_const(im);
            const int *qim_data = cpl_image_get_data_int_const(qim);

            if (qim_data[i] == MOO_BADPIX_GOOD) {
                double sig = im_data[i] - med;
                val += sig * sig;
                nbcomb++;
            }
        }
        if (nbcomb > 0) {
            res_data[i] = sqrt(val / (size - 1));
        }
        else {
            res_data[i] = NAN;
        }
    }
    cpl_image_set_bpm(res, cpl_mask_duplicate(cpl_image_get_bpm(img)));

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Compute mask of rejected pixels using kappa sigma algorithm
  @param    sigma_img image on which we used kappa sigma
  @param    niter maximum number of iteration
  @param    kappa multiple of sigma
  @param    cdiff minimum relative change in sigma
  @param    maxfrac maximum fraction of bad pixels allowed
  @return mask of bad pixels

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the size is equal to zero
 */
/*----------------------------------------------------------------------------*/
cpl_mask *
moo_kappa_sigma_clipping(cpl_image *sigma_img,
                         int niter,
                         double kappa,
                         double cdiff,
                         double maxfrac)
{
    cpl_mask *res = NULL;
    cpl_ensure(sigma_img != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(niter >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(kappa >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    int iter, i;

    int nx = cpl_image_get_size_x(sigma_img);
    int ny = cpl_image_get_size_y(sigma_img);
    cpl_mask *mask = cpl_image_get_bpm(sigma_img);

    res = cpl_mask_new(nx, ny);

    cpl_binary *orig_mask_data = cpl_mask_get_data(mask);
    cpl_binary *res_mask_data = cpl_mask_get_data(res);

    double *sigma_data = cpl_image_get_data_double(sigma_img);

    int initbad = cpl_mask_count(mask);
    int initgood = nx * ny - initbad;
    double maxrejected = maxfrac * initgood;

    double sig = 0;
    double median = cpl_image_get_median_dev(sigma_img, &sig);

    for (iter = 0; iter < niter; iter++) {
        cpl_msg_info(__func__,
                     "Iteration %d: median value = %f stdev value = %f",
                     iter + 1, median, sig);
        int nbrej = 0;

        double ksig = kappa * sig;
        for (i = 0; i < nx * ny; i++) {
            if (orig_mask_data[i] == CPL_BINARY_0 &&
                fabs(sigma_data[i] - median) > ksig) {
                nbrej++;
                if (nbrej < maxrejected) {
                    res_mask_data[i] = CPL_BINARY_1;
                    orig_mask_data[i] = CPL_BINARY_1;
                }
                else {
                    cpl_msg_info(
                        __func__,
                        "Number of rejected pixel reached maximum value %d/%d",
                        nbrej, initgood);
                    break;
                }
            }
        }
        if (nbrej == 0) {
            break;
        }
        cpl_msg_info(__func__, "Number of detected bad pixels %d", nbrej);
        double sig2 = 0;
        median = cpl_image_get_median_dev(sigma_img, &sig2);

        if ((sig - sig2) > cdiff) {
            sig = sig2;
        }
        else {
            cpl_msg_info(__func__,
                         "Difference in sigma = %f reached minimum value = %f",
                         (sig - sig2), cdiff);
            break;
        }
    }

    return res;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Fit positions using weighted fluxes
  @param    points to fit (ordered ASC)
  @param    center the fitted center
  @param    width  the width
  @return   the error code or CPL_ERROR_NONE

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the size is equal to zero
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_barycenter_fit(cpl_bivector *points, double *center, double *width)
{
    cpl_ensure_code(points != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(center != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(width != NULL, CPL_ERROR_NULL_INPUT);

    int size = cpl_bivector_get_size(points);

    cpl_ensure_code(size > 0, CPL_ERROR_ILLEGAL_INPUT);

    cpl_vector *x = cpl_bivector_get_x(points);
    cpl_vector *y = cpl_bivector_get_y(points);

    double x1 = cpl_vector_get(x, 0);
    double x2 = cpl_vector_get(x, size - 1);

    double res = 0.0;
    double sum = 0.0;
    int i;
    for (i = 0; i < size; i++) {
        double p = cpl_vector_get(x, i);
        double f = cpl_vector_get(y, i);
        if (f > 0) {
            res += p * f;
            sum += f;
        }
    }
    if (sum == 0) {
        return CPL_ERROR_DIVISION_BY_ZERO;
    }
    res /= sum;
    *center = res;
    *width = x2 - x1;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Fit the data with a gaussian
   @param points the data to fit
   @param fit_pars specifies which parameters participate in the fit
 * (any other parameters will be held constant).
 * Possible values are CPL_FIT_CENTROID, CPL_FIT_STDEV, CPL_FIT_AREA,
 * CPL_FIT_OFFSET and any bitwise combination of these.
 * As a shorthand for including all four parameters in the fit,
 * use CPL_FIT_ALL.
   @param center [OUT] center of the fitted gaussian
   @param width [OUT] width of the fitted gaussian
   @param background [OUT] background of the fitted gaussian
   @param area [OUT] area of the fitted gaussian
   @return CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
*/
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_gaussian_fit(cpl_bivector *points,
                 cpl_fit_mode fit_pars,
                 double *center,
                 double *width,
                 double *background,
                 double *area)
{
    cpl_ensure_code(points != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(center != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(width != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(background != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(area != NULL, CPL_ERROR_NULL_INPUT);

    int size = cpl_bivector_get_size(points);

    cpl_ensure_code(size > 0, CPL_ERROR_ILLEGAL_INPUT);

    cpl_vector *x = cpl_bivector_get_x(points);
    cpl_vector *y = cpl_bivector_get_y(points);

    cpl_error_code status =
        cpl_vector_fit_gaussian(x, NULL, y, NULL, fit_pars, center, width, area,
                                background, NULL, NULL, NULL);

    return status;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Evaluate the gaussian at the given x position
   @param x  x position
   @param x0 center of gaussian.
   @param sigma width of gaussian
   @param offset background level of gaussian
   @param area area of gaussian
   @return Y gaussian result for x
*/
/*----------------------------------------------------------------------------*/
double
moo_gaussian_eval(double x, double x0, double sigma, double offset, double area)
{
    double res;

    res = area / sqrt(CPL_MATH_2PI * sigma * sigma) *
              exp(-(x - x0) * (x - x0) / (2 * sigma * sigma)) +
          offset;

    return res;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Find the x positions of the gaussian at the given y position
   @param y  y position
   @param x0 center of gaussian.
   @param sigma width of gaussian
   @param offset background level of gaussian
   @param area area of gaussian
   @param x1 [OUT] first x solution (minimum)
   @param x2 [OUT] second x solution (maximum)
   @return CPL_ERROR_NONE
*/
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_gaussian_eval_inv(double y,
                      double x0,
                      double sigma,
                      double offset,
                      double area,
                      double *x1,
                      double *x2)
{
    double n = log((y - offset) * (sqrt(CPL_MATH_2PI * sigma * sigma) / area)) *
               (-2 * sigma * sigma);

    *x1 = x0 - sqrt(n);
    *x2 = x0 + sqrt(n);

    return CPL_ERROR_NONE;
}

static double
_moo_interpolate_x(cpl_bivector *points, int i, double val)
{
    cpl_ensure_code(points != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(i >= 0, CPL_ERROR_NULL_INPUT);

    int size = cpl_bivector_get_size(points);


    cpl_ensure_code(size > i + 1, CPL_ERROR_NULL_INPUT);

    double *y = cpl_bivector_get_y_data(points);
    double *x = cpl_bivector_get_x_data(points);

    double x1 = x[i];
    double y1 = y[i];
    double x2 = x[i + 1];
    double y2 = y[i + 1];

    return x1 + (val - y1) / (y2 - y1) * (x2 - x1);
}

/*----------------------------------------------------------------------------*/
/**
   @brief Find threshold limits of a 1D signal
   @param points  data to analyse
   @param threshold the threshold
   @param xmin  [OUT] minimum X position of the threshold
   @param xmax  [OUT] maximum X position of the threshold
   @return CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
*/
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_find_threshold_limits(cpl_bivector *points,
                          double threshold,
                          double *xmin,
                          double *xmax)
{
    cpl_ensure_code(points != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(xmin != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(xmax != NULL, CPL_ERROR_NULL_INPUT);

    int size = cpl_bivector_get_size(points);

    cpl_ensure_code(size > 1, CPL_ERROR_NULL_INPUT);

    double *y = cpl_bivector_get_y_data(points);
    int i;

    for (i = 1; i < size; i++) {
        if (y[i] > threshold) {
            *xmin = _moo_interpolate_x(points, i - 1, threshold);
            break;
        }
    }

    for (i = size - 2; i >= 0; i++) {
        if (y[i] > threshold) {
            *xmax = _moo_interpolate_x(points, i, threshold);
            break;
        }
    }
    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
   @brief Computes Tchebitchev transformation
   @param pos  vector position
   @param min  minimum
   @param max  maximum
   @return Tchebischev transformation applies on input vector
*/
/*----------------------------------------------------------------------------*/
static cpl_error_code
_moo_tchebychev_transform(cpl_vector *posv, double min, double max)
{
    int i;
    double a, b;

    cpl_ensure_code(posv != NULL, CPL_ERROR_NULL_INPUT);
    int size = cpl_vector_get_size(posv);
    cpl_ensure_code(size > 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(min < max, CPL_ERROR_ILLEGAL_INPUT);

    double *pos = cpl_vector_get_data(posv);

    a = 2 / (max - min);
    b = 1 - 2 * max / (max - min);

    for (i = 0; i < size; i++) {
        double res = pos[i] * a + b;
        if (res < -1.) {
            pos[i] = -1.;
        }
        else if (res > 1.) {
            pos[i] = 1.;
        }
        else {
            pos[i] = res;
        }
    }
    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_tchebychev_reverse_transform(cpl_vector *posv, double min, double max)
{
    double a, b;
    int i;

    cpl_ensure_code(min < max, CPL_ERROR_ILLEGAL_INPUT);

    int size = cpl_vector_get_size(posv);

    a = 2 / (max - min);
    b = 1 - 2 * max / (max - min);

    double *pos = cpl_vector_get_data(posv);

    for (i = 0; i < size; i++) {
        double res = (pos[i] - b) / a;
        pos[i] = res;
    }

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief
    Compute tchebitchev Tn(X) first coefficient for tchebitchev polynomial
  @param n
    index of the last term to be computed (0 to n)
  @param X
    value of X in evaluation of the polynomial
  @return
    the result of evaluation of the n first terms of tchebitchev polynomial
 */
/*----------------------------------------------------------------------------*/
static cpl_vector *
_moo_tchebychev_poly_eval(int n, double X)
{
    cpl_vector *result = NULL;

    cpl_ensure(n >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    result = cpl_vector_new(n + 1);

    cpl_vector_set(result, 0, 1.0);

    if (n > 0) {
        int indice = 2;
        cpl_vector_set(result, 1, X);
        while (indice <= n) {
            double T_indice = 0.0;
            double T_indice_1 = 0.0;
            double T_indice_2 = 0.0;

            T_indice_1 = cpl_vector_get(result, indice - 1);
            T_indice_2 = cpl_vector_get(result, indice - 2);
            T_indice = 2 * X * T_indice_1 - T_indice_2;
            cpl_vector_set(result, indice, T_indice);
            indice++;
        }
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Computes Tchebitchev transformation of data
   @param data   bivector containing data to fit
   @param degree degree used for polynomial fit
   @param xmin Minimum range along X axis
   @param xmax Maximum range along X axis
   @return the tchebychev polynomial

 This function fit the input data using tchebitchev transformation. The result
 replace the original values.
*/
/*----------------------------------------------------------------------------*/
moo_tcheby_polynomial *
moo_tcheby_polynomial_fit(cpl_bivector *data,
                          int degree,
                          double xmin,
                          double xmax)
{
    cpl_ensure(data != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(degree > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    int i, j;
    double chisq;
    gsl_matrix *X = NULL, *cov = NULL;
    gsl_vector *y = NULL, *c = NULL;
    gsl_multifit_linear_workspace *work = NULL;
    cpl_size pows[1];

    cpl_ensure(data != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int size = cpl_bivector_get_size(data);

    cpl_ensure(size > degree, CPL_ERROR_ILLEGAL_INPUT, NULL);

    moo_tcheby_polynomial *res = cpl_calloc(1, sizeof(moo_tcheby_polynomial));

    res->solution = cpl_polynomial_new(1);

    cpl_vector *vx = cpl_bivector_get_x(data);
    cpl_vector *vy = cpl_bivector_get_y(data);

    res->xmax = xmax;
    res->xmin = xmin;
    res->ymin = cpl_vector_get_min(vy);
    res->ymax = cpl_vector_get_max(vy);

    res->degree = degree;

    _moo_tchebychev_transform(vx, res->xmin, res->xmax);
    _moo_tchebychev_transform(vy, res->ymin, res->ymax);
    double *xdata = cpl_bivector_get_x_data(data);
    double *ydata = cpl_bivector_get_y_data(data);

    int nbcoeffs = degree + 1;

    X = gsl_matrix_alloc(size, nbcoeffs);
    y = gsl_vector_alloc(size);
    c = gsl_vector_alloc(nbcoeffs);
    cov = gsl_matrix_alloc(nbcoeffs, nbcoeffs);

    for (i = 0; i < size; i++) {
        for (j = 0; j < degree + 1; j++) {
            double Tj = cos(j * acos(xdata[i]));
            gsl_matrix_set(X, i, j, Tj);
        }
        gsl_vector_set(y, i, ydata[i]);
    }

    work = gsl_multifit_linear_alloc(size, nbcoeffs);
    gsl_multifit_linear(X, y, c, cov, &chisq, work);

    for (j = 0; j < degree + 1; j++) {
        pows[0] = j;
        cpl_polynomial_set_coeff(res->solution, pows, gsl_vector_get(c, j));
    }

    _moo_tchebychev_reverse_transform(vy, res->ymin, res->ymax);
    _moo_tchebychev_reverse_transform(vx, res->xmin, res->xmax);
    gsl_multifit_linear_free(work);
    gsl_matrix_free(X);
    gsl_vector_free(y);
    gsl_vector_free(c);
    gsl_matrix_free(cov);

    return res;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Computes Tchebitchev transformation of data
   @param in_x X data between 1 - 4096
   @param xdegree degree of 2D polynomial for X
   @param xmin the minimum x value
   @param xmax the maximum x value
   @param in_y Y data
   @param ydegree degree of 2D polynomial for Y
   @param ymin the minimum y value
   @param ymax the maximum y value
   @param in_l L data
   @param lmin the minimum l value
   @param lmax the maximum l value
   @return tchebitchev polynomial 2D solution

 This function fit the input data using tchebitchev transformation.
*/
/*----------------------------------------------------------------------------*/
moo_tcheby2d_polynomial *
moo_tcheby2d_polynomial_fit(cpl_vector *in_x,
                            int xdegree,
                            double xmin,
                            double xmax,
                            cpl_vector *in_y,
                            int ydegree,
                            double ymin,
                            double ymax,
                            cpl_vector *in_l,
                            double lmin,
                            double lmax)
{
    cpl_ensure(in_x != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(xdegree > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(in_y != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(ydegree > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(in_l != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int i, j, k;
    double chisq;
    gsl_matrix *X = NULL, *cov = NULL;
    gsl_vector *y = NULL, *c = NULL;
    gsl_multifit_linear_workspace *work = NULL;
    cpl_size pows[2];

    moo_tcheby2d_polynomial *res =
        cpl_calloc(1, sizeof(moo_tcheby2d_polynomial));

    res->solution = cpl_polynomial_new(2);

    int size = cpl_vector_get_size(in_x);

    res->xmin = xmin;
    res->xmax = xmax;
    res->ymin = ymin;
    res->ymax = ymax;
    res->lmin = lmin;
    res->lmax = lmax;
    res->degree_x = xdegree;
    res->degree_y = ydegree;

    cpl_vector *vx = cpl_vector_duplicate(in_x);
    cpl_vector *vy = cpl_vector_duplicate(in_y);
    cpl_vector *vl = cpl_vector_duplicate(in_l);

    _moo_tchebychev_transform(vx, res->xmin, res->xmax);
    _moo_tchebychev_transform(vy, res->ymin, res->ymax);
    _moo_tchebychev_transform(vl, res->lmin, res->lmax);

    double *xdata = cpl_vector_get_data(vx);
    double *ydata = cpl_vector_get_data(vy);
    double *ldata = cpl_vector_get_data(vl);

    int nbcoeffs = (xdegree + 1) * (ydegree + 1);

    X = gsl_matrix_alloc(size, nbcoeffs);
    y = gsl_vector_alloc(size);
    c = gsl_vector_alloc(nbcoeffs);
    cov = gsl_matrix_alloc(nbcoeffs, nbcoeffs);

    for (i = 0; i < size; i++) {
        for (j = 0; j < xdegree + 1; j++) {
            for (k = 0; k < ydegree + 1; k++) {
                double px = cos(j * acos(xdata[i]));
                double py = cos(k * acos(ydata[i]));
                gsl_matrix_set(X, i, k + j * (ydegree + 1), px * py);
            }
        }
        gsl_vector_set(y, i, ldata[i]);
    }
    work = gsl_multifit_linear_alloc(size, nbcoeffs);
    gsl_multifit_linear(X, y, c, cov, &chisq, work);

    for (j = 0; j < xdegree + 1; j++) {
        for (k = 0; k < ydegree + 1; k++) {
            pows[0] = k;
            pows[1] = j;
            cpl_polynomial_set_coeff(res->solution, pows,
                                     gsl_vector_get(c, k + j * (ydegree + 1)));
        }
    }

    gsl_multifit_linear_free(work);
    gsl_matrix_free(X);
    gsl_vector_free(y);
    gsl_vector_free(c);
    gsl_matrix_free(cov);

    cpl_vector_delete(vx);
    cpl_vector_delete(vy);
    cpl_vector_delete(vl);

    return res;
}

void
moo_tcheby_polynomial_delete(moo_tcheby_polynomial *self)
{
    if (self != NULL) {
        if (self->solution != NULL) {
            cpl_polynomial_delete(self->solution);
        }
        cpl_free(self);
    }
}

void
moo_tcheby2d_polynomial_delete(moo_tcheby2d_polynomial *self)
{
    if (self != NULL) {
        if (self->solution != NULL) {
            cpl_polynomial_delete(self->solution);
        }
        cpl_free(self);
    }
}

double
moo_tcheby_polynomial_eval(moo_tcheby_polynomial *self, double x)
{
    cpl_size pows[1];
    double yval = 0;

    cpl_vector *vx = cpl_vector_new(1);
    cpl_vector *vy = cpl_vector_new(1);

    cpl_vector_set(vx, 0, x);

    _moo_tchebychev_transform(vx, self->xmin, self->xmax);
    double tx = cpl_vector_get(vx, 0);

    cpl_vector *tcheb = _moo_tchebychev_poly_eval(self->degree, tx);
    for (int j = 0; j < self->degree + 1; j++) {
        pows[0] = j;
        double tval = cpl_vector_get(tcheb, j);
        double coef = cpl_polynomial_get_coeff(self->solution, pows);
        yval += coef * tval;
    }
    cpl_vector_set(vy, 0, yval);

    _moo_tchebychev_reverse_transform(vy, self->ymin, self->ymax);

    yval = cpl_vector_get(vy, 0);

    cpl_vector_delete(tcheb);
    cpl_vector_delete(vx);
    cpl_vector_delete(vy);

    return yval;
}

double
moo_tcheby2d_polynomial_eval(moo_tcheby2d_polynomial *self, double x, double y)
{
    cpl_size pows[2];
    double yval = 0;

    cpl_vector *vx = cpl_vector_new(1);
    cpl_vector *vy = cpl_vector_new(1);
    cpl_vector *vl = cpl_vector_new(1);

    cpl_vector_set(vx, 0, x);
    cpl_vector_set(vy, 0, y);

    _moo_tchebychev_transform(vx, self->xmin, self->xmax);
    _moo_tchebychev_transform(vy, self->ymin, self->ymax);
    double tx = cpl_vector_get(vx, 0);
    double ty = cpl_vector_get(vy, 0);

    cpl_vector *tchebx = _moo_tchebychev_poly_eval(self->degree_x, tx);
    cpl_vector *tcheby = _moo_tchebychev_poly_eval(self->degree_y, ty);

    for (int j = 0; j < self->degree_x + 1; j++) {
        for (int k = 0; k < self->degree_y + 1; k++) {
            pows[0] = k;
            pows[1] = j;
            double tvalx = cpl_vector_get(tchebx, j);
            double tvaly = cpl_vector_get(tcheby, k);
            double coef = cpl_polynomial_get_coeff(self->solution, pows);
            yval += coef * tvalx * tvaly;
        }
    }

    cpl_vector_set(vl, 0, yval);

    _moo_tchebychev_reverse_transform(vl, self->lmin, self->lmax);

    yval = cpl_vector_get(vl, 0);

    cpl_vector_delete(tchebx);
    cpl_vector_delete(tcheby);
    cpl_vector_delete(vx);
    cpl_vector_delete(vy);
    cpl_vector_delete(vl);
    return yval;
}
/*----------------------------------------------------------------------------*/
/**
   @brief Computes Tchebitchev transformation of data
   @param data   bivector containing data to fit
   @param flag   array of flag (same size of data)
   @param degree degree used for polynomial fit
   @param xmin   minimum value used in x part of data
   @param xmax   maximum value used in x part of data
   @param ymin   minmum value used in y part of data
   @param ymax   maximum value used in y part of data
   @return CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

 This function fit the input data using tchebitchev transformation. The result
 replace the original values.
*/
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_tchebychev_fit(cpl_bivector *data,
                   int *flag,
                   int degree,
                   double xmin,
                   double xmax,
                   double ymin,
                   double ymax)
{
    cpl_ensure_code(data != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(flag != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(degree > 0, CPL_ERROR_ILLEGAL_INPUT);
    int i, j;
    double chisq;
    gsl_matrix *X = NULL, *cov = NULL;
    gsl_vector *y = NULL, *c = NULL;
    gsl_multifit_linear_workspace *work = NULL;
    cpl_size pows[1];
    cpl_polynomial *result = cpl_polynomial_new(1);

    cpl_vector *vx = cpl_bivector_get_x(data);
    cpl_vector *vy = cpl_bivector_get_y(data);

    _moo_tchebychev_transform(vx, xmin, xmax);
    _moo_tchebychev_transform(vy, ymin, ymax);


    double *xdata = cpl_bivector_get_x_data(data);
    double *ydata = cpl_bivector_get_y_data(data);

    int nbcoeffs = degree + 1;

    int nb_valid = 0;
    for (i = xmin - 1; i < xmax; i++) {
        if (flag[i] == 0) {
            nb_valid++;
        }
    }

    X = gsl_matrix_alloc(nb_valid, nbcoeffs);
    y = gsl_vector_alloc(nb_valid);
    c = gsl_vector_alloc(nbcoeffs);
    cov = gsl_matrix_alloc(nbcoeffs, nbcoeffs);

    int id = 0;
    for (i = xmin - 1; i < xmax; i++) {
        if (flag[i] == 0) {
            for (j = 0; j < degree + 1; j++) {
                double Tj = cos(j * acos(xdata[i]));
                gsl_matrix_set(X, id, j, Tj);
            }
            gsl_vector_set(y, id, ydata[i]);
            id++;
        }
    }

    work = gsl_multifit_linear_alloc(nb_valid, nbcoeffs);
    gsl_multifit_linear(X, y, c, cov, &chisq, work);

    for (j = 0; j < degree + 1; j++) {
        pows[0] = j;
        cpl_polynomial_set_coeff(result, pows, gsl_vector_get(c, j));
    }

    for (i = xmin - 1; i < xmax; i++) {
        double yval = 0;
        double xval = xdata[i];
        cpl_vector *tcheb = _moo_tchebychev_poly_eval(degree, xval);
        for (j = 0; j < degree + 1; j++) {
            pows[0] = j;
            double tval = cpl_vector_get(tcheb, j);
            double coef = cpl_polynomial_get_coeff(result, pows);
            yval += coef * tval;
            ydata[i] = yval;
        }
        cpl_vector_delete(tcheb);
    }

    _moo_tchebychev_reverse_transform(vx, xmin, xmax);
    _moo_tchebychev_reverse_transform(vy, ymin, ymax);

    gsl_multifit_linear_free(work);
    gsl_matrix_free(X);
    gsl_vector_free(y);
    gsl_vector_free(c);
    gsl_matrix_free(cov);
    cpl_polynomial_delete(result);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Find minimum values in a vector using flags
   @param v the vector to use
   @param flags array of flag (same size of v)
   @return the minimum value of the vector
*/
/*----------------------------------------------------------------------------*/
double
moo_vector_get_min(const cpl_vector *v, int *flags)
{
    int size = cpl_vector_get_size(v);

    double res = DBL_MAX;

    for (int i = 0; i < size; i++) {
        if (flags[i] == 0) {
            double val = cpl_vector_get(v, i);
            res = CPL_MIN(res, val);
        }
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Find maximum values in a vector using flags
   @param v the vector to use
   @param flags array of flags (same size of v)
   @return the maximum value of the vector
*/
/*----------------------------------------------------------------------------*/
double
moo_vector_get_max(const cpl_vector *v, int *flags)
{
    int size = cpl_vector_get_size(v);

    double res = DBL_MIN;


    for (int i = 0; i < size; i++) {
        if (flags[i] == 0) {
            double val = cpl_vector_get(v, i);
            res = CPL_MAX(res, val);
        }
    }
    return res;
}

cpl_vector *
moo_hpfilter(cpl_vector *Yv, double S)
{
    int N = cpl_vector_get_size(Yv);
    double *Y = cpl_vector_get_data(Yv);

    cpl_vector *Tv = cpl_vector_new(N);
    double *T = cpl_vector_get_data(Tv);
    double **V = cpl_calloc(N, sizeof(double *));

    for (int i = 0; i < N; i++) {
        V[i] = cpl_calloc(3, sizeof(double));
    }
    double *D = cpl_calloc(N, sizeof(double));
    double SS = 0.0, M1 = 0.0, M2 = 0.0, V11 = 0.0, V12 = 0.0, V22 = 0.0,
           X = 0.0, Z = 0.0, B11 = 0.0, B12 = 0.0, B22 = 0.0, DET = 0.0,
           E1 = 0.0, E2 = 0.0;
    int NN = 0, I1 = 0, IB = 0;

    if (NN != N || S != SS) {
        SS = S;
        NN = N;
        V11 = 1;
        V22 = 1;
        V12 = 0;
        for (int i = 2; i <= N - 1; i++) {
            X = V11;
            Z = V12;
            V11 = 1.0 / S + 4.0 * (X - Z) + V22;
            V12 = 2.0 * X - Z;
            V22 = X;
            DET = V11 * V22 - V12 * V12;
            V[i][0] = V22 / DET;
            V[i][1] = -V12 / DET;
            V[i][2] = V11 / DET;
            X = V11 + 1;
            Z = V11;
            V11 = V11 - V11 * V11 / X;
            V22 = V22 - V12 * V12 / X;
            V12 = V12 - Z * V12 / X;
        }
    }

    // this is the forward pass
    M1 = Y[1];
    M2 = Y[0];

    for (int i = 2; i <= N - 1; i++) {
        X = M1;
        M1 = 2.0 * M1 - M2;
        M2 = X;
        T[i - 1] = V[i][0] * M1 + V[i][1] * M2;
        D[i - 1] = V[i][1] * M1 + V[i][2] * M2;
        DET = V[i][0] * V[i][2] - V[i][1] * V[i][1];
        V11 = V[i][2] / DET;
        V12 = -V[i][1] / DET;
        Z = (Y[i] - M1) / (V11 + 1);
        M1 = M1 + V11 * Z;
        M2 = M2 + V12 * Z;
    }
    T[N - 1] = M1;
    T[N - 2] = M2;

    //this is the backward pass
    M1 = Y[N - 2];
    M2 = Y[N - 1];

    for (int i = N - 3; i >= 0; i--) {
        I1 = i + 1;
        IB = (N - i) - 1;
        X = M1;
        M1 = 2 * M1 - M2;
        M2 = X;
        // combine info for y(.lt.i) with info for y(.ge.i)
        if (i > 2) {
            E1 = V[IB][2] * M2 + V[IB][1] * M1 + T[i];
            E2 = V[IB][1] * M2 + V[IB][0] * M1 + D[i];
            B11 = V[IB][2] + V[I1][0];
            B12 = V[IB][1] + V[I1][1];
            B22 = V[IB][0] + V[I1][2];
            DET = B11 * B22 - B12 * B12;
            T[i] = (-B12 * E1 + B11 * E2) / DET;
        }
        //  end of combining
        DET = V[IB][0] * V[IB][2] - V[IB][1] * V[IB][1];
        V11 = V[IB][2] / DET;
        V12 = -V[IB][1] / DET;
        Z = (Y[i] - M1) / (V11 + 1.);
        M1 = M1 + V11 * Z;
        M2 = M2 + V12 * Z;
    }

    T[0] = M1;
    T[1] = M2;

    for (int i = 0; i < N; i++) {
        cpl_free(V[i]);
    }
    cpl_free(V);
    cpl_free(D);

    return Tv;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Linear interpolation of a 1d-function
  @param    fout   Preallocated with X-vector set, to hold interpolation in Y
  @param    fout_errs   Preallocated to hold interpolation in Y errs
  @param    fout_qual   Preallocated to hold interpolation in Y qual
  @param    fref   Reference 1d-function
  @param    fref_errs   Reference 1d-function errs
  @param    fref_qual   Reference 1d-function qual
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_

  fref must have both its abscissa and ordinate defined.
  fout must have its abscissa defined and its ordinate allocated.

  The linear interpolation will be done from the values in fref to the abscissa
  points in fout.

  For each abscissa point in fout, fref must either have two neigboring abscissa
  points such that xref_i < xout_j < xref{i+1}, or a single identical abscissa
  point, such that xref_i == xout_j.

  This is ensured by monotonely growing abscissa points in both fout and fref
  (and by min(xref) <= min(xout) and max(xout) < max(xref)).

  However, for efficiency reasons (since fref can be very long) the monotonicity
  is only verified to the extent necessary to actually perform the interpolation.

  This input requirement implies that extrapolation is not allowed.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND if fout has an endpoint which is out of range
  - CPL_ERROR_ILLEGAL_INPUT if the monotonicity requirement on the 2 input
    abscissa vectors is not met.
 */
/*----------------------------------------------------------------------------*/

cpl_error_code
moo_interpolate_linear(cpl_bivector *fout,
                       cpl_vector *fout_errs,
                       int *fout_qual,
                       const cpl_bivector *fref,
                       const cpl_vector *fref_errs,
                       const int *fref_qual)
{
    const cpl_size m = cpl_bivector_get_size(fref);
    const cpl_size n = cpl_bivector_get_size(fout);
    const double *xref = cpl_bivector_get_x_data_const(fref);
    const double *yref = cpl_bivector_get_y_data_const(fref);
    const double *ref_errs = cpl_vector_get_data_const(fref_errs);
    double *xout = cpl_bivector_get_x_data(fout);
    double *yout = cpl_bivector_get_y_data(fout);
    double *out_errs = cpl_vector_get_data(fout_errs);

    int qual = 0;
    double y0_err = DBL_MAX;
    double grad = DBL_MAX;     /* Avoid (false) uninit warning */
    double grad_err = DBL_MAX; /* Avoid (false) uninit warning */
    double y_0 = DBL_MAX;      /* Avoid (false) uninit warning */
    cpl_size ibelow, iabove;
    cpl_size i;


    cpl_ensure_code(xref != NULL, cpl_error_get_code());
    cpl_ensure_code(yref != NULL, cpl_error_get_code());
    cpl_ensure_code(ref_errs != NULL, cpl_error_get_code());
    cpl_ensure_code(xout != NULL, cpl_error_get_code());
    cpl_ensure_code(yout != NULL, cpl_error_get_code());
    cpl_ensure_code(out_errs != NULL, cpl_error_get_code());

    /* Upper extrapolation not allowed */
    cpl_ensure_code(xout[n - 1] <= xref[m - 1], CPL_ERROR_DATA_NOT_FOUND);

    /* Start interpolation from below */
    ibelow = cpl_vector_find(cpl_bivector_get_x_const(fref), xout[0]);

    if (xout[0] < xref[ibelow]) {
        /* Lower extrapolation also not allowed */
        cpl_ensure_code(ibelow > 0, CPL_ERROR_DATA_NOT_FOUND);
        ibelow--;
    }

    iabove = ibelow; /* Ensure grad initialization, for 1st interpolation */

    /* assert( xref[iabove] <= xout[0] ); */
    for (i = 0; i < n; i++) {
        /* When possible reuse reference function abscissa points */
        if (xref[iabove] < xout[i]) {
            /* No, need new points */
            while (xref[++iabove] < xout[i])
                ;
            ibelow = iabove - 1;

            /* Verify that the pair of reference abscissa points are valid */
            if (xref[iabove] <= xref[ibelow])
                break;

            qual = fref_qual[ibelow] | fref_qual[iabove];

            grad =
                (yref[iabove] - yref[ibelow]) / (xref[iabove] - xref[ibelow]);
            y_0 = yref[ibelow] - grad * xref[ibelow];

            grad_err = (ref_errs[iabove] * ref_errs[iabove] -
                        ref_errs[ibelow] * ref_errs[ibelow]) /
                       (xref[iabove] - xref[ibelow]);
            y0_err =
                ref_errs[ibelow] * ref_errs[ibelow] - grad_err * xref[ibelow];
        }

        if (xref[iabove] > xout[i]) {
            if (xref[ibelow] < xout[i]) {
                fout_qual[i] = qual;
                out_errs[i] = sqrt(y0_err + grad_err * xout[i]);
                yout[i] = y_0 + grad * xout[i];
            }
            else {
                /* assert( xout[i] == xref[ibelow] ); */
                yout[i] = yref[ibelow];
                out_errs[i] = ref_errs[ibelow];
                fout_qual[i] = fref_qual[ibelow];
            }
        }
        else {
            yout[i] = yref[iabove];
            out_errs[i] = ref_errs[iabove];
            fout_qual[i] = fref_qual[iabove];
        }
    }

    return i == n ? CPL_ERROR_NONE : CPL_ERROR_ILLEGAL_INPUT;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get percentile of input vector
  @param    v input vector
  @param    f Fraction between 0 and 1
  @return   the quantile value
*/
double
moo_vector_get_percentile(cpl_vector *v, double f)
{
    double res = NAN;

    cpl_ensure(v != NULL, CPL_ERROR_NULL_INPUT, res);
    cpl_ensure(f >= 0 && f <= 1, CPL_ERROR_ILLEGAL_INPUT, res);

    int size = cpl_vector_get_size(v);
    int nbnan = 0;

    for (int i = 0; i < size; i++) {
        double val = cpl_vector_get(v, i);
        if (isnan(val)) {
            nbnan++;
        }
    }
    cpl_vector *temp = cpl_vector_new(size - nbnan);
    int idx = 0;
    for (int i = 0; i < size; i++) {
        double val = cpl_vector_get(v, i);
        if (!isnan(val)) {
            cpl_vector_set(temp, idx, val);
            idx++;
        }
    }

    cpl_vector_sort(temp, CPL_SORT_ASCENDING);

    double *data = cpl_vector_get_data(temp);
    int dsize = cpl_vector_get_size(temp);

    res = gsl_stats_quantile_from_sorted_data(data, 1, dsize, f);

    cpl_vector_delete(temp);

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create new vector with nan values filter
  @param    v input vector
  @return   new vector with no nan values
*/

cpl_vector *
moo_vector_filter_nan(cpl_vector *v)
{
    cpl_vector *res = NULL;
    cpl_ensure(v != NULL, CPL_ERROR_NULL_INPUT, res);
    int size = cpl_vector_get_size(v);

    int nbnan = 0;
    for (int i = 0; i < size; i++) {
        double val = cpl_vector_get(v, i);
        if (isnan(val)) {
            nbnan++;
        }
    }
    int nsize = size - nbnan;
    if (nsize > 0) {
        res = cpl_vector_new(nsize);
        int j = 0;
        for (int i = 0; i < size; i++) {
            double val = cpl_vector_get(v, i);
            if (!isnan(val)) {
                cpl_vector_set(res, j, val);
                j++;
            }
        }
    }
    return res;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Create new bi vector with nan values filter
  @param    v input bivector
  @return   new vector with no nan values
*/
cpl_bivector *
moo_bivector_filter_nan(cpl_bivector *v)
{
    cpl_bivector *res = NULL;
    cpl_ensure(v != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int size = cpl_bivector_get_size(v);
    cpl_vector *y = cpl_bivector_get_y(v);
    cpl_vector *x = cpl_bivector_get_x(v);
    int nbnan = 0;
    for (int i = 0; i < size; i++) {
        double xval = cpl_vector_get(x, i);
        double yval = cpl_vector_get(y, i);
        if (isnan(xval) || isnan(yval)) {
            nbnan++;
        }
    }
    int nsize = size - nbnan;
    if (nsize > 0) {
        res = cpl_bivector_new(nsize);
        cpl_vector *ry = cpl_bivector_get_y(res);
        cpl_vector *rx = cpl_bivector_get_x(res);

        int j = 0;
        for (int i = 0; i < size; i++) {
            double xval = cpl_vector_get(x, i);
            double yval = cpl_vector_get(y, i);
            if (!isnan(xval) && !isnan(yval)) {
                cpl_vector_set(ry, j, xval);
                cpl_vector_set(rx, j, yval);
                j++;
            }
        }
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Collapse row of an image using a median and compute associate error
  @param    image input image
  @return   image of median
*/
hdrl_image *
moo_image_collapse_median_create(hdrl_image *image)
{
    cpl_ensure(image != NULL, CPL_ERROR_NULL_INPUT, NULL);

    hdrl_image *res = NULL;

    int nx = hdrl_image_get_size_x(image);
    int ny = hdrl_image_get_size_y(image);

    res = hdrl_image_new(nx, 1);

    for (int i = 1; i <= nx; i++) {
        hdrl_image *col = hdrl_image_extract(image, i, 1, i, ny);

        hdrl_value median = hdrl_image_get_median(col);
        hdrl_image_set_pixel(res, i, 1, median);
        hdrl_image_delete(col);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Compute sky distance (in rad)
  @param    alpha1 alpha corrdinates of position 1 in degree
  @param    delta1 delta corrdinates of position 1 in degree
  @param    alpha2 alpha corrdinates of position 2 in degree
  @param    delta2 delta corrdinates of position 2 in degree
  @return   result in rad
*/
double
moo_sky_distance(double alpha1, double delta1, double alpha2, double delta2)
{
    double res = 0;

    double alpha1_rad = CPL_MATH_RAD_DEG * alpha1;
    double delta1_rad = CPL_MATH_RAD_DEG * delta1;
    double alpha2_rad = CPL_MATH_RAD_DEG * alpha2;
    double delta2_rad = CPL_MATH_RAD_DEG * delta2;

    res =
        acos(sin(delta1_rad) * sin(delta2_rad) +
             cos(delta1_rad) * cos(delta2_rad) * cos(alpha1_rad - alpha2_rad));
    return res;
}

moo_spline *
moo_spline_create(cpl_bivector *data)
{
    cpl_ensure(data != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int size = cpl_bivector_get_size(data);
    double *x = cpl_bivector_get_x_data(data);
    double *y = cpl_bivector_get_y_data(data);

    moo_spline *res = cpl_calloc(1, sizeof(moo_spline));
    res->min = x[0];
    res->max = x[size - 1];
    res->acc = gsl_interp_accel_alloc();
    res->spline = gsl_spline_alloc(gsl_interp_cspline, size);
    gsl_spline_init(res->spline, x, y, size);

    return res;
}
double
moo_spline_eval(moo_spline *self, double x)
{
    double res = NAN;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, 0.0);
    if (x >= self->min && x <= self->max) {
        res = gsl_spline_eval(self->spline, x, self->acc);
    }
    return res;
}

void
moo_spline_delete(moo_spline *self)
{
    if (self != NULL) {
        gsl_interp_accel_free(self->acc);
        gsl_spline_free(self->spline);
        cpl_free(self);
    }
}


static int
_moo_gsl_fit_wmul(const double *x,
                  const double *w,
                  const double *y,
                  const double *ey,
                  const size_t n,
                  double *c1,
                  double *sig_c1,
                  double *cov_11,
                  double *chisq)
{
    /* compute the weighted means and weighted deviations from the means */

    /* wm denotes a "weighted mean", wm(f) = (sum_i w_i f_i) / (sum_i w_i) */

    double W = 0, wm_x = 0, wm_y = 0, wm_dx2 = 0, wm_dxdy = 0;
    double sig_wm_y2 = 0, sig_wm_dxdy2 = 0; /* FRED DEBUG */

    /* ey[i] denotes l'erreur sur y - FRED DEBUG */

    size_t i;

    for (i = 0; i < n; i++) {
        const double wi = w[i];

        if (wi > 0) {
            W += wi;
            wm_x += (x[i] - wm_x) * (wi / W);
            wm_y += (y[i] - wm_y) * (wi / W);
            sig_wm_y2 += (wi * wi * ey[i] * ey[i]); /* FRED DEBUG */
        }
    }

    W = 0; /* reset the total weight */

    for (i = 0; i < n; i++) {
        const double wi = w[i];

        if (wi > 0) {
            const double dx = x[i] - wm_x;
            const double dy = y[i] - wm_y;

            W += wi;
            wm_dx2 += (dx * dx - wm_dx2) * (wi / W);
            wm_dxdy += (dx * dy - wm_dxdy) * (wi / W);
            sig_wm_dxdy2 += wi * wi * dx * dx *
                            (ey[i] * ey[i] + sig_wm_y2); /* FRED DEBUG */
        }
    }

    /* In terms of y = b x */

    {
        double d2 = 0;
        double b = (wm_x * wm_y + wm_dxdy) / (wm_x * wm_x + wm_dx2);

        double sig_b = sqrt(wm_x * wm_x * sig_wm_y2 + sig_wm_dxdy2) /
                       (wm_x * wm_x + wm_dx2); /* FRED DEBUG */

        *c1 = b;
        *sig_c1 = sig_b;
        *cov_11 = 1 / (W * (wm_x * wm_x + wm_dx2));

        /* Compute chi^2 = \sum w_i (y_i - b * x_i)^2 */

        for (i = 0; i < n; i++) {
            const double wi = w[i];

            if (wi > 0) {
                const double dx = x[i] - wm_x;
                const double dy = y[i] - wm_y;
                const double d = (wm_y - b * wm_x) + (dy - b * dx);
                d2 += wi * d * d;
            }
        }

        *chisq = d2;
    }

    return GSL_SUCCESS;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    This function computes the best-fit linear regression coefficient c1
    of the model Y = c_1 X for the weighted datasets (x, y)
  @param    vx
  @param    vw
  @param    vy
  @param    vy_err
  @param    c fit result
  @param    sig_c error on fit
  @return cpl_error_code
*/
cpl_error_code
moo_fit_mul(const cpl_vector *vx,
            const cpl_vector *vw,
            const cpl_vector *vy,
            const cpl_vector *vy_err,
            double *c,
            double *sig_c)
{
    cpl_ensure_code(vx != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(vy != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(vw != NULL, CPL_ERROR_NULL_INPUT);

    int n = cpl_vector_get_size(vx);

    const double *x = cpl_vector_get_data_const(vx);
    const double *y = cpl_vector_get_data_const(vy);
    const double *yerr = cpl_vector_get_data_const(vy_err);
    const double *w = cpl_vector_get_data_const(vw);

    double cov11, chisq;

    _moo_gsl_fit_wmul(x, w, y, yerr, n, c, sig_c, &cov11, &chisq);


    return CPL_ERROR_NONE;
}

static cpl_polynomial *
_moo_fit(cpl_vector *v, int polyorder)
{
    gsl_matrix *X = NULL, *cov = NULL;
    gsl_vector *y = NULL, *c = NULL;
    gsl_multifit_linear_workspace *work = NULL;
    cpl_size pows[1];

    cpl_ensure(v != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_polynomial *pol = cpl_polynomial_new(1);
    int i, j;
    double chisq;
    double *data = cpl_vector_get_data(v);
    int size = cpl_vector_get_size(v);

    int nbcoeffs = polyorder + 1;
    X = gsl_matrix_alloc(size, nbcoeffs);
    y = gsl_vector_alloc(size);
    c = gsl_vector_alloc(nbcoeffs);
    cov = gsl_matrix_alloc(nbcoeffs, nbcoeffs);

    int nbnan = 0;
    for (i = 0; i < size; i++) {
        double d = data[i];
        if (!isnan(d)) {
            for (j = 0; j < polyorder + 1; j++) {
                gsl_matrix_set(X, i, j, pow(i, j));
            }
            gsl_vector_set(y, i, d);
        }
        else {
            nbnan++;
        }
    }

    if ((size - nbnan) > (nbcoeffs + 1)) {
        work = gsl_multifit_linear_alloc(size, nbcoeffs);
        gsl_multifit_linear(X, y, c, cov, &chisq, work);

        for (j = 0; j < polyorder + 1; j++) {
            pows[0] = j;
            cpl_polynomial_set_coeff(pol, pows, gsl_vector_get(c, j));
        }
        gsl_multifit_linear_free(work);
    }
    gsl_matrix_free(X);
    gsl_vector_free(y);
    gsl_vector_free(c);
    gsl_matrix_free(cov);
    return pol;
}
/*----------------------------------------------------------------------------*/
/**
  @brief Apply a Savitzky-Golay filter to a vector.
  @param    v input vector
  @param    window_length The length of the filter window
   (i.e. the number of coefficients). window_length must be a positive odd integer.
  @param    polyorder The order of the polynomial used to fit the samples. polyorder
   must be less than window_length.
  @return the filtered vector
*/
cpl_vector *
moo_savgol_filter(cpl_vector *v, int window_length, int polyorder)
{
    cpl_ensure(v != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(polyorder >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(polyorder < window_length, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(window_length >= 1, CPL_ERROR_ILLEGAL_INPUT, NULL);

    int size = cpl_vector_get_size(v);
    cpl_vector *result = cpl_vector_new(size);
    double *resdata = cpl_vector_get_data(result);

    cpl_polynomial *fit = NULL;
    cpl_vector *data = NULL;
    cpl_size pows[1];

    int hsize = (window_length - 1) / 2;
    /* first values */
    data = cpl_vector_extract(v, 0, window_length - 1, 1);

    fit = _moo_fit(data, polyorder);

    for (int i = 0; i < hsize; i++) {
        double yval = 0;
        for (int j = 0; j < polyorder + 1; j++) {
            pows[0] = j;
            double coef = cpl_polynomial_get_coeff(fit, pows);
            yval += coef * pow(i, j);
        }
        resdata[i] = yval;
    }
    cpl_vector_delete(data);
    cpl_polynomial_delete(fit);
    /* end values */
    data = cpl_vector_extract(v, size - window_length, size - 1, 1);
    fit = _moo_fit(data, polyorder);

    for (int i = hsize; i < window_length; i++) {
        double yval = 0;
        for (int j = 0; j < polyorder + 1; j++) {
            pows[0] = j;
            double coef = cpl_polynomial_get_coeff(fit, pows);
            yval += coef * pow(i, j);
        }
        resdata[size - window_length + i] = yval;
    }
    cpl_vector_delete(data);
    cpl_polynomial_delete(fit);

    /* loop */
    for (int i = hsize; i < size - hsize; i++) {
        data =
            cpl_vector_extract(v, i - hsize, i - hsize - 1 + window_length, 1);
        fit = _moo_fit(data, polyorder);
        double yval = 0;
        for (int j = 0; j < polyorder + 1; j++) {
            pows[0] = j;
            double coef = cpl_polynomial_get_coeff(fit, pows);
            yval += coef * pow(hsize, j);
        }
        cpl_vector_delete(data);
        cpl_polynomial_delete(fit);
        resdata[i] = yval;
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Apply a median filter to a vector.
  @param    v input vector
  @param    winhsize window half size
  @return the filtered vector
*/
cpl_vector *
moo_median_filter(cpl_vector *v, int winhsize)
{
    cpl_ensure(v != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(winhsize >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    cpl_vector *result = NULL;
    int size = cpl_vector_get_size(v);
    int K = winhsize * 2 + 1;

    result = cpl_vector_new(size);

    double *windata = cpl_calloc(K, sizeof(double));
    /* generate input signal */
    for (int i = 0; i < size; ++i) {
        int idx = 0;

        int kmin = i - winhsize;
        if (kmin < 0) {
            kmin = 0;
        }

        int kmax = i + winhsize;
        if (kmax >= size) {
            kmax = size - 1;
        }
        for (int k = kmin; k <= kmax; k++) {
            double val = cpl_vector_get(v, k);
            if (!isnan(val)) {
                windata[idx] = val;
                idx++;
            }
        }
        if (idx > 0) {
            cpl_vector *winv = cpl_vector_wrap(idx, windata);

            double med = cpl_vector_get_median(winv);
            cpl_vector_set(result, i, med);
            cpl_vector_unwrap(winv);
        }
        else {
            cpl_vector_set(result, i, NAN);
        }
    }

    cpl_free(windata);

    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief get the QUAL resulting in a bitwise OR operation on the QUAL list
  @param    list image list
  @param    datalist the image data list
  @return the QUAL collapse with bitwise OR
*
* Possible error code :
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL or list size == 0
*
*/
cpl_image *
moo_imagelist_collapse_bitwiseor(cpl_imagelist *list, hdrl_imagelist *datalist)
{
    cpl_image *result = NULL;
    int size = 0;

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(datalist != NULL, CPL_ERROR_NULL_INPUT, NULL);
    size = cpl_imagelist_get_size(list);
    cpl_ensure(size > 0, CPL_ERROR_NULL_INPUT, NULL);

    cpl_image *first = cpl_imagelist_get(list, 0);
    int nx = cpl_image_get_size_x(first);
    int ny = cpl_image_get_size_y(first);

    result = cpl_image_new(nx, ny, CPL_TYPE_INT);

    for (int y = 1; y <= ny; y++) {
        for (int x = 1; x <= nx; x++) {
            int v = 0;
            int av = 0;
            int nbcomb = 0;
            for (int k = 0; k < size; k++) {
                cpl_image *qual = cpl_imagelist_get(list, k);
                hdrl_image *data = hdrl_imagelist_get(datalist, k);
                int rej;
                int qv = cpl_image_get(qual, x, y, &rej);
                hdrl_image_get_pixel(data, x, y, &rej);
                av |= qv;
                if (rej == 0) {
                    v |= qv;
                    nbcomb++;
                }
            }
            if (nbcomb == 0) {
                v = av;
            }
            cpl_image_set(result, x, y, v);
        }
    }


    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_image_delete(result);
        result = NULL;
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief compute ron in a diff image using boxes
  @param diff input diff image
  @param llx lower left x coord
  @param lly lower left y coord
  @param urx upper right x coord
  @param ury upper right y coord
  @param nb_boxes number of boxes
  @param box_hsize half size of a box
  @param max_error_frac max error fraction on ron
  @param max_niter max number of iteration to find ron
  @return the ron value
*
* Possible error code :
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL or list size == 0
*
*/
double
moo_image_get_ron(cpl_image *diff,
                  int llx,
                  int lly,
                  int urx,
                  int ury,
                  int nb_boxes,
                  int box_hsize,
                  double max_error_frac,
                  int max_niter)
{
    double ron = 0.0;
    double ronerr = 0.0;
    int niter = 0;

    cpl_ensure(diff, CPL_ERROR_NULL_INPUT, 0.0);
    cpl_size *window = (cpl_size *)cpl_calloc(sizeof(cpl_size), 4);
    window[0] += llx;
    window[1] += urx;
    window[2] += lly;
    window[3] += ury;

    do {
        cpl_flux_get_noise_window(diff, window, box_hsize, nb_boxes, &ron,
                                  &ronerr);
        niter++;
    } while (ronerr > max_error_frac * ron && niter < max_niter);

    cpl_free(window);
    return ron;
}

/*----------------------------------------------------------------------------*/
/**
  @brief compute first and last quartile from an image
  @param image input image
  @param qmin the first quartile
  @param qmax the last quartile

* Possible error code :
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL or list size == 0
*
*/
cpl_error_code
moo_image_get_quartile(cpl_image *image, double *qmin, double *qmax)
{
    cpl_ensure_code(image != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qmin != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qmax != NULL, CPL_ERROR_NULL_INPUT);

    int nx = cpl_image_get_size_x(image);
    int ny = cpl_image_get_size_y(image);
    int size = nx * ny;
    cpl_image *stats = cpl_image_cast(image, CPL_TYPE_DOUBLE);
    cpl_image_reject_value(stats, CPL_VALUE_NAN);
    int nb = cpl_image_count_rejected(stats);
    int nsize = size - nb;
    cpl_vector *statsv = cpl_vector_new(nsize);
    double *data = cpl_image_get_data_double(stats);
    int nbnan = 0;
    for (int i = 0; i < size; i++) {
        double v = data[i];
        if (isnan(v)) {
            nbnan++;
        }
        else {
            cpl_vector_set(statsv, i - nbnan, v);
        }
    }
    double *stats_data = cpl_vector_get_data(statsv);
    gsl_sort(stats_data, 1, nsize);
    double upperq =
        gsl_stats_quantile_from_sorted_data(stats_data, 1, nsize, 0.8415);
    double lowerq =
        gsl_stats_quantile_from_sorted_data(stats_data, 1, nsize, 0.1585);
    *qmin = lowerq;
    *qmax = upperq;

    cpl_vector_delete(statsv);
    cpl_image_delete(stats);
    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
#define MACHEP DBL_EPSILON
#define MAXLOG log(FLT_MAX)
#define mtherr(x, y)

static double igam(double, double);
static double igamc(double, double);

static double big = 4.503599627370496e15;
static double biginv = 2.22044604925031308085e-16;

static double
igam(double a, double x)
{
    double ans, ax, c, r;

    /* Check zero integration limit first */
    if (x == 0) {
        return (0.);
    }

    if ((x < 0) || (a <= 0)) {
        mtherr("igam", DOMAIN);
        return (NAN);
    }

    if ((x > 1.) && (x > a)) {
        return (1. - igamc(a, x));
    }

    /* Compute  x**a * exp(-x) / gamma(a)  */
    ax = a * log(x) - x - lgamma(a);
    if (ax < -MAXLOG) {
        mtherr("igam", UNDERFLOW);
        return (0.);
    }
    ax = exp(ax);

    /* power series */
    r = a;
    c = 1.;
    ans = 1.;

    do {
        r += 1.;
        c *= x / r;
        ans += c;

    } while (c / ans > MACHEP);

    return (ans * ax / a);
}

static double
igamc(double a, double x)
{
    double ans, ax, c, r, t, y, z;
    double pkm1, pkm2, qkm1, qkm2;

    if ((x < 0) || (a <= 0)) {
        mtherr("igamc", DOMAIN);
        return (NAN);
    }

    if ((x < 1.0) || (x < a)) {
        return (1. - igam(a, x));
    }
    ax = a * log(x) - x - lgamma(a);

    if (ax < -MAXLOG) {
        mtherr("igamc", UNDERFLOW);
        return (0.);
    }
    ax = exp(ax);

    /* continued fraction */
    y = 1. - a;
    z = x + y + 1.;
    c = 0.;
    pkm2 = 1.;
    qkm2 = x;
    pkm1 = x + 1.;
    qkm1 = z * x;
    ans = pkm1 / qkm1;

    do {
        c += 1.;
        y += 1.;
        z += 2.;

        double yc = y * c;
        double pk = pkm1 * z - pkm2 * yc;
        double qk = qkm1 * z - qkm2 * yc;

        if (qk != 0) {
            r = pk / qk;
            t = fabs((ans - r) / r);
            ans = r;
        }
        else {
            t = 1.;
        }

        pkm2 = pkm1;
        pkm1 = pk;
        qkm2 = qkm1;
        qkm1 = qk;

        if (fabs(pk) > big) {
            pkm2 *= biginv;
            pkm1 *= biginv;
            qkm2 *= biginv;
            qkm1 *= biginv;
        }

    } while (t > MACHEP);

    return (ans * ax);
}

static cpl_image *
bpm_from_rel(cpl_image *img, double kappa_low, double kappa_high, int mad)
{
    cpl_image *r;
    double m, s;
    if (mad) {
        m = cpl_image_get_mad(img, &s);
        s *= CPL_MATH_STD_MAD;
        s = CX_MAX(DBL_EPSILON, s);
    }
    else {
        m = cpl_image_get_mean(img);
        s = cpl_image_get_stdev(img);
    }
    cpl_mask *bpm = cpl_mask_threshold_image_create(img, m - s * kappa_low,
                                                    m + s * kappa_high);
    cpl_mask_not(bpm);
    r = cpl_image_new_from_mask(bpm);
    cpl_mask_delete(bpm);
    return r;
}
cpl_error_code
moo_hdrl_bpm_fit_compute(const hdrl_parameter *par,
                         const hdrl_imagelist *data,
                         const cpl_vector *sample_pos,
                         cpl_image **out_mask)
{
    cpl_error_code status = CPL_ERROR_NONE;
    cpl_image *out_chi2 = NULL, *out_dof = NULL;
    hdrl_imagelist *out_coef = NULL;

    status = hdrl_bpm_fit_parameter_verify(par);
    if (status != CPL_ERROR_NONE)
        return status;

    int degree = hdrl_bpm_fit_parameter_get_degree(par);

    /* TODO check for 0 error,
     * in that case set to 1 and do not allow pval */

    status = hdrl_fit_polynomial_imagelist(data, sample_pos, degree, &out_coef,
                                           &out_chi2, &out_dof);

    if (status) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
                                     "Fit failed");
    }

    if (cpl_image_count_rejected(out_chi2) ==
        (cpl_image_get_size_x(out_chi2) * cpl_image_get_size_y(out_chi2))) {
        cpl_msg_error(cpl_func,
                      "Too few good pixels to fit polynomial of "
                      "degree %d in all pixels",
                      degree);

        goto end;
    }

    {
        cpl_image *bpm = NULL;
        double pval = hdrl_bpm_fit_parameter_get_pval(par);
        double rel_chi_l = hdrl_bpm_fit_parameter_get_rel_chi_low(par);
        double rel_chi_h = hdrl_bpm_fit_parameter_get_rel_chi_high(par);
        double rel_coef_l = hdrl_bpm_fit_parameter_get_rel_coef_low(par);
        double rel_coef_h = hdrl_bpm_fit_parameter_get_rel_coef_high(par);

        if (rel_chi_l >= 0) {
            /* chi is symmetric */
            cpl_image_power(out_chi2, 0.5);
            bpm = bpm_from_rel(out_chi2, rel_chi_l, rel_chi_h, 1);
        }
        else if (rel_coef_l >= 0) {
            for (cpl_size i = 0; i < hdrl_imagelist_get_size(out_coef); i++) {
                hdrl_image *coef = hdrl_imagelist_get(out_coef, i);
                cpl_image *b = bpm_from_rel(hdrl_image_get_image(coef),
                                            rel_coef_l, rel_coef_h, 1);

                /* bits of bpm defines which coefficient "bad" */
                if (bpm) {
                    cpl_image_multiply_scalar(b, pow(2, i));
                    cpl_image_add(bpm, b);
                    cpl_image_delete(b);
                }
                else {
                    bpm = b;
                }
            }
        }
        else if (pval >= 0) {
            pval /= 100.;
            bpm = cpl_image_new(cpl_image_get_size_x(out_chi2),
                                cpl_image_get_size_y(out_chi2), CPL_TYPE_INT);
            int *md = cpl_image_get_data_int(bpm);
            hdrl_data_t *cd = cpl_image_get_data(out_chi2);
            hdrl_data_t *dd = cpl_image_get_data(out_dof);
            for (size_t i = 0; i < (size_t)cpl_image_get_size_x(out_chi2) *
                                       cpl_image_get_size_y(out_chi2);
                 i++) {
                double pv = igamc(dd[i] / 2., cd[i] / 2.);
                md[i] = (pv < pval);
            }
        }
        *out_mask = bpm;
    }

end:
    hdrl_imagelist_delete(out_coef);
    cpl_image_delete(out_chi2);
    cpl_image_delete(out_dof);

    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
  @brief This function computes the signal to noise ratio DER_SNR following the
    definition set forth by the Spectral Container Working Group of ST-ECF,
    MAST and CADC.
  @param ve input flux
  @return the estimated signal-to-noise ratio
* Possible error code :
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL or list size == 0
*
*/
double
moo_vector_get_dersnr(const cpl_vector *ve)
{
    double snr = 0.0;
    int size = cpl_vector_get_size(ve);

    if (size > 4) {
        cpl_vector *sve = cpl_vector_duplicate(ve);
        double signal = cpl_vector_get_median(sve);
        cpl_vector_delete(sve);
        cpl_vector *ve1 = cpl_vector_extract(ve, 2, size - 3, 1);
        cpl_vector *ve2 = cpl_vector_extract(ve, 0, size - 5, 1);
        cpl_vector *ve3 = cpl_vector_extract(ve, 4, size - 1, 1);
        int size1 = cpl_vector_get_size(ve1);

        for (int i = 0; i < size1; i++) {
            double v1 = cpl_vector_get(ve1, i);
            double v2 = cpl_vector_get(ve2, i);
            double v3 = cpl_vector_get(ve3, i);
            double v = fabs(2 * v1 - v2 - v3);
            cpl_vector_set(ve1, i, v);
        }
        double noise = 0.6052697 * cpl_vector_get_median(ve1);

        snr = signal / noise;

        cpl_vector_delete(ve1);
        cpl_vector_delete(ve2);
        cpl_vector_delete(ve3);
    }

    return snr;
}
/*----------------------------------------------------------------------------*/
/**
    @brief This function compares to string to see if the two string are stricly
    equal
    @param a first string to compare
    @param b second string to compare
    @return 1 if a===b else 0
  * Possible error code :
    - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  *
*/
int
moo_string_is_strictly_equal(const char *a, const char *b)
{
    int res = 0;
    cpl_ensure(a != NULL, CPL_ERROR_NULL_INPUT, 0);
    cpl_ensure(b != NULL, CPL_ERROR_NULL_INPUT, 0);

    int sizea = strlen(a);
    int sizeb = strlen(b);

    if (sizea != sizeb) {
        res = 0;
    }
    else {
        res = 1;
        for (int i = 0; i < sizea; i++) {
            if (a[i] != b[i]) {
                res = 0;
                break;
            }
        }
    }
    return res;
}

moo_date
moo_get_date_from_string(const char *string)
{
    moo_date date;
    int year, month, day, hour, min;
    float sec;
    int nb = sscanf(string, "%d-%d-%dT%d:%d:%f", &year, &month, &day, &hour,
                    &min, &sec);
    date.day = day;
    date.year = year;
    date.month = month;
    date.hour = hour;
    date.min = min;
    date.sec = sec;
    return date;
}
/*----------------------------------------------------------------------------*/
/**
    @brief Select from unselected table rows, by comparing column values with a constant.
    @param table	Pointer to table
    @param name	Column name
    @param string	Reference value
    @return Current number of selected rows, or a negative number in case of error.
  * Possible error code :
    - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  *
*/
cpl_size
moo_table_or_selected_sequal_string(cpl_table *table,
                                    const char *name,
                                    const char *string)
{
    int nrow = cpl_table_get_nrow(table);
    cpl_size nbsel = 0;
    const char **names = cpl_table_get_data_string_const(table, name);
    for (int i = 0; i < nrow; i++) {
        if (!cpl_table_is_selected(table, i)) {
            if (moo_string_is_strictly_equal(names[i], string)) {
                cpl_table_select_row(table, i);
                nbsel++;
            }
        }
        else {
            nbsel++;
        }
    }
    return nbsel;
}
/*----------------------------------------------------------------------------*/
/**
    @brief Select from unselected table rows, by comparing column values with a constant.
    @param table	Pointer to table
    @param name	Column name
    @param string	Reference value
    @return Current number of selected rows, or a negative number in case of error.
  * Possible error code :
    - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  *
*/
cpl_size
moo_table_and_selected_sequal_string(cpl_table *table,
                                     const char *name,
                                     const char *string)
{
    char *regexp = cpl_sprintf("^%s$", string);
    int size = cpl_table_and_selected_string(table, name, CPL_EQUAL_TO, regexp);
    cpl_free(regexp);
    return size;
}
/*******************************************************************************/
/* only for gsl2.6 */
#if 0
cpl_vector* moo_median_filter(cpl_vector* v,int winhsize)
{
    cpl_ensure(v!=NULL,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(winhsize>=0,CPL_ERROR_ILLEGAL_INPUT,NULL);

    cpl_vector* result = NULL;
    int size = cpl_vector_get_size(v);
    int K = winhsize*2+1;


    gsl_filter_median_workspace *median_p = gsl_filter_median_alloc(K);
    gsl_vector *x = gsl_vector_alloc(size);           /* input vector */
    gsl_vector *y_median = gsl_vector_alloc(size);    /* median filtered output */

    /* generate input signal */
    for (int i = 0; i < size; ++i)
    {
      double val = cpl_vector_get(v,i);
      gsl_vector_set(x, i, val);
    }

    gsl_filter_median(GSL_FILTER_END_PADZERO, x, y_median, median_p);

    result = cpl_vector_new(size);

    for (int i = 0; i < size; ++i)
    {
      double val = gsl_vector_get(y_median,i);
      cpl_vector_set(result, i, val);
    }

    gsl_vector_free(x);
    gsl_vector_free(y_median);
    gsl_filter_median_free(median_p);

    return result;
}


#endif

/**@}*/
