/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MOO_UTILS_H
#define MOO_UTILS_H


/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <hdrl.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_spline.h>

#define moo_drand() ((double)rand() / (double)RAND_MAX)
#define moo_ensure_status(CMD, STATUS)                                   \
    cpl_error_ensure(((STATUS = CMD), status == CPL_ERROR_NONE), STATUS, \
                     goto moo_try_cleanup, " ")


#define moo_try_assure(BOOL, CODE, ...) \
    cpl_error_ensure(BOOL, CODE, goto moo_try_cleanup, __VA_ARGS__)

#define moo_try_check(CMD, ...)                                     \
    cpl_error_ensure((CMD, cpl_error_get_code() == CPL_ERROR_NONE), \
                     cpl_error_get_code(), goto moo_try_cleanup, __VA_ARGS__)
/*-----------------------------------------------------------------------------
                                       Prototypes
 -----------------------------------------------------------------------------*/

typedef struct
{
    cpl_polynomial *solution;
    int degree;
    double xmin;
    double xmax;
    double ymin;
    double ymax;
} moo_tcheby_polynomial;

typedef struct
{
    cpl_polynomial *solution;
    int degree_x;
    int degree_y;
    double xmin;
    double xmax;
    double ymin;
    double ymax;
    double lmin;
    double lmax;
} moo_tcheby2d_polynomial;

typedef struct
{
    gsl_interp_accel *acc;
    gsl_spline *spline;
    double min;
    double max;
} moo_spline;

typedef struct
{
    int year;
    int month;
    int day;
    int hour;
    int min;
    float sec;
} moo_date;

const char *moo_get_license(void);

cpl_image *moo_compute_sigma_map(cpl_imagelist *list,
                                 cpl_imagelist *qlist,
                                 cpl_image *img);

cpl_mask *moo_kappa_sigma_clipping(cpl_image *sigma_img,
                                   int niter,
                                   double kappa,
                                   double cdiff,
                                   double maxfrac);

cpl_error_code
moo_barycenter_fit(cpl_bivector *points, double *center, double *width);
cpl_error_code moo_gaussian_fit(cpl_bivector *points,
                                cpl_fit_mode fit_pars,
                                double *center,
                                double *width,
                                double *background,
                                double *area);
double moo_gaussian_eval(double x,
                         double center,
                         double width,
                         double background,
                         double area);
cpl_error_code moo_gaussian_eval_inv(double y,
                                     double center,
                                     double width,
                                     double background,
                                     double area,
                                     double *x1,
                                     double *x2);

cpl_error_code moo_find_threshold_limits(cpl_bivector *points,
                                         double treshold,
                                         double *ymin,
                                         double *ymax);
cpl_error_code moo_tchebychev_fit(cpl_bivector *data,
                                  int *cflag,
                                  int degree,
                                  double xmin,
                                  double xmax,
                                  double ymin,
                                  double ymax);

moo_tcheby_polynomial *moo_tcheby_polynomial_fit(cpl_bivector *data,
                                                 int degree,
                                                 double xmin,
                                                 double xmax);
moo_tcheby2d_polynomial *moo_tcheby2d_polynomial_fit(cpl_vector *x,
                                                     int xdegree,
                                                     double xmin,
                                                     double xmax,
                                                     cpl_vector *y,
                                                     int ydegree,
                                                     double ymin,
                                                     double ymax,
                                                     cpl_vector *l,
                                                     double lmin,
                                                     double lmax);
void moo_tcheby_polynomial_delete(moo_tcheby_polynomial *self);
void moo_tcheby2d_polynomial_delete(moo_tcheby2d_polynomial *self);
double moo_tcheby_polynomial_eval(moo_tcheby_polynomial *self, double x);
double
moo_tcheby2d_polynomial_eval(moo_tcheby2d_polynomial *self, double x, double y);
double moo_vector_get_min(const cpl_vector *v, int *flags);
double moo_vector_get_max(const cpl_vector *v, int *flags);
double moo_vector_get_dersnr(const cpl_vector *v);
cpl_vector *moo_hpfilter(cpl_vector *v, double s);

cpl_error_code moo_interpolate_linear(cpl_bivector *fout,
                                      cpl_vector *fout_errs,
                                      int *fout_qual,
                                      const cpl_bivector *fref,
                                      const cpl_vector *fref_errs,
                                      const int *fref_qual);

double moo_vector_get_percentile(cpl_vector *v, double f);

cpl_vector *moo_vector_filter_nan(cpl_vector *v);
cpl_bivector *moo_bivector_filter_nan(cpl_bivector *v);

hdrl_image *moo_image_collapse_median_create(hdrl_image *image);

double
moo_sky_distance(double alpha1, double delta1, double alpha2, double delta2);

moo_spline *moo_spline_create(cpl_bivector *data);
double moo_spline_eval(moo_spline *self, double data);
void moo_spline_delete(moo_spline *self);

cpl_error_code moo_fit_mul(const cpl_vector *vx,
                           const cpl_vector *vw,
                           const cpl_vector *vy,
                           const cpl_vector *vy_err,
                           double *c,
                           double *sig_c);

cpl_vector *moo_savgol_filter(cpl_vector *v, int window_length, int poly_order);
cpl_vector *moo_median_filter(cpl_vector *v, int winhsize);


cpl_image *
moo_imagelist_collapse_bitwiseor(cpl_imagelist *list, hdrl_imagelist *input);

double moo_image_get_ron(cpl_image *image,
                         int llx,
                         int lly,
                         int urx,
                         int ury,
                         int nb_boxes,
                         int box_hsize,
                         double max_error_frac,
                         int max_niter);

cpl_error_code
moo_image_get_quartile(cpl_image *image, double *qmin, double *qmax);

cpl_error_code moo_hdrl_bpm_fit_compute(const hdrl_parameter *par,
                                        const hdrl_imagelist *data,
                                        const cpl_vector *sample_pos,
                                        cpl_image **out_mask);
int moo_string_is_strictly_equal(const char *a, const char *b);
cpl_size moo_table_or_selected_sequal_string(cpl_table *table,
                                             const char *name,
                                             const char *string);
cpl_size moo_table_and_selected_sequal_string(cpl_table *table,
                                              const char *name,
                                              const char *string);
moo_date moo_get_date_from_string(const char *string);
#endif
