/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>
#include "moo_badpix.h"
#include "moo_sci.h"
#include "moo_flx.h"
#include "moo_atm.h"
#include "moo_resp.h"
#include "moo_calib_flux.h"
#include "moo_params.h"

#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions
        Testing of functions used in moo_calib_flux
 */
/*----------------------------------------------------------------------------*/

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

char *
result_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", RESULT_DIR, name);
}

cpl_frame *
create_frame(const char *name)
{
    char *filename = path_from_filename(name);

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_free(filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    return frame;
}

static void
_moo_calib_flux_hr(void)
{
    cpl_frame *rbn_frame =
        create_frame("XSWITCH_OBJECTA_RBNSPECTRA_OFFSET1_HR_0.fits");
    cpl_frame *resp_frame = create_frame("RESPONSE_HR.fits");
    cpl_frame *atm_frame = create_frame("extinction_table.fits");

    moo_rbn *rbn = moo_rbn_create(rbn_frame);
    moo_resp *resp = moo_resp_load(resp_frame);
    moo_atm *atm = moo_atm_load(atm_frame);
    moo_calib_flux(rbn, atm, resp);
    char *resname = result_from_filename("CALIB_FLUX_HR.fits");
    moo_rbn_save(rbn, resname);
    cpl_free(resname);

    moo_resp_delete(resp);
    moo_atm_delete(atm);
    moo_rbn_delete(rbn);
    cpl_frame_delete(atm_frame);
    cpl_frame_delete(resp_frame);
    cpl_frame_delete(rbn_frame);
}

static void
_moo_calib_flux_lr(void)
{
    cpl_frame *rbn_frame = create_frame("STD_RBNSPECTRA_LR_0.fits");
    cpl_frame *resp_frame = create_frame("RESPONSE_LR.fits");
    cpl_frame *atm_frame = create_frame("extinction_table.fits");

    moo_rbn *rbn = moo_rbn_create(rbn_frame);
    moo_resp *resp = moo_resp_load(resp_frame);
    moo_atm *atm = moo_atm_load(atm_frame);
    moo_calib_flux(rbn, atm, resp);
    char *resname = result_from_filename("CALIB_FLUX_LR.fits");
    moo_rbn_save(rbn, resname);
    cpl_free(resname);

    moo_resp_delete(resp);
    moo_atm_delete(atm);
    moo_rbn_delete(rbn);
    cpl_frame_delete(atm_frame);
    cpl_frame_delete(resp_frame);
    cpl_frame_delete(rbn_frame);
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of hdrl_image module
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    _moo_calib_flux_hr();
    return cpl_test_end(0);
}
