/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>
#include "moo_badpix.h"
#include "moo_scilist.h"
#include "moo_sci.h"
#include "moo_coadd.h"
#include "moo_params.h"

#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions localise test
        Testing of functions used in moo_localise
 */
/*----------------------------------------------------------------------------*/

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

char *
result_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", RESULT_DIR, name);
}

cpl_frame *
create_frame(const char *name)
{
    char *filename = path_from_filename(name);

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_free(filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    return frame;
}

void
_moo_coadd_sum(void)
{
    cpl_frameset *set = cpl_frameset_new();
    cpl_frame *sci0_frame =
        create_frame("SCIENCE_SKSSPECTRA_OFFSET0_LR_0.fits");
    cpl_frameset_insert(set, sci0_frame);
    cpl_frame *sci1_frame =
        create_frame("SCIENCE_SKSSPECTRA_OFFSET0_LR_1.fits");
    cpl_frameset_insert(set, sci1_frame);

    moo_coadd_params *params = moo_coadd_params_new();
    params->method = MOO_COADD_METHOD_MEAN;
    params->weight = MOO_COADD_WEIGHT_NONE;

    moo_scilist *list = moo_scilist_create(set);

    char *resname = result_from_filename("COADD.fits");
    moo_sci *sci = moo_coadd(list, params, resname);
    cpl_free(resname);

    moo_sci_delete(sci);
    moo_scilist_delete(list);
    moo_coadd_params_delete(params);
    cpl_frameset_delete(set);
}

void
_moo_coadd_starenod(void)
{
    moo_sci *sci = NULL;
    cpl_frameset *set = cpl_frameset_new();
    for (int i = 0; i <= 5; i++) {
        char *sci_name =
            cpl_sprintf("STARENOD_NOD_SCI1D_OFFSET1_HR_%d.fits", i);
        cpl_frame *sci_frame = create_frame(sci_name);
        cpl_frameset_insert(set, sci_frame);
        cpl_free(sci_name);
    }

    cpl_frame *sci2_frame = create_frame("STARENOD_NOD_SCI2D_OFFSET1_HR.fits");
    cpl_frameset_insert(set, sci2_frame);

    moo_coadd_params *params = moo_coadd_params_new();
    params->method = MOO_COADD_METHOD_MEAN;
    params->weight = MOO_COADD_WEIGHT_NONE;

    moo_scilist *list = moo_scilist_create(set);

    char *resname = result_from_filename("STARENOD_COADD.fits");
    sci = moo_coadd(list, params, resname);
    cpl_free(resname);
    moo_sci_delete(sci);
    moo_scilist_delete(list);
    moo_coadd_params_delete(params);
    cpl_frameset_delete(set);
}

void
_moo_coadd_starenod2(void)
{
    moo_sci *sci = NULL;
    cpl_frameset *set = cpl_frameset_new();
    cpl_frame *sci0_frame =
        create_frame("STARENOD2_SCI1D_SCIENCE_STARENOD_SKSSPECTRA_HR.fits");
    cpl_frameset_insert(set, sci0_frame);
    cpl_frame *sci1_frame =
        create_frame("STARENOD2_SCI2D_SCIENCE_STARENOD_SKSSPECTRA_HR.fits");
    cpl_frameset_insert(set, sci1_frame);

    moo_coadd_params *params = moo_coadd_params_new();
    params->method = MOO_COADD_METHOD_MEDIAN;
    params->weight = MOO_COADD_WEIGHT_NONE;

    moo_scilist *list = moo_scilist_create(set);

    char *resname = result_from_filename("STARENOD2_COADD.fits");
    sci = moo_coadd(list, params, resname);
    cpl_free(resname);

    moo_sci_delete(sci);
    moo_scilist_delete(list);
    moo_coadd_params_delete(params);
    cpl_frameset_delete(set);
}

void
_moo_coadd_starenod3(void)
{
    moo_sci *sci = NULL;
    cpl_frameset *set = cpl_frameset_new();
    cpl_frame *sci0_frame =
        create_frame("STARENOD_SCI1D_STARENOD_SKSSPECTRA_LR.fits");
    cpl_frameset_insert(set, sci0_frame);
    cpl_frame *sci1_frame =
        create_frame("STARENOD_SCI2D_STARENOD_SKSSPECTRA_LR.fits");
    cpl_frameset_insert(set, sci1_frame);

    moo_coadd_params *params = moo_coadd_params_new();
    params->method = MOO_COADD_METHOD_MEDIAN;
    params->weight = MOO_COADD_WEIGHT_NONE;

    moo_scilist *list = moo_scilist_create(set);

    char *resname = result_from_filename("STARENOD3_COADD.fits");
    sci = moo_coadd(list, params, resname);
    cpl_free(resname);

    moo_sci_delete(sci);
    moo_scilist_delete(list);
    moo_coadd_params_delete(params);
    cpl_frameset_delete(set);
}

void
_moo_coadd_xswitch(void)
{
    moo_sci *sci = NULL;
    moo_sky_lines_list *skylines = NULL;

    cpl_frameset *set = cpl_frameset_new();
    for (int i = 0; i <= 2; i++) {
        char *name = cpl_sprintf("XSWITCH_SCI_OFFSET0_LR_%d.fits", i);
        cpl_frame *sci0_frame = create_frame(name);
        cpl_frameset_insert(set, sci0_frame);
        cpl_free(name);
    }
    cpl_frame *skylines_frame = create_frame("SKY_LINE_LIST.fits");

    moo_coadd_params *params = moo_coadd_params_new();
    params->method = MOO_COADD_METHOD_SIGCLIP;
    params->clip_niter = 5;
    params->clip_kappa = 3.;
    params->weight = MOO_COADD_WEIGHT_NONE;

    moo_scilist *list = moo_scilist_create(set);

    char *resname = result_from_filename("COADD_XSWITCH_SIGCLIP.fits");
    sci = moo_coadd(list, params, resname);
    skylines = moo_sky_lines_list_load(skylines_frame);

    moo_sci_compute_snr(sci, skylines);
    moo_sci_save(sci, resname);

    cpl_free(resname);

    moo_sky_lines_list_delete(skylines);
    moo_sci_delete(sci);

    moo_scilist_delete(list);
    moo_coadd_params_delete(params);

    cpl_frame_delete(skylines_frame);
    cpl_frameset_delete(set);
}

void
_moo_coadd_median(void)
{
    cpl_frameset *set = cpl_frameset_new();
    cpl_frame *sci0_frame = create_frame("1/SCIENCE_SKSSPECTRA_0.fits");
    cpl_frameset_insert(set, sci0_frame);
    cpl_frame *sci1_frame = create_frame("1/SCIENCE_SKSSPECTRA_1.fits");
    cpl_frameset_insert(set, sci1_frame);
    cpl_frame *sci2_frame = create_frame("1/SCIENCE_SKSSPECTRA_2.fits");
    cpl_frameset_insert(set, sci2_frame);
    cpl_frame *sci3_frame = create_frame("1/SCIENCE_SKSSPECTRA_3.fits");
    cpl_frameset_insert(set, sci3_frame);
    cpl_frame *sci4_frame = create_frame("1/SCIENCE_SKSSPECTRA_4.fits");
    cpl_frameset_insert(set, sci4_frame);
    cpl_frame *sci5_frame = create_frame("1/SCIENCE_SKSSPECTRA_5.fits");
    cpl_frameset_insert(set, sci5_frame);
    cpl_frame *sci6_frame = create_frame("1/SCIENCE_SKSSPECTRA_6.fits");
    cpl_frameset_insert(set, sci6_frame);
    cpl_frame *sci7_frame = create_frame("1/SCIENCE_SKSSPECTRA_7.fits");
    cpl_frameset_insert(set, sci7_frame);
    cpl_frame *sci8_frame = create_frame("1/SCIENCE_SKSSPECTRA_8.fits");
    cpl_frameset_insert(set, sci8_frame);
    cpl_frame *sci9_frame = create_frame("1/SCIENCE_SKSSPECTRA_9.fits");
    cpl_frameset_insert(set, sci9_frame);
    cpl_frame *sci10_frame = create_frame("1/SCIENCE_SKSSPECTRA_10.fits");
    cpl_frameset_insert(set, sci10_frame);
    cpl_frame *sci11_frame = create_frame("1/SCIENCE_SKSSPECTRA_11.fits");
    cpl_frameset_insert(set, sci11_frame);

    moo_coadd_params *params = moo_coadd_params_new();
    params->method = MOO_COADD_METHOD_MEDIAN;
    params->weight = MOO_COADD_WEIGHT_NONE;

    moo_scilist *list = moo_scilist_create(set);

    char *resname = result_from_filename("COADD_median.fits");
    moo_sci *sci = moo_coadd(list, params, resname);
    cpl_free(resname);

    moo_sci_delete(sci);
    moo_scilist_delete(list);
    moo_coadd_params_delete(params);
    cpl_frameset_delete(set);
}

void
_moo_coadd_sum2(void)
{
    cpl_frameset *set = cpl_frameset_new();
    cpl_frame *sci0_frame = create_frame("SCIENCE_SKSSPECTRA.fits");
    cpl_frameset_insert(set, sci0_frame);

    moo_coadd_params *params = moo_coadd_params_new();
    params->method = MOO_COADD_METHOD_MEAN;
    params->weight = MOO_COADD_WEIGHT_NONE;

    moo_scilist *list = moo_scilist_create(set);

    char *resname = result_from_filename("COADD.fits");
    moo_sci *sci = moo_coadd(list, params, resname);
    cpl_free(resname);

    moo_sci_delete(sci);
    moo_scilist_delete(list);
    moo_coadd_params_delete(params);
    cpl_frameset_delete(set);
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of hdrl_image module
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    _moo_coadd_starenod2();
    return cpl_test_end(0);
}
