/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>

#include "moo_detlist.h"
#include "moo_saturate_map.h"
#include "moo_compute_linearity.h"

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

char *
result_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", RESULT_DIR, name);
}
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions compute_linearity test
        Testing of functions used in moo_compute_p2p
 */
/*----------------------------------------------------------------------------*/
void
_moo_compute_saturate_pixels_test(void)
{
    for (int offset = 0; offset <= 0; offset++) {
        cpl_frameset *set = cpl_frameset_new();
        for (int i = 0; i <= 9; i++) {
            char *name =
                cpl_sprintf("LINEARITY_EXPTIME_COMB_%d_%d.fits", offset, i);
            char *filename = path_from_filename(name);
            cpl_frame *frame1 = cpl_frame_new();
            cpl_frame_set_filename(frame1, filename);
            cpl_frame_set_tag(frame1, "TAG");
            cpl_frame_set_group(frame1, CPL_FRAME_GROUP_PRODUCT);
            cpl_frameset_insert(set, cpl_frame_duplicate(frame1));
            cpl_frame_delete(frame1);
            cpl_free(name);
            cpl_free(filename);
        }
        moo_linear_params *params = moo_linear_params_new();
        double kappa = 2;
        params->saturate_kappa[0] = kappa;
        params->saturate_kappa[1] = kappa;
        params->saturate_kappa[2] = kappa;
        params->saturate_kappa[3] = kappa;
        params->saturate_kappa[4] = kappa;
        params->saturate_kappa[5] = kappa;

        params->saturate_threshold = 1E10;
        moo_detlist *list = moo_detlist_create(set);

        char *name = cpl_sprintf("FF_TRACE_GUESS_OFFSET%d_LR.fits", offset);
        char *loc_name = path_from_filename(name);
        cpl_frame *loc_frame = cpl_frame_new();
        cpl_frame_set_filename(loc_frame, loc_name);
        cpl_frame_set_tag(loc_frame, "TAG");
        cpl_frame_set_group(loc_frame, CPL_FRAME_GROUP_PRODUCT);
        cpl_free(name);
        cpl_free(loc_name);
        moo_loc *loc = moo_loc_load(loc_frame);

        moo_saturate_map *map = moo_compute_saturate_pixels(list, loc, params);
        name = cpl_sprintf("SATURATE_OFFSET%d.fits", offset);
        moo_saturate_map_save(map, name);
        cpl_free(name);
        moo_saturate_map_delete(map);
        moo_linear_params_delete(params);
        moo_detlist_delete(list);
        moo_loc_delete(loc);
        cpl_frame_delete(loc_frame);
        cpl_frameset_delete(set);
    }
}

void
_moo_compute_linearity_test(void)
{
    for (int offset = 1; offset <= 1; offset++) {
        cpl_frameset *set = cpl_frameset_new();
        for (int i = 0; i <= 9; i++) {
            char *name =
                cpl_sprintf("LINEARITY_EXPTIME_COMB_%d_%d.fits", offset - 1, i);
            char *filename = path_from_filename(name);
            cpl_frame *frame1 = cpl_frame_new();
            cpl_frame_set_filename(frame1, filename);
            cpl_frame_set_tag(frame1, "TAG");
            cpl_frame_set_group(frame1, CPL_FRAME_GROUP_PRODUCT);
            cpl_frameset_insert(set, cpl_frame_duplicate(frame1));
            cpl_frame_delete(frame1);
            cpl_free(name);
            cpl_free(filename);
        }
        char *name = cpl_sprintf("FF_TRACE_GUESS_OFFSET%d_LR.fits", offset);
        char *loc_name = path_from_filename(name);
        cpl_frame *loc_frame = cpl_frame_new();
        cpl_frame_set_filename(loc_frame, loc_name);
        cpl_frame_set_tag(loc_frame, "TAG");
        cpl_frame_set_group(loc_frame, CPL_FRAME_GROUP_PRODUCT);
        cpl_free(name);
        moo_loc *loc = moo_loc_load(loc_frame);

        name = cpl_sprintf("LINEARITY_SATURATE_OFFSET%d.fits", offset);
        char *saturate_name = path_from_filename(name);
        cpl_free(name);

        cpl_frame *saturate_frame = cpl_frame_new();
        cpl_frame_set_filename(saturate_frame, saturate_name);
        cpl_frame_set_tag(saturate_frame, "TAG");
        cpl_frame_set_group(saturate_frame, CPL_FRAME_GROUP_PRODUCT);

        moo_detlist *list = moo_detlist_create(set);

        char *cube_name =
            cpl_sprintf("LINEARITY_COEFF_CUBE_OFFSET%d.fits", offset);
        moo_cube *cube = NULL;
        moo_saturate_map *saturate_map = moo_saturate_map_load(saturate_frame);

        cube = moo_compute_linearity(list, loc, saturate_map, cube_name);
        cpl_free(cube_name);

        moo_saturate_map_delete(saturate_map);
        moo_cube_delete(cube);
        moo_loc_delete(loc);
        cpl_frame_delete(loc_frame);
        cpl_frame_delete(saturate_frame);
        cpl_free(saturate_name);
        cpl_free(loc_name);
        moo_detlist_delete(list);
        cpl_frameset_delete(set);
    }
}

void
_moo_cube_normalise_test(void)
{
    for (int offset = 0; offset <= 1; offset++) {
        char *name = cpl_sprintf("LINEARITY_COEFF_CUBE_OFFSET%d.fits", offset);
        char *cube_name = path_from_filename(name);
        cpl_frame *cube_frame = cpl_frame_new();
        cpl_frame_set_filename(cube_frame, cube_name);
        cpl_frame_set_tag(cube_frame, "TAG");
        cpl_frame_set_group(cube_frame, CPL_FRAME_GROUP_PRODUCT);
        cpl_free(cube_name);
        cpl_free(name);

        moo_cube *cube = NULL;
        cpl_frame *saturate_frame = NULL;
        char *saturate_name = NULL;

        cube = moo_cube_load(cube_frame);

        name = cpl_sprintf("SATURATE_OFFSET%d.fits", offset);
        saturate_name = path_from_filename(name);
        cpl_free(name);
        saturate_frame = cpl_frame_new();
        cpl_frame_set_filename(saturate_frame, saturate_name);
        cpl_frame_set_tag(saturate_frame, "TAG");
        cpl_frame_set_group(saturate_frame, CPL_FRAME_GROUP_PRODUCT);

        moo_saturate_map *saturate_map = moo_saturate_map_load(saturate_frame);

        moo_cube_normalise(cube, saturate_map);
        cube_name =
            cpl_sprintf("LINEARITY_NORM_COEFF_CUBE_OFFSET%d.fits", offset);
        moo_cube_save(cube, cube_name);
        cpl_free(cube_name);
        moo_saturate_map_delete(saturate_map);
        moo_cube_delete(cube);
        cpl_frame_delete(cube_frame);
        cpl_frame_delete(saturate_frame);
        cpl_free(saturate_name);
    }
}

void
_moo_compute_bpm_linearity_test(void)
{
    moo_bpm **bpm_tab = cpl_calloc(2, sizeof(moo_bpm));
    int size = 0;
    for (int offset = 0; offset <= 1; offset++) {
        char *bpm_name = NULL;
        cpl_frame *saturate_frame = NULL;
        char *saturate_name = NULL;
        cpl_msg_info("test", "DO offset %d", offset);
        char *name = cpl_sprintf("FF_TRACE_OFFSET%d_LR.fits", offset);
        char *loc_name = path_from_filename(name);
        cpl_free(name);
        cpl_frame *loc_frame = cpl_frame_new();
        cpl_frame_set_filename(loc_frame, loc_name);
        cpl_frame_set_tag(loc_frame, "TAG");
        cpl_frame_set_group(loc_frame, CPL_FRAME_GROUP_PRODUCT);
        moo_loc *loc = moo_loc_load(loc_frame);

        name = cpl_sprintf("SATURATE_OFFSET%d.fits", offset);
        saturate_name = path_from_filename(name);
        cpl_free(name);
        saturate_frame = cpl_frame_new();
        cpl_frame_set_filename(saturate_frame, saturate_name);
        cpl_frame_set_tag(saturate_frame, "TAG");
        cpl_frame_set_group(saturate_frame, CPL_FRAME_GROUP_PRODUCT);

        moo_saturate_map *saturate_map = moo_saturate_map_load(saturate_frame);

        name = cpl_sprintf("LINEARITY_NORM_COEFF_CUBE_OFFSET%d.fits", offset);
        char *cube_name = path_from_filename(name);
        cpl_frame *cube_frame = cpl_frame_new();
        cpl_frame_set_filename(cube_frame, cube_name);
        cpl_frame_set_tag(cube_frame, "TAG");
        cpl_frame_set_group(cube_frame, CPL_FRAME_GROUP_PRODUCT);
        cpl_free(cube_name);
        cpl_free(name);

        moo_cube *cube = NULL;

        cube = moo_cube_load(cube_frame);


        moo_linear_params *params = moo_linear_params_new();
        params->kappa[0] = 40;
        params->kappa[1] = 70;
        params->kappa[2] = 50;
        params->kappa[3] = 40;
        params->kappa[4] = 70;
        params->kappa[5] = 50;

        params->min_snr[0] = 10;
        params->min_snr[1] = 10;
        params->min_snr[2] = 10;
        params->min_snr[3] = 10;
        params->min_snr[4] = 10;
        params->min_snr[5] = 10;
        bpm_name = cpl_sprintf("BP_MAP_NL_OFFSET%d.fits", offset);
        bpm_tab[offset] = moo_compute_bpm_linearity(cube, loc, saturate_map,
                                                    params, bpm_name);

        size++;
        cpl_free(bpm_name);
        moo_linear_params_delete(params);
        moo_loc_delete(loc);
        moo_cube_delete(cube);
        moo_saturate_map_delete(saturate_map);
        cpl_frame_delete(loc_frame);
        cpl_frame_delete(saturate_frame);
        cpl_frame_delete(cube_frame);
        cpl_free(loc_name);
        cpl_free(saturate_name);
    }
    if (size == 2) {
        moo_bpm_merge(bpm_tab[0], bpm_tab[1]);
        moo_bpm_save(bpm_tab[0], "BP_MAP_NL.fits");
    }
    for (int i = 0; i < 2; i++) {
        if (bpm_tab[i] != NULL) {
            moo_bpm_delete(bpm_tab[i]);
        }
    }
    cpl_free(bpm_tab);
}


/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    //_moo_compute_saturate_pixels_test();
    _moo_compute_linearity_test();
    //_moo_cube_normalise_test();
    //_moo_compute_bpm_linearity_test();
    return cpl_test_end(0);
}
