/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>

#include "moo_det.h"
#include "moo_psf.h"
#include "moo_compute_p2p.h"
#include "moo_params.h"

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

char *
result_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", RESULT_DIR, name);
}
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions compute_p2p test
        Testing of functions used in moo_compute_p2p
 */
/*----------------------------------------------------------------------------*/

void
_moo_reproject_model_offset1_lr(void)
{
    char *filename1 = path_from_filename("MEDFLAT_OFFSET1_LR.fits");
    char *locname1 = path_from_filename("FF_TRACE_OFFSET1_LR.fits");
    char *psfname1 = path_from_filename("MASTER_FLAT_OFFSET1_LR.fits");

    cpl_frame *frame1 = cpl_frame_new();
    cpl_frame_set_filename(frame1, filename1);
    cpl_frame_set_tag(frame1, "TAG");
    cpl_frame_set_group(frame1, CPL_FRAME_GROUP_PRODUCT);
    moo_det *det1 = moo_det_create(frame1);

    cpl_frame *locframe1 = cpl_frame_new();
    cpl_frame_set_filename(locframe1, locname1);
    cpl_frame_set_tag(locframe1, "TAG");
    cpl_frame_set_group(locframe1, CPL_FRAME_GROUP_PRODUCT);
    moo_loc *loc1 = moo_loc_load(locframe1);

    cpl_frame *psfframe1 = cpl_frame_new();
    cpl_frame_set_filename(psfframe1, psfname1);
    cpl_frame_set_tag(psfframe1, "TAG");
    cpl_frame_set_group(psfframe1, CPL_FRAME_GROUP_PRODUCT);
    moo_psf *psf1 = moo_psf_load(psfframe1);

    //moo_det_set_single(det,MOO_TYPE_RI,1,NULL);
    moo_det_set_single(det1, MOO_TYPE_RI, 2, NULL);
    moo_det_set_single(det1, MOO_TYPE_YJ, 1, NULL);
    moo_det_set_single(det1, MOO_TYPE_YJ, 2, NULL);
    moo_det_set_single(det1, MOO_TYPE_H, 1, NULL);
    moo_det_set_single(det1, MOO_TYPE_H, 2, NULL);

    char *resname = result_from_filename("MODEL_PROJ_OFFSET1_LR.fits");
    moo_reproject_model(det1, loc1, psf1, resname);
    cpl_free(resname);


    moo_psf_delete(psf1);
    moo_loc_delete(loc1);
    moo_det_delete(det1);

    cpl_frame_delete(frame1);
    cpl_frame_delete(locframe1);
    cpl_frame_delete(psfframe1);

    cpl_free(filename1);
    cpl_free(locname1);
    cpl_free(psfname1);
}

void
_moo_reproject_model_offset1_hr(void)
{
    char *filename1 = path_from_filename("MEDFLAT_OFFSET1_HR.fits");
    char *locname1 = path_from_filename("FF_TRACE_OFFSET1_HR.fits");
    char *psfname1 = path_from_filename("MASTER_FLAT_OFFSET1_HR.fits");

    cpl_frame *frame1 = cpl_frame_new();
    cpl_frame_set_filename(frame1, filename1);
    cpl_frame_set_tag(frame1, "TAG");
    cpl_frame_set_group(frame1, CPL_FRAME_GROUP_PRODUCT);
    moo_det *det1 = moo_det_create(frame1);

    cpl_frame *locframe1 = cpl_frame_new();
    cpl_frame_set_filename(locframe1, locname1);
    cpl_frame_set_tag(locframe1, "TAG");
    cpl_frame_set_group(locframe1, CPL_FRAME_GROUP_PRODUCT);
    moo_loc *loc1 = moo_loc_load(locframe1);

    cpl_frame *psfframe1 = cpl_frame_new();
    cpl_frame_set_filename(psfframe1, psfname1);
    cpl_frame_set_tag(psfframe1, "TAG");
    cpl_frame_set_group(psfframe1, CPL_FRAME_GROUP_PRODUCT);
    moo_psf *psf1 = moo_psf_load(psfframe1);

    //moo_det_set_single(det,MOO_TYPE_RI,1,NULL);
    moo_det_set_single(det1, MOO_TYPE_RI, 2, NULL);
    moo_det_set_single(det1, MOO_TYPE_YJ, 1, NULL);
    moo_det_set_single(det1, MOO_TYPE_YJ, 2, NULL);
    moo_det_set_single(det1, MOO_TYPE_H, 1, NULL);
    moo_det_set_single(det1, MOO_TYPE_H, 2, NULL);

    char *resname = result_from_filename("MODEL_PROJ_OFFSET1_HR.fits");
    moo_reproject_model(det1, loc1, psf1, resname);
    cpl_free(resname);


    moo_psf_delete(psf1);
    moo_loc_delete(loc1);
    moo_det_delete(det1);

    cpl_frame_delete(frame1);
    cpl_frame_delete(locframe1);
    cpl_frame_delete(psfframe1);

    cpl_free(filename1);
    cpl_free(locname1);
    cpl_free(psfname1);
}

void
_moo_compute_p2p_lr(void)
{
    char *filename1 = path_from_filename("FLAT_REMOVECRH_OFFSET1_LR.fits");
    char *locname1 = path_from_filename("FF_TRACE_OFFSET1_LR.fits");
    char *psfname1 = path_from_filename("MASTER_FLAT_OFFSET1_LR.fits");
    char *filename2 = path_from_filename("FLAT_REMOVECRH_OFFSET2_LR.fits");
    char *locname2 = path_from_filename("FF_TRACE_OFFSET2_LR.fits");
    char *psfname2 = path_from_filename("MASTER_FLAT_OFFSET2_LR.fits");

    cpl_frame *frame1 = cpl_frame_new();
    cpl_frame_set_filename(frame1, filename1);
    cpl_frame_set_tag(frame1, "TAG");
    cpl_frame_set_group(frame1, CPL_FRAME_GROUP_PRODUCT);
    moo_det *det1 = moo_det_create(frame1);

    cpl_frame *frame2 = cpl_frame_new();
    cpl_frame_set_filename(frame2, filename2);
    cpl_frame_set_tag(frame2, "TAG");
    cpl_frame_set_group(frame2, CPL_FRAME_GROUP_PRODUCT);
    moo_det *det2 = moo_det_create(frame2);

    cpl_frame *locframe1 = cpl_frame_new();
    cpl_frame_set_filename(locframe1, locname1);
    cpl_frame_set_tag(locframe1, "TAG");
    cpl_frame_set_group(locframe1, CPL_FRAME_GROUP_PRODUCT);
    moo_loc *loc1 = moo_loc_load(locframe1);

    cpl_frame *locframe2 = cpl_frame_new();
    cpl_frame_set_filename(locframe2, locname2);
    cpl_frame_set_tag(locframe2, "TAG");
    cpl_frame_set_group(locframe2, CPL_FRAME_GROUP_PRODUCT);
    moo_loc *loc2 = moo_loc_load(locframe2);

    cpl_frame *psfframe1 = cpl_frame_new();
    cpl_frame_set_filename(psfframe1, psfname1);
    cpl_frame_set_tag(psfframe1, "TAG");
    cpl_frame_set_group(psfframe1, CPL_FRAME_GROUP_PRODUCT);
    moo_psf *psf1 = moo_psf_load(psfframe1);

    cpl_frame *psfframe2 = cpl_frame_new();
    cpl_frame_set_filename(psfframe2, psfname2);
    cpl_frame_set_tag(psfframe2, "TAG");
    cpl_frame_set_group(psfframe2, CPL_FRAME_GROUP_PRODUCT);
    moo_psf *psf2 = moo_psf_load(psfframe2);

    char *resname = result_from_filename("P2P_LR.fits");
    moo_det *res = NULL;
    res = moo_compute_p2p(det1, loc1, psf1, det2, loc2, psf2);
    moo_det_save(res, resname);
    cpl_free(resname);


    moo_psf_delete(psf1);
    moo_loc_delete(loc1);
    moo_det_delete(det1);
    moo_psf_delete(psf2);
    moo_loc_delete(loc2);
    moo_det_delete(det2);
    moo_det_delete(res);

    cpl_frame_delete(frame1);
    cpl_frame_delete(locframe1);
    cpl_frame_delete(psfframe1);
    cpl_frame_delete(frame2);
    cpl_frame_delete(locframe2);
    cpl_frame_delete(psfframe2);

    cpl_free(filename1);
    cpl_free(locname1);
    cpl_free(psfname1);

    cpl_free(filename2);
    cpl_free(locname2);
    cpl_free(psfname2);
}

void
_moo_compute_p2p_hr(void)
{
    char *filename1 = path_from_filename("FLAT_REMOVECRH_OFFSET1_HR.fits");
    char *locname1 = path_from_filename("FF_TRACE_OFFSET1_HR.fits");
    char *psfname1 = path_from_filename("MASTER_FLAT_OFFSET1_HR.fits");
    char *filename2 = path_from_filename("FLAT_REMOVECRH_OFFSET2_HR.fits");
    char *locname2 = path_from_filename("FF_TRACE_OFFSET2_HR.fits");
    char *psfname2 = path_from_filename("MASTER_FLAT_OFFSET2_HR.fits");

    cpl_frame *frame1 = cpl_frame_new();
    cpl_frame_set_filename(frame1, filename1);
    cpl_frame_set_tag(frame1, "TAG");
    cpl_frame_set_group(frame1, CPL_FRAME_GROUP_PRODUCT);
    moo_det *det1 = moo_det_create(frame1);

    cpl_frame *frame2 = cpl_frame_new();
    cpl_frame_set_filename(frame2, filename2);
    cpl_frame_set_tag(frame2, "TAG");
    cpl_frame_set_group(frame2, CPL_FRAME_GROUP_PRODUCT);
    moo_det *det2 = moo_det_create(frame2);

    cpl_frame *locframe1 = cpl_frame_new();
    cpl_frame_set_filename(locframe1, locname1);
    cpl_frame_set_tag(locframe1, "TAG");
    cpl_frame_set_group(locframe1, CPL_FRAME_GROUP_PRODUCT);
    moo_loc *loc1 = moo_loc_load(locframe1);

    cpl_frame *locframe2 = cpl_frame_new();
    cpl_frame_set_filename(locframe2, locname2);
    cpl_frame_set_tag(locframe2, "TAG");
    cpl_frame_set_group(locframe2, CPL_FRAME_GROUP_PRODUCT);
    moo_loc *loc2 = moo_loc_load(locframe2);

    cpl_frame *psfframe1 = cpl_frame_new();
    cpl_frame_set_filename(psfframe1, psfname1);
    cpl_frame_set_tag(psfframe1, "TAG");
    cpl_frame_set_group(psfframe1, CPL_FRAME_GROUP_PRODUCT);
    moo_psf *psf1 = moo_psf_load(psfframe1);

    cpl_frame *psfframe2 = cpl_frame_new();
    cpl_frame_set_filename(psfframe2, psfname2);
    cpl_frame_set_tag(psfframe2, "TAG");
    cpl_frame_set_group(psfframe2, CPL_FRAME_GROUP_PRODUCT);
    moo_psf *psf2 = moo_psf_load(psfframe2);

    char *resname = result_from_filename("P2P_HR.fits");
    moo_det *res = NULL;
    res = moo_compute_p2p(det1, loc1, psf1, det2, loc2, psf2);
    moo_det_save(res, resname);
    cpl_free(resname);


    moo_psf_delete(psf1);
    moo_loc_delete(loc1);
    moo_det_delete(det1);
    moo_psf_delete(psf2);
    moo_loc_delete(loc2);
    moo_det_delete(det2);
    moo_det_delete(res);

    cpl_frame_delete(frame1);
    cpl_frame_delete(locframe1);
    cpl_frame_delete(psfframe1);
    cpl_frame_delete(frame2);
    cpl_frame_delete(locframe2);
    cpl_frame_delete(psfframe2);

    cpl_free(filename1);
    cpl_free(locname1);
    cpl_free(psfname1);

    cpl_free(filename2);
    cpl_free(locname2);
    cpl_free(psfname2);
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    _moo_compute_p2p_lr();
    //_moo_reproject_model_offset1_lr();
    //_moo_reproject_model_offset1_hr();
    return cpl_test_end(0);
}
