/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>
#include "moo_badpix.h"
#include "moo_sci.h"
#include "moo_flx.h"
#include "moo_atm.h"
#include "moo_resp.h"
#include "moo_compute_resp.h"
#include "moo_params.h"

#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions localise test
        Testing of functions used in moo_localise
 */
/*----------------------------------------------------------------------------*/

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

char *
result_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", RESULT_DIR, name);
}

cpl_frame *
create_frame(const char *name)
{
    char *filename = path_from_filename(name);

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_free(filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    return frame;
}


static void
_moo_compute_resp_lr(void)
{
    moo_resp *resp = NULL;
    moo_compute_resp_params *params = NULL;
    cpl_frameset *sci_frameset = cpl_frameset_new();
    for (int i = 0; i <= 5; i++) {
        char *name = cpl_sprintf("STD_SCI_LR_%d.fits", i);
        cpl_frame *sci_frame = create_frame(name);
        cpl_frameset_insert(sci_frameset, sci_frame);
        cpl_free(name);
    }

    cpl_frame *flx_frame = create_frame("star_catalog.fits");
    cpl_frame *atm_frame = create_frame("extinction_table.fits");
    moo_scilist *scilist = moo_scilist_create(sci_frameset);
    moo_flx *flx = moo_flx_load(flx_frame);
    moo_atm *atm = moo_atm_load(atm_frame);

    params = moo_compute_resp_params_new();
    params->filter_winhsize[0] = 25;
    params->filter_winhsize[1] = 25;
    params->filter_winhsize[2] = 0;
    params->degree[0] = 4;
    params->degree[1] = 4;
    params->degree[2] = 4;
    params->kappa_lo = 1;
    params->kappa_up = 5;
    params->niter = 4;
    params->frac = 0.7;
    resp = moo_compute_resp(scilist, atm, flx, params);

    char *resname = result_from_filename("RESPONSE_LR.fits");
    moo_resp_save(resp, resname);
    cpl_free(resname);

    moo_resp_delete(resp);
    moo_scilist_delete(scilist);
    moo_atm_delete(atm);
    moo_flx_delete(flx);

    moo_compute_resp_params_delete(params);
    cpl_frame_delete(atm_frame);
    cpl_frame_delete(flx_frame);
    cpl_frameset_delete(sci_frameset);
}

static void
_moo_compute_resp_hr(void)
{
    moo_resp *resp = NULL;
    moo_compute_resp_params *params = NULL;

    cpl_frameset *sci_frameset = cpl_frameset_new();
    for (int i = 0; i < 6; i++) {
        char *name = cpl_sprintf("STD_SCI_HR_%d.fits", i);
        cpl_frame *sci_frame = create_frame(name);
        cpl_frameset_insert(sci_frameset, sci_frame);
        cpl_free(name);
    }

    cpl_frame *flx_frame = create_frame("star_catalog.fits");
    cpl_frame *atm_frame = create_frame("extinction_table.fits");
    moo_scilist *scilist = moo_scilist_create(sci_frameset);
    moo_flx *flx = moo_flx_load(flx_frame);
    moo_atm *atm = moo_atm_load(atm_frame);
    params = moo_compute_resp_params_new();
    params->filter_winhsize[0] = 25;
    params->filter_winhsize[1] = 25;
    params->filter_winhsize[2] = 25;

    resp = moo_compute_resp(scilist, atm, flx, params);

    char *resname = result_from_filename("RESPONSE_HR.fits");
    moo_resp_save(resp, resname);
    cpl_free(resname);


    moo_resp_delete(resp);
    moo_scilist_delete(scilist);
    moo_atm_delete(atm);
    moo_flx_delete(flx);
    moo_compute_resp_params_delete(params);
    cpl_frame_delete(atm_frame);
    cpl_frame_delete(flx_frame);
    cpl_frameset_delete(sci_frameset);
}

static void
_moo_compute_resp_hr_stdflux(void)
{
    moo_resp *resp = NULL;
    moo_compute_resp_params *params = NULL;

    cpl_frameset *sci_frameset = cpl_frameset_new();
    for (int i = 0; i < 6; i++) {
        char *name = cpl_sprintf("STDFLUX_SKSSPECTRA_OFFSET1_HR_%d.fits", i);
        cpl_frame *sci_frame = create_frame(name);
        cpl_frameset_insert(sci_frameset, sci_frame);
        cpl_free(name);
    }

    cpl_frame *flx_frame = create_frame("star_catalog.fits");
    cpl_frame *atm_frame = create_frame("extinction_table.fits");
    moo_scilist *scilist = moo_scilist_create_clean(sci_frameset);
    moo_flx *flx = moo_flx_load(flx_frame);
    moo_atm *atm = moo_atm_load(atm_frame);
    params = moo_compute_resp_params_new();
    params->filter_winhsize[0] = 0;
    params->filter_winhsize[1] = 25;
    params->filter_winhsize[2] = 0;
    params->degree[0] = 4;
    params->degree[1] = 4;
    params->degree[2] = 4;
    params->frac = 0.7;
    params->kappa_lo = 1;
    params->kappa_up = 4.;
    resp = moo_compute_resp(scilist, atm, flx, params);

    char *resname = result_from_filename("STDFLUX_RESPONSE_HR.fits");
    moo_resp_save(resp, resname);
    cpl_free(resname);

    moo_resp_delete(resp);
    moo_scilist_delete(scilist);
    moo_atm_delete(atm);
    moo_flx_delete(flx);
    moo_compute_resp_params_delete(params);
    cpl_frame_delete(atm_frame);
    cpl_frame_delete(flx_frame);
    cpl_frameset_delete(sci_frameset);
}
/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of hdrl_image module
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    _moo_compute_resp_hr_stdflux();
    return cpl_test_end(0);
}
