/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>
#include "moo_badpix.h"
#include "moo_det.h"
#include "moo_loc.h"
#include "moo_compute_slitoffset.h"
#include "moo_params.h"

#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions compute slit offset test
        Testing of functions used in moo_compute_slitoffset
 */
/*----------------------------------------------------------------------------*/

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

char *
result_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", RESULT_DIR, name);
}

cpl_frame *
create_frame(const char *name)
{
    char *filename = path_from_filename(name);

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_free(filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    return frame;
}


static void
_moo_compute_slitoffset(void)
{
    cpl_frame *det_frame = create_frame("FLAT_REMOVECRH_OFFSET1_LR.fits");
    cpl_frame *loc_frame = create_frame("FF_TRACE_GUESS_OFFSET1_LR.fits");

    moo_det *det = moo_det_create(det_frame);
    moo_loc *loc = moo_loc_load(loc_frame);

    moo_compute_slitoffset_params *params = moo_compute_slitoffset_params_new();
    params->step = 0.1;
    params->min = -3;
    params->max = 3;

    float *offsets = moo_compute_slitoffset(det, loc, params);
    for (int i = 0; i < 6; i++) {
        cpl_msg_info("test", "offsets[%d]=%f", i, offsets[i]);
    }
    cpl_free(offsets);
    moo_compute_slitoffset_params_delete(params);
    moo_loc_delete(loc);
    moo_det_delete(det);
    cpl_frame_delete(det_frame);
    cpl_frame_delete(loc_frame);
}


/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of hdrl_image module
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    _moo_compute_slitoffset();
    return cpl_test_end(0);
}
