/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
//#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>

#include <cpl.h>
#include <hdrl.h>
#include "moo_det.h"
#include "moo_cube.h"
#include <moo_drl.h>
#include <moo_dfs.h>
#include "moo_testbed_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_prepare_test  Unit test of moo_correct_detlin
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/


/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of moons_dfs_set_groups
 */
/*----------------------------------------------------------------------------*/

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

char *
result_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", RESULT_DIR, name);
}

static void
_moons_test_correct_detlin(void)
{
    char *cube_name = path_from_filename("LINEARITY_COEFF_CUBE_OFFSET1.fits");
    cpl_frame *cube_frame = cpl_frame_new();
    cpl_frame_set_filename(cube_frame, cube_name);
    cpl_frame_set_tag(cube_frame, MOONS_TAG_LINEARITY_COEFF_CUBE);
    cpl_frame_set_group(cube_frame, CPL_FRAME_GROUP_CALIB);

    for (int i = 0; i <= 9; i++) {
        char *name = cpl_sprintf("LINEARITY_EXPTIME_COMB_0_%d.fits", i);
        char *filename = path_from_filename(name);

        cpl_frame *frame = cpl_frame_new();
        cpl_frame_set_filename(frame, filename);
        cpl_frame_set_tag(frame, "TAG");
        cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);


        moo_det *res = moo_det_create(frame);
        moo_correct_detlin(res, cube_frame);

        char *rname = cpl_sprintf("LINCORR_0_%d_d3.fits", i);
        char *resname = result_from_filename(rname);
        moo_det_save(res, resname);
        cpl_free(rname);
        cpl_free(resname);
        moo_det_delete(res);
        cpl_frame_delete(frame);
        cpl_free(filename);
        cpl_free(name);
    }
    cpl_free(cube_name);
    cpl_frame_delete(cube_frame);
    return;
}

static void
_moons_test_correct_detlin2(void)
{
    char *cube_name = path_from_filename("LINEARITY_COEFF_CUBE_OFFSET1.fits");
    cpl_frame *cube_frame = cpl_frame_new();
    cpl_frame_set_filename(cube_frame, cube_name);
    cpl_frame_set_tag(cube_frame, MOONS_TAG_LINEARITY_COEFF_CUBE);
    cpl_frame_set_group(cube_frame, CPL_FRAME_GROUP_CALIB);


    char *filename = path_from_filename("ADU.fits");

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);


    moo_det *res = moo_det_create(frame);
    moo_correct_detlin(res, cube_frame);


    char *resname = result_from_filename("ADUlin.fits");
    moo_det_save(res, resname);
    cpl_free(resname);
    moo_det_delete(res);
    cpl_frame_delete(frame);
    cpl_free(filename);

    cpl_free(cube_name);
    cpl_frame_delete(cube_frame);
    return;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of moons_dfs module
 */
/*----------------------------------------------------------------------------*/

int
main()
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);

    _moons_test_correct_detlin2();
    return cpl_test_end(0);
}

/**@}*/
