/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>

#include <cpl.h>

#include <moo_dfs.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_dfs_test  Unit test of moons_dfs
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Textual representation of CPL frame group
  @param    group     to convert
  @return   textual representation
 */
/*----------------------------------------------------------------------------*/
static const char *
_moo_frame_group_to_string(cpl_frame_group group)
{
    const char *self = "???"; /* Only used if cpl_frame_group is extended */

    switch(group) {
    case CPL_FRAME_GROUP_RAW:
        self = CPL_FRAME_GROUP_RAW_ID;
        break;
    case CPL_FRAME_GROUP_NONE:
        self = "NONE";
        break;
    case CPL_FRAME_GROUP_CALIB:
        self = CPL_FRAME_GROUP_CALIB_ID;
        break;
    case CPL_FRAME_GROUP_PRODUCT:
        self = CPL_FRAME_GROUP_PRODUCT_ID;
        break;
    default:
        break;
    }

    return self;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of moons_dfs_set_groups
 */
/*----------------------------------------------------------------------------*/
static void 
_moo_test_set_groups(void)
{
    const char *const fctid = "_moo_test_set_groups";
    const char *const test_subject = "moo_dfs_set_groups";
    cpl_errorstate prestate = cpl_errorstate_get();

    /* Test with invalid input */
    if (moo_dfs_set_groups(NULL) == 0) {
        cpl_msg_error(fctid, "Function %s did not fail on NULL input",
                      test_subject);
        cpl_end();
        exit(EXIT_FAILURE);       
    }

    cpl_errorstate_set(prestate);

    /* Test with valid input */
    {
        /* Simulate data */
        const int N = 7;
        const char *const filename[] = {"raw1.fits", 
                                        "raw2.fits",
                                        "raw3.fits",
                                        "raw4.fits",
                                        "calib1.fits",
                                        "calib2.fits",
                                        "calib3.fits"};
        const char *const tag[] = {MOONS_TAG_BIAS,
                                   MOONS_TAG_OBJECT,
                                   MOONS_TAG_DARK_VIS,
                                   MOONS_TAG_DARK_NIR,
                                   MOONS_TAG_MASTER_BIAS,
                                   MOONS_TAG_BP_MAP_RP,
                                   MOONS_TAG_BP_MAP_NL};
        cpl_frame_group const expected_group[] = {CPL_FRAME_GROUP_RAW,
                                                  CPL_FRAME_GROUP_RAW,
                                                  CPL_FRAME_GROUP_RAW,
                                                  CPL_FRAME_GROUP_RAW,
                                                  CPL_FRAME_GROUP_CALIB,
                                                  CPL_FRAME_GROUP_CALIB,    
                                                  CPL_FRAME_GROUP_CALIB};
        cpl_frameset *frames = cpl_frameset_new();
        int i;
    
        for (i = 0; i < N; i++) {
            cpl_frame *frame = cpl_frame_new();

            cpl_frame_set_filename(frame, filename[i]);
            cpl_frame_set_tag(frame, tag[i]);
            cpl_frameset_insert(frames, frame);
        }

        /* Call the function */
        if (moo_dfs_set_groups(frames) != 0) {
            cpl_msg_error(fctid, "Function %s failed",
                          test_subject);
            cpl_errorstate_dump(prestate, CPL_FALSE, NULL);
            cpl_frameset_delete(frames);
            cpl_end();
            exit(EXIT_FAILURE);       
        }
        
        /* Verify results */
        cpl_frameset_iterator *it = cpl_frameset_iterator_new(frames);

        for (i = 0; i < N; i++) {
            cpl_frame *frame = cpl_frameset_iterator_get(it);

            if (frame == NULL) {
                cpl_msg_error(fctid, "Missing frame number %d", i);
                cpl_errorstate_dump(prestate, CPL_FALSE, NULL);
                cpl_frameset_iterator_delete(it);
                cpl_frameset_delete(frames);
                cpl_end();
                exit(EXIT_FAILURE);       
            }

            cpl_frame_group group = cpl_frame_get_group(frame);
            if (group != expected_group[i]) {
                cpl_msg_error(fctid, "Frame number %d has group %s, "
                              "%s expected",
                              i,
                              _moo_frame_group_to_string(group),
                              _moo_frame_group_to_string(expected_group[i]));
                cpl_errorstate_dump(prestate, CPL_FALSE, NULL);
                cpl_frameset_iterator_delete(it);
                cpl_frameset_delete(frames);
                cpl_end();
                exit(EXIT_FAILURE);
            }
            cpl_frameset_iterator_advance(it, 1);
        }
        
        cpl_frameset_iterator_delete(it);
        cpl_frameset_delete(frames);
    }
    
    return;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of moons_dfs module
 */
/*----------------------------------------------------------------------------*/

int
main(void)
{
#if defined CPL_VERSION_CODE && CPL_VERSION_CODE >= CPL_VERSION(4, 0, 0)
    cpl_init(CPL_INIT_DEFAULT);
#else
    cpl_init();
#endif

    _moo_test_set_groups();

    cpl_end();
    exit(EXIT_SUCCESS);
}

/**@}*/
