/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <string.h>

#include "moo_det.h"
#include "moo_loc.h"
#include <moo_drl.h>
#include <moo_dfs.h>
#include "moo_testbed_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions extract test
        Testing of functions used in moo_extract
 */
/*----------------------------------------------------------------------------*/

static char *
_moo_path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

void
_moo_extract_sum_lr_test(void)
{
    char *filename = _moo_path_from_filename("MEDFLAT_OFFSET1_LR.fits");
    char *locname = _moo_path_from_filename("FF_TRACE_OFFSET1_LR.fits");
    cpl_frame *flatframe = cpl_frame_new();
    cpl_frame_set_filename(flatframe, filename);
    cpl_frame_set_tag(flatframe, MOONS_TAG_FLAT);
    cpl_frame_set_group(flatframe, CPL_FRAME_GROUP_PRODUCT);

    moo_extract_params *params = moo_extract_params_new();
    params->method = MOO_EXTRACT_METHOD_SUM;
    params->aperture[0] = 2.5;
    params->aperture[1] = 2.5;
    params->aperture[2] = 2.5;
    moo_det *det = moo_det_create(flatframe);
    moo_det_load(det, 1);

    cpl_frame *locframe = cpl_frame_new();
    cpl_frame_set_filename(locframe, locname);
    cpl_frame_set_tag(locframe, MOONS_TAG_FF_TRACE);
    cpl_frame_set_group(locframe, CPL_FRAME_GROUP_PRODUCT);
    moo_loc *loc = moo_loc_load(locframe);

    moo_ext *ext =
        moo_extract(det, loc, NULL, params, "EXT_OFFSET1_LR_sum.fits");

    moo_ext_delete(ext);
    moo_loc_delete(loc);
    moo_det_delete(det);
    moo_extract_params_delete(params);
    cpl_frame_delete(flatframe);
    cpl_frame_delete(locframe);
    cpl_free(filename);
    cpl_free(locname);
}

void
_moo_extract_sum_hr_std_test(void)
{
    cpl_msg_info("test", "std test");
    char *filename = _moo_path_from_filename("STD_DET_HR_0.fits");
    char *locname = _moo_path_from_filename("FF_TRACE_OFFSET1_HR.fits");
    cpl_frame *flatframe = cpl_frame_new();
    cpl_frame_set_filename(flatframe, filename);
    cpl_frame_set_tag(flatframe, MOONS_TAG_FLAT);
    cpl_frame_set_group(flatframe, CPL_FRAME_GROUP_PRODUCT);

    moo_extract_params *params = moo_extract_params_new();
    params->method = MOO_EXTRACT_METHOD_SUM;
    double aperture = -1;
    params->aperture[0] = aperture;
    params->aperture[1] = 2.5;
    params->aperture[2] = 2.5;
    moo_det *det = moo_det_create(flatframe);

    cpl_frame *locframe = cpl_frame_new();
    cpl_frame_set_filename(locframe, locname);
    cpl_frame_set_tag(locframe, MOONS_TAG_FF_TRACE);
    cpl_frame_set_group(locframe, CPL_FRAME_GROUP_PRODUCT);
    moo_loc *loc = moo_loc_load(locframe);
    char *resname = cpl_sprintf("EXT_OFFSET1_HR_%f_sum.fits", aperture);
    moo_ext *ext = moo_extract(det, loc, NULL, params, resname);

    cpl_free(resname);
    moo_ext_delete(ext);
    moo_loc_delete(loc);
    moo_det_delete(det);
    moo_extract_params_delete(params);
    cpl_frame_delete(flatframe);
    cpl_frame_delete(locframe);
    cpl_free(filename);
    cpl_free(locname);
}

void
_moo_extract_sum_offset1_hr(void)
{
    cpl_msg_info("test", "flat test");
    //char *filename = _moo_path_from_filename("FLAT_REMOVECRH_OFFSET1_HR.fits");
    char *filename = _moo_path_from_filename("FLAT_APPLYP2P_OFFSET1_HR.fits");
    char *locname = _moo_path_from_filename("FF_TRACE_OFFSET1_HR.fits");
    cpl_frame *flatframe = cpl_frame_new();
    cpl_frame_set_filename(flatframe, filename);
    cpl_frame_set_tag(flatframe, MOONS_TAG_FLAT);
    cpl_frame_set_group(flatframe, CPL_FRAME_GROUP_PRODUCT);

    moo_extract_params *params = moo_extract_params_new();
    params->method = MOO_EXTRACT_METHOD_SUM;
    double aperture = -1;
    params->aperture[0] = aperture;
    params->aperture[1] = aperture;
    params->aperture[2] = aperture;
    params->aperture[3] = aperture;
    params->aperture[4] = aperture;
    params->aperture[5] = aperture;
    moo_det *det = moo_det_create(flatframe);
    /*moo_single_delete(det->ri[0]);
    det->ri[0] = NULL;*/
    moo_single_delete(det->yj[0]);
    det->yj[0] = NULL;
    moo_single_delete(det->h[0]);
    det->h[0] = NULL;

    moo_single_delete(det->ri[1]);
    det->ri[1] = NULL;
    moo_single_delete(det->yj[1]);
    det->yj[1] = NULL;
    moo_single_delete(det->h[1]);
    det->h[1] = NULL;

    cpl_frame *locframe = cpl_frame_new();
    cpl_frame_set_filename(locframe, locname);
    cpl_frame_set_tag(locframe, MOONS_TAG_FF_TRACE);
    cpl_frame_set_group(locframe, CPL_FRAME_GROUP_PRODUCT);
    moo_loc *loc = moo_loc_load(locframe);
    char *resname =
        cpl_sprintf("FLAT_EXTSPECTRA_OFFSET1_HR_%f_sum.fits", aperture);
    moo_ext *ext = moo_extract(det, loc, NULL, params, resname);

    cpl_free(resname);
    moo_ext_delete(ext);
    moo_loc_delete(loc);
    moo_det_delete(det);
    moo_extract_params_delete(params);
    cpl_frame_delete(flatframe);
    cpl_frame_delete(locframe);
    cpl_free(filename);
    cpl_free(locname);
}

void
_moo_extract_opt_hr_test(void)
{
    char *filename =
        _moo_path_from_filename("opt/FLAT_REMOVECRH_OFFSET1_HR.fits");
    char *locname = _moo_path_from_filename("opt/FF_TRACE_OFFSET1_HR.fits");
    cpl_frame *flatframe = cpl_frame_new();
    cpl_frame_set_filename(flatframe, filename);
    cpl_frame_set_tag(flatframe, MOONS_TAG_FLAT);
    cpl_frame_set_group(flatframe, CPL_FRAME_GROUP_PRODUCT);

    char *mflatname =
        _moo_path_from_filename("opt/MASTER_FLAT_OFFSET1_HR.fits");
    cpl_frame *mflatframe = cpl_frame_new();
    cpl_frame_set_filename(mflatframe, mflatname);
    cpl_frame_set_tag(mflatframe, MOONS_TAG_MASTER_FLAT);
    cpl_frame_set_group(mflatframe, CPL_FRAME_GROUP_CALIB);

    moo_extract_params *params = moo_extract_params_new();
    params->method = MOO_EXTRACT_METHOD_OPTIMAL;
    double aperture = -1;
    params->aperture[0] = aperture;
    params->aperture[1] = 2.5;
    params->aperture[2] = 2.5;
    moo_det *det = moo_det_create(flatframe);
    moo_det_load(det, 1);
    cpl_frame *locframe = cpl_frame_new();
    cpl_frame_set_filename(locframe, locname);
    cpl_frame_set_tag(locframe, MOONS_TAG_FF_TRACE);
    cpl_frame_set_group(locframe, CPL_FRAME_GROUP_PRODUCT);
    moo_loc *loc = moo_loc_load(locframe);
    moo_psf *mflat = moo_psf_load(mflatframe);

    char *resname = cpl_sprintf("EXT_OFFSET1_HR_%f_opt.fits", aperture);
    moo_ext *ext = moo_extract(det, loc, mflat, params, resname);
    cpl_free(resname);
    moo_ext_delete(ext);

    moo_psf_delete(mflat);
    moo_loc_delete(loc);
    moo_det_delete(det);
    moo_extract_params_delete(params);
    cpl_frame_delete(flatframe);
    cpl_frame_delete(locframe);
    cpl_frame_delete(mflatframe);

    cpl_free(filename);
    cpl_free(locname);
    cpl_free(mflatname);
}

void
_moo_extract_optimal_lr_test(void)
{
    char *filename = _moo_path_from_filename("MEDFLAT_OFFSET0_LR.fits");
    char *locname = _moo_path_from_filename("FF_TRACE_OFFSET0_LR.fits");
    char *mflatname = _moo_path_from_filename("MASTER_FLAT_OFFSET0_LR.fits");
    cpl_frame *flatframe = cpl_frame_new();
    cpl_frame_set_filename(flatframe, filename);
    cpl_frame_set_tag(flatframe, MOONS_TAG_FLAT);
    cpl_frame_set_group(flatframe, CPL_FRAME_GROUP_PRODUCT);

    moo_extract_params *params = moo_extract_params_new();
    params->method = MOO_EXTRACT_METHOD_OPTIMAL;
    moo_det *det = moo_det_create(flatframe);
    moo_det_load(det, 1);

    cpl_frame *locframe = cpl_frame_new();
    cpl_frame_set_filename(locframe, locname);
    cpl_frame_set_tag(locframe, MOONS_TAG_FF_TRACE);
    cpl_frame_set_group(locframe, CPL_FRAME_GROUP_PRODUCT);
    moo_loc *loc = moo_loc_load(locframe);

    cpl_frame *mflatframe = cpl_frame_new();
    cpl_frame_set_filename(mflatframe, mflatname);
    cpl_frame_set_tag(mflatframe, MOONS_TAG_MASTER_FLAT);
    cpl_frame_set_group(mflatframe, CPL_FRAME_GROUP_CALIB);

    moo_psf *mflat = moo_psf_load(mflatframe);
    //moo_det_set_single(det,MOO_TYPE_RI,1,NULL);
    moo_det_set_single(det, MOO_TYPE_RI, 2, NULL);
    //moo_det_set_single(det,MOO_TYPE_YJ,1,NULL);
    moo_det_set_single(det, MOO_TYPE_YJ, 2, NULL);
    //moo_det_set_single(det,MOO_TYPE_H,1,NULL);
    moo_det_set_single(det, MOO_TYPE_H, 2, NULL);

    moo_ext *ext = moo_extract(det, loc, mflat, params, "EXT_OPT.fits");
    moo_ext_delete(ext);

    moo_psf_delete(mflat);
    moo_loc_delete(loc);
    moo_det_delete(det);
    moo_extract_params_delete(params);
    cpl_frame_delete(flatframe);
    cpl_frame_delete(locframe);
    cpl_frame_delete(mflatframe);
    cpl_free(filename);
    cpl_free(locname);
    cpl_free(mflatname);
}

static void
_moons_extract(cpl_frameset *sof)
{
    cpl_frame *raw = NULL;
    cpl_frame *mbias = NULL;
    const char *bpmap_rp = NULL;
    const char *bpmap_nl = NULL;
    cpl_frame *ff_trace = NULL;
    moo_prepare_params *prepare_params = moo_prepare_params_new();

    int size = cpl_frameset_get_size(sof);
    for (int i = 0; i < size; i++) {
        cpl_frame *frame = cpl_frameset_get_position(sof, i);
        const char *tag = cpl_frame_get_tag(frame);
        if (strcmp(tag, MOONS_TAG_RAW) == 0) {
            raw = frame;
            cpl_frame_set_group(raw, CPL_FRAME_GROUP_RAW);
        }
        else if (strcmp(tag, MOONS_TAG_FLAT) == 0) {
            raw = frame;
            cpl_frame_set_group(raw, CPL_FRAME_GROUP_RAW);
        }
        else if (strcmp(tag, MOONS_TAG_MASTER_BIAS) == 0) {
            mbias = frame;
        }
        else if (strcmp(tag, MOONS_TAG_BP_MAP_RP) == 0) {
            bpmap_rp = cpl_frame_get_filename(frame);
        }
        else if (strcmp(tag, MOONS_TAG_BP_MAP_NL) == 0) {
            bpmap_nl = cpl_frame_get_filename(frame);
        }
        else if (strcmp(tag, MOONS_TAG_FF_TRACE) == 0) {
            ff_trace = frame;
        }
    }
    moo_extract_params *params = moo_extract_params_new();
    params->method = MOO_EXTRACT_METHOD_SUM;

    moo_det *det =
        moo_prepare(raw, bpmap_rp, bpmap_nl, mbias, NULL, prepare_params);
    moo_loc *loc = moo_loc_load(ff_trace);

    moo_ext *ext = moo_extract(det, loc, NULL, params, "EXT.fits");


    moo_extract_params_delete(params);
    moo_det_delete(det);
    moo_loc_delete(loc);
    moo_ext_delete(ext);
    moo_prepare_params_delete(prepare_params);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of hdrl_image module
 **/
/*----------------------------------------------------------------------------*/
int
main(int argc, char *argv[])
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_DEBUG);
    if (argc == 2) {
        cpl_msg_info("test", "Use data SOF: %s", argv[1]);
        cpl_frameset *set = moo_testbed_frameset_load(argv[1]);
        _moons_extract(set);
        cpl_frameset_delete(set);
    }
    else {
        _moo_extract_sum_offset1_hr();
        //_moo_extract_opt_hr_test();
    }
    return cpl_test_end(0);
}
