/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>

#include "moo_det.h"
#include "moo_psf.h"
#include "moo_model_flat.h"
#include "moo_utils.h"
#include "moo_params.h"

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

char *
result_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", RESULT_DIR, name);
}
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions model flat test
        Testing of functions used in moo_model_flat
 */
/*----------------------------------------------------------------------------*/

void
_moo_model_flat_test_giraffe(void)
{
    char *filename = path_from_filename("MEDFLAT_OFFSET0_giraffe.fits");
    char *locname = path_from_filename("FF_TRACE_giraffe.fits");

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);
    moo_det *det = moo_det_create(frame);

    cpl_frame *locframe = cpl_frame_new();
    cpl_frame_set_filename(locframe, locname);
    cpl_frame_set_tag(locframe, "TAG");
    cpl_frame_set_group(locframe, CPL_FRAME_GROUP_PRODUCT);
    moo_loc *loc = moo_loc_load(locframe);

    moo_model_flat_params *params = moo_model_flat_params_new();

    params->oversamp = 1;
    //moo_det_set_single(det,MOO_TYPE_RI,1,NULL);
    moo_det_set_single(det, MOO_TYPE_RI, 2, NULL);
    moo_det_set_single(det, MOO_TYPE_YJ, 1, NULL);
    moo_det_set_single(det, MOO_TYPE_YJ, 2, NULL);
    moo_det_set_single(det, MOO_TYPE_H, 1, NULL);
    moo_det_set_single(det, MOO_TYPE_H, 2, NULL);

    char *resname = result_from_filename("MASTER_FLAT_OFFSET0_giraffe.fits");
    moo_psf *psf = moo_model_flat(det, loc, params, resname);
    cpl_free(resname);

    moo_psf_delete(psf);
    moo_loc_delete(loc);
    moo_det_delete(det);
    moo_model_flat_params_delete(params);
    cpl_frame_delete(frame);
    cpl_frame_delete(locframe);
    cpl_free(filename);
    cpl_free(locname);
}

void
_moo_model_flat_lr_test(void)
{
    char *filename = path_from_filename("MEDFLAT_OFFSET0_LR.fits");
    char *locname = path_from_filename("FF_TRACE_OFFSET0_LR.fits");

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);
    moo_det *det = moo_det_create(frame);

    cpl_frame *locframe = cpl_frame_new();
    cpl_frame_set_filename(locframe, locname);
    cpl_frame_set_tag(locframe, "TAG");
    cpl_frame_set_group(locframe, CPL_FRAME_GROUP_PRODUCT);
    moo_loc *loc = moo_loc_load(locframe);

    moo_model_flat_params *params = moo_model_flat_params_new();

    params->oversamp = 5;
    params->xstep = 10;
    params->winhsize = 0;
    //moo_det_set_single(det,MOO_TYPE_RI,1,NULL);
    moo_det_set_single(det, MOO_TYPE_RI, 2, NULL);
    moo_det_set_single(det, MOO_TYPE_YJ, 1, NULL);
    moo_det_set_single(det, MOO_TYPE_YJ, 2, NULL);
    moo_det_set_single(det, MOO_TYPE_H, 1, NULL);
    moo_det_set_single(det, MOO_TYPE_H, 2, NULL);

    char *resname = result_from_filename("MASTER_FLAT_OFFSET0.fits");
    moo_psf *psf = moo_model_flat(det, loc, params, resname);
    cpl_free(resname);

    moo_psf_delete(psf);
    moo_loc_delete(loc);
    moo_det_delete(det);
    moo_model_flat_params_delete(params);
    cpl_frame_delete(frame);
    cpl_frame_delete(locframe);
    cpl_free(filename);
    cpl_free(locname);
}

void
_moo_model_flat_hr_test(void)
{
    char *filename = path_from_filename("FLAT_APPLYP2P_OFFSET1_HR.fits");
    char *locname = path_from_filename("FF_TRACE_OFFSET1_HR.fits");

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);
    moo_det *det = moo_det_create(frame);
    //moo_single_delete(det->ri[0]);
    //det->ri[0] = NULL;
    moo_single_delete(det->yj[0]);
    det->yj[0] = NULL;
    moo_single_delete(det->h[0]);
    det->h[0] = NULL;

    moo_single_delete(det->ri[1]);
    det->ri[1] = NULL;
    moo_single_delete(det->yj[1]);
    det->yj[1] = NULL;
    moo_single_delete(det->h[1]);
    det->h[1] = NULL;

    cpl_frame *locframe = cpl_frame_new();
    cpl_frame_set_filename(locframe, locname);
    cpl_frame_set_tag(locframe, "TAG");
    cpl_frame_set_group(locframe, CPL_FRAME_GROUP_PRODUCT);
    moo_loc *loc = moo_loc_load(locframe);

    moo_model_flat_params *params = moo_model_flat_params_new();

    params->oversamp = 5;
    params->xstep = 10;
    params->winhsize = 20;

    char *resname = result_from_filename("MASTER_FLAT_OFFSET1_HR.fits");
    moo_psf *psf = moo_model_flat(det, loc, params, resname);
    cpl_free(resname);

    moo_psf_delete(psf);
    moo_loc_delete(loc);
    moo_det_delete(det);
    moo_model_flat_params_delete(params);
    cpl_frame_delete(frame);
    cpl_frame_delete(locframe);
    cpl_free(filename);
    cpl_free(locname);
}

void
_moo_hpfilter_test(void)
{
    const char *filename = "/home/haigron/moons/data/in.dat";

    cpl_vector *v = cpl_vector_read(filename);


    cpl_vector *res = moo_hpfilter(v, 1600);
    FILE *test = fopen("res.dat", "w");
    cpl_vector_dump(res, test);
    fclose(test);

    cpl_vector_delete(res);
    cpl_vector_delete(v);
}

void
_moo_model_test(void)
{
    double sigma = 1.1;
    double cexp = 2.6;
    double A = 1.06;
    double c0 = 0.5;
    double b = 0.17;
    moo_model *m = moo_model_new(sigma, cexp, A, c0, b);

    for (double y = -3; y <= 3; y += 0.01) {
        double v = moo_model_dcexp_eval(m, y);
        cpl_msg_info("test", "DEBUG %f %f", y, v);
    }

    moo_model_delete(m);
}


/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of hdrl_image module
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    _moo_model_flat_hr_test();
    //_moo_model_flat_test_giraffe();
    //_moo_model_test();
    //_moo_hpfilter_test();
    return cpl_test_end(0);
}
