/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>
#include "moo_badpix.h"
#include "moo_sci.h"
#include "moo_molecfit.h"
#include "moo_molectable.h"
#include "moo_detector.h"
#include "moo_params.h"

#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions molecfit test
        Testing of functions used in moo_molecfit
 */
/*----------------------------------------------------------------------------*/

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

char *
result_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", RESULT_DIR, name);
}

cpl_frame *
create_frame(const char *name)
{
    char *filename = path_from_filename(name);

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_free(filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    return frame;
}


static void
_moo_molecfit_model(void)
{
    cpl_frame *sci_frame = NULL;
    cpl_frame *molecule_frame = NULL;
    cpl_frame *winc_frame = NULL;
    moo_molecfit_model_params *molecfit_model_params = NULL;

    moo_molectable *atm_fitted = NULL;
    moo_molectable *best_fitted_params = NULL;
    moo_molectable *best_fitted_model = NULL;
    moo_mode_type mode;
    const char *modename = NULL;

    sci_frame = create_frame("STDTELL_SKSSPECTRA_OFFSET1_HR_0.fits");
    molecule_frame = create_frame("moons_molecules.fits");
    winc_frame = create_frame("moons_wl_inc.fits");
    cpl_msg_info("test", "-------------before model_params");
    cpl_memory_dump();


    molecfit_model_params = moo_molecfit_model_params_new();
    molecfit_model_params->kern_fac = 3;
    molecfit_model_params->kern_mode = CPL_FALSE;
    molecfit_model_params->var_kern = CPL_FALSE;
    molecfit_model_params->continuum_const = .9492509365081787;

    mode = moo_mode_get(sci_frame);

    modename = moo_mode_get_name(mode);
    best_fitted_params =
        moo_molecfit_model(sci_frame, modename, molecule_frame, winc_frame,
                           &atm_fitted, &best_fitted_model,
                           molecfit_model_params);
    moo_molectable_save(best_fitted_model, "BEST_FITTED_MODEL.fits");
    moo_molectable_save(best_fitted_model, "ATM_FITTED.fits");
    moo_molectable_save(best_fitted_model, "BEST_FITTED_PARAMS.fits");
    moo_molectable_delete(best_fitted_model);
    moo_molectable_delete(atm_fitted);
    moo_molectable_delete(best_fitted_params);
    moo_molecfit_model_params_delete(molecfit_model_params);
    cpl_frame_delete(sci_frame);
    cpl_frame_delete(molecule_frame);
    cpl_frame_delete(winc_frame);
}

static void
_moo_molecfit_calctrans(void)
{
    cpl_frame *rbn_frame = NULL;
    cpl_frame *molecule_frame = NULL;
    cpl_frame *atm_frame = NULL;
    cpl_frame *best_fit_frame = NULL;
    moo_molecfit_calctrans_params *molecfit_calctrans_params = NULL;

    moo_telluric *telluric = NULL;
    moo_mode_type mode;
    const char *modename = NULL;

    rbn_frame = create_frame("STDFLUX_RBNSPECTRA_OFFSET1_HR_0.fits");
    molecule_frame = create_frame("moons_molecules.fits");
    atm_frame = create_frame("ATM_PARAMETERS_HR.fits");
    best_fit_frame = create_frame("BEST_FIT_PARAMETERS_HR.fits");

    molecfit_calctrans_params = moo_molecfit_calctrans_params_new();
    molecfit_calctrans_params->filter_skyfibre = 1;
    molecfit_calctrans_params->min_snr[0] = 0.0;

    mode = moo_mode_get(rbn_frame);

    modename = moo_mode_get_name(mode);

    telluric =
        moo_molecfit_calctrans(rbn_frame, modename, molecule_frame, atm_frame,
                               best_fit_frame, NULL, molecfit_calctrans_params);
    moo_telluric_delete(telluric);
    moo_molecfit_calctrans_params_delete(molecfit_calctrans_params);
    cpl_frame_delete(rbn_frame);
    cpl_frame_delete(molecule_frame);
    cpl_frame_delete(atm_frame);
    cpl_frame_delete(best_fit_frame);
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of hdrl_image module
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_DEBUG);
    _moo_molecfit_model();
    //_moo_molecfit_calctrans();
    return cpl_test_end(0);
}
