/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
//#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>

#include <cpl.h>
#include <hdrl.h>
#include "moo_single.h"
#include "moo_det.h"
#include "moo_detlist.h"
#include <moo_drl.h>
#include <moo_dfs.h>
#include "moo_testbed_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_prepare_test  Unit test of moons_prepare
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/


/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of moons_dfs_set_groups
 */
/*----------------------------------------------------------------------------*/

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

char *
result_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", RESULT_DIR, name);
}

static void
_moons_test_prepare(void)
{
    char *filename = path_from_filename("RAW1.fits");
    char *bprp_name = path_from_filename("BPMAP.fits");
    char *bpnl_name = NULL;
    char *mbias_name = path_from_filename("MASTER_BIAS.fits");

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);

    cpl_frame *mbias_frame = cpl_frame_new();
    cpl_frame_set_filename(mbias_frame, mbias_name);
    cpl_frame_set_tag(mbias_frame, "TAG");
    cpl_frame_set_group(mbias_frame, CPL_FRAME_GROUP_CALIB);

    // 1 - null input RAW frame
    moo_prepare_params *prepare_params = moo_prepare_params_new();

    moo_det *res = moo_prepare(frame, bprp_name, bpnl_name, mbias_frame, NULL,
                               prepare_params);

    char *resname = result_from_filename("DET1.fits");
    moo_det_save(res, resname);
    cpl_free(resname);
    moo_det_delete(res);

    cpl_free(filename);
    cpl_free(bprp_name);
    cpl_free(bpnl_name);
    cpl_free(mbias_name);

    cpl_frame_delete(frame);
    cpl_frame_delete(mbias_frame);
    moo_prepare_params_delete(prepare_params);
    return;
}

static void
_moons_test_prepare3(void)
{
    moo_prepare_params *prepare_params = moo_prepare_params_new();

    char *filename = path_from_filename("RAW.fits");
    char *bprp_name = path_from_filename("BPMAP.fits");
    char *bpnl_name = path_from_filename("BP_MAP_NL.fits");

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);

    // 1 - null input RAW frame
    moo_det *res =
        moo_prepare(frame, bprp_name, bpnl_name, NULL, NULL, prepare_params);

    char *resname = result_from_filename("DET.fits");
    moo_det_save(res, resname);
    cpl_free(resname);
    moo_det_delete(res);

    cpl_free(filename);
    cpl_free(bprp_name);
    cpl_free(bpnl_name);
    cpl_frame_delete(frame);

    moo_prepare_params_delete(prepare_params);
    return;
}

static void
_moons_prepare(cpl_frameset *sof)
{
    cpl_frame *raw = NULL;
    cpl_frame *mbias = NULL;
    const char *bpmap_rp = NULL;
    const char *bpmap_nl = NULL;
    moo_det *res = NULL;
    moo_prepare_params *prepare_params = moo_prepare_params_new();

    int size = cpl_frameset_get_size(sof);
    for (int i = 0; i < size; i++) {
        cpl_frame *frame = cpl_frameset_get_position(sof, i);
        const char *tag = cpl_frame_get_tag(frame);
        if (strcmp(tag, MOONS_TAG_RAW) == 0) {
            raw = frame;
            cpl_frame_set_group(raw, CPL_FRAME_GROUP_RAW);
        }
        else if (strcmp(tag, MOONS_TAG_MASTER_BIAS) == 0) {
            mbias = frame;
        }
        else if (strcmp(tag, MOONS_TAG_BP_MAP_RP) == 0) {
            bpmap_rp = cpl_frame_get_filename(frame);
        }
        else if (strcmp(tag, MOONS_TAG_BP_MAP_NL) == 0) {
            bpmap_nl = cpl_frame_get_filename(frame);
        }
    }

    res = moo_prepare(raw, bpmap_rp, bpmap_nl, mbias, NULL, prepare_params);
    cpl_msg_info("test", "Producing DET.fits");
    moo_det_save(res, "DET.fits");
    moo_det_delete(res);
    moo_prepare_params_delete(prepare_params);

    return;
}

static void
_moons_test_prepare2(void)
{
    cpl_frame *raw = NULL;
    cpl_frame *mbias = NULL;
    moo_det *res = NULL;
    moo_prepare_params *prepare_params = moo_prepare_params_new();

    // 1 - null input RAW frame
    res = moo_prepare(raw, NULL, NULL, NULL, NULL, prepare_params);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    // 2 - empty frame
    raw = cpl_frame_new();
    res = moo_prepare(raw, NULL, NULL, mbias, NULL, prepare_params);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    // 3 - Invalid group
    const char *bands[3] = { "RI", "YJ", "H" };
    moo_testbed_create_raw(10, bands, 3, 2, "det1.fits", 1, 1, 0.9, 5, 150);
    cpl_frame_set_tag(raw, "TEST");
    cpl_frame_set_filename(raw, "det1.fits");

    res = moo_prepare(raw, NULL, NULL, NULL, NULL, prepare_params);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    // 4 - valid test
    cpl_frame_set_group(raw, CPL_FRAME_GROUP_RAW);
    res = moo_prepare(raw, NULL, NULL, mbias, NULL, prepare_params);
    cpl_test_nonnull(res);
    moo_det_delete(res);

    // 5 - add bpmap data
    int badpix[2][2] = { { 5, 5 }, { 5, 6 } };
    moo_testbed_create_bpmap(10, bands, 3, 2, "bpmaprp.fits", &badpix, 2);
    res = moo_prepare(raw, "bpmaprp.fits", NULL, NULL, NULL, prepare_params);
    cpl_test_nonnull(res);
    moo_det_delete(res);

    cpl_frame_delete(raw);
    moo_prepare_params_delete(prepare_params);

    return;
}

static void
_moons_test_removeCRH()
{
    cpl_frameset *frameset = cpl_frameset_new();
    moo_prepare_params *prepare_params = moo_prepare_params_new();
    // 1 -create RAW frames
    int badpix[2][2] = { { 5, 5 }, { 5, 6 } };
    const char *bands[3] = { "RI" };
    moo_testbed_create_bpmap(10, bands, 3, 2, "bpmaprp.fits", &badpix, 2);
    int badpix2[2][2] = { { 6, 6 }, { 6, 7 } };
    moo_testbed_create_bpmap(10, bands, 3, 2, "bpmaprp2.fits", &badpix2, 2);

    moo_testbed_create_raw(10, bands, 1, 2, "bias1.fits", 1, 1, 0.9, 5, 150);
    cpl_frame *raw1 = cpl_frame_new();
    cpl_frame_set_filename(raw1, "bias1.fits");
    cpl_frame_set_tag(raw1, "BIAS");
    cpl_frame_set_group(raw1, CPL_FRAME_GROUP_RAW);

    moo_testbed_create_raw(10, bands, 1, 2, "bias2.fits", 1, 1, 0.9, 5, 250);
    cpl_frame *raw2 = cpl_frame_new();
    cpl_frame_set_filename(raw2, "bias2.fits");
    cpl_frame_set_tag(raw2, "BIAS");
    cpl_frame_set_group(raw2, CPL_FRAME_GROUP_RAW);

    moo_testbed_create_raw(10, bands, 1, 2, "bias3.fits", 1, 1, 0.9, 5, 300);
    cpl_frame *raw3 = cpl_frame_new();
    cpl_frame_set_filename(raw3, "bias3.fits");
    cpl_frame_set_tag(raw3, "BIAS");
    cpl_frame_set_group(raw3, CPL_FRAME_GROUP_RAW);

    // 2- prepare RAW frames
    moo_det *det1 =
        moo_prepare(raw1, "bpmaprp.fits", NULL, NULL, NULL, prepare_params);
    cpl_test_nonnull(det1);
    moo_det *det2 =
        moo_prepare(raw2, "bpmaprp2.fits", NULL, NULL, NULL, prepare_params);
    cpl_test_nonnull(det2);
    moo_det *det3 =
        moo_prepare(raw3, "bpmaprp2.fits", NULL, NULL, NULL, prepare_params);
    cpl_test_nonnull(det3);
    cpl_frame_delete(raw1);
    cpl_frame_delete(raw2);
    cpl_frame_delete(raw3);

    // 3- Create DET list
    moo_detlist *detlist = moo_detlist_new();
    moo_detlist_push(detlist, det1);
    moo_detlist_push(detlist, det2);
    moo_detlist_push(detlist, det3);

    // 4- call remove CRH
    moo_crh_params *params = moo_crh_params_new();
    params->method = MOO_CRH_METHOD_MEDIAN;
    moo_det *detres = moo_remove_CRH(detlist, NULL, params);

    // 5- Free memory
    moo_det_delete(detres);
    moo_crh_params_delete(params);
    moo_detlist_delete(detlist);

    moo_prepare_params_delete(prepare_params);
    cpl_frameset_delete(frameset);

    return;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of moons_dfs module
 */
/*----------------------------------------------------------------------------*/

int
main(int argc, char *argv[])
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    /*
    if ( argc == 2){
        cpl_msg_info("test","Use data SOF: %s",argv[1]);
        cpl_frameset* set = moo_testbed_frameset_load(argv[1]);
        _moons_prepare(set);
        cpl_frameset_delete(set);
    }
    else{
        _moons_test_prepare();    
        _moons_test_removeCRH();    
    } */
    _moons_test_prepare();
    return cpl_test_end(0);
}

/**@}*/
