/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>
#include "moo_badpix.h"
#include "moo_det.h"
#include "moo_spectral_format.h"
#include "moo_map.h"
#include "moo_rbn.h"
#include "moo_f2f.h"
#include "moo_sub_sky_stare.h"
#include "moo_params.h"

#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions localise test
        Testing of functions used in moo_localise
 */
/*----------------------------------------------------------------------------*/

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

cpl_frame *
create_frame(const char *name)
{
    char *filename = path_from_filename(name);

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_free(filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    return frame;
}


static void
_moo_sub_sky_stare_local_median_std(void)
{
    moo_sci *sci = NULL;
    moo_f2f *f2f = NULL;
    cpl_frame *rbn_frame = create_frame("STD_RBNSPECTRA_0_LR.fits");
    cpl_frame *target_table_frame = create_frame("STD_TARGET_TABLE_0_LR.fits");
    cpl_frame *f2f_frame = create_frame("F2F_OFFSET0_LR.fits");

    moo_sub_sky_stare_params *params = moo_sub_sky_stare_params_new();

    moo_rbn *rbn = moo_rbn_create(rbn_frame);
    moo_target_table *target_table = moo_target_table_load(target_table_frame);
    f2f = moo_f2f_load(f2f_frame);

    params->method = MOO_SUB_SKY_STARE_METHOD_SIMPLE;
    params->radius_sky = 1.5;
    params->step_r = 0.15;
    params->min_sky = 5;
    params->min_trans = 0.;
    params->maxdistslit = 20;

    cpl_msg_info("", "Use target table %s",
                 cpl_frame_get_filename(target_table_frame));
    char *resname =
        cpl_sprintf("SCI_STARE_SIMPLE_mdslit%d.fits", params->maxdistslit);
    sci = moo_sub_sky_stare(rbn, target_table, rbn, f2f, NULL, NULL, NULL,
                            params, resname, 0);

    cpl_free(resname);
    moo_sci_delete(sci);
    moo_rbn_delete(rbn);
    moo_target_table_delete(target_table);
    moo_f2f_delete(f2f);
    moo_sub_sky_stare_params_delete(params);

    cpl_frame_delete(rbn_frame);
    cpl_frame_delete(target_table_frame);
    cpl_frame_delete(f2f_frame);
}

static void
_moo_sub_sky_stare_local_median_sc(void)
{
    moo_sci *sci = NULL;
    moo_f2f *f2f = NULL;
    cpl_frame *rbn_frame =
        create_frame("STARENOD_OBJ_RBNSPECTRA_OFFSET1_HR_0.fits");
    cpl_frame *target_table_frame =
        create_frame("STARENOD_NOD_TARGET_TABLE_OFFSET1_HR_0.fits");
    cpl_frame *f2f_frame = create_frame("F2F_TABLE_OFFSET1_HR.fits");

    moo_sub_sky_stare_params *params = moo_sub_sky_stare_params_new();

    moo_rbn *rbn = moo_rbn_create(rbn_frame);
    moo_target_table *target_table = moo_target_table_load(target_table_frame);
    f2f = moo_f2f_load(f2f_frame);

    params->method = MOO_SUB_SKY_STARE_METHOD_SKYCORR;
    params->radius_sky = 150;
    params->step_r = 0.15;
    params->min_sky = 0;
    params->maxdistslit = 35;

    cpl_frame *airglow_group_frame = create_frame("airglow_groups.dat");
    cpl_frame *airglow_var_frame = create_frame("airglow_var.dat");
    cpl_frame *solflux_frame = create_frame("solflux_monthly_average.txt");

    moo_skycorr_params *sk = moo_skycorr_params_new();
    sk->ltol = 0.01;
    sk->min_line_dist_fac = 2.5;
    sk->min_line_flux_fac = 0.;
    sk->fluxlim = -1;
    sk->ftol = 1e-3;
    sk->xtol = 1e-3;
    sk->wtol = 1e-3;
    sk->cheby_max = 7;
    sk->cheby_min = 3;
    sk->cheby_const = 0;
    sk->rebintype = 1;
    sk->weightlim = 0.67;
    sk->siglim = 15.;
    sk->fitlim = 0;
    params->sk = sk;

    char *resname =
        cpl_sprintf("SCI_STARE_SKYCORR_mdslit%d.fits", params->maxdistslit);
    cpl_msg_info("", "Use target table %s",
                 cpl_frame_get_filename(target_table_frame));

    sci = moo_sub_sky_stare(rbn, target_table, rbn, f2f, solflux_frame,
                            airglow_group_frame, airglow_var_frame, params,
                            resname, 0);

    cpl_free(resname);

    moo_sci_delete(sci);
    moo_rbn_delete(rbn);
    moo_target_table_delete(target_table);
    moo_f2f_delete(f2f);
    moo_sub_sky_stare_params_delete(params);

    cpl_frame_delete(rbn_frame);
    cpl_frame_delete(target_table_frame);
    cpl_frame_delete(f2f_frame);
    cpl_frame_delete(airglow_var_frame);
    cpl_frame_delete(airglow_group_frame);
    cpl_frame_delete(solflux_frame);
}

static void
_moo_sub_sky_stare_nosky(void)
{
    moo_sci *sci = NULL;
    moo_f2f *f2f = NULL;
    cpl_frame *rbn_frame =
        create_frame("nosky/SCIENCE_RBNSPECTRA_OFFSET0_HR_0.fits");
    cpl_frame *target_table_frame =
        create_frame("nosky/SCIENCE_TARGET_TABLE_OFFSET0_HR_0.fits");
    cpl_frame *f2f_frame = create_frame("nosky/F2F_OFFSET0_HR.fits");

    moo_sub_sky_stare_params *params = moo_sub_sky_stare_params_new();

    moo_rbn *rbn = moo_rbn_create(rbn_frame);
    moo_rbn_single_delete(rbn->yj);
    rbn->yj = NULL;
    moo_rbn_single_delete(rbn->h);
    rbn->h = NULL;

    moo_target_table *target_table = moo_target_table_load(target_table_frame);
    f2f = moo_f2f_load(f2f_frame);
    params->method = MOO_SUB_SKY_STARE_METHOD_SIMPLE;
    params->radius_sky = 150;
    params->step_r = 0.15;
    params->min_sky = 0;
    params->min_trans = 0;
    params->maxdistslit = 0;

    cpl_msg_info("", "Use target table %s",
                 cpl_frame_get_filename(target_table_frame));
    sci = moo_sub_sky_stare(rbn, target_table, NULL, f2f, NULL, NULL, NULL,
                            params, "SCI_STARE.fits", 0);

    moo_sci_delete(sci);
    moo_rbn_delete(rbn);
    moo_target_table_delete(target_table);
    moo_f2f_delete(f2f);
    moo_sub_sky_stare_params_delete(params);

    cpl_frame_delete(rbn_frame);
    cpl_frame_delete(target_table_frame);
    cpl_frame_delete(f2f_frame);
}

void
_moo_sub_sky_stare_local_median_nod(void)
{
    moo_sci *sci = NULL;
    moo_f2f *f2f = NULL;
    cpl_frame *obj_rbn_frame =
        create_frame("STARENOD_OBJ_RBNSPECTRA_OFFSET0_LR_0.fits");
    cpl_frame *target_table_frame =
        create_frame("STARENOD_TARGET_TABLE_OFFSET0_LR.fits");
    cpl_frame *sky_rbn_frame =
        create_frame("STARENOD_SKY_RBNSPECTRA_OFFSET0_LR_0.fits");
    cpl_frame *f2f_frame = create_frame("F2F_OFFSET0_LR.fits");

    moo_sub_sky_stare_params *params = moo_sub_sky_stare_params_new();

    moo_rbn *obj_rbn = moo_rbn_create(obj_rbn_frame);
    moo_rbn_single_delete(obj_rbn->yj);
    obj_rbn->yj = NULL;
    moo_rbn_single_delete(obj_rbn->h);
    obj_rbn->h = NULL;

    moo_rbn *sky_rbn = moo_rbn_create(sky_rbn_frame);
    moo_rbn_single_delete(sky_rbn->yj);
    sky_rbn->yj = NULL;
    moo_rbn_single_delete(sky_rbn->h);
    sky_rbn->h = NULL;

    moo_target_table *target_table = moo_target_table_load(target_table_frame);
    f2f = moo_f2f_load(f2f_frame);

    params->method = MOO_SUB_SKY_STARE_METHOD_SIMPLE;
    params->radius_sky = 150;
    params->step_r = 0.15;
    params->min_sky = 0;
    params->min_trans = 0.;
    params->maxdistslit = 15;
    sci = moo_sub_sky_stare(obj_rbn, target_table, sky_rbn, f2f, NULL, NULL,
                            NULL, params, "SCI_NOD.fits", 0);

    moo_sci_delete(sci);
    moo_rbn_delete(obj_rbn);
    moo_rbn_delete(sky_rbn);
    moo_target_table_delete(target_table);
    moo_f2f_delete(f2f);
    moo_sub_sky_stare_params_delete(params);

    cpl_frame_delete(obj_rbn_frame);
    cpl_frame_delete(sky_rbn_frame);
    cpl_frame_delete(target_table_frame);
    cpl_frame_delete(f2f_frame);
}

void
_moo_sub_sky_stare_local_median(void)
{
    moo_sci *sci = NULL;
    moo_f2f *f2f = NULL;
    cpl_frame *obj_rbn_frame =
        create_frame("STARENOD_OBJ_RBNSPECTRA_OFFSET0_LR_0.fits");
    cpl_frame *target_table_frame =
        create_frame("STARENOD_OBJ_TARGET_TABLE_OFFSET0_LR_0.fits");
    cpl_frame *f2f_frame = create_frame("F2F_OFFSET0_LR_h.fits");

    moo_sub_sky_stare_params *params = moo_sub_sky_stare_params_new();

    moo_rbn *obj_rbn = moo_rbn_create(obj_rbn_frame);
    moo_rbn_single_delete(obj_rbn->ri);
    obj_rbn->ri = NULL;
    moo_rbn_single_delete(obj_rbn->yj);
    obj_rbn->yj = NULL;

    moo_target_table *target_table = moo_target_table_load(target_table_frame);
    f2f = moo_f2f_load(f2f_frame);

    params->method = MOO_SUB_SKY_STARE_METHOD_SIMPLE;
    params->radius_sky = 150;
    params->step_r = 0.15;
    params->min_sky = 0;
    params->min_trans = 0.;
    params->maxdistslit = 30;
    sci = moo_sub_sky_stare(obj_rbn, target_table, obj_rbn, f2f, NULL, NULL,
                            NULL, params, "SCI_STARE.fits", 0);

    moo_sci_delete(sci);
    moo_rbn_delete(obj_rbn);
    moo_target_table_delete(target_table);
    moo_f2f_delete(f2f);
    moo_sub_sky_stare_params_delete(params);

    cpl_frame_delete(obj_rbn_frame);
    cpl_frame_delete(target_table_frame);
    cpl_frame_delete(f2f_frame);
}

void
_moo_sub_sky_stare_local_median_nod_sc(void)
{
    moo_sci *sci = NULL;
    moo_f2f *f2f = NULL;
    cpl_frame *rbn_frame =
        create_frame("STARENOD_OBJ_RBNSPECTRA_OFFSET1_HR_0.fits");
    cpl_frame *target_table_frame =
        create_frame("STARENOD_TARGET_TABLE_OFFSET1_HR.fits");
    cpl_frame *sky_rbn_frame =
        create_frame("STARENOD_SKY_RBNSPECTRA_OFFSET1_HR_0.fits");
    cpl_frame *f2f_frame = create_frame("F2F_TABLE_OFFSET1_HR.fits");
    cpl_frame *solflux_frame = create_frame("solflux_monthly_average.txt");
    cpl_frame *airglow_group_frame = create_frame("airglow_groups.dat");
    cpl_frame *airglow_var_frame = create_frame("airglow_var.dat");
    moo_sub_sky_stare_params *params = moo_sub_sky_stare_params_new();

    moo_skycorr_params *sk = moo_skycorr_params_new();
    sk->ltol = 0.01;
    sk->min_line_dist_fac = 2.5;
    sk->min_line_flux_fac = 0.;
    sk->fluxlim = -1;
    sk->ftol = 1e-3;
    sk->xtol = 1e-3;
    sk->wtol = 1e-3;
    sk->cheby_max = 7;
    sk->cheby_min = 3;
    sk->cheby_const = 0;
    sk->rebintype = 1;
    sk->weightlim = 0.97;
    sk->siglim = 15.;
    sk->fitlim = 0;

    moo_rbn *rbn = moo_rbn_create(rbn_frame);
    moo_rbn_single_delete(rbn->yj);
    rbn->yj = NULL;
    moo_rbn_single_delete(rbn->h);
    rbn->h = NULL;

    moo_rbn *sky_rbn = moo_rbn_create(sky_rbn_frame);
    moo_rbn_single_delete(sky_rbn->yj);
    sky_rbn->yj = NULL;
    moo_rbn_single_delete(sky_rbn->h);
    sky_rbn->h = NULL;

    moo_target_table *target_table = moo_target_table_load(target_table_frame);
    f2f = moo_f2f_load(f2f_frame);

    params->method = MOO_SUB_SKY_STARE_METHOD_SKYCORR;
    params->radius_sky = 1;
    params->step_r = 0.1;
    params->min_sky = 3;
    params->min_trans = 0.5;
    params->maxdistslit = 10;
    params->sk = sk;

    sci = moo_sub_sky_stare(rbn, target_table, sky_rbn, f2f, solflux_frame,
                            airglow_group_frame, airglow_var_frame, params,
                            "SCI_NOD_SC.fits", 0);

    moo_sci_delete(sci);
    moo_rbn_delete(rbn);
    moo_rbn_delete(sky_rbn);
    moo_target_table_delete(target_table);
    moo_f2f_delete(f2f);
    moo_sub_sky_stare_params_delete(params);

    cpl_frame_delete(rbn_frame);
    cpl_frame_delete(sky_rbn_frame);
    cpl_frame_delete(target_table_frame);
    cpl_frame_delete(f2f_frame);
    cpl_frame_delete(airglow_var_frame);
    cpl_frame_delete(airglow_group_frame);
    cpl_frame_delete(solflux_frame);
}

void
_moo_sub_sky_stare_local_median2(void)
{
    moo_sci *sci = NULL;
    moo_f2f *f2f = NULL;
    cpl_frame *rbn_frame = create_frame("SCIENCE_RBNSPECTRA_0.fits");
    cpl_frame *target_table_frame = create_frame("SCIENCE_TARGET_TABLE_0.fits");
    cpl_frame *f2f_frame = create_frame("F2F_OFFSET0_LR.fits");


    moo_sub_sky_stare_params *params = moo_sub_sky_stare_params_new();

    moo_rbn *rbn = moo_rbn_create(rbn_frame);
    moo_rbn_single_delete(rbn->yj);
    rbn->yj = NULL;
    moo_rbn_single_delete(rbn->h);
    rbn->h = NULL;

    moo_target_table *target_table = moo_target_table_load(target_table_frame);

    f2f = moo_f2f_load(f2f_frame);

    params->method = MOO_SUB_SKY_STARE_METHOD_SIMPLE;
    params->radius_sky = 1.5;
    params->step_r = 0.15;
    params->min_sky = 5;
    params->maxdistslit = 80;

    sci = moo_sub_sky_stare(rbn, target_table, NULL, f2f, NULL, NULL, NULL,
                            params, "SCI.fits", 0);
    moo_sci_delete(sci);

    moo_rbn_delete(rbn);
    moo_target_table_delete(target_table);
    moo_f2f_delete(f2f);
    moo_sub_sky_stare_params_delete(params);

    cpl_frame_delete(rbn_frame);
    cpl_frame_delete(target_table_frame);
    cpl_frame_delete(f2f_frame);
}

void
_moo_sub_sky_stare_local_median_xswitch(void)
{
    moo_sci *sciAB = NULL;
    moo_sci *sciBA = NULL;
    moo_f2f *f2f = NULL;
    cpl_frame *rbn_frameA =
        create_frame("XSWITCH_OBJECTA_RBNSPECTRA_OFFSET1_HR_0.fits");
    cpl_frame *rbn_frameB =
        create_frame("XSWITCH_OBJECTB_RBNSPECTRA_OFFSET1_HR_0.fits");
    cpl_frame *target_table_frame =
        create_frame("XSWITCH_TARGET_TABLE_OFFSET1_HR.fits");
    cpl_frame *f2f_frame = create_frame("F2F_TABLE_OFFSET1_HR.fits");


    moo_sub_sky_stare_params *params = moo_sub_sky_stare_params_new();

    moo_rbn *rbnA = moo_rbn_create(rbn_frameA);
    moo_rbn *rbnB = moo_rbn_create(rbn_frameB);

    moo_target_table *target_table = moo_target_table_load(target_table_frame);

    f2f = moo_f2f_load(f2f_frame);

    params->method = MOO_SUB_SKY_STARE_METHOD_SIMPLE;
    params->radius_sky = 180;
    params->step_r = 0.15;
    params->min_sky = 5;
    params->maxdistslit = 20;

    sciAB = moo_sub_sky_stare(rbnA, target_table, rbnB, f2f, NULL, NULL, NULL,
                              params, "SCIAB.fits", 0);
    sciBA = moo_sub_sky_stare(rbnA, target_table, rbnB, f2f, NULL, NULL, NULL,
                              params, "SCIBA.fits", 1);
    moo_sci_delete(sciAB);
    moo_sci_delete(sciBA);
    moo_rbn_delete(rbnA);
    moo_rbn_delete(rbnB);
    moo_target_table_delete(target_table);
    moo_f2f_delete(f2f);
    moo_sub_sky_stare_params_delete(params);

    cpl_frame_delete(rbn_frameA);
    cpl_frame_delete(rbn_frameB);
    cpl_frame_delete(target_table_frame);
    cpl_frame_delete(f2f_frame);
}

void
_moo_sub_sky_stare_qlook(void)
{
    moo_sci *sci = NULL;
    moo_f2f *f2f = NULL;
    cpl_frame *rbn_frame = create_frame("QLOOK_RBNSPECTRA.fits");
    cpl_frame *target_table_frame = create_frame("QLOOK_TARGET_TABLE.fits");
    cpl_frame *f2f_frame = create_frame("F2F_OFFSET0_HR.fits");


    moo_sub_sky_stare_params *params = moo_sub_sky_stare_params_new();

    moo_rbn *rbn = moo_rbn_create(rbn_frame);
    /*moo_rbn_single_delete(rbn->yj);
    rbn->yj = NULL;
    moo_rbn_single_delete(rbn->h);
    rbn->h = NULL;
    */
    moo_target_table *target_table = moo_target_table_load(target_table_frame);

    f2f = moo_f2f_load(f2f_frame);

    params->method = MOO_SUB_SKY_STARE_METHOD_SIMPLE;
    params->radius_sky = 150;
    params->step_r = 0.15;
    params->min_sky = 0;
    params->maxdistslit = 15;
    params->min_trans = 0;

    sci = moo_sub_sky_stare(rbn, target_table, rbn, f2f, NULL, NULL, NULL,
                            params, "QLOOK_SCI.fits", 0);

    moo_sci_delete(sci);
    moo_rbn_delete(rbn);
    moo_target_table_delete(target_table);
    moo_f2f_delete(f2f);
    moo_sub_sky_stare_params_delete(params);

    cpl_frame_delete(rbn_frame);
    cpl_frame_delete(target_table_frame);
    cpl_frame_delete(f2f_frame);
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of hdrl_image module
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    //_moo_sub_sky_stare_local_median_std();
    //_moo_sub_sky_stare_qlook();
    _moo_sub_sky_stare_local_median_sc();
    //_moo_sub_sky_stare_nosky();
    //_moo_sub_sky_stare_local_median();
    //_moo_sub_sky_stare_local_median_xswitch();
    return cpl_test_end(0);
}
