/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "moo_utils.h"
#include "moo_pfits.h"
#include "moo_dfs.h"
#include "moo_params.h"
#include "moo_drl.h"
#include "moo_badpix.h"
#include "moo_extlist.h"
#include "moo_subtract_nod.h"
#include "moo_combine_pair.h"
#include "moo_products.h"
#include <cpl.h>

#include <string.h>

/*-----------------------------------------------------------------------------
                              Plugin registration
 -----------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list);

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

static int _moons_science_xswitch_create(cpl_plugin *plugin);
static int _moons_science_xswitch_exec(cpl_plugin *plugin);
static int _moons_science_xswitch_destroy(cpl_plugin *plugin);
static int _moons_science_xswitch(cpl_frameset *frameset,
                                  const cpl_parameterlist *parlist);

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/
static const char *const _moons_science_xswitch_description =
    "From an observation sequence in XSwitch mode, this recipe removes "
    "the instrumental signature and provides calibrated science products. "
    "During the sequence, the telescope nods from one position on the sky "
    "(labeled as A in Fig. 5-11) to another (B). In the sequence, different "
    "slit"
    " offset positions can be used: as represented in Fig. 5-11, the "
    "combination "
    "is done with 1D wavelength calibrated sky subtracted spectra. Using "
    "different"
    " slit offset positions implies that different sets of calibration "
    "(FF, wave maps) are given as inputs.\n"
    "INPUT FRAMES\n"
    "  * science exposure frames n>=1 file(s)               (RAW) with tag "
    "OBJECT_XSWITCH : "
    "science frame in xswitch mode file\n"
    "  * [OPTIONAL] ReferenceBadPixMask 1 file              (QUA) with tag "
    "BP_MAP_RP : "
    "cosmetic bad pixel map\n"
    "  * [OPTIONAL] NonLinearityBadPixMask 1 file           (QUA) with tag "
    "BP_MAP_NL : "
    "cosmetic bad pixel map coming from linearity recipe\n"
    "  * MasterBias 1 file                                  (DET) with tag "
    "MASTER_BIAS : "
    "master bias file\n"
    "  * MasterDarkNir 1 file                               (DET) with tag "
    "MASTER_DARK_NIR : "
    "master dark nir for object\n"
    "  * [OPTIONAL] MasterDarkNir 1 file                    (DET) with tag "
    "MASTER_DARK_NIR : "
    "master dark nir file for sky (if exposure time is different from object)\n"
    "  * [OPTIONAL] MasterDarkVis 1 file                    (DET) with tag "
    "MASTER_DARK_VIS : "
    "master dark vis file\n"
    "  * [OPTIONAL] CoeffsCube 1 or 2 file(s) one by offset (3D) with tag "
    "LINEARITY_COEFF_CUBE : "
    "coefficients to correct pixels detector linearity\n"
    "  * [OPTIONAL] P2pMap 1 file                           (DET) with tag "
    "P2P_MAP : "
    "pixel to pixel map\n"
    "  * Fftrace 1 or 2 file(s) one by offset               (LOC) with tag "
    "FF_TRACE :"
    "the localisation table\n"
    "  * [OPTIONAL] MasterFlat 1 or 2 file(s) one by offset (PSF) with tag "
    "MASTER_FLAT : "
    "the master flat used for optimal extraction\n"
    "  * Ffext  1 or 2 file(s) one by offset                (EXT) with tag "
    "FF_EXTSPECTRA : "
    "the extracted flat field\n"
    "  * F2f [OPTIONAL] 1 or 2 file(s) one by offset        (F2F) with tag "
    "F2F_TABLE :"
    "the fibre-to-fibre relative response\n"
    "  * SFormat 1 file                                     (FMT) with tag "
    "SPECTRAL_FORMAT : "
    "the spectral format table\n"
    "  * WaveMap 1 or 2 file(s) one by offset               (WMAP) with tag "
    "WAVE_MAP : "
    "the wavelength map\n"
    "  * SkyLines 1 file                                    (CAT) with tag "
    "SKY_LINE_LIST : "
    "the sky lines list table\n"
    "  * [OPTIONAL] response 1 file                         (RESPONSE) with "
    "tag RESPONSE : "
    "the response spectra in the extracted pixel space\n"
    "  * [OPTIONAL] atmosExt 1 file                         (ATMO) with tag "
    "ATMOS_EXT : "
    "the atmospheric extinction table\n"
    "PRODUCTS\n"
    "  * SCIENCE_XSWITCH_RBNSPECTRA_OFFSET[offset]_[insmode]_i.fits (RBN) with "
    "tag "
    "xswitch_RBNSPECTRA : "
    "Science product individual calibrated frames\n"
    "  * SCIENCE_XSWITCH_SKSSPECTRA_OFFSET[offset]_[insmode]_i.fits (SCI) with "
    "tag "
    "xswitch_SKSSPECTRA : "
    "Science product individual sky subtracted frames\n"
    "  * SCIENCE_XSWITCH_SKSSPECTRA_[insmode].fits                  (SCI) with "
    "tag "
    "xswitch_SKSSPECTRA : "
    "Science product coadd calibrated frame\n"
    "  * [OPTIONAL] SCIENCE_XSWITCH_FLXSPECTRA_[insmode].fits       (SCI) with "
    "tag "
    "xswitch_FLXSPECTRA : "
    "Science product flux calibrated frame\n"
    "\n";

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok, 1 otherwise
  @note     Only this function is exported

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*----------------------------------------------------------------------------*/

int
cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
    cpl_plugin *plugin = &recipe->interface;

    if (cpl_plugin_init(plugin, CPL_PLUGIN_API, MOONS_BINARY_VERSION,
                        CPL_PLUGIN_TYPE_RECIPE, "moons_science_xswitch",
                        "Reduces science exposure in xswitch mode",
                        _moons_science_xswitch_description, "Regis Haigron",
                        PACKAGE_BUGREPORT, moo_get_license(),
                        _moons_science_xswitch_create,
                        _moons_science_xswitch_exec,
                        _moons_science_xswitch_destroy)) {
        cpl_msg_error(cpl_func, "Plugin initialization failed");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    if (cpl_pluginlist_append(list, plugin)) {
        cpl_msg_error(cpl_func, "Error adding plugin to list");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/

static int
_moons_science_xswitch_create(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    /* Do not create the recipe if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Create the parameters list in the cpl_recipe object */
    recipe->parameters = cpl_parameterlist_new();
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Parameter list allocation failed");
        cpl_ensure_code(0, (int)CPL_ERROR_ILLEGAL_OUTPUT);
    }

    moo_params *params = moo_params_new("moons", "moons_science_xswitch");

    /* Fill the parameters list */
    moo_params_add_keep_temp(params, recipe->parameters);
    moo_params_add_prepare(params, recipe->parameters);
    moo_params_add_correct_bias(params, recipe->parameters,
                                MOO_CORRECT_BIAS_METHOD_MASTER);
    moo_params_add_extract(params, recipe->parameters);
    moo_params_add_science_wavesol(params, recipe->parameters);
    moo_params_add_rebin(params, recipe->parameters);
    moo_params_add_target_table(params, recipe->parameters, CPL_FALSE);
    moo_params_add_combine_pair(params, recipe->parameters, CPL_TRUE);
    moo_params_add_sub_sky_stare(params, recipe->parameters);
    moo_params_add_coadd(params, recipe->parameters);
    moo_params_add_create_s1d(params, recipe->parameters);
    moo_params_delete(params);

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_science_xswitch_exec(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    int recipe_status;
    cpl_errorstate initial_errorstate = cpl_errorstate_get();

    /* Return immediately if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Verify parameter and frame lists */
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL parameter list");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }
    if (recipe->frames == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL frame set");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Invoke the recipe */
    recipe_status = _moons_science_xswitch(recipe->frames, recipe->parameters);

    /* Ensure DFS-compliance of the products */
    if (cpl_dfs_update_product_header(recipe->frames)) {
        if (!recipe_status)
            recipe_status = (int)cpl_error_get_code();
    }

    if (!cpl_errorstate_is_equal(initial_errorstate)) {
        /* Dump the error history since recipe execution start.
           At this point the recipe cannot recover from the error */
        cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
    }

    return recipe_status;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_science_xswitch_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    cpl_parameterlist_delete(recipe->parameters);

    return 0;
}

static cpl_error_code
_moons_science_xswitch_check_sof(cpl_frameset *frameset,
                                 cpl_frameset *offset_frameset,
                                 const char **bpmap_rp_name,
                                 const char **bpmap_nl_name,
                                 const cpl_frame **masterbias,
                                 const cpl_frame **masterdark_vis,
                                 cpl_frameset *masterdark_nir,
                                 const cpl_frame **p2pmap,
                                 const cpl_frame **sformat,
                                 const cpl_frame **sky_lines,
                                 const cpl_frame **atmo,
                                 const cpl_frame **resp,
                                 const cpl_frame **airglow_group,
                                 const cpl_frame **airglow_var,
                                 const cpl_frame **solflux,
                                 const moo_sub_sky_stare_params *subsky_params)
{
    cpl_ensure_code(moo_dfs_set_groups(frameset) == CPL_ERROR_NONE,
                    cpl_error_get_code());
    int i;

    for (i = 0; i < cpl_frameset_get_size(frameset); ++i) {
        cpl_frame *current_frame = cpl_frameset_get_position(frameset, i);
        if (!strcmp(cpl_frame_get_tag(current_frame),
                    MOONS_TAG_OBJECT_XSWITCH)) {
            cpl_frameset_insert(offset_frameset,
                                cpl_frame_duplicate(current_frame));
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_BP_MAP_RP)) {
            *bpmap_rp_name = cpl_frame_get_filename(current_frame);
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_BP_MAP_NL)) {
            *bpmap_nl_name = cpl_frame_get_filename(current_frame);
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_BIAS)) {
            *masterbias = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_DARK_VIS)) {
            *masterdark_vis = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_DARK_NIR)) {
            cpl_frameset_insert(masterdark_nir,
                                cpl_frame_duplicate(current_frame));
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_FLAT)) {
            cpl_frameset_insert(offset_frameset,
                                cpl_frame_duplicate(current_frame));
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_P2P_MAP)) {
            *p2pmap = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FF_TRACE)) {
            cpl_frameset_insert(offset_frameset,
                                cpl_frame_duplicate(current_frame));
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_LINEARITY_COEFF_CUBE)) {
            cpl_frameset_insert(offset_frameset,
                                cpl_frame_duplicate(current_frame));
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_SPECTRAL_FORMAT)) {
            *sformat = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_WAVEMAP)) {
            cpl_frameset_insert(offset_frameset,
                                cpl_frame_duplicate(current_frame));
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_SKY_LINE_LIST)) {
            *sky_lines = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FF_EXTSPECTRA)) {
            cpl_frameset_insert(offset_frameset,
                                cpl_frame_duplicate(current_frame));
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_F2F_TABLE)) {
            cpl_frameset_insert(offset_frameset,
                                cpl_frame_duplicate(current_frame));
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_ATMOS_EXT)) {
            *atmo = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_RESPONSE)) {
            *resp = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_AIRGLOW_GROUP)) {
            *airglow_group = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_AIRGLOW_VAR)) {
            *airglow_var = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_SOLFLUX)) {
            *solflux = current_frame;
        }
    }

    if (*sformat == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_SPECTRAL_FORMAT);
    }

    if (*sky_lines == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_SKY_LINE_LIST);
    }

    if (*airglow_group == NULL &&
        strcmp(subsky_params->method, MOO_SUB_SKY_STARE_METHOD_SKYCORR) == 0) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_AIRGLOW_GROUP);
    }
    if (*airglow_var == NULL &&
        strcmp(subsky_params->method, MOO_SUB_SKY_STARE_METHOD_SKYCORR) == 0) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_AIRGLOW_VAR);
    }
    if (*solflux == NULL &&
        strcmp(subsky_params->method, MOO_SUB_SKY_STARE_METHOD_SKYCORR) == 0) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_SOLFLUX);
    }
    return CPL_ERROR_NONE;
}

static cpl_error_code
_moons_science_xswitch_check_sof_byoffset(cpl_frameset *frameset,
                                          cpl_frameset *node_frameset,
                                          const cpl_frame **master_flat,
                                          const cpl_frame **fftrace,
                                          const cpl_frame **coeffs_cube,
                                          const cpl_frame **wmap,
                                          const cpl_frame **flat,
                                          const cpl_frame **f2f,
                                          moo_extract_params *ext_params)
{
    cpl_ensure_code(moo_dfs_set_groups(frameset) == CPL_ERROR_NONE,
                    cpl_error_get_code());
    int nb_object = 0;
    int i;

    for (i = 0; i < cpl_frameset_get_size(frameset); ++i) {
        cpl_frame *current_frame = cpl_frameset_get_position(frameset, i);
        if (!strcmp(cpl_frame_get_tag(current_frame),
                    MOONS_TAG_OBJECT_XSWITCH)) {
            cpl_frameset_insert(node_frameset,
                                cpl_frame_duplicate(current_frame));
            nb_object++;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_FLAT)) {
            *master_flat = current_frame;
        }

        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FF_TRACE)) {
            *fftrace = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_LINEARITY_COEFF_CUBE)) {
            *coeffs_cube = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_WAVEMAP)) {
            *wmap = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FF_EXTSPECTRA)) {
            *flat = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_F2F_TABLE)) {
            *f2f = current_frame;
        }
    }


    if (nb_object <= 0) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_OBJECT_XSWITCH);
    }

    if (*fftrace == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_FF_TRACE);
    }

    if (*wmap == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_WAVEMAP);
    }
    if (*flat == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_FF_EXTSPECTRA);
    }
    if (*master_flat == NULL &&
        strcmp(ext_params->method, MOO_EXTRACT_METHOD_OPTIMAL) == 0) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_MASTER_FLAT);
    }
    return CPL_ERROR_NONE;
}


static cpl_error_code
_moons_p2pmap(moo_det *arc, const cpl_frame *p2pmap)
{
    cpl_error_code status = CPL_ERROR_NONE;
    moo_det *p2p = NULL;

    if (p2pmap != NULL) {
        moo_try_check(p2p = moo_det_create(p2pmap), " ");
        moo_try_check(moo_apply_p2p(arc, p2p), " ");
    }
moo_try_cleanup:
    moo_det_delete(p2p);
    return status;
}

static cpl_error_code
_moons_apply_flat(moo_ext *ext,
                  const cpl_frame *flat_frame,
                  const cpl_frame *f2f_frame)
{
    cpl_error_code status = CPL_ERROR_NONE;
    moo_ext *flat = NULL;
    moo_f2f *f2f = NULL;

    moo_try_check(flat = moo_ext_create(flat_frame), " ");
    if (f2f_frame != NULL) {
        f2f = moo_f2f_load(f2f_frame);
    }
    moo_try_check(moo_apply_flat(ext, flat, f2f), " ");

moo_try_cleanup:
    moo_ext_delete(flat);
    moo_f2f_delete(f2f);
    return status;
}

static moo_sci *
_moons_combine_pair(moo_rbn *rbn,
                    moo_rbn *sky_rbn,
                    moo_rbn *obj_rbn,
                    moo_target_table *target_table,
                    const cpl_frame *f2f_frame,
                    const cpl_frame *solflux_frame,
                    const cpl_frame *airglow_group_frame,
                    const cpl_frame *airglow_var_frame,
                    moo_combine_pair_params *combine_pair_params,
                    moo_sub_sky_stare_params *sky_params,
                    const char *sci_filename,
                    moo_products *products,
                    const cpl_frame *ref_frame)
{
    moo_sci *result = NULL;

    moo_f2f *f2f = NULL;

    if (f2f_frame != NULL) {
        f2f = moo_f2f_load(f2f_frame);
    }
    moo_try_check(result =
                      moo_combine_pair(rbn, sky_rbn, obj_rbn, target_table, f2f,
                                       solflux_frame, airglow_group_frame,
                                       airglow_var_frame, combine_pair_params,
                                       sky_params, sci_filename, products,
                                       ref_frame),
                  " ");

moo_try_cleanup:
    moo_f2f_delete(f2f);
    return result;
}


static cpl_frame *
_moons_science_xswitch_frame(int inum,
                             const char *mode,
                             int offset,
                             moo_ext *objA_ext,
                             moo_ext *objB_ext,
                             moo_ext *nod_ext,
                             const cpl_frame *objA_ref_frame,
                             const cpl_frame *objB_ref_frame,
                             moo_spectral_format *sformat,
                             moo_sky_lines_list *sky_lines,
                             moo_map *wmap,
                             const cpl_frame *f2f_frame,
                             const cpl_frame *airglow_group_frame,
                             const cpl_frame *airglow_var_frame,
                             const cpl_frame *solflux_frame,
                             const cpl_frame *atmo_frame,
                             const cpl_frame *resp_frame,
                             moo_rebin_params *rbn_params,
                             moo_target_table_params *target_table_params,
                             moo_combine_pair_params *combine_pair_params,
                             moo_sub_sky_stare_params *sky_params,
                             moo_products *products)
{
    cpl_frame *sci_frame = NULL;
    moo_rbn *nod_rbn = NULL;
    moo_rbn *objB_rbn = NULL;
    moo_rbn *objA_rbn = NULL;
    moo_atm *atm = NULL;
    moo_resp *resp = NULL;

    moo_target_table *target_table = NULL;
    moo_sci *sci = NULL;

    char *nod_rbn_filename = NULL;
    char *objB_rbn_filename = NULL;
    char *objA_rbn_filename = NULL;
    char *nod_calibflux_filename = NULL;
    char *objB_calibflux_filename = NULL;
    char *objA_calibflux_filename = NULL;

    char *ttable_filename = NULL;
    char *sci_filename = NULL;

    cpl_ensure(nod_ext != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(objA_ext != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(objB_ext != NULL, CPL_ERROR_NULL_INPUT, NULL);

    nod_rbn_filename =
        cpl_sprintf("%s_NOD_OFFSET%d_%s_%d.fits",
                    MOONS_TAG_SCIENCE_XSWITCH_RBNSPECTRA, offset, mode, inum);
    moo_try_check(nod_rbn = moo_rebin(nod_ext, wmap, sformat, rbn_params,
                                      nod_rbn_filename),
                  " ");
    moo_try_check(moo_rbn_compute_snr(nod_rbn, sky_lines), " ");
    moo_try_check(moo_products_add_rbn(products, nod_rbn, CPL_FRAME_LEVEL_FINAL,
                                       MOONS_TAG_SCIENCE_XSWITCH_RBNSPECTRA,
                                       nod_rbn_filename, objA_ref_frame),
                  " ");

    objB_rbn_filename =
        cpl_sprintf("%s_B_OFFSET%d_%s_%d.fits",
                    MOONS_TAG_SCIENCE_XSWITCH_RBNSPECTRA, offset, mode, inum);
    moo_try_check(objB_rbn = moo_rebin(objB_ext, wmap, sformat, rbn_params,
                                       objB_rbn_filename),
                  " ");
    moo_try_check(moo_rbn_compute_snr(objB_rbn, sky_lines), " ");
    moo_try_check(moo_products_add_rbn(products, objB_rbn,
                                       CPL_FRAME_LEVEL_INTERMEDIATE,
                                       MOONS_TAG_SCIENCE_XSWITCH_RBNSPECTRA,
                                       objB_rbn_filename, objB_ref_frame),
                  " ");
    objA_rbn_filename =
        cpl_sprintf("%s_A_OFFSET%d_%s_%d.fits",
                    MOONS_TAG_SCIENCE_XSWITCH_RBNSPECTRA, offset, mode, inum);
    moo_try_check(objA_rbn = moo_rebin(objA_ext, wmap, sformat, rbn_params,
                                       objA_rbn_filename),
                  " ");
    moo_try_check(moo_rbn_compute_snr(objA_rbn, sky_lines), " ");
    moo_try_check(moo_products_add_rbn(products, objA_rbn,
                                       CPL_FRAME_LEVEL_INTERMEDIATE,
                                       MOONS_TAG_SCIENCE_XSWITCH_RBNSPECTRA,
                                       objA_rbn_filename, objA_ref_frame),
                  " ");

    if (atmo_frame != NULL && resp_frame != NULL) {
        moo_try_check(atm = moo_atm_load(atmo_frame), " ");
        moo_try_check(resp = moo_resp_load(resp_frame), " ");

        nod_calibflux_filename =
            cpl_sprintf("%s_NOD_OFFSET%d_%s_%d.fits",
                        MOONS_TAG_SCIENCE_XSWITCH_RBNFLXSPECTRA, offset, mode,
                        inum);
        moo_try_check(moo_calib_flux(nod_rbn, atm, resp), " ");
        moo_try_check(
            moo_products_add_rbn(products, nod_rbn, CPL_FRAME_LEVEL_FINAL,
                                 MOONS_TAG_SCIENCE_XSWITCH_RBNFLXSPECTRA,
                                 nod_calibflux_filename, objA_ref_frame),
            " ");

        objA_calibflux_filename =
            cpl_sprintf("%s_A_OFFSET%d_%s_%d.fits",
                        MOONS_TAG_SCIENCE_XSWITCH_RBNFLXSPECTRA, offset, mode,
                        inum);
        moo_try_check(moo_calib_flux(objA_rbn, atm, resp), " ");
        moo_try_check(moo_products_add_rbn(
                          products, objA_rbn, CPL_FRAME_LEVEL_INTERMEDIATE,
                          MOONS_TAG_SCIENCE_XSWITCH_RBNFLXSPECTRA,
                          objA_calibflux_filename, objA_ref_frame),
                      " ");

        objB_calibflux_filename =
            cpl_sprintf("%s_B_OFFSET%d_%s_%d.fits",
                        MOONS_TAG_SCIENCE_XSWITCH_RBNFLXSPECTRA, offset, mode,
                        inum);
        moo_try_check(moo_calib_flux(objB_rbn, atm, resp), " ");
        moo_try_check(moo_products_add_rbn(
                          products, objB_rbn, CPL_FRAME_LEVEL_INTERMEDIATE,
                          MOONS_TAG_SCIENCE_XSWITCH_RBNFLXSPECTRA,
                          objB_calibflux_filename, objB_ref_frame),
                      " ");
    }
    else {
        cpl_msg_info(__func__, "No flux calibration");
    }
    moo_try_check(target_table = moo_create_target_table(objA_rbn, objB_rbn,
                                                         MOO_MODE_XSWITCH,
                                                         target_table_params),
                  " ");
    ttable_filename =
        cpl_sprintf("%s_OFFSET%d_%s_%d.fits", MOONS_TAG_XSWITCH_TARGET_TABLE,
                    offset, mode, inum);
    moo_try_check(moo_products_add_target_table(products, target_table,
                                                CPL_FRAME_LEVEL_INTERMEDIATE,
                                                MOONS_TAG_XSWITCH_TARGET_TABLE,
                                                ttable_filename,
                                                objA_ref_frame),
                  " ");

    sci_filename =
        cpl_sprintf("%s_OFFSET%d_%s_%d.fits",
                    MOONS_TAG_SCIENCE_XSWITCH_SKSSPECTRA, offset, mode, inum);

    moo_try_check(sci = _moons_combine_pair(nod_rbn, objB_rbn, objA_rbn,
                                            target_table, f2f_frame,
                                            solflux_frame, airglow_group_frame,
                                            airglow_var_frame,
                                            combine_pair_params, sky_params,
                                            sci_filename, products,
                                            objA_ref_frame),
                  " ");

    moo_try_check(moo_sci_compute_snr(sci, sky_lines), " ");

    moo_try_check(sci_frame = cpl_frame_duplicate(
                      moo_products_add_sci(products, sci, CPL_FRAME_LEVEL_FINAL,
                                           MOONS_TAG_SCIENCE_XSWITCH_SKSSPECTRA,
                                           sci_filename, objA_ref_frame)),
                  " ");

moo_try_cleanup:
    cpl_free(objB_calibflux_filename);
    cpl_free(nod_calibflux_filename);
    cpl_free(objA_calibflux_filename);
    cpl_free(objB_rbn_filename);
    cpl_free(nod_rbn_filename);
    cpl_free(objA_rbn_filename);
    cpl_free(ttable_filename);
    cpl_free(sci_filename);

    moo_atm_delete(atm);
    moo_resp_delete(resp);
    moo_target_table_delete(target_table);
    moo_sci_delete(sci);
    moo_rbn_delete(nod_rbn);
    moo_rbn_delete(objB_rbn);
    moo_rbn_delete(objA_rbn);

    return sci_frame;
}

static const cpl_frame *
find_masterdark_nir(moo_det *det, cpl_frameset *set)
{
    const cpl_frame *result = NULL;
    double dit = 0.0;
    int ndit = 0;
    double exptime = 0.0;
    int num;
    int type;
    int find = 0;
    moo_det *current_det = NULL;

    cpl_ensure(det != NULL, CPL_ERROR_NULL_INPUT, NULL);

    for (num = 1; num <= 2; num++) {
        for (type = 1; type <= 2; type++) {
            moo_single *single = moo_det_get_single(det, type, num);
            if (single != NULL) {
                moo_try_check(ndit = moo_pfits_get_ndit(single->header), " ");
                moo_try_check(dit = moo_pfits_get_dit(single->header), " ");
                exptime = dit * ndit;
                find = 1;
                break;
            }
        }
        if (find) {
            break;
        }
    }

    if (find && set != NULL) {
        int size = cpl_frameset_get_size(set);
        for (int i = 0; i < size; i++) {
            const cpl_frame *current = cpl_frameset_get_position_const(set, i);
            current_det = moo_det_create(current);
            moo_single *current_single =
                moo_det_get_single(current_det, type, num);
            moo_try_check(ndit = moo_pfits_get_ndit(current_single->header),
                          " ");
            moo_try_check(dit = moo_pfits_get_dit(current_single->header), " ");
            double current_exptime = dit * ndit;
            if (fabs(exptime - current_exptime) <= 1E-9) {
                result = current;
                break;
            }
            moo_det_delete(current_det);
            current_det = NULL;
        }
    }
moo_try_cleanup:
    moo_det_delete(current_det);
    return result;
}
static cpl_error_code
_moons_xswitch_extract(int inum,
                       const char *mode,
                       int offset,
                       const cpl_frame *objectA_frame,
                       const cpl_frame *objectB_frame,
                       const char *bpmap_rp_name,
                       const char *bpmap_nl_name,
                       const cpl_frame *masterbias,
                       const cpl_frame *masterdark_vis,
                       cpl_frameset *masterdark_nir_frameset,
                       const cpl_frame *master_flat,
                       const cpl_frame *p2pmap,
                       moo_loc *loc,
                       moo_map *wmap,
                       moo_spectral_format *sformat,
                       moo_sky_lines_list *skylines,
                       const cpl_frame *coeffs_cube,
                       const cpl_frame *flat_frame,
                       const cpl_frame *f2f_frame,
                       moo_extract_params *extract_params,
                       moo_prepare_params *prepare_params,
                       moo_correct_bias_params *cbias_params,
                       moo_products *products,
                       cpl_frameset *objA_ext_frameset,
                       cpl_frameset *objB_ext_frameset,
                       cpl_frameset *nod_ext_frameset)
{
    cpl_frame *nod_ext_frame = NULL;
    cpl_frame *objB_ext_frame = NULL;
    cpl_frame *objA_ext_frame = NULL;
    const cpl_frame *masterdark_nir = NULL;
    moo_det *objA_det = NULL;
    moo_det *objB_det = NULL;
    moo_det *nod_det = NULL;
    moo_psf *psf = NULL;
    moo_ext *nod_ext = NULL;
    moo_ext *objB_ext = NULL;
    moo_ext *objA_ext = NULL;

    char *nod_det_filename = NULL;
    char *nod_det_p2p_filename = NULL;
    char *nod_ext_filename = NULL;

    char *objA_det_filename = NULL;
    char *objA_det_p2p_filename = NULL;
    char *objA_ext_filename = NULL;

    char *objB_det_filename = NULL;
    char *objB_det_p2p_filename = NULL;
    char *objB_ext_filename = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_ensure_code(objectA_frame != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(objectB_frame != NULL, CPL_ERROR_NULL_INPUT);

    moo_try_check(objA_det =
                      moo_prepare(objectA_frame, bpmap_rp_name, bpmap_nl_name,
                                  masterbias, coeffs_cube, prepare_params),
                  " ");
    moo_try_check(moo_correct_bias(objA_det, masterbias, cbias_params), " ");
    masterdark_nir = find_masterdark_nir(objA_det, masterdark_nir_frameset);
    moo_try_check(moo_correct_dark(objA_det, NULL, masterdark_vis,
                                   masterdark_nir),
                  " ");
    objA_det_filename =
        cpl_sprintf("%s_A_OFFSET%d_%s_%d.fits", MOONS_TAG_XSWITCH_CORRECTDARK,
                    offset, mode, inum);
    moo_try_check(moo_products_add(products, objA_det,
                                   CPL_FRAME_LEVEL_INTERMEDIATE,
                                   MOONS_TAG_XSWITCH_CORRECTDARK,
                                   objA_det_filename, objectA_frame),
                  " ");

    moo_try_check(objB_det =
                      moo_prepare(objectB_frame, bpmap_rp_name, bpmap_nl_name,
                                  masterbias, coeffs_cube, prepare_params),
                  " ");
    moo_try_check(moo_correct_bias(objB_det, masterbias, cbias_params), " ");
    masterdark_nir = find_masterdark_nir(objB_det, masterdark_nir_frameset);
    moo_try_check(moo_correct_dark(objB_det, NULL, masterdark_vis,
                                   masterdark_nir),
                  " ");

    objB_det_filename =
        cpl_sprintf("%s_B_OFFSET%d_%s_%d.fits", MOONS_TAG_XSWITCH_CORRECTDARK,
                    offset, mode, inum);
    moo_try_check(moo_products_add(products, objB_det,
                                   CPL_FRAME_LEVEL_INTERMEDIATE,
                                   MOONS_TAG_XSWITCH_CORRECTDARK,
                                   objB_det_filename, objectB_frame),
                  " ");

    nod_det_filename =
        cpl_sprintf("%s_OFFSET%d_%s_%d.fits", MOONS_TAG_XSWITCH_SUBTRACTNOD,
                    offset, mode, inum);

    moo_try_check(nod_det =
                      moo_subtract_nod(objA_det, objB_det, nod_det_filename),
                  " ");
    moo_try_check(moo_products_add(products, nod_det,
                                   CPL_FRAME_LEVEL_INTERMEDIATE,
                                   MOONS_TAG_XSWITCH_SUBTRACTNOD,
                                   nod_det_filename, objectA_frame),
                  " ");

    moo_try_check(_moons_p2pmap(nod_det, p2pmap), " ");
    nod_det_p2p_filename =
        cpl_sprintf("%s_NOD_OFFSET%d_%s_%d.fits", MOONS_TAG_XSWITCH_APPLYP2P,
                    offset, mode, inum);
    moo_try_check(moo_products_add(products, nod_det,
                                   CPL_FRAME_LEVEL_INTERMEDIATE,
                                   MOONS_TAG_XSWITCH_APPLYP2P,
                                   nod_det_p2p_filename, objectA_frame),
                  " ");
    nod_ext_filename =
        cpl_sprintf("%s_NOD_OFFSET%d_%s_%d.fits",
                    MOONS_TAG_SCIENCE_XSWITCH_EXTSPECTRA, offset, mode, inum);
    if (master_flat != NULL) {
        psf = moo_psf_load(master_flat);
    }
    moo_try_check(nod_ext = moo_extract(nod_det, loc, psf, extract_params,
                                        nod_ext_filename),
                  " ");
    moo_try_check(_moons_apply_flat(nod_ext, flat_frame, f2f_frame), " ");
    moo_try_check(moo_ext_compute_snr(nod_ext, wmap, sformat, skylines), " ");
    moo_try_check(nod_ext_frame = cpl_frame_duplicate(
                      moo_products_add_ext(products, nod_ext,
                                           CPL_FRAME_LEVEL_INTERMEDIATE,
                                           MOONS_TAG_SCIENCE_XSWITCH_EXTSPECTRA,
                                           nod_ext_filename, objectA_frame)),
                  " ");

    objA_ext_filename =
        cpl_sprintf("%s_A_OFFSET%d_%s_%d.fits",
                    MOONS_TAG_SCIENCE_XSWITCH_EXTSPECTRA, offset, mode, inum);
    moo_try_check(_moons_p2pmap(objA_det, p2pmap), " ");
    objA_det_p2p_filename =
        cpl_sprintf("%s_A_OFFSET%d_%s_%d.fits", MOONS_TAG_XSWITCH_APPLYP2P,
                    offset, mode, inum);
    moo_try_check(moo_products_add(products, objA_det,
                                   CPL_FRAME_LEVEL_INTERMEDIATE,
                                   MOONS_TAG_XSWITCH_APPLYP2P,
                                   objA_det_p2p_filename, objectA_frame),
                  " ");
    moo_try_check(objA_ext = moo_extract(objA_det, loc, psf, extract_params,
                                         objA_ext_filename),
                  " ");
    moo_try_check(_moons_apply_flat(objA_ext, flat_frame, f2f_frame), " ");
    moo_try_check(moo_ext_compute_snr(objA_ext, wmap, sformat, skylines), " ");
    moo_try_check(objA_ext_frame = cpl_frame_duplicate(
                      moo_products_add_ext(products, objA_ext,
                                           CPL_FRAME_LEVEL_INTERMEDIATE,
                                           MOONS_TAG_SCIENCE_XSWITCH_EXTSPECTRA,
                                           objA_ext_filename, objectA_frame)),
                  " ");

    objB_ext_filename =
        cpl_sprintf("%s_B_OFFSET%d_%s_%d.fits",
                    MOONS_TAG_SCIENCE_XSWITCH_EXTSPECTRA, offset, mode, inum);
    moo_try_check(_moons_p2pmap(objB_det, p2pmap), " ");
    objB_det_p2p_filename =
        cpl_sprintf("%s_B_OFFSET%d_%s_%d.fits", MOONS_TAG_XSWITCH_APPLYP2P,
                    offset, mode, inum);
    moo_try_check(moo_products_add(products, objB_det,
                                   CPL_FRAME_LEVEL_INTERMEDIATE,
                                   MOONS_TAG_XSWITCH_APPLYP2P,
                                   objB_det_p2p_filename, objectB_frame),
                  " ");
    moo_try_check(objB_ext = moo_extract(objB_det, loc, psf, extract_params,
                                         objB_ext_filename),
                  " ");
    moo_try_check(_moons_apply_flat(objB_ext, flat_frame, f2f_frame), " ");
    moo_try_check(moo_ext_compute_snr(objB_ext, wmap, sformat, skylines), " ");
    moo_try_check(objB_ext_frame = cpl_frame_duplicate(
                      moo_products_add_ext(products, objB_ext,
                                           CPL_FRAME_LEVEL_INTERMEDIATE,
                                           MOONS_TAG_SCIENCE_XSWITCH_EXTSPECTRA,
                                           objB_ext_filename, objectB_frame)),
                  " ");
moo_try_cleanup:
    cpl_free(nod_det_filename);
    cpl_free(nod_det_p2p_filename);
    cpl_free(nod_ext_filename);

    cpl_free(objA_det_filename);
    cpl_free(objA_det_p2p_filename);
    cpl_free(objA_ext_filename);

    cpl_free(objB_det_filename);
    cpl_free(objB_det_p2p_filename);
    cpl_free(objB_ext_filename);
    moo_ext_delete(nod_ext);
    moo_ext_delete(objB_ext);
    moo_ext_delete(objA_ext);
    moo_psf_delete(psf);
    moo_det_delete(nod_det);
    moo_det_delete(objB_det);
    moo_det_delete(objA_det);
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_frame_delete(nod_ext_frame);
        cpl_frame_delete(objB_ext_frame);
        cpl_frame_delete(objA_ext_frame);
    }
    else {
        cpl_frameset_insert(nod_ext_frameset, nod_ext_frame);
        cpl_frameset_insert(objB_ext_frameset, objB_ext_frame);
        cpl_frameset_insert(objA_ext_frameset, objA_ext_frame);
    }
    return CPL_ERROR_NONE;
}

static int
_moons_sort_mjdobs(const cpl_frame *a, const cpl_frame *b)
{
    int res = 0;

    const char *filenameA = cpl_frame_get_filename(a);
    const char *filenameB = cpl_frame_get_filename(b);

    cpl_propertylist *lista = cpl_propertylist_load(filenameA, 0);
    cpl_propertylist *listb = cpl_propertylist_load(filenameB, 0);

    if (cpl_propertylist_has(lista, MOO_PFITS_MJDOBS) &&
        cpl_propertylist_has(listb, MOO_PFITS_MJDOBS)) {
        double mjdA = moo_pfits_get_mjdobs(lista);
        double mjdB = moo_pfits_get_mjdobs(listb);
        if (mjdA < mjdB) {
            res = -1;
        }
        else if (mjdA > mjdB) {
            res = 1;
        }
        else {
            res = 0;
        }
    }
    cpl_propertylist_delete(listb);
    cpl_propertylist_delete(lista);
    return res;
}

static int
_moons_compare_offset(const cpl_frame *a, const cpl_frame *b)
{
    int res = 0;
    const char *filenameA = cpl_frame_get_filename(a);
    cpl_propertylist *lista = cpl_propertylist_load(filenameA, 0);

    double oA = moo_pfits_get_slit_offset(lista);
    const char *filenameB = cpl_frame_get_filename(b);
    cpl_propertylist *listb = cpl_propertylist_load(filenameB, 0);

    double oB = moo_pfits_get_slit_offset(listb);
    if (fabs(oA - oB) < 1E-12) {
        res = 1;
    }

    cpl_propertylist_delete(lista);
    cpl_propertylist_delete(listb);
    return res;
}

static int
_moons_science_xswitch_byoffset(cpl_frameset *frameset,
                                cpl_frameset *sci_set,
                                cpl_frameset *objA_ref_frameset,
                                const char *bpmap_rp_name,
                                const char *bpmap_nl_name,
                                const cpl_frame *masterbias,
                                const cpl_frame *masterdark_vis,
                                cpl_frameset *masterdark_nir,
                                const cpl_frame *p2pmap,
                                const cpl_frame *sformat_frame,
                                moo_sky_lines_list *sky_lines,
                                const cpl_frame *airglow_group_frame,
                                const cpl_frame *solflux_frame,
                                const cpl_frame *airglow_var_frame,
                                const cpl_frame *atmo_frame,
                                const cpl_frame *resp_frame,
                                moo_prepare_params *prepare_params,
                                moo_correct_bias_params *correct_bias_params,
                                moo_extract_params *extract_params,
                                moo_wavesol_params *wavesol_params,
                                moo_rebin_params *rbn_params,
                                moo_target_table_params *target_table_params,
                                moo_combine_pair_params *combine_pair_params,
                                moo_sub_sky_stare_params *sky_params,
                                moo_products *products)
{
    /* parameters */
    moo_loc *loc = NULL;
    moo_spectral_format *sformat = NULL;
    moo_map *wmap = NULL;
    moo_map *wmap_tuned = NULL;
    moo_extlist *objA_extlist = NULL;
    moo_extlist *objB_extlist = NULL;
    moo_extlist *nod_extlist = NULL;
    moo_extlist *wavemap_extlist = NULL;
    moo_ext *sum_ext = NULL;

    cpl_frameset *node_frameset = NULL;
    cpl_frameset *nod_ext_frameset = NULL;
    cpl_frameset *objB_ext_frameset = NULL;
    cpl_frameset *objA_ext_frameset = NULL;
    cpl_frameset *objB_ref_frameset = NULL;

    const cpl_frame *master_flat = NULL;
    const cpl_frame *fftrace = NULL;
    const cpl_frame *coeffs_cube = NULL;
    const cpl_frame *wmap_frame = NULL;
    const cpl_frame *flat_frame = NULL;
    const cpl_frame *f2f_frame = NULL;

    char *sum_ext_filename = NULL;
    const char *ref_filename = NULL;
    cpl_propertylist *ref_header = NULL;

    moo_mode_type mode;
    const char *mode_name = NULL;
    int offset;
    char *wmap_tuned_filename = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    node_frameset = cpl_frameset_new();
    nod_ext_frameset = cpl_frameset_new();
    objB_ext_frameset = cpl_frameset_new();
    objA_ext_frameset = cpl_frameset_new();
    objB_ref_frameset = cpl_frameset_new();

    moo_try_check(_moons_science_xswitch_check_sof_byoffset(
                      frameset, node_frameset, &master_flat, &fftrace,
                      &coeffs_cube, &wmap_frame, &flat_frame, &f2f_frame,
                      extract_params),
                  " ");
    int nb_nod = cpl_frameset_get_size(node_frameset);
    cpl_frame *frame = cpl_frameset_get_position(node_frameset, 0);
    ref_filename = cpl_frame_get_filename(frame);
    moo_try_check(mode = moo_mode_get(frame), " ");
    mode_name = moo_mode_get_name(mode);
    moo_try_check(ref_header = cpl_propertylist_load(ref_filename, 0), " ");
    moo_try_check(offset = moo_pfits_get_slit_offset(ref_header), " ");
    cpl_propertylist_delete(ref_header);
    ref_header = NULL;
    cpl_msg_info(__func__, "Do offset %d use mode %s", offset, mode_name);
    moo_try_check(sformat = moo_spectral_format_load(sformat_frame, mode), " ");
    moo_try_check(wmap = moo_map_load(wmap_frame), " ");
    moo_try_check(loc = moo_loc_load(fftrace), " ");

    for (int i = 0; i < nb_nod; i += 2) {
        cpl_frame *objectA_frame = NULL;
        cpl_frame *objectB_frame = NULL;

        moo_try_check(objectA_frame =
                          cpl_frameset_get_position(node_frameset, i),
                      " ");
        moo_try_check(objectB_frame =
                          cpl_frameset_get_position(node_frameset, i + 1),
                      " ");

        cpl_frameset_insert(objA_ref_frameset,
                            cpl_frame_duplicate(objectA_frame));
        cpl_frameset_insert(objB_ref_frameset,
                            cpl_frame_duplicate(objectB_frame));

        moo_try_check(_moons_xswitch_extract(
                          i / 2, mode_name, offset, objectA_frame,
                          objectB_frame, bpmap_rp_name, bpmap_nl_name,
                          masterbias, masterdark_vis, masterdark_nir,
                          master_flat, p2pmap, loc, wmap, sformat, sky_lines,
                          coeffs_cube, flat_frame, f2f_frame, extract_params,
                          prepare_params, correct_bias_params, products,
                          objA_ext_frameset, objB_ext_frameset,
                          nod_ext_frameset),
                      " ");
    }

    objB_extlist = moo_extlist_create(objB_ext_frameset);
    objA_extlist = moo_extlist_create(objA_ext_frameset);
    nod_extlist = moo_extlist_create(nod_ext_frameset);
    wavemap_extlist = moo_extlist_new();

    nb_nod = moo_extlist_get_size(nod_extlist);
    for (int i = 0; i < nb_nod; i++) {
        int k = i * 2;
        moo_ext *objB_ext = moo_extlist_get(objB_extlist, i);
        moo_ext *objA_ext = moo_extlist_get(objA_extlist, i);
        moo_extlist_set(wavemap_extlist, objA_ext, k);
        moo_extlist_set(wavemap_extlist, objB_ext, k + 1);
    }

    sum_ext_filename =
        cpl_sprintf("XSWITCH_EXT_SUM_OFFSET%d_%s.fits", offset, mode_name);
    sum_ext = moo_extlist_sum(wavemap_extlist, sum_ext_filename);
    cpl_frame *ref_frame = cpl_frameset_get_position(objA_ref_frameset, 0);
    moo_try_check(moo_products_add_ext(products, sum_ext,
                                       CPL_FRAME_LEVEL_TEMPORARY,
                                       MOONS_TAG_SCIENCE_XSWITCH_EXTSPECTRA,
                                       sum_ext_filename, ref_frame),
                  " ");
    moo_map *used_wmap = NULL;
    const char *wavesol_control = wavesol_params->control;
    if ((strcmp(wavesol_control, MOO_WAVESOL_CONTROL_CHECK) == 0) ||
        (strcmp(wavesol_control, MOO_WAVESOL_CONTROL_UPDATE) == 0)) {
        const char *sky_lines_name = sky_lines->filename;

        moo_try_check(wmap_tuned = moo_wavesol(sum_ext, sky_lines_name, sformat,
                                               loc, wmap, wavesol_params),
                      " ");
        if (strcmp(wavesol_control, MOO_WAVESOL_CONTROL_UPDATE) == 0) {
            used_wmap = wmap_tuned;
            wmap_tuned_filename =
                cpl_sprintf("XSWITCH_WMAP_TUNED_OFFSET%d_%s.fits", offset,
                            mode_name);
            moo_map_save(wmap_tuned, wmap_tuned_filename);
        }
        else {
            used_wmap = wmap;
        }
    }
    else {
        used_wmap = wmap;
    }

    for (int i = 0; i < nb_nod; i++) {
        moo_ext *nod_ext = moo_extlist_get(nod_extlist, i);
        moo_ext *objB_ext = moo_extlist_get(objB_extlist, i);
        moo_ext *objA_ext = moo_extlist_get(objA_extlist, i);
        cpl_frame *sci_frame = NULL;
        cpl_frame *objA_ref_frame =
            cpl_frameset_get_position(objA_ref_frameset, i);
        cpl_frame *objB_ref_frame =
            cpl_frameset_get_position(objB_ref_frameset, i);

        moo_try_check(sci_frame = _moons_science_xswitch_frame(
                          i, mode_name, offset, objA_ext, objB_ext, nod_ext,
                          objA_ref_frame, objB_ref_frame, sformat, sky_lines,
                          used_wmap, f2f_frame, airglow_group_frame,
                          airglow_var_frame, solflux_frame, atmo_frame,
                          resp_frame, rbn_params, target_table_params,
                          combine_pair_params, sky_params, products),
                      " ");
        cpl_frameset_insert(sci_set, sci_frame);
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_propertylist_delete(ref_header);
    }
    moo_extlist_delete(objA_extlist);
    moo_extlist_delete(objB_extlist);
    moo_extlist_delete(nod_extlist);
    if (wavemap_extlist != NULL) {
        for (int i = 0; i < nb_nod * 2; i++) {
            moo_extlist_unset(wavemap_extlist, 0);
        }
    }
    moo_extlist_delete(wavemap_extlist);
    moo_ext_delete(sum_ext);
    moo_loc_delete(loc);
    moo_map_delete(wmap);
    moo_map_delete(wmap_tuned);
    cpl_free(sum_ext_filename);
    cpl_free(wmap_tuned_filename);
    moo_spectral_format_delete(sformat);

    cpl_frameset_delete(node_frameset);
    cpl_frameset_delete(nod_ext_frameset);
    cpl_frameset_delete(objB_ext_frameset);
    cpl_frameset_delete(objA_ext_frameset);
    cpl_frameset_delete(objB_ref_frameset);

    return (int)cpl_error_get_code();
}

static int
_moons_science_xswitch(cpl_frameset *frameset, const cpl_parameterlist *parlist)
{
    /* parameters */
    moo_prepare_params *prepare_params = NULL;
    moo_correct_bias_params *correct_bias_params = NULL;
    moo_extract_params *extract_params = NULL;
    moo_wavesol_params *wavesol_params = NULL;
    moo_rebin_params *rbn_params = NULL;
    moo_target_table_params *target_table_params = NULL;
    moo_combine_pair_params *combine_pair_params = NULL;
    moo_sub_sky_stare_params *sky_params = NULL;
    moo_coadd_params *coadd_params = NULL;
    moo_create_s1d_params *s1d_params = NULL;

    moo_scilist *sci_list = NULL;
    moo_sci *sci_coadd = NULL;

    cpl_frameset *offset_frameset = NULL;
    cpl_frameset *set = NULL;
    cpl_frameset *sci_set = NULL;
    cpl_frameset *ref_frameset = NULL;

    const char *bpmap_rp_name = NULL;
    const char *bpmap_nl_name = NULL;
    const cpl_frame *masterbias = NULL;
    const cpl_frame *masterdark_vis = NULL;
    cpl_frameset *masterdark_nir = NULL;
    const cpl_frame *p2pmap = NULL;
    const cpl_frame *sformat_frame = NULL;
    const cpl_frame *sky_lines_frame = NULL;
    const cpl_frame *atmo_frame = NULL;
    const cpl_frame *resp_frame = NULL;
    const cpl_frame *airglow_group_frame = NULL;
    const cpl_frame *solflux_frame = NULL;
    const cpl_frame *airglow_var_frame = NULL;
    const cpl_frame *ref_frame = NULL;
    cpl_size nsel = 0;
    cpl_size *selection = NULL;
    moo_sky_lines_list *skylines = NULL;
    char *coadd_filename = NULL;
    char *calibflux_filename = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    moo_products *products =
        moo_products_new(frameset, parlist, "moons_science_xswitch",
                         PACKAGE "/" PACKAGE_VERSION);
    const moo_params *params = moo_products_get_params(products);
    moo_try_check(prepare_params = moo_params_get_prepare(params, parlist),
                  " ");
    moo_try_check(correct_bias_params =
                      moo_params_get_correct_bias(params, parlist),
                  " ");
    moo_try_check(extract_params = moo_params_get_extract(params, parlist),
                  " ");
    moo_try_check(rbn_params = moo_params_get_rebin(params, parlist), " ");
    moo_try_check(target_table_params =
                      moo_params_get_target_table(params, parlist),
                  " ");
    moo_try_check(combine_pair_params =
                      moo_params_get_combine_pair(params, parlist),
                  " ");
    moo_try_check(sky_params = moo_params_get_sub_sky_stare(params, parlist),
                  " ");
    moo_try_check(coadd_params = moo_params_get_coadd(params, parlist), " ");
    moo_try_check(wavesol_params =
                      moo_params_get_science_wavesol(params, parlist),
                  " ");
    moo_try_check(s1d_params = moo_params_get_create_s1d(params, parlist), " ");
    /* SOF file */
    offset_frameset = cpl_frameset_new();
    sci_set = cpl_frameset_new();
    ref_frameset = cpl_frameset_new();
    masterdark_nir = cpl_frameset_new();

    moo_try_check(_moons_science_xswitch_check_sof(
                      frameset, offset_frameset, &bpmap_rp_name, &bpmap_nl_name,
                      &masterbias, &masterdark_vis, masterdark_nir, &p2pmap,
                      &sformat_frame, &sky_lines_frame, &atmo_frame,
                      &resp_frame, &airglow_group_frame, &airglow_var_frame,
                      &solflux_frame, sky_params),
                  " ");

    //cpl_frameset_sort(offset_frameset,_moons_sort_mjdobs);

    moo_try_check(selection =
                      cpl_frameset_labelise(offset_frameset,
                                            &_moons_compare_offset, &nsel),
                  " ");

    moo_try_check(skylines = moo_sky_lines_list_load(sky_lines_frame), " ");

    for (int i = 0; i < nsel; i++) {
        moo_try_check(set = cpl_frameset_extract(offset_frameset, selection, i),
                      " ");
        moo_try_check(_moons_science_xswitch_byoffset(
                          set, sci_set, ref_frameset, bpmap_rp_name,
                          bpmap_nl_name, masterbias, masterdark_vis,
                          masterdark_nir, p2pmap, sformat_frame, skylines,
                          airglow_group_frame, solflux_frame, airglow_var_frame,
                          atmo_frame, resp_frame, prepare_params,
                          correct_bias_params, extract_params, wavesol_params,
                          rbn_params, target_table_params, combine_pair_params,
                          sky_params, products),
                      " ");
        cpl_frameset_delete(set);
        set = NULL;
    }

    moo_try_check(ref_frame = cpl_frameset_get_position_const(ref_frameset, 0),
                  " ");
    moo_mode_type mode = moo_mode_get(ref_frame);

    sci_list = moo_scilist_create(sci_set);
    coadd_filename =
        cpl_sprintf("%s_%s.fits", MOONS_TAG_XSWITCH_COMBINED_SPECTRA,
                    moo_mode_get_name(mode));

    sci_coadd = moo_coadd(sci_list, coadd_params, coadd_filename);

    moo_try_check(moo_sci_compute_snr(sci_coadd, skylines), " ");
    cpl_frame *sci_final = NULL;
    moo_try_check(sci_final =
                      moo_products_add_sci(products, sci_coadd,
                                           CPL_FRAME_LEVEL_FINAL,
                                           MOONS_TAG_XSWITCH_COMBINED_SPECTRA,
                                           coadd_filename, ref_frame),
                  " ");
    moo_create_s1d(sci_final, s1d_params, MOONS_TAG_SCIENCE_XSWITCH_1DSPECTRUM,
                   products);

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_frameset_delete(set);
    }
    cpl_free(coadd_filename);
    cpl_free(calibflux_filename);
    cpl_free(selection);
    moo_sci_delete(sci_coadd);
    moo_scilist_delete(sci_list);
    moo_sky_lines_list_delete(skylines);

    cpl_frameset_delete(sci_set);
    cpl_frameset_delete(ref_frameset);
    cpl_frameset_delete(offset_frameset);
    cpl_frameset_delete(masterdark_nir);

    moo_extract_params_delete(extract_params);
    moo_wavesol_params_delete(wavesol_params);
    moo_rebin_params_delete(rbn_params);
    moo_target_table_params_delete(target_table_params);

    moo_combine_pair_params_delete(combine_pair_params);
    moo_sub_sky_stare_params_delete(sky_params);
    moo_coadd_params_delete(coadd_params);
    moo_correct_bias_params_delete(correct_bias_params);
    moo_prepare_params_delete(prepare_params);
    moo_create_s1d_params_delete(s1d_params);
    moo_products_delete(products);
    return (int)cpl_error_get_code();
}
