/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "moo_utils.h"
#include "moo_pfits.h"
#include "moo_dfs.h"
#include "moo_params.h"
#include "moo_drl.h"
#include "moo_badpix.h"
#include "moo_extlist.h"
#include "moo_rbnlist.h"
#include "moo_correct_tell.h"
#include "moo_subtract_nod.h"
#include "moo_combine_sky.h"
#include "moo_combine_pair.h"
#include "moo_products.h"
#include <cpl.h>

#include <string.h>

/*-----------------------------------------------------------------------------
                              Plugin registration
 -----------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list);

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

static int _moons_science_xswitch2_create(cpl_plugin *plugin);
static int _moons_science_xswitch2_exec(cpl_plugin *plugin);
static int _moons_science_xswitch2_destroy(cpl_plugin *plugin);
static int _moons_science_xswitch2(cpl_frameset *frameset,
                                   const cpl_parameterlist *parlist);

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/
static const char *const _moons_science_xswitch2_description =
    "From an observation sequence in XSwitch mode, this recipe removes "
    "the instrumental signature and provides calibrated science products. "
    "During the sequence, the telescope nods from one position on the sky "
    "(labeled as A in Fig. 5-11) to another (B). In the sequence, different "
    "slit"
    " offset positions can be used: as represented in Fig. 5-11, the "
    "combination "
    "is done with 1D wavelength calibrated sky subtracted spectra. Using "
    "different"
    " slit offset positions implies that different sets of calibration "
    "(FF, wave maps) are given as inputs.\n"
    "INPUT FRAMES\n"
    "  * science exposure frames n>=1 file(s)               (RAW) with tag "
    "OBJECT_XSWITCH : "
    "science frame in xswitch mode file\n"
    "  * [OPTIONAL] ReferenceBadPixMask 1 file              (QUA) with tag "
    "BP_MAP_RP : "
    "cosmetic bad pixel map\n"
    "  * [OPTIONAL] NonLinearityBadPixMask 1 file           (QUA) with tag "
    "BP_MAP_NL : "
    "cosmetic bad pixel map coming from linearity recipe\n"
    "  * MasterBias 1 file                                  (DET) with tag "
    "MASTER_BIAS : "
    "master bias file\n"
    "  * MasterDarkNir 1 file                               (DET) with tag "
    "MASTER_DARK_NIR : "
    "master dark nir for object\n"
    "  * [OPTIONAL] MasterDarkNir 1 file                    (DET) with tag "
    "MASTER_DARK_NIR : "
    "master dark nir file for sky (if exposure time is different from object)\n"
    "  * [OPTIONAL] MasterDarkVis 1 file                    (DET) with tag "
    "MASTER_DARK_VIS : "
    "master dark vis file\n"
    "  * [OPTIONAL] CoeffsCube 1 or 2 file(s) one by offset (3D) with tag "
    "LINEARITY_COEFF_CUBE : "
    "coefficients to correct pixels detector linearity\n"
    "  * [OPTIONAL] P2pMap 1 file                           (DET) with tag "
    "P2P_MAP : "
    "pixel to pixel map\n"
    "  * Fftrace 1 or 2 file(s) one by offset               (LOC) with tag "
    "FF_TRACE :"
    "the localisation table\n"
    "  * [OPTIONAL] MasterFlat 1 or 2 file(s) one by offset (PSF) with tag "
    "MASTER_FLAT : "
    "the master flat used for optimal extraction\n"
    "  * Ffext  1 or 2 file(s) one by offset                (EXT) with tag "
    "FF_EXTSPECTRA : "
    "the extracted flat field\n"
    "  * F2f [OPTIONAL] 1 or 2 file(s) one by offset        (F2F) with tag "
    "F2F_TABLE :"
    "the fibre-to-fibre relative response\n"
    "  * SFormat 1 file                                     (FMT) with tag "
    "SPECTRAL_FORMAT : "
    "the spectral format table\n"
    "  * WaveMap 1 or 2 file(s) one by offset               (WMAP) with tag "
    "WAVE_MAP : "
    "the wavelength map\n"
    "  * SkyLines 1 file                                    (CAT) with tag "
    "SKY_LINE_LIST : "
    "the sky lines list table\n"
    "  * [OPTIONAL] response 1 file                         (RESPONSE) with "
    "tag RESPONSE : "
    "the response spectra in the extracted pixel space\n"
    "  * [OPTIONAL] atmosExt 1 file                         (ATMO) with tag "
    "ATMOS_EXT : "
    "the atmospheric extinction table\n"
    "PRODUCTS\n"
    "  * SCIENCE_XSWITCHRBNSPECTRA_OFFSET[offset]_[insmode]_i.fits (RBN) with "
    "tag "
    "xswitch_RBNSPECTRA : "
    "Science product individual calibrated frames\n"
    "  * SCIENCE_XSWITCHSKSSPECTRA_OFFSET[offset]_[insmode]_i.fits (SCI) with "
    "tag "
    "xswitch_SKSSPECTRA : "
    "Science product individual sky subtracted frames\n"
    "  * SCIENCE_XSWITCHSKSSPECTRA_[insmode].fits                  (SCI) with "
    "tag "
    "xswitch_SKSSPECTRA : "
    "Science product coadd calibrated frame\n"
    "  * [OPTIONAL] SCIENCE_XSWITCHFLXSPECTRA_[insmode].fits       (SCI) with "
    "tag "
    "xswitch_FLXSPECTRA : "
    "Science product flux calibrated frame\n"
    "\n";

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok, 1 otherwise
  @note     Only this function is exported

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*----------------------------------------------------------------------------*/

int
cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
    cpl_plugin *plugin = &recipe->interface;

    if (cpl_plugin_init(plugin, CPL_PLUGIN_API, MOONS_BINARY_VERSION,
                        CPL_PLUGIN_TYPE_RECIPE, "moons_science_xswitch2",
                        "Reduces science exposure in xswitch mode",
                        _moons_science_xswitch2_description, "Regis Haigron",
                        PACKAGE_BUGREPORT, moo_get_license(),
                        _moons_science_xswitch2_create,
                        _moons_science_xswitch2_exec,
                        _moons_science_xswitch2_destroy)) {
        cpl_msg_error(cpl_func, "Plugin initialization failed");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    if (cpl_pluginlist_append(list, plugin)) {
        cpl_msg_error(cpl_func, "Error adding plugin to list");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/

static int
_moons_science_xswitch2_create(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    /* Do not create the recipe if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Create the parameters list in the cpl_recipe object */
    recipe->parameters = cpl_parameterlist_new();
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Parameter list allocation failed");
        cpl_ensure_code(0, (int)CPL_ERROR_ILLEGAL_OUTPUT);
    }

    moo_params *params = moo_params_new("moons", "moons_science_xswitch2");

    /* Fill the parameters list */
    moo_params_add_keep_temp(params, recipe->parameters);
    moo_params_add_prepare(params, recipe->parameters);
    moo_params_add_correct_bias(params, recipe->parameters,
                                MOO_CORRECT_BIAS_METHOD_MASTER);
    moo_params_add_extract(params, recipe->parameters);
    moo_params_add_science_wavesol(params, recipe->parameters);
    moo_params_add_rebin(params, recipe->parameters);
    moo_params_add_target_table(params, recipe->parameters, CPL_FALSE);
    moo_params_add_combine_pair(params, recipe->parameters, CPL_TRUE);
    moo_params_add_sub_sky_stare(params, recipe->parameters);
    moo_params_add_coadd(params, recipe->parameters);
    moo_params_add_create_s1d(params, recipe->parameters);
    moo_params_delete(params);

    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_science_xswitch2_exec(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    int recipe_status;
    cpl_errorstate initial_errorstate = cpl_errorstate_get();

    /* Return immediately if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Verify parameter and frame lists */
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL parameter list");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }
    if (recipe->frames == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL frame set");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Invoke the recipe */
    recipe_status = _moons_science_xswitch2(recipe->frames, recipe->parameters);

    /* Ensure DFS-compliance of the products */
    if (cpl_dfs_update_product_header(recipe->frames)) {
        if (!recipe_status)
            recipe_status = (int)cpl_error_get_code();
    }

    if (!cpl_errorstate_is_equal(initial_errorstate)) {
        /* Dump the error history since recipe execution start.
           At this point the recipe cannot recover from the error */
        cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
    }

    return recipe_status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_science_xswitch2_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    cpl_parameterlist_delete(recipe->parameters);

    return 0;
}

static cpl_error_code
_moons_p2pmap(moo_det *arc, const cpl_frame *p2pmap)
{
    cpl_error_code status = CPL_ERROR_NONE;
    moo_det *p2p = NULL;
    if (p2pmap != NULL) {
        moo_try_check(p2p = moo_det_create(p2pmap), " ");
        moo_try_check(moo_apply_p2p(arc, p2p), " ");
    }
moo_try_cleanup:
    moo_det_delete(p2p);
    return status;
}

static const cpl_frame *
find_masterdark_nir(moo_det *det, cpl_frameset *set)
{
    const cpl_frame *result = NULL;
    double dit = 0.0;
    int ndit = 0;
    double exptime = 0.0;
    int num;
    int type;
    int find = 0;
    moo_det *current_det = NULL;

    cpl_ensure(det != NULL, CPL_ERROR_NULL_INPUT, NULL);

    for (num = 1; num <= 2; num++) {
        for (type = 1; type <= 2; type++) {
            moo_single *single = moo_det_get_single(det, type, num);
            if (single != NULL) {
                moo_try_check(ndit = moo_pfits_get_ndit(single->header), " ");
                moo_try_check(dit = moo_pfits_get_dit(single->header), " ");
                exptime = dit * ndit;
                find = 1;
                break;
            }
        }
        if (find) {
            break;
        }
    }

    if (find && set != NULL) {
        int size = cpl_frameset_get_size(set);
        for (int i = 0; i < size; i++) {
            const cpl_frame *current = cpl_frameset_get_position_const(set, i);
            current_det = moo_det_create(current);
            moo_single *current_single =
                moo_det_get_single(current_det, type, num);
            moo_try_check(ndit = moo_pfits_get_ndit(current_single->header),
                          " ");
            moo_try_check(dit = moo_pfits_get_dit(current_single->header), " ");
            double current_exptime = dit * ndit;
            if (fabs(exptime - current_exptime) <= 1E-9) {
                result = current;
                break;
            }
            moo_det_delete(current_det);
            current_det = NULL;
        }
    }
moo_try_cleanup:
    moo_det_delete(current_det);
    return result;
}

static cpl_frame *
_moons_xswitch_subtractnod(int inum,
                           const char *mode,
                           int offset,
                           const cpl_frame *obj_frame,
                           const cpl_frame *obj_det_frame,
                           const cpl_frame *sky_det_frame,
                           moo_products *products)
{
    cpl_frame *result = NULL;
    char *nod_det_filename = NULL;
    moo_det *nod_det = NULL;
    unsigned int badpix_level = MOO_BADPIX_GOOD;

    moo_det *obj_det = moo_det_create(obj_det_frame);
    moo_det *sky_det = moo_det_create(sky_det_frame);
    moo_try_check(moo_det_load(obj_det, badpix_level), " ");
    moo_try_check(moo_det_load(sky_det, badpix_level), " ");

    nod_det_filename =
        cpl_sprintf("%s_OFFSET%d_%s_%d.fits", MOONS_TAG_XSWITCH_SUBTRACTNOD,
                    offset, mode, inum);

    moo_try_check(nod_det =
                      moo_subtract_nod(obj_det, sky_det, nod_det_filename),
                  " ");
    moo_try_check(result = moo_products_add(products, nod_det,
                                            CPL_FRAME_LEVEL_INTERMEDIATE,
                                            MOONS_TAG_XSWITCH_SUBTRACTNOD,
                                            nod_det_filename, obj_frame),
                  " ");

moo_try_cleanup:
    cpl_free(nod_det_filename);
    moo_det_delete(obj_det);
    moo_det_delete(sky_det);
    moo_det_delete(nod_det);

    return result;
}

static cpl_frame *
_moons_xswitch_prepare(int inum,
                       const char *mode,
                       int offset,
                       const cpl_frame *frame,
                       const char *bpmap_rp_name,
                       const char *bpmap_nl_name,
                       const cpl_frame *masterbias,
                       const cpl_frame *masterdark_vis,
                       cpl_frameset *masterdark_nir_frameset,
                       const cpl_frame *p2pmap,
                       const cpl_frame *coeffs_cube,
                       moo_prepare_params *prepare_params,
                       moo_correct_bias_params *cbias_params,
                       moo_products *products,
                       const char *prefix)
{
    cpl_frame *result = NULL;
    const cpl_frame *masterdark_nir = NULL;
    moo_det *det = NULL;
    char *det_filename = NULL;
    char *det_p2p_filename = NULL;

    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, NULL);

    moo_try_check(det = moo_prepare(frame, bpmap_rp_name, bpmap_nl_name,
                                    masterbias, coeffs_cube, prepare_params),
                  " ");
    moo_try_check(moo_correct_bias(det, masterbias, cbias_params), " ");

    masterdark_nir = find_masterdark_nir(det, masterdark_nir_frameset);

    moo_try_check(moo_correct_dark(det, NULL, masterdark_vis, masterdark_nir),
                  " ");
    det_filename =
        cpl_sprintf("%s_%s_OFFSET%d_%s_%d.fits", MOONS_TAG_XSWITCH_CORRECTDARK,
                    prefix, offset, mode, inum);
    moo_try_check(moo_products_add(products, det, CPL_FRAME_LEVEL_INTERMEDIATE,
                                   MOONS_TAG_XSWITCH_CORRECTDARK, det_filename,
                                   frame),
                  " ");
    moo_try_check(_moons_p2pmap(det, p2pmap), " ");
    det_p2p_filename =
        cpl_sprintf("%s_%s_OFFSET%d_%s_%d.fits", MOONS_TAG_XSWITCH_APPLYP2P,
                    prefix, offset, mode, inum);
    moo_try_check(result = moo_products_add(products, det,
                                            CPL_FRAME_LEVEL_INTERMEDIATE,
                                            MOONS_TAG_XSWITCH_APPLYP2P,
                                            det_p2p_filename, frame),
                  " ");

moo_try_cleanup:
    cpl_free(det_filename);
    cpl_free(det_p2p_filename);
    moo_det_delete(det);
    return result;
}

static cpl_error_code
_moons_science_xswitch2_check_sof(cpl_frameset *frameset,
                                  cpl_frameset *offset_frameset,
                                  const char **bpmap_rp_name,
                                  const char **bpmap_nl_name,
                                  const cpl_frame **masterbias,
                                  const cpl_frame **masterdark_vis,
                                  cpl_frameset *masterdark_nir,
                                  const cpl_frame **p2pmap,
                                  const cpl_frame **sformat,
                                  const cpl_frame **sky_lines,
                                  const cpl_frame **atmo,
                                  const cpl_frame **resp,
                                  const cpl_frame **tell,
                                  const cpl_frame **airglow_group,
                                  const cpl_frame **airglow_var,
                                  const cpl_frame **solflux,
                                  const moo_sub_sky_stare_params *subsky_params)
{
    cpl_ensure_code(moo_dfs_set_groups(frameset) == CPL_ERROR_NONE,
                    cpl_error_get_code());
    int i;

    for (i = 0; i < cpl_frameset_get_size(frameset); ++i) {
        cpl_frame *current_frame = cpl_frameset_get_position(frameset, i);
        if (!strcmp(cpl_frame_get_tag(current_frame),
                    MOONS_TAG_OBJECT_XSWITCH)) {
            cpl_frameset_insert(offset_frameset,
                                cpl_frame_duplicate(current_frame));
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_BP_MAP_RP)) {
            *bpmap_rp_name = cpl_frame_get_filename(current_frame);
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_BP_MAP_NL)) {
            *bpmap_nl_name = cpl_frame_get_filename(current_frame);
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_BIAS)) {
            *masterbias = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_DARK_VIS)) {
            *masterdark_vis = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_DARK_NIR)) {
            cpl_frameset_insert(masterdark_nir,
                                cpl_frame_duplicate(current_frame));
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_FLAT)) {
            cpl_frameset_insert(offset_frameset,
                                cpl_frame_duplicate(current_frame));
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_P2P_MAP)) {
            *p2pmap = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FF_TRACE)) {
            cpl_frameset_insert(offset_frameset,
                                cpl_frame_duplicate(current_frame));
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_LINEARITY_COEFF_CUBE)) {
            cpl_frameset_insert(offset_frameset,
                                cpl_frame_duplicate(current_frame));
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_SPECTRAL_FORMAT)) {
            *sformat = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_WAVEMAP)) {
            cpl_frameset_insert(offset_frameset,
                                cpl_frame_duplicate(current_frame));
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_SKY_LINE_LIST)) {
            *sky_lines = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FF_EXTSPECTRA)) {
            cpl_frameset_insert(offset_frameset,
                                cpl_frame_duplicate(current_frame));
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_F2F_TABLE)) {
            cpl_frameset_insert(offset_frameset,
                                cpl_frame_duplicate(current_frame));
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_ATMOS_EXT)) {
            *atmo = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_RESPONSE)) {
            *resp = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_TELLURIC_CORR)) {
            *tell = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_AIRGLOW_GROUP)) {
            *airglow_group = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_AIRGLOW_VAR)) {
            *airglow_var = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_SOLFLUX)) {
            *solflux = current_frame;
        }
    }

    if (*sformat == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_SPECTRAL_FORMAT);
    }

    if (*sky_lines == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_SKY_LINE_LIST);
    }

    if (*airglow_group == NULL &&
        strcmp(subsky_params->method, MOO_SUB_SKY_STARE_METHOD_SKYCORR) == 0) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_AIRGLOW_GROUP);
    }
    if (*airglow_var == NULL &&
        strcmp(subsky_params->method, MOO_SUB_SKY_STARE_METHOD_SKYCORR) == 0) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_AIRGLOW_VAR);
    }
    if (*solflux == NULL &&
        strcmp(subsky_params->method, MOO_SUB_SKY_STARE_METHOD_SKYCORR) == 0) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_SOLFLUX);
    }
    return CPL_ERROR_NONE;
}

static cpl_error_code
_moons_science_xswitch2_check_sof_byoffset(cpl_frameset *frameset,
                                           cpl_frameset *objA_frameset,
                                           cpl_frameset *objB_frameset,
                                           const cpl_frame **master_flat,
                                           const cpl_frame **fftrace,
                                           const cpl_frame **coeffs_cube,
                                           const cpl_frame **wmap,
                                           const cpl_frame **flat,
                                           const cpl_frame **f2f,
                                           moo_extract_params *ext_params)
{
    cpl_ensure_code(moo_dfs_set_groups(frameset) == CPL_ERROR_NONE,
                    cpl_error_get_code());
    int nb_object = 0;
    int i;
    for (i = 0; i < cpl_frameset_get_size(frameset); ++i) {
        cpl_frame *current_frame = cpl_frameset_get_position(frameset, i);
        if (!strcmp(cpl_frame_get_tag(current_frame),
                    MOONS_TAG_OBJECT_XSWITCH)) {
            if (nb_object % 2 == 0) {
                cpl_frameset_insert(objA_frameset,
                                    cpl_frame_duplicate(current_frame));
            }
            else {
                cpl_frameset_insert(objB_frameset,
                                    cpl_frame_duplicate(current_frame));
            }
            nb_object++;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_FLAT)) {
            *master_flat = current_frame;
        }

        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FF_TRACE)) {
            *fftrace = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_LINEARITY_COEFF_CUBE)) {
            *coeffs_cube = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_WAVEMAP)) {
            *wmap = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FF_EXTSPECTRA)) {
            *flat = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_F2F_TABLE)) {
            *f2f = current_frame;
        }
    }

    if (nb_object <= 0) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_OBJECT_XSWITCH);
    }

    if (*fftrace == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_FF_TRACE);
    }

    if (*wmap == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_WAVEMAP);
    }
    if (*flat == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_FF_EXTSPECTRA);
    }
    if (*master_flat == NULL &&
        strcmp(ext_params->method, MOO_EXTRACT_METHOD_OPTIMAL) == 0) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_MASTER_FLAT);
    }
    return CPL_ERROR_NONE;
}

static cpl_error_code
_moons_apply_flat(moo_ext *ext,
                  const cpl_frame *flat_frame,
                  const cpl_frame *f2f_frame)
{
    cpl_error_code status = CPL_ERROR_NONE;
    moo_ext *flat = NULL;
    moo_f2f *f2f = NULL;

    moo_try_check(flat = moo_ext_create(flat_frame), " ");
    if (f2f_frame != NULL) {
        f2f = moo_f2f_load(f2f_frame);
    }
    moo_try_check(moo_apply_flat(ext, flat, f2f), " ");

moo_try_cleanup:
    moo_ext_delete(flat);
    moo_f2f_delete(f2f);
    return status;
}

static cpl_frame *
_moons_xswitch_rebin(int inum,
                     const char *mode,
                     int offset,
                     moo_ext *ext,
                     const cpl_frame *ref_frame,
                     moo_spectral_format *sformat,
                     moo_sky_lines_list *sky_lines,
                     moo_map *wmap,
                     const cpl_frame *atmo_frame,
                     const cpl_frame *resp_frame,
                     const cpl_frame *tell_frame,
                     moo_rebin_params *rbn_params,
                     moo_products *products,
                     const char *prefix)
{
    cpl_frame *rbn_frame = NULL;
    moo_atm *atm = NULL;
    moo_resp *resp = NULL;
    moo_telluric *telluric = NULL;
    moo_rbn *rbn = NULL;

    char *rbn_filename = NULL;
    char *calibflux_filename = NULL;
    char *tellcorr_filename = NULL;

    cpl_ensure(ext != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(ref_frame != NULL, CPL_ERROR_NULL_INPUT, NULL);

    rbn_filename = cpl_sprintf("%s_RBNSPECTRA_OFFSET%d_%s_%d.fits", prefix,
                               offset, mode, inum);
    moo_try_check(rbn = moo_rebin(ext, wmap, sformat, rbn_params, rbn_filename),
                  " ");
    moo_try_check(moo_rbn_compute_snr(rbn, sky_lines), " ");
    moo_try_check(rbn_frame =
                      moo_products_add_rbn(products, rbn, CPL_FRAME_LEVEL_FINAL,
                                           MOONS_TAG_SCIENCE_XSWITCH_RBNSPECTRA,
                                           rbn_filename, ref_frame),
                  " ");

    if (atmo_frame != NULL && resp_frame != NULL) {
        moo_try_check(atm = moo_atm_load(atmo_frame), " ");
        moo_try_check(resp = moo_resp_load(resp_frame), " ");

        calibflux_filename = cpl_sprintf("%s_RBNFLXSPECTRA_OFFSET%d_%s_%d.fits",
                                         prefix, offset, mode, inum);
        moo_try_check(moo_calib_flux(rbn, atm, resp), " ");
        moo_try_check(rbn_frame = moo_products_add_rbn(
                          products, rbn, CPL_FRAME_LEVEL_FINAL,
                          MOONS_TAG_SCIENCE_XSWITCH_RBNFLXSPECTRA,
                          calibflux_filename, ref_frame),
                      " ");
    }
    else {
        cpl_msg_info(__func__, "No flux calibration");
    }

    if (tell_frame != NULL) {
        moo_try_check(telluric = moo_telluric_load(tell_frame), " ");

        tellcorr_filename =
            cpl_sprintf("%s_RBNTELLCORRSPECTRA_OFFSET%d_%s_%d.fits", prefix,
                        offset, mode, inum);
        moo_try_check(moo_correct_tell(rbn, telluric), " ");
        moo_try_check(rbn_frame = moo_products_add_rbn(
                          products, rbn, CPL_FRAME_LEVEL_FINAL,
                          MOONS_TAG_SCIENCE_XSWITCH_RBNTELLCORRSPECTRA,
                          tellcorr_filename, ref_frame),
                      " ");
    }
    else {
        cpl_msg_info(__func__, "No telluric correction");
    }

moo_try_cleanup:
    cpl_free(tellcorr_filename);
    cpl_free(calibflux_filename);
    cpl_free(rbn_filename);
    moo_atm_delete(atm);
    moo_resp_delete(resp);
    moo_telluric_delete(telluric);
    moo_rbn_delete(rbn);
    return rbn_frame;
}

static cpl_frame *
_moons_xswitch_extract(int inum,
                       const char *mode,
                       int offset,
                       const cpl_frame *ref_frame,
                       const cpl_frame *det_frame,
                       moo_loc *loc,
                       moo_map *wmap,
                       moo_spectral_format *sformat,
                       moo_sky_lines_list *skylines,
                       const cpl_frame *masterflat_frame,
                       const cpl_frame *flat_frame,
                       const cpl_frame *f2f_frame,
                       moo_extract_params *extract_params,
                       moo_products *products,
                       const char *prefix)
{
    cpl_frame *ext_frame = NULL;
    moo_det *det = NULL;
    moo_psf *psf = NULL;
    moo_ext *ext = NULL;

    char *ext_filename = NULL;
    char *extff_filename = NULL;

    cpl_ensure(det_frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    det = moo_det_create(det_frame);
    ext_filename = cpl_sprintf("%s_EXTSPECTRA_OFFSET%d_%s_%d.fits", prefix,
                               offset, mode, inum);
    if (masterflat_frame != NULL) {
        psf = moo_psf_load(masterflat_frame);
    }

    moo_try_check(ext =
                      moo_extract(det, loc, psf, extract_params, ext_filename),
                  " ");
    moo_try_check(ext_frame =
                      moo_products_add_ext(products, ext,
                                           CPL_FRAME_LEVEL_INTERMEDIATE,
                                           MOONS_TAG_SCIENCE_XSWITCH_EXTSPECTRA,
                                           ext_filename, ref_frame),
                  " ");

    moo_try_check(_moons_apply_flat(ext, flat_frame, f2f_frame), " ");
    moo_try_check(moo_ext_compute_snr(ext, wmap, sformat, skylines), " ");
    extff_filename = cpl_sprintf("%s_EXTFFSPECTRA_OFFSET%d_%s_%d.fits", prefix,
                                 offset, mode, inum);

    moo_try_check(ext_frame =
                      moo_products_add_ext(products, ext,
                                           CPL_FRAME_LEVEL_INTERMEDIATE,
                                           MOONS_TAG_SCIENCE_XSWITCH_EXTSPECTRA,
                                           extff_filename, ref_frame),
                  " ");

moo_try_cleanup:
    cpl_free(ext_filename);
    cpl_free(extff_filename);
    moo_det_delete(det);
    moo_ext_delete(ext);
    moo_psf_delete(psf);
    return ext_frame;
}

static int
_moons_sort_mjdobs(const cpl_frame *a, const cpl_frame *b)
{
    int res = 0;

    const char *filenameA = cpl_frame_get_filename(a);
    const char *filenameB = cpl_frame_get_filename(b);

    cpl_propertylist *lista = cpl_propertylist_load(filenameA, 0);
    cpl_propertylist *listb = cpl_propertylist_load(filenameB, 0);

    if (cpl_propertylist_has(lista, MOO_PFITS_MJDOBS) &&
        cpl_propertylist_has(listb, MOO_PFITS_MJDOBS)) {
        double mjdA = moo_pfits_get_mjdobs(lista);
        double mjdB = moo_pfits_get_mjdobs(listb);
        if (mjdA < mjdB) {
            res = -1;
        }
        else if (mjdA > mjdB) {
            res = 1;
        }
        else {
            res = 0;
        }
    }
    cpl_propertylist_delete(listb);
    cpl_propertylist_delete(lista);
    return res;
}

static int
_moons_compare_offset(const cpl_frame *a, const cpl_frame *b)
{
    int res = 0;
    const char *filenameA = cpl_frame_get_filename(a);
    cpl_propertylist *lista = cpl_propertylist_load(filenameA, 0);

    double oA = moo_pfits_get_slit_offset(lista);
    const char *filenameB = cpl_frame_get_filename(b);
    cpl_propertylist *listb = cpl_propertylist_load(filenameB, 0);

    double oB = moo_pfits_get_slit_offset(listb);
    if (fabs(oA - oB) < 1E-12) {
        res = 1;
    }

    cpl_propertylist_delete(lista);
    cpl_propertylist_delete(listb);
    return res;
}

static moo_target_table *
_xswitch_target_table(cpl_frame *frameA,
                      cpl_frame *frameB,
                      moo_target_table_mode mode,
                      moo_target_table_params *params)
{
    moo_target_table *result = NULL;

    moo_rbn *objA_rbn = NULL;
    moo_rbn *objB_rbn = NULL;

    moo_try_check(objA_rbn = moo_rbn_create(frameA), " ");
    moo_try_check(objB_rbn = moo_rbn_create(frameB), " ");

    moo_try_check(result =
                      moo_create_target_table(objA_rbn, objB_rbn, mode, params),
                  " ");
moo_try_cleanup:
    moo_rbn_delete(objA_rbn);
    moo_rbn_delete(objB_rbn);

    return result;
}
static cpl_frame *
_xswitch_nod_rebin(cpl_frameset *det_frameset,
                   const char *mode,
                   int offset,
                   const cpl_frame *ref_frame,
                   moo_loc *loc,
                   moo_map *wmap,
                   moo_spectral_format *sformat,
                   moo_sky_lines_list *skylines,
                   const cpl_frame *masterflat_frame,
                   const cpl_frame *flat_frame,
                   const cpl_frame *f2f_frame,
                   const cpl_frame *atmo_frame,
                   const cpl_frame *resp_frame,
                   const cpl_frame *tell_frame,
                   moo_extract_params *extract_params,
                   moo_rebin_params *rbn_params,
                   moo_products *products,
                   const char *prefix)
{
    cpl_frame *result = NULL;
    moo_detlist *detlist = NULL;
    moo_det *combined = NULL;

    moo_psf *psf = NULL;
    moo_ext *ext = NULL;
    moo_crh_params *crh_params = NULL;
    moo_atm *atm = NULL;
    moo_resp *resp = NULL;
    moo_telluric *telluric = NULL;
    moo_rbn *rbn = NULL;

    char *comb_filename = NULL;
    char *ext_filename = NULL;
    char *extff_filename = NULL;

    char *rbn_filename = NULL;
    char *calibflux_filename = NULL;
    char *tellcorr_filename = NULL;

    cpl_ensure(det_frameset != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();
    crh_params = moo_crh_params_new();
    crh_params->method = MOO_CRH_METHOD_MEDIAN;
    moo_try_check(detlist = moo_detlist_create(det_frameset), " ");
    moo_try_check(combined = moo_remove_CRH(detlist, NULL, crh_params), " ");
    comb_filename =
        cpl_sprintf("%s_COMB2D_OFFSET%d_%s.fits", prefix, offset, mode);

    moo_try_check(moo_products_add(products, combined,
                                   CPL_FRAME_LEVEL_INTERMEDIATE,
                                   MOONS_TAG_XSWITCH_REMOVECRH, comb_filename,
                                   ref_frame),
                  " ");

    ext_filename =
        cpl_sprintf("%s_EXTSPECTRA_OFFSET%d_%s.fits", prefix, offset, mode);
    if (masterflat_frame != NULL) {
        psf = moo_psf_load(masterflat_frame);
    }

    moo_try_check(ext = moo_extract(combined, loc, psf, extract_params,
                                    ext_filename),
                  " ");
    moo_try_check(moo_products_add_ext(products, ext,
                                       CPL_FRAME_LEVEL_INTERMEDIATE,
                                       MOONS_TAG_SCIENCE_XSWITCH_EXTSPECTRA,
                                       ext_filename, ref_frame),
                  " ");

    moo_try_check(_moons_apply_flat(ext, flat_frame, f2f_frame), " ");
    moo_try_check(moo_ext_compute_snr(ext, wmap, sformat, skylines), " ");
    extff_filename =
        cpl_sprintf("%s_EXTFFSPECTRA_OFFSET%d_%s.fits", prefix, offset, mode);

    moo_try_check(moo_products_add_ext(products, ext,
                                       CPL_FRAME_LEVEL_INTERMEDIATE,
                                       MOONS_TAG_SCIENCE_XSWITCH_EXTSPECTRA,
                                       extff_filename, ref_frame),
                  " ");

    rbn_filename =
        cpl_sprintf("%s_RBNSPECTRA_OFFSET%d_%s.fits", prefix, offset, mode);
    moo_try_check(rbn = moo_rebin(ext, wmap, sformat, rbn_params, rbn_filename),
                  " ");
    moo_try_check(moo_rbn_compute_snr(rbn, skylines), " ");
    moo_try_check(result =
                      moo_products_add_rbn(products, rbn, CPL_FRAME_LEVEL_FINAL,
                                           MOONS_TAG_SCIENCE_XSWITCH_RBNSPECTRA,
                                           rbn_filename, ref_frame),
                  " ");

    if (atmo_frame != NULL && resp_frame != NULL) {
        moo_try_check(atm = moo_atm_load(atmo_frame), " ");
        moo_try_check(resp = moo_resp_load(resp_frame), " ");

        calibflux_filename = cpl_sprintf("%s_RBNFLXSPECTRA_OFFSET%d_%s.fits",
                                         prefix, offset, mode);
        moo_try_check(moo_calib_flux(rbn, atm, resp), " ");
        moo_try_check(result = moo_products_add_rbn(
                          products, rbn, CPL_FRAME_LEVEL_FINAL,
                          MOONS_TAG_SCIENCE_XSWITCH_RBNFLXSPECTRA,
                          calibflux_filename, ref_frame),
                      " ");
    }
    else {
        cpl_msg_info(__func__, "No flux calibration");
    }

    if (tell_frame != NULL) {
        moo_try_check(telluric = moo_telluric_load(tell_frame), " ");

        tellcorr_filename =
            cpl_sprintf("%s_RBNTELLCORRSPECTRA_OFFSET%d_%s.fits", prefix,
                        offset, mode);
        moo_try_check(moo_correct_tell(rbn, telluric), " ");
        moo_try_check(result = moo_products_add_rbn(
                          products, rbn, CPL_FRAME_LEVEL_FINAL,
                          MOONS_TAG_SCIENCE_XSWITCH_RBNTELLCORRSPECTRA,
                          tellcorr_filename, ref_frame),
                      " ");
    }
    else {
        cpl_msg_info(__func__, "No telluric correction");
    }
    result = cpl_frame_duplicate(result);

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_frame_delete(result);
        result = NULL;
    }
    moo_detlist_delete(detlist);
    moo_crh_params_delete(crh_params);
    moo_det_delete(combined);
    cpl_free(comb_filename);

    moo_ext_delete(ext);
    moo_psf_delete(psf);
    cpl_free(ext_filename);
    cpl_free(extff_filename);
    cpl_free(rbn_filename);
    moo_rbn_delete(rbn);

    cpl_free(tellcorr_filename);
    cpl_free(calibflux_filename);

    moo_atm_delete(atm);
    moo_resp_delete(resp);
    moo_telluric_delete(telluric);

    return result;
}

static cpl_frame *
_moons_xswitch_sci1d(int inum,
                     const char *mode,
                     int offset,
                     const cpl_frame *obja_frame,
                     const cpl_frame *obja_rbn_frame,
                     const cpl_frame *objb_frame,
                     const cpl_frame *objb_rbn_frame,
                     moo_sky_lines_list *sky_lines,
                     const cpl_frame *f2f_frame,
                     const cpl_frame *airglow_group_frame,
                     const cpl_frame *airglow_var_frame,
                     const cpl_frame *solflux_frame,
                     moo_target_table *target_table,
                     moo_sub_sky_stare_params *sky_params,
                     moo_products *products,
                     const char *prefix,
                     cpl_frame **result_psci1d)
{
    cpl_frame *result_sci1d = NULL;
    moo_rbn *objb_rbn = NULL;
    moo_rbn *obja_rbn = NULL;
    moo_sci *sci1d = NULL;
    moo_sci *psci1d = NULL;
    moo_f2f *f2f = NULL;

    char *sci1d_filename = NULL;
    char *psci1d_filename = NULL;

    cpl_ensure(obja_rbn_frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(objb_rbn_frame != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    moo_try_check(obja_rbn = moo_rbn_create(obja_rbn_frame), " ");
    moo_try_check(objb_rbn = moo_rbn_create(objb_rbn_frame), " ");

    sci1d_filename =
        cpl_sprintf("%s_SCI1D_OFFSET%d_%s_%d.fits", prefix, offset, mode, inum);

    if (f2f_frame != NULL) {
        f2f = moo_f2f_load(f2f_frame);
    }
    moo_try_check(sci1d =
                      moo_sub_sky_stare(obja_rbn, target_table, objb_rbn, f2f,
                                        solflux_frame, airglow_group_frame,
                                        airglow_var_frame, sky_params,
                                        sci1d_filename, MOO_SCI1D_NOT_PAIRED),
                  " ");

    moo_try_check(moo_sci_compute_snr(sci1d, sky_lines), " ");

    moo_try_check(result_sci1d =
                      moo_products_add_sci(products, sci1d,
                                           CPL_FRAME_LEVEL_FINAL,
                                           MOONS_TAG_SCIENCE_XSWITCH_SKSSPECTRA,
                                           sci1d_filename, obja_frame),
                  " ");
    psci1d_filename = cpl_sprintf("%s_SCI1D_PAIRED_OFFSET%d_%s_%d.fits", prefix,
                                  offset, mode, inum);
    moo_try_check(psci1d =
                      moo_sub_sky_stare(obja_rbn, target_table, objb_rbn, f2f,
                                        solflux_frame, airglow_group_frame,
                                        airglow_var_frame, sky_params,
                                        psci1d_filename, MOO_SCI1D_PAIRED),
                  " ");

    moo_try_check(moo_sci_compute_snr(psci1d, sky_lines), " ");

    moo_try_check(*result_psci1d =
                      moo_products_add_sci(products, psci1d,
                                           CPL_FRAME_LEVEL_FINAL,
                                           MOONS_TAG_SCIENCE_XSWITCH_SKSSPECTRA,
                                           psci1d_filename, objb_frame),
                  " ");

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_frame_delete(result_sci1d);
        result_sci1d = NULL;
        *result_psci1d = NULL;
    }
    cpl_free(sci1d_filename);
    cpl_free(psci1d_filename);
    moo_sci_delete(sci1d);
    moo_sci_delete(psci1d);
    moo_rbn_delete(objb_rbn);
    moo_rbn_delete(obja_rbn);
    moo_f2f_delete(f2f);
    return result_sci1d;
}

static cpl_frame *
_moons_xswitch_sci2d(const char *mode,
                     int offset,
                     const cpl_frame *ref_frame,
                     const cpl_frame *nod_rbn_frame,
                     const cpl_frameset *objB_rbn_frameset,
                     const cpl_frameset *objA_rbn_frameset,
                     moo_sky_lines_list *sky_lines,
                     moo_target_table *target_table,
                     int ispaired,
                     moo_products *products,
                     const char *prefix)
{
    cpl_frame *result = NULL;
    moo_rbnlist *objA_rbnlist = NULL;
    moo_rbnlist *objB_rbnlist = NULL;
    moo_rbn *nod_rbn = NULL;
    moo_rbn *objB_rbn = NULL;
    moo_rbn *objA_rbn = NULL;

    moo_sci *sci2d = NULL;
    char *sci2d_filename = NULL;

    cpl_ensure(nod_rbn_frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(objA_rbn_frameset != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(objB_rbn_frameset != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    moo_try_check(nod_rbn = moo_rbn_create(nod_rbn_frame), " ");
    moo_try_check(objA_rbnlist = moo_rbnlist_create(objA_rbn_frameset), " ");
    moo_try_check(objB_rbnlist = moo_rbnlist_create(objB_rbn_frameset), " ");
    moo_try_check(objA_rbn = moo_combine_sky(objA_rbnlist), " ");
    moo_try_check(objB_rbn = moo_combine_sky(objB_rbnlist), " ");

    sci2d_filename =
        cpl_sprintf("%s_SCI2D_OFFSET%d_%s.fits", prefix, offset, mode);

    moo_try_check(sci2d =
                      moo_create_sci2d(nod_rbn, objB_rbn, objA_rbn,
                                       target_table, ispaired, sci2d_filename),
                  " ");

    moo_try_check(moo_sci_compute_snr(sci2d, sky_lines), " ");

    moo_try_check(result =
                      moo_products_add_sci(products, sci2d,
                                           CPL_FRAME_LEVEL_FINAL,
                                           MOONS_TAG_SCIENCE_XSWITCH_SKSSPECTRA,
                                           sci2d_filename, ref_frame),
                  " ");

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_frame_delete(result);
        result = NULL;
    }
    cpl_free(sci2d_filename);
    moo_sci_delete(sci2d);
    moo_rbn_delete(nod_rbn);
    moo_rbn_delete(objA_rbn);
    moo_rbn_delete(objB_rbn);
    moo_rbnlist_delete(objA_rbnlist);
    moo_rbnlist_delete(objB_rbnlist);
    return result;
}

static cpl_frame *
_moons_science_xswitch2_byoffset(cpl_frameset *frameset,
                                 cpl_frameset *ref_frameset,
                                 const char *bpmap_rp_name,
                                 const char *bpmap_nl_name,
                                 const cpl_frame *masterbias,
                                 const cpl_frame *masterdark_vis,
                                 cpl_frameset *masterdark_nir,
                                 const cpl_frame *p2pmap,
                                 const cpl_frame *sformat_frame,
                                 moo_sky_lines_list *sky_lines,
                                 const cpl_frame *airglow_group_frame,
                                 const cpl_frame *solflux_frame,
                                 const cpl_frame *airglow_var_frame,
                                 const cpl_frame *atmo_frame,
                                 const cpl_frame *resp_frame,
                                 const cpl_frame *tell_frame,
                                 moo_prepare_params *prepare_params,
                                 moo_correct_bias_params *correct_bias_params,
                                 moo_extract_params *extract_params,
                                 moo_wavesol_params *wavesol_params,
                                 moo_rebin_params *rbn_params,
                                 moo_target_table_params *target_table_params,
                                 moo_sub_sky_stare_params *sky_params,
                                 moo_coadd_params *coadd_params,
                                 moo_products *products)
{
    /* parameters */
    cpl_frame *result = NULL;
    moo_loc *loc = NULL;
    moo_spectral_format *sformat = NULL;
    moo_map *wmap = NULL;
    moo_map *wmap_tuned = NULL;
    moo_extlist *objA_extlist = NULL;
    moo_extlist *objB_extlist = NULL;
    moo_extlist *wavemap_extlist = NULL;
    moo_ext *sum_ext = NULL;

    cpl_frameset *objA_frameset = NULL;
    cpl_frameset *objB_frameset = NULL;

    cpl_frameset *objA_det_frameset = NULL;
    cpl_frameset *objB_det_frameset = NULL;
    cpl_frameset *nod_det_frameset = NULL;

    cpl_frameset *objB_ext_frameset = NULL;
    cpl_frameset *objA_ext_frameset = NULL;

    cpl_frameset *objB_rbn_frameset = NULL;
    cpl_frameset *objA_rbn_frameset = NULL;
    cpl_frame *nod_rbn_frame = NULL;

    cpl_frameset *objB_ref_frameset = NULL;
    cpl_frameset *objA_sci1d_frameset = NULL;
    cpl_frameset *objB_sci1d_frameset = NULL;

    const cpl_frame *master_flat = NULL;
    const cpl_frame *fftrace = NULL;
    const cpl_frame *coeffs_cube = NULL;
    const cpl_frame *wmap_frame = NULL;
    const cpl_frame *flat_frame = NULL;
    const cpl_frame *f2f_frame = NULL;

    char *sum_ext_filename = NULL;
    const char *refA_filename = NULL;
    cpl_propertylist *refA_header = NULL;

    moo_mode_type mode;
    const char *mode_name = NULL;
    int offset;
    char *wmap_tuned_filename = NULL;
    int wavemap_extlist_size = 0;

    moo_target_table *xswitch_target_table = NULL;
    char *xswitch_ttable_filename = NULL;

    moo_scilist *sci1d_list = NULL;
    moo_sci *sci1d_coadd = NULL;
    char *coadd1d_filename = NULL;

    moo_scilist *psci1d_list = NULL;
    moo_sci *psci1d_coadd = NULL;
    char *pcoadd1d_filename = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();
    objA_frameset = cpl_frameset_new();
    objB_frameset = cpl_frameset_new();

    objA_det_frameset = cpl_frameset_new();
    objB_det_frameset = cpl_frameset_new();
    nod_det_frameset = cpl_frameset_new();

    objA_ext_frameset = cpl_frameset_new();
    objB_ext_frameset = cpl_frameset_new();

    objA_rbn_frameset = cpl_frameset_new();
    objB_rbn_frameset = cpl_frameset_new();

    objA_sci1d_frameset = cpl_frameset_new();
    objB_sci1d_frameset = cpl_frameset_new();

    objB_ref_frameset = cpl_frameset_new();

    moo_try_check(_moons_science_xswitch2_check_sof_byoffset(
                      frameset, objA_frameset, objB_frameset, &master_flat,
                      &fftrace, &coeffs_cube, &wmap_frame, &flat_frame,
                      &f2f_frame, extract_params),
                  " ");

    int nb_objA = cpl_frameset_get_size(objA_frameset);
    int nb_objB = cpl_frameset_get_size(objB_frameset);
    cpl_frame *ref_frameA = cpl_frameset_get_position(objA_frameset, 0);
    cpl_frameset_insert(ref_frameset, cpl_frame_duplicate(ref_frameA));

    cpl_frame *ref_frameB = cpl_frameset_get_position(objB_frameset, 0);
    refA_filename = cpl_frame_get_filename(ref_frameA);

    moo_try_check(mode = moo_mode_get(ref_frameA), " ");
    mode_name = moo_mode_get_name(mode);

    moo_try_check(refA_header = cpl_propertylist_load(refA_filename, 0), " ");
    moo_try_check(offset = moo_pfits_get_slit_offset(refA_header), " ");
    cpl_propertylist_delete(refA_header);
    refA_header = NULL;

    cpl_msg_info(__func__, "Do offset %d use mode %s", offset, mode_name);

    for (int i = 0; i < nb_objA; i++) {
        cpl_msg_info(__func__, "---Prepare object A: %d/%d", i + 1, nb_objA);
        cpl_frame *objA_frame = NULL;
        moo_try_check(objA_frame = cpl_frameset_get_position(objA_frameset, i),
                      " ");
        cpl_frame *frame =
            _moons_xswitch_prepare(i, mode_name, offset, objA_frame,
                                   bpmap_rp_name, bpmap_nl_name, masterbias,
                                   masterdark_vis, masterdark_nir, p2pmap,
                                   coeffs_cube, prepare_params,
                                   correct_bias_params, products, "A");
        cpl_frameset_insert(objA_det_frameset, cpl_frame_duplicate(frame));
    }

    for (int i = 0; i < nb_objB; i++) {
        cpl_msg_info(__func__, "---Prepare object B: %d/%d", i + 1, nb_objA);
        cpl_frame *objB_frame = NULL;
        moo_try_check(objB_frame = cpl_frameset_get_position(objB_frameset, i),
                      " ");
        cpl_frame *frame =
            _moons_xswitch_prepare(i, mode_name, offset, objB_frame,
                                   bpmap_rp_name, bpmap_nl_name, masterbias,
                                   masterdark_vis, masterdark_nir, p2pmap,
                                   coeffs_cube, prepare_params,
                                   correct_bias_params, products, "B");
        cpl_frameset_insert(objB_det_frameset, cpl_frame_duplicate(frame));
    }
    moo_try_check(sformat = moo_spectral_format_load(sformat_frame, mode), " ");
    moo_try_check(wmap = moo_map_load(wmap_frame), " ");
    moo_try_check(loc = moo_loc_load(fftrace), " ");

    for (int i = 0; i < nb_objA; i++) {
        cpl_frame *objA_frame = NULL;
        cpl_frame *objA_det_frame = NULL;
        cpl_frame *objB_det_frame = NULL;
        moo_try_check(objA_frame = cpl_frameset_get_position(objA_frameset, i),
                      " ");
        moo_try_check(objA_det_frame =
                          cpl_frameset_get_position(objA_det_frameset, i),
                      " ");
        moo_try_check(objB_det_frame =
                          cpl_frameset_get_position(objB_det_frameset, i),
                      " ");
        cpl_msg_info(__func__, "---Prepare nod %d/%d using object A:%s B:%s",
                     i + 1, nb_objA, cpl_frame_get_filename(objA_det_frame),
                     cpl_frame_get_filename(objB_det_frame));
        cpl_frame *frame =
            _moons_xswitch_subtractnod(i, mode_name, offset, objA_frame,
                                       objA_det_frame, objB_det_frame,
                                       products);
        cpl_frameset_insert(nod_det_frameset, cpl_frame_duplicate(frame));
    }
    int nb_nod = cpl_frameset_get_size(nod_det_frameset);

    for (int i = 0; i < nb_objA; i++) {
        cpl_frame *objA_frame = NULL;
        cpl_frame *objA_det_frame = NULL;
        cpl_msg_info(__func__, "---Extract object A: %d/%d", i + 1, nb_objA);
        moo_try_check(objA_frame = cpl_frameset_get_position(objA_frameset, i),
                      " ");
        moo_try_check(objA_det_frame =
                          cpl_frameset_get_position(objA_det_frameset, i),
                      " ");
        cpl_frame *frame =
            _moons_xswitch_extract(i, mode_name, offset, objA_frame,
                                   objA_det_frame, loc, wmap, sformat,
                                   sky_lines, master_flat, flat_frame,
                                   f2f_frame, extract_params, products,
                                   "XSWITCH_OBJECTA");
        cpl_frameset_insert(objA_ext_frameset, cpl_frame_duplicate(frame));
    }

    for (int i = 0; i < nb_objB; i++) {
        cpl_frame *objB_frame = NULL;
        cpl_frame *objB_det_frame = NULL;
        cpl_msg_info(__func__, "---Extract object B: %d/%d", i + 1, nb_objB);
        moo_try_check(objB_frame = cpl_frameset_get_position(objB_frameset, i),
                      " ");
        moo_try_check(objB_det_frame =
                          cpl_frameset_get_position(objB_det_frameset, i),
                      " ");
        cpl_frame *frame =
            _moons_xswitch_extract(i, mode_name, offset, objB_frame,
                                   objB_det_frame, loc, wmap, sformat,
                                   sky_lines, master_flat, flat_frame,
                                   f2f_frame, extract_params, products,
                                   "XSWITCH_OBJECTB");
        cpl_frameset_insert(objB_ext_frameset, cpl_frame_duplicate(frame));
    }

    objA_extlist = moo_extlist_create(objA_ext_frameset);
    objB_extlist = moo_extlist_create(objB_ext_frameset);
    wavemap_extlist = moo_extlist_new();

    for (int i = 0; i < nb_objA; i++) {
        moo_ext *ext = moo_extlist_get(objA_extlist, i);
        moo_extlist_set(wavemap_extlist, ext, wavemap_extlist_size);
        wavemap_extlist_size++;
    }
    for (int i = 0; i < nb_objB; i++) {
        moo_ext *ext = moo_extlist_get(objB_extlist, i);
        moo_extlist_set(wavemap_extlist, ext, wavemap_extlist_size);
        wavemap_extlist_size++;
    }

    sum_ext_filename =
        cpl_sprintf("XSWITCH_EXT_SUM_OFFSET%d_%s.fits", offset, mode_name);
    sum_ext = moo_extlist_sum(wavemap_extlist, sum_ext_filename);
    moo_try_check(moo_products_add_ext(products, sum_ext,
                                       CPL_FRAME_LEVEL_TEMPORARY,
                                       MOONS_TAG_SCIENCE_XSWITCH_EXTSPECTRA,
                                       sum_ext_filename, ref_frameA),
                  " ");
    moo_map *used_wmap = NULL;
    const char *wavesol_control = wavesol_params->control;
    if ((strcmp(wavesol_control, MOO_WAVESOL_CONTROL_CHECK) == 0) ||
        (strcmp(wavesol_control, MOO_WAVESOL_CONTROL_UPDATE) == 0)) {
        const char *sky_lines_name = sky_lines->filename;
        moo_try_check(wmap_tuned = moo_wavesol(sum_ext, sky_lines_name, sformat,
                                               loc, wmap, wavesol_params),
                      " ");
        if (strcmp(wavesol_control, MOO_WAVESOL_CONTROL_UPDATE) == 0) {
            used_wmap = wmap_tuned;
            wmap_tuned_filename =
                cpl_sprintf("XSWICTH_WMAP_TUNED_OFFSET%d_%s.fits", offset,
                            mode_name);
            moo_map_save(wmap_tuned, wmap_tuned_filename);
        }
    }
    else {
        used_wmap = wmap;
    }

    for (int i = 0; i < nb_objA; i++) {
        cpl_frame *objA_frame = NULL;
        moo_ext *objA_ext = moo_extlist_get(objA_extlist, i);
        cpl_msg_info(__func__, "---Rebin object A: %d/%d", i + 1, nb_objA);
        moo_try_check(objA_frame = cpl_frameset_get_position(objA_frameset, i),
                      " ");
        cpl_frame *frame =
            _moons_xswitch_rebin(i, mode_name, offset, objA_ext, objA_frame,
                                 sformat, sky_lines, used_wmap, atmo_frame,
                                 resp_frame, tell_frame, rbn_params, products,
                                 "XSWITCH_OBJECTA");
        cpl_frameset_insert(objA_rbn_frameset, cpl_frame_duplicate(frame));
    }

    for (int i = 0; i < nb_objB; i++) {
        cpl_frame *objB_frame = NULL;
        moo_ext *objB_ext = moo_extlist_get(objB_extlist, i);
        cpl_msg_info(__func__, "---Rebin object B: %d/%d", i + 1, nb_objB);
        moo_try_check(objB_frame = cpl_frameset_get_position(objB_frameset, i),
                      " ");
        cpl_frame *frame =
            _moons_xswitch_rebin(i, mode_name, offset, objB_ext, objB_frame,
                                 sformat, sky_lines, used_wmap, atmo_frame,
                                 resp_frame, tell_frame, rbn_params, products,
                                 "XSWITCH_OBJECTB");
        cpl_frameset_insert(objB_rbn_frameset, cpl_frame_duplicate(frame));
    }

    if (nb_nod > 0) {
        moo_try_check(nod_rbn_frame =
                          _xswitch_nod_rebin(nod_det_frameset, mode_name,
                                             offset, ref_frameA, loc, used_wmap,
                                             sformat, sky_lines, master_flat,
                                             flat_frame, f2f_frame, atmo_frame,
                                             resp_frame, tell_frame,
                                             extract_params, rbn_params,
                                             products, "XSWITCH_NOD"),
                      " ");

        cpl_frame *objA_rbn_frame = NULL;
        cpl_frame *objB_rbn_frame = NULL;
        cpl_frame *sci_frameA = NULL;
        cpl_frame *sci_frameB = NULL;

        moo_try_check(objA_rbn_frame =
                          cpl_frameset_get_position(objA_rbn_frameset, 0),
                      " ");
        moo_try_check(objB_rbn_frame =
                          cpl_frameset_get_position(objB_rbn_frameset, 0),
                      " ");

        moo_try_check(xswitch_target_table =
                          _xswitch_target_table(objA_rbn_frame, objB_rbn_frame,
                                                MOO_MODE_XSWITCH,
                                                target_table_params),
                      " ");
        xswitch_ttable_filename =
            cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_XSWITCH_TARGET_TABLE,
                        offset, mode_name);

        moo_try_check(
            moo_products_add_target_table(products, xswitch_target_table,
                                          CPL_FRAME_LEVEL_INTERMEDIATE,
                                          MOONS_TAG_XSWITCH_TARGET_TABLE,
                                          xswitch_ttable_filename, ref_frameA),
            " ");

        moo_try_check(sci_frameA =
                          _moons_xswitch_sci2d(mode_name, offset, ref_frameA,
                                               nod_rbn_frame, objB_rbn_frameset,
                                               objA_rbn_frameset, sky_lines,
                                               xswitch_target_table,
                                               MOO_SCI2D_NOT_PAIRED, products,
                                               "XSWITCH_NOD"),
                      " ");
        moo_try_check(sci_frameB =
                          _moons_xswitch_sci2d(mode_name, offset, ref_frameB,
                                               nod_rbn_frame, objB_rbn_frameset,
                                               objA_rbn_frameset, sky_lines,
                                               xswitch_target_table,
                                               MOO_SCI2D_PAIRED, products,
                                               "XSWITCH_NOD_PAIRED"),
                      " ");
        for (int i = 0; i < nb_objA; i++) {
            cpl_frame *objA_frame = NULL;
            cpl_frame *objA_rbn_framel = NULL;
            cpl_frame *objB_frame = NULL;
            cpl_frame *objB_rbn_framel = NULL;
            cpl_frame *sci_frame = NULL;
            cpl_frame *psci_frame = NULL;
            moo_try_check(objA_frame =
                              cpl_frameset_get_position(objA_frameset, i),
                          " ");
            moo_try_check(objA_rbn_framel =
                              cpl_frameset_get_position(objA_rbn_frameset, i),
                          " ");
            moo_try_check(objB_frame =
                              cpl_frameset_get_position(objB_frameset, i),
                          " ");
            moo_try_check(objB_rbn_framel =
                              cpl_frameset_get_position(objB_rbn_frameset, i),
                          " ");

            cpl_msg_info(__func__,
                         "---Create sci1d: %d/%d use objectA:%s objectB: %s",
                         i + 1, nb_objA,
                         cpl_frame_get_filename(objA_rbn_framel),
                         cpl_frame_get_filename(objB_rbn_framel));

            moo_try_check(sci_frame = _moons_xswitch_sci1d(
                              i, mode_name, offset, objA_frame, objA_rbn_framel,
                              objB_frame, objB_rbn_framel, sky_lines, f2f_frame,
                              airglow_group_frame, airglow_var_frame,
                              solflux_frame, xswitch_target_table, sky_params,
                              products, "XSWITCH_OBJECTA", &psci_frame),
                          " ");
            cpl_frameset_insert(objA_sci1d_frameset,
                                cpl_frame_duplicate(sci_frame));
            cpl_frameset_insert(objB_sci1d_frameset,
                                cpl_frame_duplicate(psci_frame));
        }
        int sci1d_size = cpl_frameset_get_size(objA_sci1d_frameset);
        if (sci1d_size > 0) {
            sci1d_list = moo_scilist_create(objA_sci1d_frameset);
            coadd1d_filename = cpl_sprintf("XSWITCH_SCI1D_OFFSET%d_%s.fits",
                                           offset, moo_mode_get_name(mode));

            sci1d_coadd = moo_coadd(sci1d_list, coadd_params, coadd1d_filename);

            moo_try_check(moo_sci_compute_snr(sci1d_coadd, sky_lines), " ");

            moo_try_check(moo_products_add_sci(products, sci1d_coadd,
                                               CPL_FRAME_LEVEL_FINAL,
                                               MOONS_TAG_XSWITCH_SKSSPECTRA,
                                               coadd1d_filename, ref_frameA),
                          " ");
        }
        int psci1d_size = cpl_frameset_get_size(objA_sci1d_frameset);
        if (psci1d_size > 0) {
            psci1d_list = moo_scilist_create(objB_sci1d_frameset);
            pcoadd1d_filename =
                cpl_sprintf("XSWITCH_SCI1D_PAIRED_OFFSET%d_%s.fits", offset,
                            moo_mode_get_name(mode));

            psci1d_coadd =
                moo_coadd(psci1d_list, coadd_params, pcoadd1d_filename);

            moo_try_check(moo_sci_compute_snr(psci1d_coadd, sky_lines), " ");

            moo_try_check(moo_products_add_sci(products, psci1d_coadd,
                                               CPL_FRAME_LEVEL_FINAL,
                                               MOONS_TAG_XSWITCH_SKSSPECTRA,
                                               pcoadd1d_filename, ref_frameB),
                          " ");
        }
        {
            moo_sci *sci2d = NULL;
            moo_sci *psci2d = NULL;
            moo_sci *scicomb = NULL;
            char *result_filename = NULL;

            sci2d = moo_sci_create(sci_frameA);
            psci2d = moo_sci_create(sci_frameB);
            result_filename = cpl_sprintf("XSWITCH_SCI_OFFSET%d_%s.fits",
                                          offset, moo_mode_get_name(mode));

            scicomb = moo_combine_pair_sci(xswitch_target_table, sci1d_coadd,
                                           psci1d_coadd, sci2d, psci2d,
                                           result_filename);

            moo_try_check(moo_sci_compute_snr(scicomb, sky_lines), " ");

            moo_try_check(result = cpl_frame_duplicate(moo_products_add_sci(
                              products, scicomb, CPL_FRAME_LEVEL_FINAL,
                              MOONS_TAG_XSWITCH_SKSSPECTRA, result_filename,
                              ref_frameA)),
                          " ");

            cpl_free(result_filename);
            moo_sci_delete(scicomb);
            moo_sci_delete(sci2d);
            moo_sci_delete(psci2d);
        }
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_propertylist_delete(refA_header);
    }
    moo_extlist_delete(objA_extlist);
    moo_extlist_delete(objB_extlist);

    if (wavemap_extlist != NULL) {
        for (int i = 0; i < nb_objA * 2; i++) {
            moo_extlist_unset(wavemap_extlist, 0);
        }
    }
    moo_extlist_delete(wavemap_extlist);
    moo_ext_delete(sum_ext);
    moo_loc_delete(loc);
    moo_map_delete(wmap);
    moo_map_delete(wmap_tuned);
    moo_target_table_delete(xswitch_target_table);
    cpl_free(xswitch_ttable_filename);

    cpl_free(sum_ext_filename);
    cpl_free(wmap_tuned_filename);
    moo_spectral_format_delete(sformat);

    cpl_frameset_delete(objA_frameset);
    cpl_frameset_delete(objB_frameset);

    cpl_frameset_delete(nod_det_frameset);
    cpl_frameset_delete(objA_det_frameset);
    cpl_frameset_delete(objB_det_frameset);

    cpl_frameset_delete(objB_ext_frameset);
    cpl_frameset_delete(objA_ext_frameset);

    cpl_frameset_delete(objB_rbn_frameset);
    cpl_frameset_delete(objA_rbn_frameset);

    cpl_frameset_delete(objA_sci1d_frameset);
    cpl_frameset_delete(objB_sci1d_frameset);
    cpl_frame_delete(nod_rbn_frame);

    cpl_frameset_delete(objB_ref_frameset);

    cpl_free(coadd1d_filename);
    moo_sci_delete(sci1d_coadd);
    moo_scilist_delete(sci1d_list);

    cpl_free(pcoadd1d_filename);
    moo_sci_delete(psci1d_coadd);
    moo_scilist_delete(psci1d_list);

    return result;
}

static int
_moons_science_xswitch2(cpl_frameset *frameset,
                        const cpl_parameterlist *parlist)
{
    /* parameters */
    moo_prepare_params *prepare_params = NULL;
    moo_correct_bias_params *correct_bias_params = NULL;
    moo_extract_params *extract_params = NULL;
    moo_wavesol_params *wavesol_params = NULL;
    moo_rebin_params *rbn_params = NULL;
    moo_target_table_params *target_table_params = NULL;
    moo_combine_pair_params *combine_pair_params = NULL;
    moo_sub_sky_stare_params *sky_params = NULL;
    moo_coadd_params *coadd_params = NULL;
    moo_create_s1d_params *s1d_params = NULL;

    moo_scilist *sci_list = NULL;
    moo_sci *sci_coadd = NULL;

    cpl_frameset *offset_frameset = NULL;
    cpl_frameset *set = NULL;
    cpl_frameset *sci_set = NULL;
    cpl_frameset *ref_frameset = NULL;

    const char *bpmap_rp_name = NULL;
    const char *bpmap_nl_name = NULL;
    const cpl_frame *masterbias = NULL;
    const cpl_frame *masterdark_vis = NULL;
    cpl_frameset *masterdark_nir = NULL;
    const cpl_frame *p2pmap = NULL;
    const cpl_frame *sformat_frame = NULL;
    const cpl_frame *sky_lines_frame = NULL;
    const cpl_frame *atmo_frame = NULL;
    const cpl_frame *resp_frame = NULL;
    const cpl_frame *tell_frame = NULL;

    const cpl_frame *airglow_group_frame = NULL;
    const cpl_frame *solflux_frame = NULL;
    const cpl_frame *airglow_var_frame = NULL;
    const cpl_frame *ref_frame = NULL;
    cpl_size nsel = 0;
    cpl_size *selection = NULL;
    moo_sky_lines_list *skylines = NULL;
    char *coadd_filename = NULL;
    char *calibflux_filename = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    moo_products *products =
        moo_products_new(frameset, parlist, "moons_science_xswitch2",
                         PACKAGE "/" PACKAGE_VERSION);

    const moo_params *params = moo_products_get_params(products);
    moo_try_check(prepare_params = moo_params_get_prepare(params, parlist),
                  " ");
    moo_try_check(correct_bias_params =
                      moo_params_get_correct_bias(params, parlist),
                  " ");
    moo_try_check(extract_params = moo_params_get_extract(params, parlist),
                  " ");
    moo_try_check(rbn_params = moo_params_get_rebin(params, parlist), " ");
    moo_try_check(target_table_params =
                      moo_params_get_target_table(params, parlist),
                  " ");
    moo_try_check(combine_pair_params =
                      moo_params_get_combine_pair(params, parlist),
                  " ");
    moo_try_check(sky_params = moo_params_get_sub_sky_stare(params, parlist),
                  " ");
    moo_try_check(coadd_params = moo_params_get_coadd(params, parlist), " ");
    moo_try_check(wavesol_params =
                      moo_params_get_science_wavesol(params, parlist),
                  " ");
    moo_try_check(s1d_params = moo_params_get_create_s1d(params, parlist), " ");
    /* SOF file */
    offset_frameset = cpl_frameset_new();
    sci_set = cpl_frameset_new();

    ref_frameset = cpl_frameset_new();
    masterdark_nir = cpl_frameset_new();

    moo_try_check(_moons_science_xswitch2_check_sof(
                      frameset, offset_frameset, &bpmap_rp_name, &bpmap_nl_name,
                      &masterbias, &masterdark_vis, masterdark_nir, &p2pmap,
                      &sformat_frame, &sky_lines_frame, &atmo_frame,
                      &resp_frame, &tell_frame, &airglow_group_frame,
                      &airglow_var_frame, &solflux_frame, sky_params),
                  " ");

    // cpl_frameset_sort(offset_frameset, _moons_sort_mjdobs);
    moo_try_check(selection =
                      cpl_frameset_labelise(offset_frameset,
                                            &_moons_compare_offset, &nsel),
                  " ");

    moo_try_check(skylines = moo_sky_lines_list_load(sky_lines_frame), " ");

    for (int i = 0; i < nsel; i++) {
        cpl_frame *result = NULL;
        moo_try_check(set = cpl_frameset_extract(offset_frameset, selection, i),
                      " ");
        moo_try_check(result = _moons_science_xswitch2_byoffset(
                          set, ref_frameset, bpmap_rp_name, bpmap_nl_name,
                          masterbias, masterdark_vis, masterdark_nir, p2pmap,
                          sformat_frame, skylines, airglow_group_frame,
                          solflux_frame, airglow_var_frame, atmo_frame,
                          resp_frame, tell_frame, prepare_params,
                          correct_bias_params, extract_params, wavesol_params,
                          rbn_params, target_table_params, sky_params,
                          coadd_params, products),
                      " ");
        cpl_frameset_insert(sci_set, result);
        cpl_frameset_delete(set);
        set = NULL;
    }

    moo_try_check(ref_frame = cpl_frameset_get_position_const(ref_frameset, 0),
                  " ");
    moo_mode_type mode = moo_mode_get(ref_frame);

    sci_list = moo_scilist_create(sci_set);
    coadd_filename =
        cpl_sprintf("%s_%s.fits", MOONS_TAG_XSWITCH2_COMBINED_SPECTRA,
                    moo_mode_get_name(mode));

    sci_coadd = moo_coadd(sci_list, coadd_params, coadd_filename);

    moo_try_check(moo_sci_compute_snr(sci_coadd, skylines), " ");
    cpl_frame *sci_final = NULL;

    moo_try_check(sci_final =
                      moo_products_add_sci(products, sci_coadd,
                                           CPL_FRAME_LEVEL_FINAL,
                                           MOONS_TAG_XSWITCH2_COMBINED_SPECTRA,
                                           coadd_filename, ref_frame),
                  " ");
    moo_create_s1d(sci_final, s1d_params, MOONS_TAG_SCIENCE_XSWITCH2_1DSPECTRUM,
                   products);


moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_frameset_delete(set);
    }
    cpl_free(coadd_filename);
    cpl_free(calibflux_filename);
    cpl_free(selection);
    moo_sci_delete(sci_coadd);
    moo_scilist_delete(sci_list);
    moo_sky_lines_list_delete(skylines);

    cpl_frameset_delete(sci_set);

    cpl_frameset_delete(ref_frameset);
    cpl_frameset_delete(offset_frameset);
    cpl_frameset_delete(masterdark_nir);

    moo_extract_params_delete(extract_params);
    moo_wavesol_params_delete(wavesol_params);
    moo_rebin_params_delete(rbn_params);
    moo_target_table_params_delete(target_table_params);

    moo_combine_pair_params_delete(combine_pair_params);
    moo_sub_sky_stare_params_delete(sky_params);
    moo_coadd_params_delete(coadd_params);
    moo_correct_bias_params_delete(correct_bias_params);
    moo_prepare_params_delete(prepare_params);
    moo_create_s1d_params_delete(s1d_params);
    moo_products_delete(products);
    return (int)cpl_error_get_code();
}
