/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "mf_molecules.h"

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup mf_molec   Tools to work with molecules.
 *
 * @brief
 *
 *  List of valid molecules in LNFL/LBLRTM:
 *
 *   1:H2O     2:CO2     3:O3      4:N2O     5:CO      6:CH4     7:O2
 *   8:NO      9:SO2    10:NO2    11:NH3    12:HNO3   13:OH     14:HF
 *  15:HCL    16:HBR    17:HI     18:CLO    19:OCS    20:H2CO   21:HOCL
 *  22:N2     23:HCN    24:CH3CL  25:H2O2   26:C2H2   27:C2H6   28:PH3
 *  29:COF2   30:SF6    31:H2S    32:HCOOH  33:HO2    34:O      35:CLONO2
 *  36:NO+    37:HOBR   38:C2H4   39:CH3OH  40:CH3Br  41:CH3CN  42:CF4
 *  43:C4H2   44:HC3N   45:H2     46:CS     47:SO3
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* ---------------------------------------------------------------------------*/
/**
 * @brief Check molecules input cpl_table
 *
 * @param table              .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_error_code mf_molecules_check(const cpl_table *table)
{
    cpl_ensure(table, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

    cpl_error_ensure(
        cpl_table_has_column(table, MF_COL_LIST_MOLECULES) != 0 &&
            cpl_table_has_column(table, MF_COL_FIT_MOLECULES) != 0 &&
            cpl_table_has_column(table, MF_COL_REL_COL) != 0 && cpl_table_get_ncol(table) == 3 &&
            cpl_table_get_nrow(table) > 0,
        CPL_ERROR_INCOMPATIBLE_INPUT, return CPL_ERROR_INCOMPATIBLE_INPUT,
        "cpl_table *%s does not have the correct columns [%s, %s, %s]", MF_INPUT_MOLECULES, MF_COL_LIST_MOLECULES,
        MF_COL_FIT_MOLECULES, MF_COL_REL_COL
    );

    return CPL_ERROR_NONE;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Updates params based on contents of filled molecules. Fills nmolec and lbl_molecs
 *
 * @param molecules          .
 * @param n_molec            .
 * @param lbl_molecs         .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_error_code mf_molecules_lbl_fill(const cpl_table *molecules, int *n_molec, char **lbl_molecs)
{
    /* check inputs */
    cpl_ensure_code(molecules && n_molec && *lbl_molecs, CPL_ERROR_NULL_INPUT);

    /* Superfluous check to placate CLion. */
    if (!n_molec) {
        return CPL_ERROR_NULL_INPUT;
    }

    *n_molec = (int)cpl_table_get_nrow(molecules);

    for (cpl_size i = 0; i < *n_molec; i++) {
        const char *molec = cpl_table_get_string(molecules, MF_COL_LIST_MOLECULES, i);

        /* Set molecule */
        if (strcmp(molec, MF_MOLECULES_H2O) == 0) {
            *(*lbl_molecs + MF_MOLECULES_H2O_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_CO2) == 0) {
            *(*lbl_molecs + MF_MOLECULES_CO2_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_O3) == 0) {
            *(*lbl_molecs + MF_MOLECULES_O3_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_N2O) == 0) {
            *(*lbl_molecs + MF_MOLECULES_N2O_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_CO) == 0) {
            *(*lbl_molecs + MF_MOLECULES_CO_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_CH4) == 0) {
            *(*lbl_molecs + MF_MOLECULES_CH4_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_O2) == 0) {
            *(*lbl_molecs + MF_MOLECULES_O2_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_NO) == 0) {
            *(*lbl_molecs + MF_MOLECULES_NO_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_SO2) == 0) {
            *(*lbl_molecs + MF_MOLECULES_SO2_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_NO2) == 0) {
            *(*lbl_molecs + MF_MOLECULES_NO2_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_NH3) == 0) {
            *(*lbl_molecs + MF_MOLECULES_NH3_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_HNO3) == 0) {
            *(*lbl_molecs + MF_MOLECULES_HNO3_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_OH) == 0) {
            *(*lbl_molecs + MF_MOLECULES_OH_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_HF) == 0) {
            *(*lbl_molecs + MF_MOLECULES_HF_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_HCL) == 0) {
            *(*lbl_molecs + MF_MOLECULES_HCL_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_HBR) == 0) {
            *(*lbl_molecs + MF_MOLECULES_HBR_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_HI) == 0) {
            *(*lbl_molecs + MF_MOLECULES_HI_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_CLO) == 0) {
            *(*lbl_molecs + MF_MOLECULES_CLO_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_OCS) == 0) {
            *(*lbl_molecs + MF_MOLECULES_OCS_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_H2CO) == 0) {
            *(*lbl_molecs + MF_MOLECULES_H2CO_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_HOCL) == 0) {
            *(*lbl_molecs + MF_MOLECULES_HOCL_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_N2) == 0) {
            *(*lbl_molecs + MF_MOLECULES_N2_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_HCN) == 0) {
            *(*lbl_molecs + MF_MOLECULES_HCN_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_CH3CL) == 0) {
            *(*lbl_molecs + MF_MOLECULES_CH3CL_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_H2O2) == 0) {
            *(*lbl_molecs + MF_MOLECULES_H2O2_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_C2H2) == 0) {
            *(*lbl_molecs + MF_MOLECULES_C2H2_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_C2H6) == 0) {
            *(*lbl_molecs + MF_MOLECULES_C2H6_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_PH3) == 0) {
            *(*lbl_molecs + MF_MOLECULES_PH3_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_COF2) == 0) {
            *(*lbl_molecs + MF_MOLECULES_COF2_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_SF6) == 0) {
            *(*lbl_molecs + MF_MOLECULES_SF6_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_H2S) == 0) {
            *(*lbl_molecs + MF_MOLECULES_H2S_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_HCOOH) == 0) {
            *(*lbl_molecs + MF_MOLECULES_HCOOH_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_HO2) == 0) {
            *(*lbl_molecs + MF_MOLECULES_HO2_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_O) == 0) {
            *(*lbl_molecs + MF_MOLECULES_O_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_CLONO2) == 0) {
            *(*lbl_molecs + MF_MOLECULES_CLONO2_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_NO_P) == 0) {
            *(*lbl_molecs + MF_MOLECULES_NO_P_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_HOBR) == 0) {
            *(*lbl_molecs + MF_MOLECULES_HOBR_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_C2H4) == 0) {
            *(*lbl_molecs + MF_MOLECULES_C2H4_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_CH3OH) == 0) {
            *(*lbl_molecs + MF_MOLECULES_CH3OH_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_CH3BR) == 0) {
            *(*lbl_molecs + MF_MOLECULES_CH3BR_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_CH3CN) == 0) {
            *(*lbl_molecs + MF_MOLECULES_CH3CN_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_CF4) == 0) {
            *(*lbl_molecs + MF_MOLECULES_CF4_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_C4H2) == 0) {
            *(*lbl_molecs + MF_MOLECULES_C4H2_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_HC3N) == 0) {
            *(*lbl_molecs + MF_MOLECULES_HC3N_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_H2) == 0) {
            *(*lbl_molecs + MF_MOLECULES_H2_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_CS) == 0) {
            *(*lbl_molecs + MF_MOLECULES_CS_POS) = '1';
        }
        else if (strcmp(molec, MF_MOLECULES_SO3) == 0) {
            *(*lbl_molecs + MF_MOLECULES_SO3_POS) = '1';
        }
    }

    return cpl_error_get_code();
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Create molecules cpl_array.
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_array *mf_molecules_create_array(void)
{
    /*!
   * \brief
   *   Return an array with all molecules accepted by LBLRTM.
   *
   * This function returns a CPL_ARRAY with all molecules included in
   * LBLRTM.
   * \b OUTPUT:
   * \return  string CPL_ARRAY molecs of type CPL_TYPE_STRING
   */

    static cpl_array *molecs;
    molecs = cpl_array_new(MF_MOLECULES_NUMBER_MAX, CPL_TYPE_STRING);

    cpl_array_set_string(molecs, MF_MOLECULES_H2O_POS, MF_MOLECULES_H2O);
    cpl_array_set_string(molecs, MF_MOLECULES_CO2_POS, MF_MOLECULES_CO2);
    cpl_array_set_string(molecs, MF_MOLECULES_O3_POS, MF_MOLECULES_O3);
    cpl_array_set_string(molecs, MF_MOLECULES_N2O_POS, MF_MOLECULES_N2O);
    cpl_array_set_string(molecs, MF_MOLECULES_CO_POS, MF_MOLECULES_CO);
    cpl_array_set_string(molecs, MF_MOLECULES_CH4_POS, MF_MOLECULES_CH4);
    cpl_array_set_string(molecs, MF_MOLECULES_O2_POS, MF_MOLECULES_O2);
    cpl_array_set_string(molecs, MF_MOLECULES_NO_POS, MF_MOLECULES_NO);
    cpl_array_set_string(molecs, MF_MOLECULES_SO2_POS, MF_MOLECULES_SO2);
    cpl_array_set_string(molecs, MF_MOLECULES_NO2_POS, MF_MOLECULES_NO2);
    cpl_array_set_string(molecs, MF_MOLECULES_NH3_POS, MF_MOLECULES_NH3);
    cpl_array_set_string(molecs, MF_MOLECULES_HNO3_POS, MF_MOLECULES_HNO3);
    cpl_array_set_string(molecs, MF_MOLECULES_OH_POS, MF_MOLECULES_OH);
    cpl_array_set_string(molecs, MF_MOLECULES_HF_POS, MF_MOLECULES_HF);
    cpl_array_set_string(molecs, MF_MOLECULES_HCL_POS, MF_MOLECULES_HCL);
    cpl_array_set_string(molecs, MF_MOLECULES_HBR_POS, MF_MOLECULES_HBR);
    cpl_array_set_string(molecs, MF_MOLECULES_HI_POS, MF_MOLECULES_HI);
    cpl_array_set_string(molecs, MF_MOLECULES_CLO_POS, MF_MOLECULES_CLO);
    cpl_array_set_string(molecs, MF_MOLECULES_OCS_POS, MF_MOLECULES_OCS);
    cpl_array_set_string(molecs, MF_MOLECULES_H2CO_POS, MF_MOLECULES_H2CO);
    cpl_array_set_string(molecs, MF_MOLECULES_HOCL_POS, MF_MOLECULES_HOCL);
    cpl_array_set_string(molecs, MF_MOLECULES_N2_POS, MF_MOLECULES_N2);
    cpl_array_set_string(molecs, MF_MOLECULES_HCN_POS, MF_MOLECULES_HCN);
    cpl_array_set_string(molecs, MF_MOLECULES_CH3CL_POS, MF_MOLECULES_CH3CL);
    cpl_array_set_string(molecs, MF_MOLECULES_H2O2_POS, MF_MOLECULES_H2O2);
    cpl_array_set_string(molecs, MF_MOLECULES_C2H2_POS, MF_MOLECULES_C2H2);
    cpl_array_set_string(molecs, MF_MOLECULES_C2H6_POS, MF_MOLECULES_C2H6);
    cpl_array_set_string(molecs, MF_MOLECULES_PH3_POS, MF_MOLECULES_PH3);
    cpl_array_set_string(molecs, MF_MOLECULES_COF2_POS, MF_MOLECULES_COF2);
    cpl_array_set_string(molecs, MF_MOLECULES_SF6_POS, MF_MOLECULES_SF6);
    cpl_array_set_string(molecs, MF_MOLECULES_H2S_POS, MF_MOLECULES_H2S);
    cpl_array_set_string(molecs, MF_MOLECULES_HCOOH_POS, MF_MOLECULES_HCOOH);
    cpl_array_set_string(molecs, MF_MOLECULES_HO2_POS, MF_MOLECULES_HO2);
    cpl_array_set_string(molecs, MF_MOLECULES_O_POS, MF_MOLECULES_O);
    cpl_array_set_string(molecs, MF_MOLECULES_CLONO2_POS, MF_MOLECULES_CLONO2);
    cpl_array_set_string(molecs, MF_MOLECULES_NO_P_POS, MF_MOLECULES_NO_P);
    cpl_array_set_string(molecs, MF_MOLECULES_HOBR_POS, MF_MOLECULES_HOBR);
    cpl_array_set_string(molecs, MF_MOLECULES_C2H4_POS, MF_MOLECULES_C2H4);
    cpl_array_set_string(molecs, MF_MOLECULES_CH3OH_POS, MF_MOLECULES_CH3OH);
    cpl_array_set_string(molecs, MF_MOLECULES_CH3BR_POS, MF_MOLECULES_CH3BR);
    cpl_array_set_string(molecs, MF_MOLECULES_CH3CN_POS, MF_MOLECULES_CH3CN);
    cpl_array_set_string(molecs, MF_MOLECULES_CF4_POS, MF_MOLECULES_CF4);
    cpl_array_set_string(molecs, MF_MOLECULES_C4H2_POS, MF_MOLECULES_C4H2);
    cpl_array_set_string(molecs, MF_MOLECULES_HC3N_POS, MF_MOLECULES_HC3N);
    cpl_array_set_string(molecs, MF_MOLECULES_H2_POS, MF_MOLECULES_H2);
    cpl_array_set_string(molecs, MF_MOLECULES_CS_POS, MF_MOLECULES_CS);
    cpl_array_set_string(molecs, MF_MOLECULES_SO3_POS, MF_MOLECULES_SO3);

    return molecs;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Check a string that contain a Molecule name in order to be sure that is a valid molecule in telluriccorr
 *
 * @param molec              .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_error_code mf_molecules_str_check(const char *molec)
{
    /* Check if the input molecule is valid */
    if (strcmp(molec, MF_MOLECULES_H2O) != 0 && strcmp(molec, MF_MOLECULES_CO2) != 0 &&
        strcmp(molec, MF_MOLECULES_O3) != 0 && strcmp(molec, MF_MOLECULES_N2O) != 0 &&
        strcmp(molec, MF_MOLECULES_CO) != 0 && strcmp(molec, MF_MOLECULES_CH4) != 0 &&
        strcmp(molec, MF_MOLECULES_O2) != 0 && strcmp(molec, MF_MOLECULES_NO) != 0 &&
        strcmp(molec, MF_MOLECULES_SO2) != 0 && strcmp(molec, MF_MOLECULES_NO2) != 0 &&
        strcmp(molec, MF_MOLECULES_NH3) != 0 && strcmp(molec, MF_MOLECULES_HNO3) != 0 &&
        strcmp(molec, MF_MOLECULES_OH) != 0 && strcmp(molec, MF_MOLECULES_HF) != 0 &&
        strcmp(molec, MF_MOLECULES_HCL) != 0 && strcmp(molec, MF_MOLECULES_HBR) != 0 &&
        strcmp(molec, MF_MOLECULES_HI) != 0 && strcmp(molec, MF_MOLECULES_CLO) != 0 &&
        strcmp(molec, MF_MOLECULES_OCS) != 0 && strcmp(molec, MF_MOLECULES_H2CO) != 0 &&
        strcmp(molec, MF_MOLECULES_HOCL) != 0 && strcmp(molec, MF_MOLECULES_N2) != 0 &&
        strcmp(molec, MF_MOLECULES_HCN) != 0 && strcmp(molec, MF_MOLECULES_CH3CL) != 0 &&
        strcmp(molec, MF_MOLECULES_H2O2) != 0 && strcmp(molec, MF_MOLECULES_C2H2) != 0 &&
        strcmp(molec, MF_MOLECULES_C2H6) != 0 && strcmp(molec, MF_MOLECULES_PH3) != 0 &&
        strcmp(molec, MF_MOLECULES_COF2) != 0 && strcmp(molec, MF_MOLECULES_SF6) != 0 &&
        strcmp(molec, MF_MOLECULES_H2S) != 0 && strcmp(molec, MF_MOLECULES_HCOOH) != 0 &&
        strcmp(molec, MF_MOLECULES_HO2) != 0 && strcmp(molec, MF_MOLECULES_O) != 0 &&
        strcmp(molec, MF_MOLECULES_CLONO2) != 0 && strcmp(molec, MF_MOLECULES_NO_P) != 0 &&
        strcmp(molec, MF_MOLECULES_HOBR) != 0 && strcmp(molec, MF_MOLECULES_C2H4) != 0 &&
        strcmp(molec, MF_MOLECULES_CH3OH) != 0 && strcmp(molec, MF_MOLECULES_CH3BR) != 0 &&
        strcmp(molec, MF_MOLECULES_CH3CN) != 0 && strcmp(molec, MF_MOLECULES_CF4) != 0 &&
        strcmp(molec, MF_MOLECULES_C4H2) != 0 && strcmp(molec, MF_MOLECULES_HC3N) != 0 &&
        strcmp(molec, MF_MOLECULES_H2) != 0 && strcmp(molec, MF_MOLECULES_CS) != 0 &&
        strcmp(molec, MF_MOLECULES_SO3) != 0) {
        return CPL_ERROR_INCOMPATIBLE_INPUT;
    }

    return CPL_ERROR_NONE;
}

/** @cond PRIVATE */

/** @endcond */


/**@}*/
