/*
 * This file is part of the Molecfit Pipeline
 * Copyright (C) 2001-2019 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "mf_wrap_data.h"

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

/* Convert IDP (1 row, m-columns : first row -> array with size n) to cpl_table with size n x m */
cpl_table *mf_wrap_data_convert_cpl_table_arrays_to_cpl_table_columns(
    const cpl_table *table_array,
    const int        chip,
    const char      *column_lam,
    const char      *column_flux,
    const char      *column_dflux,
    const char      *column_mask
);


/* Convert one Table header to one Spectrum */
cpl_error_code mf_wrap_data_convert_header_Table_to_Spectrum(
    cpl_propertylist *header,
    const cpl_size    n_waves,
    const double      crpix,
    const double      crval,
    const double      cdelt,
    const char       *ctype,
    const char       *cunit
);

/* Convert one Image1D header to one Spectrum */
cpl_error_code
mf_wrap_data_convert_header_Image1D_to_Spectrum(cpl_propertylist *header, const cpl_size n_waves, double crpix);

/* Convert one Image2D header to one Spectrum */
cpl_error_code mf_wrap_data_convert_header_Image2D_to_Spectrum(cpl_propertylist *header, const cpl_size n_waves);

/* Convert one Image3D header to one Spectrum */
cpl_error_code mf_wrap_data_convert_header_Image3D_to_Spectrum(cpl_propertylist *header, const cpl_size n_waves);

/* */
int mf_wrap_data_is_nan_or_inf(double value);

/* */
int mf_wrap_data_is_inf(double value);

/*----------------------------------------------------------------------------*/
/**
 *                 Functions
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup molecfit_data  .
 *
 * @brief
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* ---------------------------------------------------------------------------*/
/**
 * @brief Extract spectrum from cpl_table
 *
 * @param header             in/out: Header of the output spectrum.
 * @param table              .
 * @param chip               Number of chip extension
 * @param column_lambda      Name of the output cpl_table spectrum lambda column.
 * @param column_flux        Name of the output cpl_table spectrum flux   column.
 * @param column_dflux       Name of the output cpl_table spectrum dflux  column.
 * @param column_mask        Name of the output cpl_table spectrum mask   column.
 *
 * @return cpl_table         Spectrum if everything is OK or NULL in other case
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_table *mf_wrap_data_extract_spectrum_from_cpl_table(
    cpl_propertylist *header,
    const cpl_table  *table,
    const int         chip,
    const char       *column_lambda,
    const char       *column_flux,
    const char       *column_dflux,
    const char       *column_mask
)
{
    /* Check inputs */
    cpl_error_ensure(
        header && table, CPL_ERROR_NULL_INPUT, return NULL, "Null inputs in parse data cpl_table to spectrum cpl_table"
    );

    /* Checks */
    if (!cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_XTENSION) ||
        !cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_NAXIS) ||
        !cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_NAXIS1) ||
        !cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_NAXIS2) ||
        cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_NAXIS3)) {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
            "Header keywords problems to identify XTENSION TABLE : (XTENSION, NAXIS and only NAXIS1 and NAXIS2)"
        );
        return NULL;
    }
    else if (strcmp(
                 cpl_propertylist_get_string(header, MOLECFIT_FITS_KEYWORD_XTENSION), MOLECFIT_FITS_KEYWORD_BINTABLE
             ) != 0 ||
             cpl_propertylist_get_int(header, MOLECFIT_FITS_KEYWORD_NAXIS) != 2) {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "Header keywords NOT match with XTENSION=BINTABLE: 2-Axis"
        );
        return NULL;
    }

    cpl_table *spectrum_table = NULL;
    if (cpl_propertylist_get_int(header, MOLECFIT_FITS_KEYWORD_NAXIS1) > 1 &&
        cpl_propertylist_get_int(header, MOLECFIT_FITS_KEYWORD_NAXIS2) == 1) {
        /* 1-ROW ARRAY BINTABLE FORMAT : CONVERT TO NORMAL BINTABLE */

        cpl_msg_info(
            cpl_func, "%s with array format --> Convert to columns for Molecfit format : only columns[%s, %s, %s, %s]",
            MOLECFIT_FITS_KEYWORD_BINTABLE, column_lambda, column_flux, column_dflux, column_mask
        );

        cpl_msg_info(cpl_func, "original      cpl_table structure : Before convert array to table");
        cpl_table_dump_structure(table, NULL);

        //cpl_msg_info(cpl_func,"before mf_wrap_data_convert_cpl_table_arrays_to_cpl_table_columns %s",cpl_error_get_message());

        /* Storage the new table : First row --> Array row (IDP format) */
        spectrum_table = mf_wrap_data_convert_cpl_table_arrays_to_cpl_table_columns(
            table, chip, column_lambda, column_flux, column_dflux, column_mask
        );

        //cpl_msg_info(cpl_func,"after mf_wrap_data_convert_cpl_table_arrays_to_cpl_table_columns %s",cpl_error_get_message());

        cpl_msg_info(cpl_func, "spectrum_table cpl_table structure : After convert array to table");
        cpl_table_dump_structure(spectrum_table, NULL);
    }
    else {
        /* N-ROW NORMAL BINTABLE FORMAT */
        cpl_size nrow  = cpl_table_get_nrow(table);
        spectrum_table = cpl_table_new(nrow);

        cpl_table_new_column(spectrum_table, MF_COL_CHIP, CPL_TYPE_INT);
        cpl_table_fill_column_window_int(spectrum_table, MF_COL_CHIP, 0, nrow, chip);

        cpl_table_duplicate_column(spectrum_table, column_lambda, table, column_lambda);
        cpl_table_duplicate_column(spectrum_table, column_flux, table, column_flux);

        if (strcmp(column_dflux, MF_PARAMETERS_COLUMN_DFLUX_NULL) != 0) {
            cpl_table_duplicate_column(spectrum_table, column_dflux, table, column_dflux);
        }

        if (strcmp(column_mask, MF_PARAMETERS_COLUMN_MASK_NULL) != 0) {
            cpl_table_duplicate_column(spectrum_table, column_mask, table, column_mask);
        }
    }

    /* Convert header */
    cpl_error_code err = cpl_error_get_code();
    if (spectrum_table && !err) {
        int null;

        const char *ctype = column_lambda;
        const char *cunit = cpl_table_get_column_unit(table, column_lambda);

        double crval = cpl_table_get(spectrum_table, column_lambda, 0, &null);
        double cdelt = cpl_table_get(spectrum_table, column_lambda, 1, &null) - crval;

        err = cpl_error_get_code();
        if (!err) {
            double crpix = 1.;
            err          = mf_wrap_data_convert_header_Table_to_Spectrum(
                header, cpl_table_get_nrow(spectrum_table), crpix, crval, cdelt, ctype, cunit
            );
        }
    }

    /* Check preState */
    if (cpl_error_get_code() != CPL_ERROR_NONE || err != CPL_ERROR_NONE) {
        cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "Parse 1D spectrum from BINTABLE failed!");
        if (spectrum_table) {
            cpl_table_delete(spectrum_table);
        }
        return NULL;
    }

    return spectrum_table;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Extract spectrum from cpl_propertylist and cpl_vector's
 *
 * @param header             In/out: Contains the CRVAL1 and CD1_1 for compose the lambda values.
 * @param flux_in            Input flux       cpl_vector.
 * @param dflux_in           Input error flux cpl_vector.
 * @param mask_in            Input mask       cpl_vector.
 * @param chip               Number of chip extension.
 * @param column_lambda      Name of the output cpl_table spectrum lambda column.
 * @param column_flux        Name of the output cpl_table spectrum flux   column.
 * @param column_dflux       Name of the output cpl_table spectrum dflux  column.
 * @param column_mask        Name of the output cpl_table spectrum mask   column.
 *
 * @return cpl_table         Spectrum if everything is OK or NULL in other case
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_table *mf_wrap_data_extract_spectrum_from_cpl_vector(
    cpl_propertylist *header,
    const cpl_vector *flux_in,
    const cpl_vector *dflux_in,
    const cpl_vector *mask_in,
    const int         chip,
    const char       *column_lambda,
    const char       *column_flux,
    const char       *column_dflux,
    const char       *column_mask
)
{
    /* Check inputs */
    cpl_error_ensure(
        header && flux_in, CPL_ERROR_NULL_INPUT, return NULL,
        "Null inputs in parse data cpl_vector to spectrum cpl_table"
    );
    cpl_error_ensure(
        cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CRVAL1) &&
            (cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CD1_1) ||
             cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CDELT1)),
        CPL_ERROR_NULL_INPUT, return NULL, "Not correct KEYWORDS in the header [%s, %s or DEPRECATED %s]",
        MOLECFIT_FITS_KEYWORD_CRVAL1, MOLECFIT_FITS_KEYWORD_CD1_1, MOLECFIT_FITS_KEYWORD_CDELT1
    );

    /* Wavelength values: CRVAL1 (initial wavelength) and CD1_1 (step wavelength) */
    double wave_ini = cpl_propertylist_get_double(header, MOLECFIT_FITS_KEYWORD_CRVAL1);
    double wave_step;
    if (cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CD1_1)) {
        wave_step = cpl_propertylist_get_double(header, MOLECFIT_FITS_KEYWORD_CD1_1);
    }
    else {
        cpl_msg_warning(cpl_func, "Cannot find CD1_1 keyword. Will look for the DEPRECATED CDELT1 keyword instead!");
        wave_step = cpl_propertylist_get_double(header, MOLECFIT_FITS_KEYWORD_CDELT1);
    }
    /* No need to check for else here, as taken care of by cpl_error_ensure call above */

    double wave = wave_ini;

    /* Create table structure for the input spectrum;  n_waves = number of wavelengths */
    double n_wave = cpl_vector_get_size(flux_in);

    /* Convert header */
    double         crpix = 1.;
    cpl_error_code err   = mf_wrap_data_convert_header_Image1D_to_Spectrum(header, n_wave, crpix);

    /* Create output table */
    cpl_table  *spectrum_table = cpl_table_new(n_wave);
    const char *column_lambda_str =
        strcmp(column_lambda, MF_PARAMETERS_COLUMN_LAMBDA_NULL) ? column_lambda : MF_PARAMETERS_COLUMN_LAMBDA_DEFAULT;
    const char *column_flux_str =
        strcmp(column_flux, MF_PARAMETERS_COLUMN_FLUX_NULL) ? column_flux : MF_PARAMETERS_COLUMN_FLUX_DEFAULT;
    const char *column_dflux_str =
        strcmp(column_dflux, MF_PARAMETERS_COLUMN_DFLUX_NULL) ? column_dflux : MF_PARAMETERS_COLUMN_DFLUX_DEFAULT;
    const char *column_mask_str =
        strcmp(column_mask, MF_PARAMETERS_COLUMN_MASK_NULL) ? column_mask : MF_PARAMETERS_COLUMN_MASK_DEFAULT;

    cpl_boolean exist_column_dflux = strcmp(column_dflux, MF_PARAMETERS_COLUMN_DFLUX_NULL) || dflux_in;
    cpl_boolean exist_column_mask  = strcmp(column_mask, MF_PARAMETERS_COLUMN_MASK_NULL) || mask_in;

    /* Create columns */
    cpl_table_new_column(spectrum_table, MF_COL_CHIP, CPL_TYPE_INT);
    cpl_table_new_column(spectrum_table, column_lambda_str, CPL_TYPE_DOUBLE);
    cpl_table_new_column(spectrum_table, column_flux_str, CPL_TYPE_DOUBLE);
    if (exist_column_dflux) {
        cpl_table_new_column(spectrum_table, column_dflux_str, CPL_TYPE_DOUBLE);
    }
    if (exist_column_mask) {
        cpl_table_new_column(spectrum_table, column_mask_str, CPL_TYPE_INT);
    }

    cpl_msg_info(
        cpl_func,
        "Output cpl_table columns : lambda=%s, flux=%s, dflux=%s (input dflux_in ? %d), mask=%s (input mask_in ? %d)",
        column_lambda_str, column_flux_str, column_dflux, dflux_in != NULL, column_mask, mask_in != NULL
    );

    /* Set chip */
    cpl_table_fill_column_window_int(spectrum_table, MF_COL_CHIP, 0, n_wave, chip);

    /* Set lambda array */
    int null;
    for (cpl_size i = 0; i < n_wave; i++) {
        cpl_table_set_double(spectrum_table, column_lambda_str, i, wave);
        wave += wave_step;
    }

    /* Set flux array */
    cpl_table_copy_data_double(spectrum_table, column_flux_str, cpl_vector_get_data_const(flux_in));

    /* Set dflux array : flux error */
    if (exist_column_dflux) {
        if (dflux_in) {
            cpl_table_copy_data_double(spectrum_table, column_dflux_str, cpl_vector_get_data_const(dflux_in));
        }
        else {
            for (cpl_size i = 0; i < n_wave; i++) {
                cpl_table_set_double(spectrum_table, column_dflux_str, i, 0.);
                wave += wave_step;
            }
        }
    }

    /* Set mask values (bad value only if no good pixel in flux) */
    if (exist_column_mask) {
        if (dflux_in) {
            const double *mask_array = cpl_vector_get_data_const(mask_in);
            for (cpl_size i = 0; i < n_wave; i++) {
                cpl_table_set_int(spectrum_table, column_mask_str, i, (cpl_binary)mask_array[i]);
            }
        }
        else {
            for (cpl_size i = 0; i < n_wave; i++) {
                if (mf_wrap_data_is_nan_or_inf(cpl_table_get_double(spectrum_table, column_flux_str, i, &null))) {
                    cpl_table_set_int(spectrum_table, column_mask_str, i, CPL_BINARY_1); /* Bad  pixel */
                }
                else {
                    cpl_table_set_int(spectrum_table, column_mask_str, i, CPL_BINARY_0); /* Good pixel */
                }
                wave += wave_step;
            }
        }
    }


    /* Check preState */
    if (cpl_error_get_code() != CPL_ERROR_NONE || err != CPL_ERROR_NONE) {
        cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "Parse 1D spectrum form IMAGE1D failed!");
        if (spectrum_table) {
            cpl_table_delete(spectrum_table);
        }
        return NULL;
    }

    return spectrum_table;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Extract spectrum from cpl_image.
 *
 * @param header             In/out: Converted header from 2D to 1D spectrum
 * @param img                Input image      cpl_image
 * @param dflux_in           Input error flux cpl_image
 * @param mask_in            Input mask       cpl_image
 * @param chip               Number of chip extension
 * @param column_lambda      Name of the output cpl_table spectrum lambda column.
 * @param column_flux        Name of the output cpl_table spectrum flux   column.
 * @param column_dflux       Name of the output cpl_table spectrum dflux  column.
 * @param column_mask        Name of the output cpl_table spectrum mask   column.
 *
 * @return cpl_table         Spectrum if everything is OK or NULL in other case
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_table *mf_wrap_data_extract_spectrum_from_cpl_image(
    cpl_propertylist *header,
    const cpl_image  *img,
    const cpl_image  *dflux_in,
    const cpl_image  *mask_in,
    const int         chip,
    const char       *column_lambda,
    const char       *column_flux,
    const char       *column_dflux,
    const char       *column_mask
)
{
    /* Check data image exist */
    cpl_error_ensure(
        header && img, CPL_ERROR_NULL_INPUT, return NULL, "Null inputs in parse data cpl_image to spectrum cpl_table"
    );

    /* Check image size (check zeroth extension only) */
    cpl_size nx = cpl_image_get_size_x(img); /* number of lambdas */
    cpl_size ny = cpl_image_get_size_y(img);
    cpl_ensure(nx > 0 || ny > 0, CPL_ERROR_DATA_NOT_FOUND, NULL);

    /* Convert header */
    cpl_error_code err = mf_wrap_data_convert_header_Image2D_to_Spectrum(header, nx);

    /* If not BPM data, get empty MASK */
    cpl_mask       *empty_mask = cpl_mask_new(nx, ny);
    const cpl_mask *bpm        = cpl_image_get_bpm_const(img);
    if (!bpm) {
        bpm = empty_mask;
    }

    /*** Initialize local variables ****/

    /* Get profile along slit */
    cpl_vector *row  = cpl_vector_new(nx); /* X-vector */
    cpl_vector *prof = cpl_vector_new(ny); /* Y-vector */

    int null;
    for (cpl_size j = 0; j < ny; j++) {
        for (cpl_size i = 0; i < nx; i++) {
            cpl_vector_set(row, i, cpl_image_get(img, i + 1, j + 1, &null));
        }
        cpl_vector_set(prof, j, cpl_vector_get_median(row));
    }
    cpl_vector_delete(row);

    /* Check and avoid negative profile values */
    double profmin = cpl_vector_get_min(prof);
    if (profmin < 0.) {
        cpl_vector_subtract_scalar(prof, profmin);
    }

    /* Normalize profile function to get weights and check it */
    double sum = 0.;
    for (cpl_size j = 0; j < ny; j++) {
        sum += cpl_vector_get(prof, j);
    }
    if (sum == 0.) {
        cpl_vector_delete(prof);
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Invalid object value (flux sum = 0)");
        return NULL;
    }
    cpl_vector_divide_scalar(prof, sum);

    /* Calculate correction function for columns with bad pixels */
    cpl_vector *rel_sum = cpl_vector_new(nx);

    /* Check mask */
    for (cpl_size i = 0; i < nx; i++) {
        /* Initialize new value*/
        sum = 0.;

        for (cpl_size j = 0; j < ny; j++) {
            /* Add values of good pixels */
            if (cpl_mask_get(bpm, i + 1, j + 1) == CPL_BINARY_0) {
                sum += cpl_vector_get(prof, j);
            }
        }

        /* Update value take in count bad pixels */
        cpl_vector_set(rel_sum, i, sum);
    }

    /* Delete temporary profile vector */
    cpl_vector_delete(prof);


    /* Create output table */
    cpl_table  *spectrum_table = cpl_table_new(nx);
    const char *column_lambda_str =
        strcmp(column_lambda, MF_PARAMETERS_COLUMN_LAMBDA_NULL) ? column_lambda : MF_PARAMETERS_COLUMN_LAMBDA_DEFAULT;
    const char *column_flux_str =
        strcmp(column_flux, MF_PARAMETERS_COLUMN_FLUX_NULL) ? column_flux : MF_PARAMETERS_COLUMN_FLUX_DEFAULT;
    const char *column_dflux_str =
        strcmp(column_dflux, MF_PARAMETERS_COLUMN_DFLUX_NULL) ? column_dflux : MF_PARAMETERS_COLUMN_DFLUX_DEFAULT;
    const char *column_mask_str =
        strcmp(column_mask, MF_PARAMETERS_COLUMN_MASK_NULL) ? column_mask : MF_PARAMETERS_COLUMN_MASK_DEFAULT;

    cpl_boolean exist_column_dflux = strcmp(column_dflux, MF_PARAMETERS_COLUMN_DFLUX_NULL) || dflux_in;
    cpl_boolean exist_column_mask  = strcmp(column_mask, MF_PARAMETERS_COLUMN_MASK_NULL) || mask_in;

    /* Create columns */
    cpl_table_new_column(spectrum_table, MF_COL_CHIP, CPL_TYPE_INT);
    cpl_table_new_column(spectrum_table, column_lambda_str, CPL_TYPE_DOUBLE);
    cpl_table_new_column(spectrum_table, column_flux_str, CPL_TYPE_DOUBLE);
    if (exist_column_dflux) {
        cpl_table_new_column(spectrum_table, column_dflux_str, CPL_TYPE_DOUBLE);
    }
    if (exist_column_mask) {
        cpl_table_new_column(spectrum_table, column_mask_str, CPL_TYPE_INT);
    }

    cpl_msg_info(
        cpl_func,
        "Output cpl_table columns : lambda=%s, flux=%s, dflux=%s (input dflux_in ? %d), mask=%s (input mask_in ? %d)",
        column_lambda_str, column_flux_str, column_dflux, dflux_in != NULL, column_mask, mask_in != NULL
    );

    /* Set chip */
    cpl_table_fill_column_window_int(spectrum_table, MF_COL_CHIP, 0, nx, chip);

    /* Convert cpl_image (2D) to cpl_vector (1D) */
    for (cpl_size i = 0; i < nx && !err; i++) {
        double sum_lambda = 0.;
        double sum_flux   = 0.;
        double sum_dflux  = 0.;

        for (cpl_size j = 0; j < ny; j++) {
            /* Add values of good pixels */
            if (cpl_mask_get(bpm, i + 1, j + 1) == CPL_BINARY_0) {
                /* Pixel value */
                double val = cpl_image_get(img, i + 1, j + 1, &null);
                if (val <= 0.) {
                    err = CPL_ERROR_ILLEGAL_INPUT;
                }
                else {
                    sum_lambda += val;
                    sum_flux += val;
                    sum_dflux += val * val;
                }
            }
        }

        if (!err) {
            /* Get Wavelenght */
            sum_lambda = sum_lambda / (double)ny; /* Wavelength */

            /* Get scaling factor for pixel flux */
            double scale = cpl_vector_get(rel_sum, i);
            if (scale > 0.) {
                sum_flux  = sum_flux / scale;        /* Flux       */
                sum_dflux = sqrt(sum_dflux) / scale; /* Flux Error */
            }

            /*** Write resulting value into output table ***/

            cpl_table_set_double(spectrum_table, column_lambda_str, i, sum_lambda);
            cpl_table_set_double(spectrum_table, column_flux_str, i, sum_flux);

            if (exist_column_dflux) {
                cpl_table_set_double(spectrum_table, column_dflux_str, i, sum_dflux);
            }

            /* Set mask values (bad value only if no good pixel in image column) */
            if (exist_column_mask) {
                if (scale < 2 * MF_TOL) {
                    cpl_table_set_int(spectrum_table, column_mask_str, i, CPL_BINARY_1); /* Bad  pixel */
                }
                else {
                    cpl_table_set_int(spectrum_table, column_mask_str, i, CPL_BINARY_0); /* Good pixel */
                }
            }
        }
    }

    /* Cleanup */
    cpl_mask_delete(empty_mask);
    cpl_vector_delete(rel_sum);

    /* Check errors and return */
    if (!cpl_error_get_code()) {
        return spectrum_table;
    }
    else {
        cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "Parse 1D spectrum form IMAGE2D failed!");
        cpl_table_delete(spectrum_table);
        return NULL;
    }
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Extract spectrum from cpl_imagelist (datacube).
 *
 * @param header             In/out: Converted header from 3D to 1D spectrum
 * @param cube               Input data cube  cpl_imagelist
 * @param dflux_in           Input error flux cpl_image
 * @param mask_in            Input mask       cpl_image
 * @param chip               Number of chip extension
 * @param column_lambda      Name of the output cpl_table spectrum lambda column.
 * @param column_flux        Name of the output cpl_table spectrum flux   column.
 * @param column_dflux       Name of the output cpl_table spectrum dflux  column.
 * @param column_mask        Name of the output cpl_table spectrum mask   column.
 *
 * @return cpl_table         The calculated spectrum of the data or NULL on error.
 *
 * @note   Only the region defined by the mask is taken into account.
 *           The mask normally should contain values between 0.0 and 1.0.
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_table *mf_wrap_data_extract_spectrum_from_cpl_imagelist(
    cpl_propertylist    *header,
    const cpl_imagelist *cube,
    const cpl_image     *dflux_in,
    const cpl_image     *mask_in,
    const int            chip,
    const char          *column_lambda,
    const char          *column_flux,
    const char          *column_dflux,
    const char          *column_mask
)
{
    /*This function, mf_wrap_data_extract_spectrum_from_cpl_imagelist, is no longer to be used */
    cpl_error_set_message(
        cpl_func, CPL_ERROR_ILLEGAL_INPUT,
        "Molecfit extraction of 3D->1D spectrum no longer permitted. Extract to 1D another way, then run molecfit on "
        "product."
    );
    return NULL;

    /* Check inputs */
    cpl_error_ensure(
        header && cube, CPL_ERROR_NULL_INPUT, return NULL,
        "Null inputs in parse data cpl_imagelist to spectrum cpl_table"
    );
    cpl_error_ensure(
        cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CRVAL3) &&
            cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CD3_3),
        CPL_ERROR_NULL_INPUT, return NULL, "Not correct KEYWORDS in the header [%s, %s]", MOLECFIT_FITS_KEYWORD_CRVAL3,
        MOLECFIT_FITS_KEYWORD_CD3_3
    );

    /* Get and check the number of lambdas = size of datacube */
    cpl_size n_images = cpl_imagelist_get_size(cube);
    cpl_ensure(n_images > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    /* Initialize variables */
    const cpl_image *img_first = cpl_imagelist_get_const(cube, 0);
    cpl_size         nx        = cpl_image_get_size_x(img_first);
    cpl_size         ny        = cpl_image_get_size_y(img_first);
    cpl_ensure(nx > 0 && ny > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    /* Convert header */
    cpl_error_code err = mf_wrap_data_convert_header_Image3D_to_Spectrum(header, n_images);

    /* Create output table */
    cpl_table  *spectrum_table = cpl_table_new(n_images);
    const char *column_lambda_str =
        strcmp(column_lambda, MF_PARAMETERS_COLUMN_LAMBDA_NULL) ? column_lambda : MF_PARAMETERS_COLUMN_LAMBDA_DEFAULT;
    const char *column_flux_str =
        strcmp(column_flux, MF_PARAMETERS_COLUMN_FLUX_NULL) ? column_flux : MF_PARAMETERS_COLUMN_FLUX_DEFAULT;
    const char *column_dflux_str =
        strcmp(column_dflux, MF_PARAMETERS_COLUMN_DFLUX_NULL) ? column_dflux : MF_PARAMETERS_COLUMN_DFLUX_DEFAULT;
    const char *column_mask_str =
        strcmp(column_mask, MF_PARAMETERS_COLUMN_MASK_NULL) ? column_mask : MF_PARAMETERS_COLUMN_MASK_DEFAULT;

    cpl_boolean exist_column_dflux = strcmp(column_dflux, MF_PARAMETERS_COLUMN_DFLUX_NULL) || dflux_in;
    cpl_boolean exist_column_mask  = strcmp(column_mask, MF_PARAMETERS_COLUMN_MASK_NULL) || mask_in;

    /* Create columns */
    cpl_table_new_column(spectrum_table, MF_COL_CHIP, CPL_TYPE_INT);
    cpl_table_new_column(spectrum_table, column_lambda_str, CPL_TYPE_DOUBLE);
    cpl_table_new_column(spectrum_table, column_flux_str, CPL_TYPE_DOUBLE);
    if (exist_column_dflux) {
        cpl_table_new_column(spectrum_table, column_dflux_str, CPL_TYPE_DOUBLE);
    }
    if (exist_column_mask) {
        cpl_table_new_column(spectrum_table, column_mask_str, CPL_TYPE_INT);
    }

    cpl_msg_info(
        cpl_func,
        "Output cpl_table columns : lambda=%s, flux=%s, dflux=%s (input dflux_in ? %d), mask=%s (input mask_in ? %d)",
        column_lambda_str, column_flux_str, column_dflux, dflux_in != NULL, column_mask, mask_in != NULL
    );

    /* Create an empty mask and an output spectrum */
    cpl_mask *mask_empty = cpl_mask_new(nx, ny);

    /* Check input mask data */
    if (mask_in) {
        cpl_ensure(
            nx == cpl_image_get_size_x(mask_in) && ny == cpl_image_get_size_y(mask_in), CPL_ERROR_ILLEGAL_INPUT, NULL
        );
    }

    /* Wavelength values: CRVAL3 (initial wavelength) and CD3_3 (step wavelength) */
    double wave_ini  = cpl_propertylist_get_double(header, MOLECFIT_FITS_KEYWORD_CRVAL3);
    double wave_step = cpl_propertylist_get_double(header, MOLECFIT_FITS_KEYWORD_CD3_3);
    double wave      = wave_ini;

    /* Set chip */
    cpl_table_fill_column_window_int(spectrum_table, MF_COL_CHIP, 0, n_images, chip);

    /* loop over all spatial slices */
    int null;
    for (cpl_size k = 0; k < n_images; k++) {
        /* Get IMAGE data */
        const cpl_image *img = cpl_imagelist_get_const(cube, k);
        const cpl_mask  *bpm = cpl_image_get_bpm_const(img);

        /* If not BPM data, get empty MASK */
        if (!bpm) {
            bpm = mask_empty;
        }

        /* IMAGE and BPM basic data type */
        const cpl_binary *bpm_data = cpl_mask_get_data_const(bpm);

        /* Initialize accumulators */
        double weights  = 0.;
        double sum_data = 0.;

        /* Extract spectrum for data */
        for (cpl_size j = 0; j < ny; j++) {
            for (cpl_size i = 0; i < nx; i++) {
                double img_value = cpl_image_get(img, i + 1, j + 1, &null);

                /* Sum weighted pixels in spatial plane */
                if (!mf_wrap_data_is_nan_or_inf(img_value) && bpm_data[i + (j * nx)] == CPL_BINARY_0) {
                    if (mask_in) {
                        double mask_value = cpl_image_get(img, i + 1, j + 1, &null);

                        weights += mask_value;
                        sum_data += mask_value * img_value;
                    }
                    else {
                        weights += 1.;
                        sum_data += img_value;
                    }
                }
            }
        }

        /* Write resulting value into output table */
        cpl_table_set_double(spectrum_table, column_lambda_str, k, wave);
        cpl_table_set_double(spectrum_table, column_flux_str, k, sum_data / weights);

        if (exist_column_dflux) {
            cpl_table_set_double(spectrum_table, column_dflux_str, k, 0.);
        }

        /* Set mask values (bad value only if no good pixel in flux) */
        if (exist_column_mask) {
            if (mf_wrap_data_is_nan_or_inf(cpl_table_get_double(spectrum_table, column_flux_str, k, &null))) {
                cpl_table_set_int(spectrum_table, column_mask_str, k, CPL_BINARY_1); /* Bad  pixel */
            }
            else {
                cpl_table_set_int(spectrum_table, column_mask_str, k, CPL_BINARY_0); /* Good pixel */
            }
        }

        /* Increase wavelength */
        wave += wave_step;
    }

    /* Cleanup */
    if (mask_empty) {
        cpl_mask_delete(mask_empty);
    }

    /* Check and return */
    err = cpl_error_get_code();
    if (!err) {
        return spectrum_table;
    }
    else {
        cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "Parse 1D spectrum form IMAGE3D failed!");
        cpl_table_delete(spectrum_table);
        return NULL;
    }
}


/* ---------------------------------------------------------------------------*/
/**
 * @brief Update cpl_table with molecfit_data cpl_table.
 *
 * @param molecfit_data      Input molecfit table
 * @param chip_extensions    boolean
 * @param data               in/out: data to update
 * @param ext_ini            Number of extension for update
 * @param column_flux        Name of column to update in the table.
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK or cpl_error_code in other case.
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_error_code mf_wrap_data_update_table(
    const cpl_table  *molecfit_data,
    const cpl_boolean chip_extensions,
    mf_wrap_fits     *data,
    const cpl_size    ext_ini,
    const char       *column_flux
)
{
    // TODO : CHECK THIS FUNCTION

    /* Check inputs */
    cpl_ensure(molecfit_data && column_flux, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

    cpl_size n_ext      = chip_extensions ? data->n_ext : ext_ini + 1;
    cpl_size chip_index = 0;

    //    cpl_size n_waves = chip_extensions ? cpl_table_get_nrow(data->v_ext[ext_ini].table) : cpl_table_get_nrow(molecfit_data);
    cpl_size n_waves = 0;

    for (cpl_size ext = ext_ini; ext < n_ext; ext++) {
        //unused variable       cpl_size chip = ext - ext_ini;

        chip_index = chip_index + n_waves;
        n_waves    = chip_extensions ? cpl_table_get_nrow(data->v_ext[ext].table) : cpl_table_get_nrow(molecfit_data);

        //        cpl_msg_info(cpl_func, "molecfit_data n_waves %d", n_waves );
        //        cpl_msg_info(cpl_func, "molecfit_data chips  %d", (int)chip_extensions );
        //        cpl_msg_info(cpl_func, "molecfit_data table size  %d", (int)cpl_table_get_nrow(molecfit_data));


        cpl_table              *table  = data->v_ext[ext].table;
        const cpl_propertylist *header = data->v_ext[ext].header;

        /* Check inputs */
        cpl_ensure(header && table, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);


        /* Show info */
        if (cpl_propertylist_get_int(header, MOLECFIT_FITS_KEYWORD_NAXIS1) > 1 &&
            cpl_propertylist_get_int(header, MOLECFIT_FITS_KEYWORD_NAXIS2) == 1) {
            if (chip_extensions) {
                n_waves = cpl_propertylist_get_int(header, MOLECFIT_FITS_KEYWORD_NAXIS1);
            }

            /* 1-ROW ARRAY BINTABLE FORMAT */
            cpl_msg_info(
                cpl_func, "Update input array DATA (BINTABLE) cpl_table (ext = %lld, n_rows = %lld)", ext, n_waves
            );


            //            if (chip_extensions) n_waves = cpl_propertylist_get_int(header, MOLECFIT_FITS_KEYWORD_NAXIS1);
            //            cpl_size chip_index = chip * n_waves;

            /* Create temporary cpl_array */
            cpl_array *flux_corr_array = cpl_array_new(n_waves, CPL_TYPE_DOUBLE);

            /* Copy data from cpl_table to cpl_array */
            int null;
            for (cpl_size wave = 0; wave < n_waves; wave++) {
                cpl_array_set(
                    flux_corr_array, wave, cpl_table_get(molecfit_data, MF_COL_OUT_FLUX, chip_index + wave, &null)
                );
            }

            /* Copy data based on its type*/

            /* Check on the actual data type of the column_flux array */
            cpl_type array_data_type = cpl_array_get_type(cpl_table_get_array(table, column_flux, 0));

            if (array_data_type == CPL_TYPE_DOUBLE) {
                /* The data type is double as expected so use the array copy command */
                cpl_msg_info(cpl_func, "SET TO DBL err=%d", cpl_error_get_code());
                cpl_table_set_array(table, column_flux, 0, flux_corr_array);
            }
            else if (array_data_type == CPL_TYPE_FLOAT) {
                /* The data type is float and not double (as expected) so a recast is needed */
                cpl_msg_info(cpl_func, "SET TO FLOAT err=%d", cpl_error_get_code());
                cpl_array *flux_corr_array_float = cpl_array_new(n_waves, CPL_TYPE_FLOAT);
                for (cpl_size i = 0; i < n_waves; i++) {
                    double dval = cpl_array_get_double(flux_corr_array, i, NULL);
                    cpl_array_set_float(flux_corr_array_float, i, (float)dval);
                }
                cpl_table_set_array(table, column_flux, 0, flux_corr_array_float);
                cpl_array_delete(flux_corr_array_float);
            }
            else {
                /* The data type is something weird so err */
                cpl_msg_error(cpl_func, "Unexpected data type when dealing with the %s data column", column_flux);
                cpl_error_set(cpl_func, CPL_ERROR_INVALID_TYPE);
            }


            /* Remove temporary array */
            if (flux_corr_array) {
                cpl_array_delete(flux_corr_array);
            }
        }
        else {
            if (chip_extensions) {
                n_waves = cpl_table_get_nrow(data->v_ext[ext].table);
            }

            /* N-ROW NORMAL BINTABLE FORMAT */
            cpl_msg_info(
                cpl_func, "Update input normal DATA (BINTABLE) cpl_table (ext = %lld, n_rows = %lld)", ext, n_waves
            );

            //            if (chip_extensions) n_waves = cpl_table_get_nrow(data->v_ext[ext_ini].table);
            //            cpl_size chip_index = chip * n_waves;

            /* Copy data from cpl_table to cpl_table */
            int null;
            for (cpl_size wave = 0; wave < n_waves; wave++) {
                cpl_table_set(
                    table, column_flux, wave, cpl_table_get(molecfit_data, MF_COL_OUT_FLUX, chip_index + wave, &null)
                );
            }
        }
    }

    /* Return error */
    return cpl_error_get_code();
}

/* PIPE-11599
 This is a copy of the function mf_wrap_data_update_table implemented above.
 This function copies both flux and flux error columns from the molecfit results.
 The original function is kept for backward compatibility and they can be likely refactored.
 */
/* ---------------------------------------------------------------------------*/
/**
 * @brief Update cpl_table with molecfit_data cpl_table.
 *
 * @param molecfit_data      Input molecfit table
 * @param chip_extensions    boolean
 * @param data               in/out: data to update
 * @param ext_ini            Number of extension for update
 * @param column_flux        Name of column to update in the table.
 * @param column_dflux        Name of error column to update in the table.
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK or cpl_error_code in other case.
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_error_code mf_wrap_data_update_table_witherrors(
    const cpl_table  *molecfit_data,
    const cpl_boolean chip_extensions,
    mf_wrap_fits     *data,
    const cpl_size    ext_ini,
    const char       *column_flux,
    const char       *column_dflux
)
{
    // TODO : CHECK THIS FUNCTION

    /* Check inputs */
    cpl_ensure(molecfit_data && column_flux, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

    cpl_size n_ext      = chip_extensions ? data->n_ext : ext_ini + 1;
    cpl_size chip_index = 0;

    //    cpl_size n_waves = chip_extensions ? cpl_table_get_nrow(data->v_ext[ext_ini].table) : cpl_table_get_nrow(molecfit_data);
    cpl_size n_waves = 0;

    for (cpl_size ext = ext_ini; ext < n_ext; ext++) {
        //unused variable       cpl_size chip = ext - ext_ini;

        chip_index = chip_index + n_waves;
        n_waves    = chip_extensions ? cpl_table_get_nrow(data->v_ext[ext].table) : cpl_table_get_nrow(molecfit_data);

        //        cpl_msg_info(cpl_func, "molecfit_data n_waves %d", n_waves );
        //        cpl_msg_info(cpl_func, "molecfit_data chips  %d", (int)chip_extensions );
        //        cpl_msg_info(cpl_func, "molecfit_data table size  %d", (int)cpl_table_get_nrow(molecfit_data));


        cpl_table              *table  = data->v_ext[ext].table;
        const cpl_propertylist *header = data->v_ext[ext].header;

        /* Check inputs */
        cpl_ensure(header && table, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);


        /* Show info */
        if (cpl_propertylist_get_int(header, MOLECFIT_FITS_KEYWORD_NAXIS1) > 1 &&
            cpl_propertylist_get_int(header, MOLECFIT_FITS_KEYWORD_NAXIS2) == 1) {
            if (chip_extensions) {
                n_waves = cpl_propertylist_get_int(header, MOLECFIT_FITS_KEYWORD_NAXIS1);
            }

            /* 1-ROW ARRAY BINTABLE FORMAT */
            cpl_msg_info(
                cpl_func, "Update input array DATA (BINTABLE) cpl_table (ext = %lld, n_rows = %lld)", ext, n_waves
            );


            //            if (chip_extensions) n_waves = cpl_propertylist_get_int(header, MOLECFIT_FITS_KEYWORD_NAXIS1);
            //            cpl_size chip_index = chip * n_waves;

            /* Create temporary cpl_array */
            cpl_array *flux_corr_array  = cpl_array_new(n_waves, CPL_TYPE_DOUBLE);
            cpl_array *dflux_corr_array = cpl_array_new(n_waves, CPL_TYPE_DOUBLE);

            /* Copy data from cpl_table to cpl_array */
            int null;
            for (cpl_size wave = 0; wave < n_waves; wave++) {
                cpl_array_set(
                    flux_corr_array, wave, cpl_table_get(molecfit_data, MF_COL_OUT_FLUX, chip_index + wave, &null)
                );
                cpl_array_set(
                    dflux_corr_array, wave, cpl_table_get(molecfit_data, MF_COL_OUT_DFLUX, chip_index + wave, &null)
                );
            }

            /* Copy data based on its type*/

            /* Check on the actual data type of the column_flux array */
            cpl_type array_data_type  = cpl_array_get_type(cpl_table_get_array(table, column_flux, 0));
            cpl_type darray_data_type = cpl_array_get_type(cpl_table_get_array(table, column_dflux, 0));

            if (array_data_type == CPL_TYPE_DOUBLE && darray_data_type == CPL_TYPE_DOUBLE) {
                /* The data type is double as expected so use the array copy command */
                cpl_msg_info(cpl_func, "SET TO DBL err=%d", cpl_error_get_code());
                cpl_table_set_array(table, column_flux, 0, flux_corr_array);
                cpl_table_set_array(table, column_dflux, 0, dflux_corr_array);
            }
            else if (array_data_type == CPL_TYPE_FLOAT && darray_data_type == CPL_TYPE_FLOAT) {
                /* The data type is float and not double (as expected) so a recast is needed */
                cpl_msg_info(cpl_func, "SET TO FLOAT err=%d", cpl_error_get_code());
                cpl_array *flux_corr_array_float  = cpl_array_new(n_waves, CPL_TYPE_FLOAT);
                cpl_array *dflux_corr_array_float = cpl_array_new(n_waves, CPL_TYPE_FLOAT);
                for (cpl_size i = 0; i < n_waves; i++) {
                    double dval  = cpl_array_get_double(flux_corr_array, i, NULL);
                    double deval = cpl_array_get_double(dflux_corr_array, i, NULL);
                    cpl_array_set_float(flux_corr_array_float, i, (float)dval);
                    cpl_array_set_float(dflux_corr_array_float, i, (float)deval);
                }
                cpl_table_set_array(table, column_flux, 0, flux_corr_array_float);
                cpl_table_set_array(table, column_dflux, 0, dflux_corr_array_float);
                cpl_array_delete(flux_corr_array_float);
                cpl_array_delete(dflux_corr_array_float);
            }
            else {
                /* The data type is something weird so err */
                cpl_msg_error(
                    cpl_func, "Unexpected data type when dealing with the %s %s data columns", column_flux, column_dflux
                );
                cpl_error_set(cpl_func, CPL_ERROR_INVALID_TYPE);
            }


            /* Remove temporary array */
            if (flux_corr_array) {
                cpl_array_delete(flux_corr_array);
            }
            if (dflux_corr_array) {
                cpl_array_delete(dflux_corr_array);
            }
        }
        else {
            if (chip_extensions) {
                n_waves = cpl_table_get_nrow(data->v_ext[ext].table);
            }

            /* N-ROW NORMAL BINTABLE FORMAT */
            cpl_msg_info(
                cpl_func, "Update input normal DATA (BINTABLE) cpl_table (ext = %lld, n_rows = %lld)", ext, n_waves
            );

            //            if (chip_extensions) n_waves = cpl_table_get_nrow(data->v_ext[ext_ini].table);
            //            cpl_size chip_index = chip * n_waves;

            /* Copy data from cpl_table to cpl_table */
            int null;
            for (cpl_size wave = 0; wave < n_waves; wave++) {
                cpl_table_set(
                    table, column_flux, wave, cpl_table_get(molecfit_data, MF_COL_OUT_FLUX, chip_index + wave, &null)
                );
                cpl_table_set(
                    table, column_dflux, wave, cpl_table_get(molecfit_data, MF_COL_OUT_DFLUX, chip_index + wave, &null)
                );
            }
        }
    }

    /* Return error */
    return cpl_error_get_code();
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Update cpl_vector with molecfit_data cpl_table.
 *
 * @param molecfit_data      Input molecfit table
 * @param chip_extensions    boolean
 * @param data               in/out: data to update
 * @param ext_ini            Number of extension for update
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK or cpl_error_code in other case.
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_error_code mf_wrap_data_update_vector(
    const cpl_table  *molecfit_data,
    const cpl_boolean chip_extensions,
    mf_wrap_fits     *data,
    const cpl_size    ext_ini
)
{
    // TODO : CHECK THIS FUNCTION

    /* Check inputs */
    cpl_ensure(molecfit_data, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

    cpl_size n_ext = chip_extensions ? data->n_ext : ext_ini + 1;
    //    cpl_size n_waves = chip_extensions ? cpl_vector_get_size(data->v_ext[ext_ini].vector) : cpl_table_get_nrow(molecfit_data);
    cpl_size chip_index = 0;
    cpl_size n_waves    = 0;

    for (cpl_size ext = ext_ini; ext < n_ext; ext++) {
        //unused variable        cpl_size chip       = ext - ext_ini;
        //        cpl_size chip_index = chip * n_waves;
        chip_index = chip_index + n_waves;
        n_waves    = chip_extensions ? cpl_vector_get_size(data->v_ext[ext].vector) : cpl_table_get_nrow(molecfit_data);

        cpl_vector *vector = data->v_ext[ext].vector;

        /* Check inputs */
        cpl_ensure(vector, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

        /* Show info */
        cpl_msg_info(cpl_func, "Update input DATA (IMAGE-1D) cpl_vector (ext = %lld, n_rows = %lld)", ext, n_waves);

        /* Copy data from cpl_table to cpl_vector */
        int null;
        for (cpl_size wave = 0; wave < n_waves; wave++) {
            cpl_vector_set(vector, wave, cpl_table_get(molecfit_data, MF_COL_OUT_FLUX, chip_index + wave, &null));
        }
    }

    /* Return error */
    return cpl_error_get_code();
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Update cpl_image with molecfit_data cpl_table.
 *
 * @param molecfit_data      Input molecfit table
 * @param chip_extensions    boolean
 * @param data               in/out: data to update
 * @param ext_ini            Number of extension for update
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK or cpl_error_code in other case.
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_error_code mf_wrap_data_update_image(
    const cpl_table  *molecfit_data,
    const cpl_boolean chip_extensions,
    mf_wrap_fits     *data,
    const cpl_size    ext_ini
)
{
    // TODO : CHECK THIS FUNCTION

    /* Check data image exist */
    cpl_ensure(molecfit_data, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

    cpl_size n_ext = chip_extensions ? data->n_ext : ext_ini + 1;
    cpl_size n_waves =
        chip_extensions ? cpl_image_get_size_x(data->v_ext[ext_ini].image) : cpl_table_get_nrow(molecfit_data);

    for (cpl_size ext = ext_ini; ext < n_ext; ext++) {
        cpl_size chip       = ext - ext_ini;
        cpl_size chip_index = chip * n_waves;

        cpl_image *image = data->v_ext[ext].image;

        /* Check data image exist */
        cpl_ensure(image, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

        /* Show info */
        cpl_msg_info(cpl_func, "Update input DATA (IMAGE-2D) cpl_image (ext = %lld, n_rows = %lld)", ext, n_waves);

        /* Check image size (check zeroth extension only) */
        cpl_size nx = cpl_image_get_size_x(image);
        cpl_size ny = cpl_image_get_size_y(image);
        cpl_ensure(nx > 0 || ny > 0, CPL_ERROR_DATA_NOT_FOUND, CPL_ERROR_DATA_NOT_FOUND);

        /* Check number of data points between vectors and image (check zeroth extension only) */
        if (n_waves <= 0 || n_waves != nx) {
            return cpl_error_set_message(
                cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "Invalid object structure (number of data points)"
            );
        }

        /*** UPDATE mf_wrap_fits IMAGE structure ***/

        /* Use correction function from VECTORS structure to correct the flux in the IMAGE structure (set bad pixels according to mask) */
        for (cpl_size i = 0; i < nx; i++) {
            double mtrans = cpl_table_get(molecfit_data, MF_COL_OUT_TELLURIC_CORR, chip_index + i, NULL);

            int null;
            for (cpl_size j = 0; j < ny; j++) {
                double flux = cpl_image_get(image, i + 1, j + 1, &null);
                cpl_image_set(image, i + 1, j + 1, flux / mtrans);

                //double dflux = cpl_image_get(     image, i + 1, j + 1, &null         );
                //cpl_image_set(                    image, i + 1, j + 1, dflux / mtrans);
            }
        }

        cpl_mask *bpm = cpl_image_get_bpm(image);
        if (bpm) {
            for (cpl_size i = 0; i < nx; i++) {
                int ref_mask = cpl_table_get(molecfit_data, MF_COL_OUT_MASK, chip_index + i, NULL);

                for (cpl_size j = 0; j < ny; j++) {
                    cpl_binary mask = cpl_mask_get(bpm, i + 1, j + 1);
                    if (mask != ref_mask) {
                        cpl_mask_set(bpm, i + 1, j + 1, (cpl_binary)ref_mask);
                    }
                }
            }
        }
    }

    /* Return error */
    return cpl_error_get_code();
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Update cpl_imagelist with molecfit_data cpl_table.
 *
 * @param molecfit_data      Input molecfit table
 * @param chip_extensions    boolean
 * @param data               in/out: data to update
 * @param ext_ini            Number of extension for update
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK or cpl_error_code in other case.
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_error_code mf_wrap_data_update_cube(
    const cpl_table  *molecfit_data,
    const cpl_boolean chip_extensions,
    mf_wrap_fits     *data,
    const cpl_size    ext_ini
)
{
    // TODO : CHECK THIS FUNCTION

    /* Check inputs */
    cpl_ensure(molecfit_data, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

    cpl_size n_ext = chip_extensions ? data->n_ext : ext_ini + 1;
    cpl_size n_waves =
        chip_extensions ? cpl_imagelist_get_size(data->v_ext[ext_ini].cube) : cpl_table_get_nrow(molecfit_data);

    for (cpl_size ext = ext_ini; ext < n_ext; ext++) {
        cpl_size chip       = ext - ext_ini;
        cpl_size chip_index = chip * n_waves;

        cpl_imagelist *cube = data->v_ext[ext].cube;

        /* Check inputs */
        cpl_ensure(cube, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

        /* Show info */
        cpl_msg_info(cpl_func, "Update input DATA (IMAGE-3D) cpl_imagelist (ext = %lld, n_rows = %lld)", ext, n_waves);

        int null;

        /* Create data and copy values*/
        cpl_vector *flux = cpl_vector_new(n_waves);

        for (cpl_size wave = 0; wave < n_waves; wave++) {
            cpl_vector_set(flux, wave, cpl_table_get(molecfit_data, MF_COL_OUT_FLUX, chip_index + wave, &null));
        }

        /* Create mask and fill */
        cpl_vector *mask = cpl_vector_new(n_waves);
        cpl_vector_fill(mask, 1);

        /* Generate BPM */
        double *pflux = cpl_vector_get_data(flux);
        double *pmask = cpl_vector_get_data(mask);
        for (cpl_size wave = 0; wave < n_waves; wave++) {
            if (isinf(pflux[wave]) != 0 || isnan(pflux[wave]) != 0) {
                pmask[wave] = 0.;
            }
        }

        /* Process data */
        const double *flux_d = cpl_vector_get_data_const(flux);
        const double *mask_d = cpl_vector_get_data_const(mask);

        cpl_size size = cpl_imagelist_get_size(cube);

        for (cpl_size n_img = 0; n_img < size; n_img++) {
            cpl_image *img = cpl_imagelist_get(cube, n_img);

            if (mask_d[n_img] < 0.5) {
                /* Whole image rejected */
                for (cpl_size ix = 1; ix <= cpl_image_get_size_x(img); ix++) {
                    for (cpl_size iy = 1; iy <= cpl_image_get_size_y(img); iy++) {
                        cpl_image_reject(img, ix, iy);
                    }
                }
            }
            else {
                cpl_image_divide_scalar(img, flux_d[n_img]);
            }
        }

        /* Cleanup */
        cpl_vector_delete(flux);
        cpl_vector_delete(mask);
    }

    /* Return error */
    return cpl_error_get_code();
}


/** @cond PRIVATE */

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_table *mf_wrap_data_convert_cpl_table_arrays_to_cpl_table_columns(
    const cpl_table *table_array,
    const int        chip,
    const char      *column_lam,
    const char      *column_flux,
    const char      *column_dflux,
    const char      *column_mask
)
{
    /*!
     * Convert data array rows in columns (to internal management format).
     * This format consists of one row containing arrays with the data in the
     * columns
     *
     * \b INPUT:
     * \param inptable      input Spectrum table with one row containing arrays
     * \param column_lam    name of wavelength column
     * \param column_flux   name of flux column
     * \param column_dflux  name of flux error column or NULL
     * \param column_mask   name of mask column or NULL
     *
     * \b OUTPUT:
     * new table in with arrays expanded to rows
     *
     * \b ERRORS:
     * - Missing or inconsistent columns
     */

    const cpl_array *lambda_array = cpl_table_get_array(table_array, column_lam, 0);
    if (!lambda_array) {
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Expected %s array column in input", column_lam);
        ;
        return NULL;
    }

    const cpl_array *flux_array = cpl_table_get_array(table_array, column_flux, 0);
    if (!flux_array) {
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Expected %s array column in input", column_flux);
        return NULL;
    }

    const cpl_array *dflux_array = NULL;
    if (strcmp(column_dflux, MF_PARAMETERS_COLUMN_DFLUX_NULL) != 0) {
        dflux_array = cpl_table_get_array(table_array, column_dflux, 0);
        if (!dflux_array) {
            cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Expected %s array column in input", column_dflux);
            return NULL;
        }
    }

    const cpl_array *mask_array = NULL;
    if (strcmp(column_mask, MF_PARAMETERS_COLUMN_MASK_NULL) != 0) {
        mask_array = cpl_table_get_array(table_array, column_mask, 0);
        if (!mask_array) {
            cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Expected %s array column in input", column_mask);
            return NULL;
        }
    }

    if (cpl_array_get_size(lambda_array) != cpl_array_get_size(flux_array) ||
        (dflux_array && cpl_array_get_size(lambda_array) != cpl_array_get_size(dflux_array)) ||
        (mask_array && cpl_array_get_size(lambda_array) != cpl_array_get_size(mask_array))) {
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Table columns do not all have the same size");
        return NULL;
    }

    cpl_size nrow = cpl_array_get_size(lambda_array);

    /* Create cpl_table */
    cpl_table *spectrum_molecfit_table = cpl_table_new(nrow);

    /* Fill chip */
    cpl_table_new_column(spectrum_molecfit_table, MF_COL_CHIP, CPL_TYPE_INT);
    cpl_table_fill_column_window_int(spectrum_molecfit_table, MF_COL_CHIP, 0, nrow, chip);


    /* FIXME: Const casts due to PIPE-6075 */
    CPL_DIAG_PRAGMA_PUSH_IGN(-Wcast-qual);
    cpl_array *lambda_double_array = cpl_array_cast((cpl_array *)lambda_array, CPL_TYPE_DOUBLE);
    CPL_DIAG_PRAGMA_POP;

    cpl_table_new_column(spectrum_molecfit_table, column_lam, CPL_TYPE_DOUBLE);
    cpl_table_copy_data_double(
        spectrum_molecfit_table, column_lam, cpl_array_get_data_double_const(lambda_double_array)
    );
    cpl_table_set_column_unit(spectrum_molecfit_table, column_lam, cpl_table_get_column_unit(table_array, column_lam));
    cpl_array_delete(lambda_double_array);


    /* FIXME: Const casts due to PIPE-6075 */
    CPL_DIAG_PRAGMA_PUSH_IGN(-Wcast-qual);
    cpl_array *flux_double_array = cpl_array_cast((cpl_array *)flux_array, CPL_TYPE_DOUBLE);
    CPL_DIAG_PRAGMA_POP;

    cpl_table_new_column(spectrum_molecfit_table, column_flux, CPL_TYPE_DOUBLE);
    cpl_table_copy_data_double(
        spectrum_molecfit_table, column_flux, cpl_array_get_data_double_const(flux_double_array)
    );
    cpl_table_set_column_unit(spectrum_molecfit_table, column_flux, cpl_table_get_column_unit(table_array, column_flux));
    cpl_array_delete(flux_double_array);


    if (dflux_array) {
        /* FIXME: Const casts due to PIPE-6075 */
        CPL_DIAG_PRAGMA_PUSH_IGN(-Wcast-qual);
        cpl_array *dflux_double_array = cpl_array_cast((cpl_array *)dflux_array, CPL_TYPE_DOUBLE);
        CPL_DIAG_PRAGMA_POP;

        cpl_table_new_column(spectrum_molecfit_table, column_dflux, CPL_TYPE_DOUBLE);
        cpl_table_copy_data_double(
            spectrum_molecfit_table, column_dflux, cpl_array_get_data_double_const(dflux_double_array)
        );
        cpl_table_set_column_unit(spectrum_molecfit_table, column_dflux, cpl_table_get_column_unit(table_array, column_dflux));
        cpl_array_delete(dflux_double_array);
    }

    if (mask_array) {
        cpl_table_new_column(spectrum_molecfit_table, column_mask, CPL_TYPE_INT);
        cpl_table_copy_data_int(spectrum_molecfit_table, column_mask, cpl_array_get_data_int_const(mask_array));
    }


    return spectrum_molecfit_table;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Convert one Table header to one Spectrum
 *
 * @param    header          In/out header 3D and at the end of the function 1D
 * @param    n_waves         Number of lambdas
 * @param    n_waves         Number of lambdas
 * @param    n_waves         Number of lambdas
 * @param    n_waves         Number of lambdas
 * @param    n_waves         Number of lambdas
 * @param    n_waves         Number of lambdas
 *
 * @return   cpl_error_code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code mf_wrap_data_convert_header_Table_to_Spectrum(
    cpl_propertylist *header,
    const cpl_size    n_waves,
    const double      crpix,
    const double      crval,
    const double      cdelt,
    const char       *ctype,
    const char       *cunit
)
{
    /* Check inputs */
    cpl_ensure(header, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
    cpl_ensure(n_waves > 0, CPL_ERROR_ILLEGAL_INPUT, CPL_ERROR_NULL_INPUT);

    /* Change extension type */
    if (cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_XTENSION)) {
        cpl_propertylist_update_string(header, MOLECFIT_FITS_KEYWORD_XTENSION, MOLECFIT_FITS_KEYWORD_IMAGE);
    }

    /*** Change WCS ***/
    if (cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_NAXIS)) {
        cpl_propertylist_update_int(header, MOLECFIT_FITS_KEYWORD_NAXIS, 1);
    }
    else {
        cpl_propertylist_append_int(header, MOLECFIT_FITS_KEYWORD_NAXIS, 1);
    }
    if (cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_NAXIS1)) {
        cpl_propertylist_update_int(header, MOLECFIT_FITS_KEYWORD_NAXIS1, n_waves);
    }
    else {
        cpl_propertylist_append_int(header, MOLECFIT_FITS_KEYWORD_NAXIS1, n_waves);
    }
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_NAXIS2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_NAXIS3);

    if (cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CRPIX1)) {
        cpl_propertylist_update_double(header, MOLECFIT_FITS_KEYWORD_CRPIX1, crpix);
    }
    else {
        cpl_propertylist_append_double(header, MOLECFIT_FITS_KEYWORD_CRPIX1, crpix);
    }
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CRPIX2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CRPIX3);

    if (cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CRVAL1)) {
        cpl_propertylist_update_double(header, MOLECFIT_FITS_KEYWORD_CRVAL1, crval);
    }
    else {
        cpl_propertylist_append_double(header, MOLECFIT_FITS_KEYWORD_CRVAL1, crval);
    }
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CRVAL2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CRVAL3);

    if (cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CD1_1)) {
        cpl_propertylist_update_double(header, MOLECFIT_FITS_KEYWORD_CD1_1, cdelt);
    }
    else {
        cpl_propertylist_append_double(header, MOLECFIT_FITS_KEYWORD_CD1_1, cdelt);
    }
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD2_2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD3_3);

    if (cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CTYPE1)) {
        cpl_propertylist_update_string(header, MOLECFIT_FITS_KEYWORD_CTYPE1, ctype ? ctype : "");
    }
    else {
        cpl_propertylist_append_string(header, MOLECFIT_FITS_KEYWORD_CTYPE1, ctype ? ctype : "");
    }
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CTYPE2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CTYPE3);

    if (cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CUNIT1)) {
        cpl_propertylist_update_string(header, MOLECFIT_FITS_KEYWORD_CUNIT1, cunit ? cunit : "");
    }
    else {
        cpl_propertylist_append_string(header, MOLECFIT_FITS_KEYWORD_CUNIT1, cunit ? cunit : "");
    }
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CUNIT2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CUNIT3);

    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD1_1);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD1_2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD1_3);

    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD2_1);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD2_2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD2_3);

    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD3_1);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD3_2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD3_3);

    cpl_error_code err = cpl_error_get_code();
    if (err != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, err, "Cannot update the BINTABLE header to convert in spectrum header");
    }

    return err;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Convert one Image1D header to one Spectrum
 *
 * @param    header          In/out header 3D and at the end of the function 1D
 * @param    n_waves         Number of lambdas
 *
 * @return   cpl_error_code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
mf_wrap_data_convert_header_Image1D_to_Spectrum(cpl_propertylist *header, const cpl_size n_waves, double crpix)
{
    /* Check inputs */
    cpl_ensure(header, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
    cpl_ensure(n_waves > 0, CPL_ERROR_ILLEGAL_INPUT, CPL_ERROR_NULL_INPUT);

    /* Check Image1D header properties */
    if (!cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CRVAL1) ||
        (!cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CD1_1) && !cpl_propertylist_has(header, "CDELT1")) ||
        !cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CTYPE1) ||
        !cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CUNIT1)) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Not all input data are in the header!");
    }

    /*** Change WCS ***/
    if (cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_NAXIS)) {
        cpl_propertylist_update_int(header, MOLECFIT_FITS_KEYWORD_NAXIS, 1);
    }
    else {
        cpl_propertylist_append_int(header, MOLECFIT_FITS_KEYWORD_NAXIS, 1);
    }
    if (cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_NAXIS1)) {
        cpl_propertylist_update_int(header, MOLECFIT_FITS_KEYWORD_NAXIS1, n_waves);
    }
    else {
        cpl_propertylist_append_int(header, MOLECFIT_FITS_KEYWORD_NAXIS1, n_waves);
    }
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_NAXIS2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_NAXIS3);

    if (cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CRPIX1)) {
        cpl_propertylist_update_double(header, MOLECFIT_FITS_KEYWORD_CRPIX1, crpix);
    }
    else {
        cpl_propertylist_append_double(header, MOLECFIT_FITS_KEYWORD_CRPIX1, crpix);
    }
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CRPIX2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CRPIX3);

    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CRVAL2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CRVAL3);

    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD2_2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD3_3);

    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CTYPE2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CTYPE3);

    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CUNIT2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CUNIT3);

    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD1_1);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD1_2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD1_3);

    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD2_1);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD2_2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD2_3);

    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD3_1);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD3_2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD3_3);

    cpl_error_code err = cpl_error_get_code();
    if (err != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, err, "Cannot update the IMAGE1D header to convert in spectrum header");
    }

    return err;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Convert one Image2D header to one Spectrum
 *
 * @param    header          In/out header 3D and at the end of the function 1D
 * @param    n_waves         Number of lambdas
 *
 * @return   cpl_error_code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code mf_wrap_data_convert_header_Image2D_to_Spectrum(cpl_propertylist *header, const cpl_size n_waves)
{
    /* Check inputs */
    cpl_ensure(header, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
    cpl_ensure(n_waves > 0, CPL_ERROR_ILLEGAL_INPUT, CPL_ERROR_NULL_INPUT);

    /* Check Image2D header properties */
    if (!cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CRPIX2) ||
        !cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CRVAL2) ||
        !cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CD2_2) ||
        !cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CTYPE2) ||
        !cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CUNIT2)) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Not all input data are in the header!");
    }

    /*** Change WCS ***/

    cpl_propertylist_update_int(header, MOLECFIT_FITS_KEYWORD_NAXIS, 1);
    cpl_propertylist_update_int(header, MOLECFIT_FITS_KEYWORD_NAXIS1, n_waves);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_NAXIS2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_NAXIS3);

    cpl_propertylist_update_double(
        header, MOLECFIT_FITS_KEYWORD_CRPIX1, cpl_propertylist_get_double(header, MOLECFIT_FITS_KEYWORD_CRPIX2)
    );
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CRPIX2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CRPIX3);

    cpl_propertylist_update_double(
        header, MOLECFIT_FITS_KEYWORD_CRVAL1, cpl_propertylist_get_double(header, MOLECFIT_FITS_KEYWORD_CRVAL2)
    );
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CRVAL2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CRVAL3);

    cpl_propertylist_update_double(
        header, MOLECFIT_FITS_KEYWORD_CD1_1, cpl_propertylist_get_double(header, MOLECFIT_FITS_KEYWORD_CD2_2)
    );
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD2_2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD3_3);

    cpl_propertylist_update_string(
        header, MOLECFIT_FITS_KEYWORD_CTYPE1, cpl_propertylist_get_string(header, MOLECFIT_FITS_KEYWORD_CTYPE2)
    );
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CTYPE2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CTYPE3);

    cpl_propertylist_update_string(
        header, MOLECFIT_FITS_KEYWORD_CUNIT1, cpl_propertylist_get_string(header, MOLECFIT_FITS_KEYWORD_CUNIT2)
    );
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CUNIT2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CUNIT3);

    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD1_1);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD1_2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD1_3);

    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD2_1);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD2_2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD2_3);

    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD3_1);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD3_2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD3_3);

    cpl_error_code err = cpl_error_get_code();
    if (err != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, err, "Cannot update the IMAGE2D header to convert in spectrum header");
    }

    return err;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Convert one Image3D header to one Spectrum
 *
 * @param    header          In/out header 3D and at the end of the function 1D
 * @param    n_waves         Number of lambdas
 *
 * @return   cpl_error_code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code mf_wrap_data_convert_header_Image3D_to_Spectrum(cpl_propertylist *header, const cpl_size n_waves)
{
    /* Check inputs */
    cpl_ensure(header, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
    cpl_ensure(n_waves > 0, CPL_ERROR_ILLEGAL_INPUT, CPL_ERROR_NULL_INPUT);

    /* Check Image3D header properties */
    if (!cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CRPIX3) ||
        !cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CRVAL3) ||
        !cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CD3_3) ||
        !cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CTYPE3) ||
        !cpl_propertylist_has(header, MOLECFIT_FITS_KEYWORD_CUNIT3)) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Not all input data are in the header!");
    }

    /*** Change WCS ***/

    cpl_propertylist_update_int(header, MOLECFIT_FITS_KEYWORD_NAXIS, 1);
    cpl_propertylist_update_int(header, MOLECFIT_FITS_KEYWORD_NAXIS1, n_waves);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_NAXIS2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_NAXIS3);

    cpl_propertylist_update_double(
        header, MOLECFIT_FITS_KEYWORD_CRPIX1, cpl_propertylist_get_double(header, MOLECFIT_FITS_KEYWORD_CRPIX3)
    );
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CRPIX2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CRPIX3);

    cpl_propertylist_update_double(
        header, MOLECFIT_FITS_KEYWORD_CRVAL1, cpl_propertylist_get_double(header, MOLECFIT_FITS_KEYWORD_CRVAL3)
    );
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CRVAL2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CRVAL3);

    cpl_propertylist_update_double(
        header, MOLECFIT_FITS_KEYWORD_CD1_1, cpl_propertylist_get_double(header, MOLECFIT_FITS_KEYWORD_CD3_3)
    );
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD2_2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD3_3);

    cpl_propertylist_update_string(
        header, MOLECFIT_FITS_KEYWORD_CTYPE1, cpl_propertylist_get_string(header, MOLECFIT_FITS_KEYWORD_CTYPE3)
    );
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CTYPE2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CTYPE3);

    cpl_propertylist_update_string(
        header, MOLECFIT_FITS_KEYWORD_CUNIT1, cpl_propertylist_get_string(header, MOLECFIT_FITS_KEYWORD_CUNIT3)
    );
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CUNIT2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CUNIT3);

    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD1_1);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD1_2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD1_3);

    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD2_1);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD2_2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD2_3);

    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD3_1);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD3_2);
    cpl_propertylist_erase(header, MOLECFIT_FITS_KEYWORD_CD3_3);

    cpl_error_code err = cpl_error_get_code();
    if (err != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, err, "Cannot update the IMAGE3D header to convert in spectrum header");
    }

    return err;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
int mf_wrap_data_is_nan_or_inf(double value)
{
    return (isnan(value) || mf_wrap_data_is_inf(value) == 1 || mf_wrap_data_is_inf(value) == -1);
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
int mf_wrap_data_is_inf(double value)
{
    int ret = isinf(value);

    /* Double check in case we are running on MacOSX */
    if (ret == 1) {
        if (isgreater(value, 0)) {
            return 1;
        }
        else {
            return -1;
        }
    }

    return ret;
}

/** @endcond */


/**@}*/
