/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>

#include "moo_compute_resp.h"
#include "moo_fibres_table.h"
#include "moo_fits.h"
#include "moo_pfits.h"
#include "moo_badpix.h"
#include "moo_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

static moo_spline *
_get_atm_spline(moo_atm *atm, double airm)
{
    moo_spline *res = NULL;
    cpl_table *atm_flux = moo_atm_convert_mag_to_flux(atm, airm);

    int nrow = cpl_table_get_nrow(atm_flux);
    double *x = cpl_table_get_data_double(atm_flux, MOO_ATM_WAVE);
    double *y = cpl_table_get_data_double(atm_flux, MOO_ATM_FLUX);

    cpl_vector *vx = cpl_vector_wrap(nrow, x);
    cpl_vector *vy = cpl_vector_wrap(nrow, y);
    cpl_bivector *bv = cpl_bivector_wrap_vectors(vx, vy);

    res = moo_spline_create(bv);

    cpl_bivector_unwrap_vectors(bv);
    cpl_vector_unwrap(vx);
    cpl_vector_unwrap(vy);
    cpl_table_delete(atm_flux);
    return res;
}

static moo_spline *
_get_resp_spline(cpl_table *resp, const char *resp_colname)
{
    moo_spline *res = NULL;
    cpl_vector *vx = NULL;
    cpl_vector *vy = NULL;
    cpl_bivector *bv = NULL;
    cpl_table *sel = NULL;

    cpl_ensure(resp != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(resp_colname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int nrow = cpl_table_get_nrow(resp);
    cpl_table_unselect_all(resp);

    double *ya = cpl_table_get_data_double(resp, resp_colname);
    for (int i = 0; i < nrow; i++) {
        if (!isnan(ya[i])) {
            cpl_table_select_row(resp, i);
        }
    }
    sel = cpl_table_extract_selected(resp);
    int size = cpl_table_get_nrow(sel);
    moo_try_assure(size > 0, CPL_ERROR_NULL_INPUT,
                   "No data for building response %s", resp_colname);
    double *x = cpl_table_get_data_double(sel, MOO_RESP_WAVE);
    double *y = cpl_table_get_data_double(sel, resp_colname);

    vx = cpl_vector_wrap(size, x);
    vy = cpl_vector_wrap(size, y);
    bv = cpl_bivector_wrap_vectors(vx, vy);

    res = moo_spline_create(bv);

moo_try_cleanup:
    cpl_bivector_unwrap_vectors(bv);
    cpl_vector_unwrap(vx);
    cpl_vector_unwrap(vy);
    cpl_table_delete(sel);

    return res;
}

static cpl_error_code
_moo_update_primary_header(cpl_propertylist *primary_header)
{
    cpl_error_code status = CPL_ERROR_NONE;
    status = moo_pfits_set_fluxcal(primary_header, MOO_PFITS_FLUXCAL_ABSOLUTE);

    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Apply the relative flux calibration to 1D rebinned spectra
  @param    rbn RBN spectra
  @param    atm Atmospheric extinction table
  @param    resp Instrumental response
  @return   Flux calibrated frame

 * _Flags considered as bad :
 *
 * _Bad pixels flags_:

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_calib_flux(moo_rbn *rbn, moo_atm *atm, moo_resp *resp)
{
    cpl_ensure_code(rbn != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(atm != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(resp != NULL, CPL_ERROR_NULL_INPUT);
    moo_spline *atm_spline = NULL;
    moo_spline *resp_spline = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_msg_info(__func__, "Apply flux calibration");

    int badpix_level = MOO_BADPIX_OUTSIDE_DATA_RANGE;
    cpl_table *fibre_table = moo_rbn_get_fibre_table(rbn);
    cpl_propertylist *primary_header = moo_rbn_get_primary_header(rbn);
    _moo_update_primary_header(primary_header);

    int ntarget = cpl_table_get_nrow(fibre_table);
    int *idrbnt =
        cpl_table_get_data_int(fibre_table, MOO_FIBRES_TABLE_INDEXRBN);
    const int *spectrot =
        cpl_table_get_data_int_const(fibre_table, MOO_FIBRES_TABLE_SPECTRO);
    double airm_start = moo_pfits_get_tel_airm_start(rbn->primary_header);
    double airm_end = moo_pfits_get_tel_airm_end(rbn->primary_header);
    double airm = (airm_start + airm_end) / 2.;
    moo_try_check(atm_spline = _get_atm_spline(atm, airm), " ");

    const char *trans_colnames[] = { MOO_FIBRES_TABLE_TRANS_RI,
                                     MOO_FIBRES_TABLE_TRANS_YJ,
                                     MOO_FIBRES_TABLE_TRANS_H };

    for (int i = 0; i < 3; i++) {
        moo_rbn_single *rbn_single = moo_rbn_load_single(rbn, i, badpix_level);

        if (rbn_single != NULL) {
            const char *trans_colname = trans_colnames[i];

            float *transt =
                cpl_table_get_data_float(fibre_table, trans_colname);
            cpl_table *response = moo_resp_get_table(resp, i);
            cpl_propertylist *header = moo_rbn_single_get_header(rbn_single);
            double cd1_1 = moo_pfits_get_cd1_1(header);
            double crpix1 = moo_pfits_get_crpix1(header);
            double crval1 = moo_pfits_get_crval1(header);
            double exptime = NAN;
            if (i == 0) {
                exptime = moo_pfits_get_exptime(rbn->primary_header);
            }
            else {
                double dit = moo_pfits_get_dit(header);
                double ndit = moo_pfits_get_ndit(header);
                exptime = dit * ndit;
            }
            hdrl_image *himg = moo_rbn_single_get_image(rbn_single);
            cpl_image *img = hdrl_image_get_image(himg);
            cpl_image *err = hdrl_image_get_error(himg);
            int nx = cpl_image_get_size_x(img);
            double entranceLoss = 1;
            for (int t = 0; t < ntarget; t++) {
                int idrbn = idrbnt[t];
                double trans = transt[t];
                int spectro = spectrot[t];

                if (!isnan(trans) && trans > 0) {
                    const char *resp_colname =
                        moo_resp_get_colname(resp, idrbn, spectro);
                    moo_try_check(resp_spline =
                                      _get_resp_spline(response, resp_colname),
                                  "Response search for %s idrbn %d",
                                  rbn_single->extname, idrbn);

                    double div = cd1_1 * exptime / entranceLoss;

                    for (int j = 1; j <= nx; j++) {
                        int rej;
                        double w = (j - crpix1) * cd1_1 + crval1;
                        double e = cpl_image_get(err, j, idrbn, &rej);
                        double f = cpl_image_get(img, j, idrbn, &rej);
                        double respval = moo_spline_eval(resp_spline, w);
                        double atmval = moo_spline_eval(atm_spline, w);

                        if (rej == 0) {
                            double cf = f / atmval / div / respval;
                            double ce = e / atmval / div / respval;
                            cpl_image_set(img, j, idrbn, cf);
                            cpl_image_set(err, j, idrbn, ce);
                        }
                        else {
                            cpl_image_set(img, j, idrbn, NAN);
                            cpl_image_set(err, j, idrbn, NAN);
                        }
                    }
                    moo_spline_delete(resp_spline);
                    resp_spline = NULL;
                }
            }
        }
    }

moo_try_cleanup:
    moo_spline_delete(atm_spline);
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in calibrate flux");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        if (resp_spline != NULL) {
            moo_spline_delete(resp_spline);
        }
    }
    return CPL_ERROR_NONE;
}
/**@}*/
