/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>

#include "moo_combine_pair.h"
#include "moo_sub_sky_stare.h"
#include "moo_fibres_table.h"
#include "moo_fits.h"
#include "moo_pfits.h"
#include "moo_badpix.h"
#include "moo_utils.h"
#include "moo_skycorr.h"
#include "moo_products.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/
static moo_sci_single *
_moo_create_sci2d_single(moo_rbn_single *nod_rbn_single,
                         moo_rbn_single *objb_rbn_single,
                         moo_rbn_single *obja_rbn_single,
                         cpl_table *nod_fibret,
                         moo_target_table *target_table,
                         int ispaired)
{
    moo_sci_single *sci_single = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(nod_rbn_single != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(objb_rbn_single != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(target_table != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int nb_target = cpl_table_get_nrow(target_table->table);

    if (nb_target > 0) {
        moo_try_check(sci_single = moo_sci_single_new(nod_rbn_single->type),
                      " ");

        moo_pfits_append_hduclass_data(sci_single->header, sci_single->type,
                                       -1);
        cpl_propertylist_copy_property(sci_single->header,
                                       nod_rbn_single->header,
                                       MOO_PFITS_CRPIX1);
        cpl_propertylist_copy_property(sci_single->header,
                                       nod_rbn_single->header,
                                       MOO_PFITS_CRVAL1);
        cpl_propertylist_copy_property(sci_single->header,
                                       nod_rbn_single->header, MOO_PFITS_CD1_1);
        cpl_propertylist_copy_property(sci_single->header,
                                       nod_rbn_single->header, MOO_PFITS_CD1_2);
        cpl_propertylist_copy_property(sci_single->header,
                                       nod_rbn_single->header,
                                       MOO_PFITS_CTYPE1);
        cpl_propertylist_copy_property(sci_single->header,
                                       nod_rbn_single->header,
                                       MOO_PFITS_CUNIT1);
        cpl_propertylist_copy_property(sci_single->header,
                                       nod_rbn_single->header, MOO_PFITS_BUNIT);
        cpl_propertylist_copy_property(sci_single->header,
                                       nod_rbn_single->header,
                                       MOO_PFITS_CRPIX2);
        cpl_propertylist_copy_property(sci_single->header,
                                       nod_rbn_single->header,
                                       MOO_PFITS_CRVAL2);
        cpl_propertylist_copy_property(sci_single->header,
                                       nod_rbn_single->header, MOO_PFITS_CD2_1);
        cpl_propertylist_copy_property(sci_single->header,
                                       nod_rbn_single->header, MOO_PFITS_CD2_2);
        cpl_propertylist_copy_property(sci_single->header,
                                       nod_rbn_single->header,
                                       MOO_PFITS_CTYPE2);
        cpl_propertylist_copy_property(sci_single->header,
                                       nod_rbn_single->header,
                                       MOO_PFITS_WAVELMIN);
        cpl_propertylist_copy_property(sci_single->header,
                                       nod_rbn_single->header,
                                       MOO_PFITS_WAVELMAX);
        cpl_propertylist_copy_property(sci_single->header,
                                       nod_rbn_single->header,
                                       MOO_PFITS_SPECBIN);
        int nx = hdrl_image_get_size_x(nod_rbn_single->image);

        sci_single->image = hdrl_image_new(nx, nb_target);
        sci_single->sky = cpl_image_new(nx, nb_target, CPL_TYPE_DOUBLE);
        sci_single->qual = cpl_image_new(nx, nb_target, CPL_TYPE_INT);

        cpl_image *sci_img = hdrl_image_get_image(sci_single->image);
        /* create the mask if empty */
        cpl_image_get_bpm(sci_img);

        cpl_image *sci_err_img = hdrl_image_get_error(sci_single->image);
        cpl_image *sci_qual_img = sci_single->qual;

        cpl_image *objb_rbn_img = hdrl_image_get_image(objb_rbn_single->image);
        cpl_image *obja_rbn_img = NULL;

        if (obja_rbn_single != NULL) {
            obja_rbn_img = hdrl_image_get_image(obja_rbn_single->image);
        }

        cpl_image *nod_rbn_img = hdrl_image_get_image(nod_rbn_single->image);
        cpl_image *nod_rbn_err_img =
            hdrl_image_get_error(nod_rbn_single->image);
        cpl_image *nod_rbn_qual_img = nod_rbn_single->qual;

        const char *transname[] = { MOO_TARGET_TABLE_TRANSRI,
                                    MOO_TARGET_TABLE_TRANSYJ,
                                    MOO_TARGET_TABLE_TRANSH };
        const char *ptransname[] = { MOO_TARGET_TABLE_PTRANSRI,
                                     MOO_TARGET_TABLE_PTRANSYJ,
                                     MOO_TARGET_TABLE_PTRANSH };
        float *transt =
            cpl_table_get_data_float(target_table->table,
                                     transname[nod_rbn_single->type]);
        float *ptranst =
            cpl_table_get_data_float(target_table->table,
                                     ptransname[nod_rbn_single->type]);
        int *indexrbnt = cpl_table_get_data_int(target_table->table,
                                                MOO_TARGET_TABLE_INDEXRBN);
        int *pindexrbnt =
            cpl_table_get_data_int(target_table->table,
                                   MOO_TARGET_TABLE_PAIREDINDEXRBN);
        char **obstypet = cpl_table_get_data_string(target_table->table,
                                                    MOO_TARGET_TABLE_OBSTYPE);

        for (int i = 1; i <= nb_target; i++) {
            int idx = i - 1;
            int indexrbn = indexrbnt[idx];

            cpl_table_set_int(target_table->table, MOO_TARGET_TABLE_INDEXTARG,
                              idx, i);
            cpl_table_set_int(target_table->table, MOO_TARGET_TABLE_NBSKY, idx,
                              1);
            cpl_table_set_int(target_table->table,
                              MOO_TARGET_TABLE_MAX_DIST_SLIT, idx, 0);
            cpl_table_set_float(target_table->table,
                                MOO_TARGET_TABLE_MAX_DIST_SKY, idx, 0);
            cpl_table_set_string(target_table->table,
                                 MOO_TARGET_TABLE_SUBSKYMODE, idx,
                                 MOO_TARGET_TABLE_SUBSKY_NOD);

            cpl_table_select_all(nod_fibret);
            cpl_table_and_selected_int(nod_fibret, MOO_FIBRES_TABLE_INDEXRBN,
                                       CPL_EQUAL_TO, indexrbn);
            cpl_array *sel = cpl_table_where_selected(nod_fibret);
            int ft_idx = cpl_array_get_cplsize(sel, 0, NULL);
            moo_target_table_set_snr(target_table, idx, nod_fibret, ft_idx);
            cpl_array_delete(sel);
        }
/* The following addresses an issue with the gcc9 compiler series prior to
 * gcc 9.3. These compiler versions require that the variable '__func__'
 * appears in the data sharing clause if default(none) is used. However
 * adding it to the data sharing clause breaks older compiler versions
 * where these variables are pre-determined as shared.
 * This was fixed in gcc 9.3 and OpenMP 5.1.
 */
#ifdef _OPENMP
#if (__GNUC__ == 9) && (__GNUC_MINOR__ < 3)
#pragma omp parallel shared(nb_target, indexrbnt, pindexrbnt, transt, ptranst, \
                                obstypet, target_table, nod_rbn_single,        \
                                objb_rbn_img, obja_rbn_img, sci_single,        \
                                nod_rbn_qual_img, sci_qual_img, nod_rbn_img,   \
                                sci_img, nod_rbn_err_img, sci_err_img, nx,     \
                                ispaired)
#else
#pragma omp parallel default(none)                                       \
    shared(nb_target, indexrbnt, pindexrbnt, transt, ptranst, obstypet,  \
               target_table, nod_rbn_single, objb_rbn_img, obja_rbn_img, \
               sci_single, nod_rbn_qual_img, sci_qual_img, nod_rbn_img,  \
               sci_img, nod_rbn_err_img, sci_err_img, nx, ispaired)
#endif
        {
#pragma omp for
#endif
            for (int i = 1; i <= nb_target; i++) {
                int idx = i - 1;
                int indexrbn = indexrbnt[idx];
                char *obstype = obstypet[idx];

                if ((strcmp(obstype, MOO_OBSTYPE_OBSPNOD) == 0) ||
                    (strcmp(obstype, MOO_OBSTYPE_OBSPNODSTARE) == 0)) {
                    for (int j = 1; j <= nx; j++) {
                        int rej;
                        double sky_val =
                            cpl_image_get(obja_rbn_img, j, indexrbn, &rej);
                        cpl_image_set(sci_single->sky, j, i, sky_val);
                        double rbn_img_val =
                            cpl_image_get(nod_rbn_img, j, indexrbn, &rej);
                        if (rej) {
                            cpl_image_set(sci_img, j, i, NAN);
                            cpl_image_set(sci_err_img, j, i, NAN);
                            cpl_image_reject(sci_img, j, i);
                        }
                        else {
                            double sci_flux = -rbn_img_val;
                            cpl_image_set(sci_img, j, i, sci_flux);
                            double rbn_err_val =
                                cpl_image_get(nod_rbn_err_img, j, indexrbn,
                                              &rej);
                            double err = rbn_err_val;
                            cpl_image_set(sci_err_img, j, i, err);
                        }
                        double rbn_qual_val =
                            cpl_image_get(nod_rbn_qual_img, j, indexrbn, &rej);
                        cpl_image_set(sci_qual_img, j, i, rbn_qual_val);
                    }
                }
                else if (strcmp(obstype, MOO_OBSTYPE_OBSSWITCH) == 0) {
                    // double transsum = trans + ptrans;
                    double factor = 1.;
                    cpl_image *sky_img = objb_rbn_img;
                    if (ispaired) {
                        factor = -1;
                        sky_img = obja_rbn_img;
                    }
                    for (int j = 1; j <= nx; j++) {
                        int rej, rerr;

                        double sky_val =
                            cpl_image_get(sky_img, j, indexrbn, &rej);
                        cpl_image_set(sci_single->sky, j, i, sky_val);

                        double nod_rbn_img_val =
                            factor *
                            cpl_image_get(nod_rbn_img, j, indexrbn, &rej);
                        double nod_rbn_qual_val =
                            cpl_image_get(nod_rbn_qual_img, j, indexrbn, &rerr);
                        double nod_rbn_err_val =
                            cpl_image_get(nod_rbn_err_img, j, indexrbn, &rerr);

                        if (rej) {
                            cpl_image_reject(sci_img, j, i);
                            nod_rbn_img_val = NAN;
                            nod_rbn_err_val = NAN;
                        }
                        cpl_image_set(sci_img, j, i, nod_rbn_img_val);
                        cpl_image_set(sci_err_img, j, i, nod_rbn_err_val);
                        cpl_image_set(sci_qual_img, j, i, nod_rbn_qual_val);
                    }
                }
                else {
                    for (int j = 1; j <= nx; j++) {
                        int rej;

                        double sky_val =
                            cpl_image_get(objb_rbn_img, j, indexrbn, &rej);
                        cpl_image_set(sci_single->sky, j, i, sky_val);
                        double rbn_img_val =
                            cpl_image_get(nod_rbn_img, j, indexrbn, &rej);
                        if (rej) {
                            cpl_image_set(sci_img, j, i, NAN);
                            cpl_image_set(sci_err_img, j, i, NAN);
                            cpl_image_reject(sci_img, j, i);
                        }
                        else {
                            double sci_flux = rbn_img_val;
                            cpl_image_set(sci_img, j, i, sci_flux);
                            double rbn_err_val =
                                cpl_image_get(nod_rbn_err_img, j, indexrbn,
                                              &rej);
                            double err = rbn_err_val;
                            cpl_image_set(sci_err_img, j, i, err);
                        }
                        double rbn_qual_val =
                            cpl_image_get(nod_rbn_qual_img, j, indexrbn, &rej);
                        cpl_image_set(sci_qual_img, j, i, rbn_qual_val);
                    }
                }
            }
#ifdef _OPENMP
        }
#endif
    }
moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_error_set_message(__func__, CPL_ERROR_ILLEGAL_OUTPUT,
                              "Error in combine pair %s",
                              nod_rbn_single->extname);
        moo_sci_single_delete(sci_single);
        sci_single = NULL;
    }
    return sci_single;
}

static moo_sci_single *
_moo_combine_pair_2d_single(moo_rbn_single *nod_rbn_single,
                            moo_rbn_single *objb_rbn_single,
                            moo_rbn_single *obja_rbn_single,
                            cpl_table *nod_fibret,
                            moo_target_table *target_table)
{
    moo_sci_single *sci_single = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(nod_rbn_single != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(objb_rbn_single != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(target_table != NULL, CPL_ERROR_NULL_INPUT, NULL);

    moo_try_check(sci_single = moo_sci_single_new(nod_rbn_single->type), " ");

    moo_pfits_append_hduclass_data(sci_single->header, sci_single->type, -1);
    cpl_propertylist_copy_property(sci_single->header, nod_rbn_single->header,
                                   MOO_PFITS_CRPIX1);
    cpl_propertylist_copy_property(sci_single->header, nod_rbn_single->header,
                                   MOO_PFITS_CRVAL1);
    cpl_propertylist_copy_property(sci_single->header, nod_rbn_single->header,
                                   MOO_PFITS_CD1_1);
    cpl_propertylist_copy_property(sci_single->header, nod_rbn_single->header,
                                   MOO_PFITS_CD1_2);
    cpl_propertylist_copy_property(sci_single->header, nod_rbn_single->header,
                                   MOO_PFITS_CTYPE1);
    cpl_propertylist_copy_property(sci_single->header, nod_rbn_single->header,
                                   MOO_PFITS_CUNIT1);
    cpl_propertylist_copy_property(sci_single->header, nod_rbn_single->header,
                                   MOO_PFITS_BUNIT);
    cpl_propertylist_copy_property(sci_single->header, nod_rbn_single->header,
                                   MOO_PFITS_CRPIX2);
    cpl_propertylist_copy_property(sci_single->header, nod_rbn_single->header,
                                   MOO_PFITS_CRVAL2);
    cpl_propertylist_copy_property(sci_single->header, nod_rbn_single->header,
                                   MOO_PFITS_CD2_1);
    cpl_propertylist_copy_property(sci_single->header, nod_rbn_single->header,
                                   MOO_PFITS_CD2_2);
    cpl_propertylist_copy_property(sci_single->header, nod_rbn_single->header,
                                   MOO_PFITS_CTYPE2);
    cpl_propertylist_copy_property(sci_single->header, nod_rbn_single->header,
                                   MOO_PFITS_BUNIT);
    cpl_propertylist_copy_property(sci_single->header, nod_rbn_single->header,
                                   MOO_PFITS_WAVELMIN);
    cpl_propertylist_copy_property(sci_single->header, nod_rbn_single->header,
                                   MOO_PFITS_WAVELMAX);
    cpl_propertylist_copy_property(sci_single->header, nod_rbn_single->header,
                                   MOO_PFITS_SPECBIN);
    int nx = hdrl_image_get_size_x(nod_rbn_single->image);
    int nb_target = cpl_table_get_nrow(target_table->table);

    sci_single->image = hdrl_image_new(nx, nb_target);
    sci_single->sky = cpl_image_new(nx, nb_target, CPL_TYPE_DOUBLE);
    sci_single->qual = cpl_image_new(nx, nb_target, CPL_TYPE_INT);

    cpl_image *sci_img = hdrl_image_get_image(sci_single->image);
    /* create the mask if empty */
    cpl_image_get_bpm(sci_img);

    cpl_image *sci_err_img = hdrl_image_get_error(sci_single->image);
    cpl_image *sci_qual_img = sci_single->qual;

    cpl_image *objb_rbn_img = hdrl_image_get_image(objb_rbn_single->image);
    cpl_image *obja_rbn_img = hdrl_image_get_image(obja_rbn_single->image);

    cpl_image *nod_rbn_img = hdrl_image_get_image(nod_rbn_single->image);
    cpl_image *nod_rbn_err_img = hdrl_image_get_error(nod_rbn_single->image);
    cpl_image *nod_rbn_qual_img = nod_rbn_single->qual;

    const char *transname[] = { MOO_TARGET_TABLE_TRANSRI,
                                MOO_TARGET_TABLE_TRANSYJ,
                                MOO_TARGET_TABLE_TRANSH };
    const char *ptransname[] = { MOO_TARGET_TABLE_PTRANSRI,
                                 MOO_TARGET_TABLE_PTRANSYJ,
                                 MOO_TARGET_TABLE_PTRANSH };
    float *transt = cpl_table_get_data_float(target_table->table,
                                             transname[nod_rbn_single->type]);
    float *ptranst = cpl_table_get_data_float(target_table->table,
                                              ptransname[nod_rbn_single->type]);
    int *indexrbnt =
        cpl_table_get_data_int(target_table->table, MOO_TARGET_TABLE_INDEXRBN);
    int *pindexrbnt = cpl_table_get_data_int(target_table->table,
                                             MOO_TARGET_TABLE_PAIREDINDEXRBN);
    char **obstypet = cpl_table_get_data_string(target_table->table,
                                                MOO_TARGET_TABLE_OBSTYPE);
    int *nbskyt =
        cpl_table_get_data_int(target_table->table, MOO_TARGET_TABLE_NBSKY);
    int *maxdistslitt = cpl_table_get_data_int(target_table->table,
                                               MOO_TARGET_TABLE_MAX_DIST_SLIT);
    float *maxdistskyt =
        cpl_table_get_data_float(target_table->table,
                                 MOO_TARGET_TABLE_MAX_DIST_SKY);
    int *pnbskyt =
        cpl_table_get_data_int(target_table->table, MOO_TARGET_TABLE_PNBSKY);
    int *pmaxdistslitt =
        cpl_table_get_data_int(target_table->table,
                               MOO_TARGET_TABLE_PMAX_DIST_SLIT);
    float *pmaxdistskyt =
        cpl_table_get_data_float(target_table->table,
                                 MOO_TARGET_TABLE_PMAX_DIST_SKY);

    for (int i = 1; i <= nb_target; i++) {
        int indexrbn = indexrbnt[i - 1];
        int pindexrbn = pindexrbnt[i - 1];

        cpl_table_set_int(target_table->table, MOO_TARGET_TABLE_INDEXTARG,
                          i - 1, i);
        cpl_table_set_string(target_table->table, MOO_TARGET_TABLE_SUBSKYMODE,
                             i - 1, MOO_TARGET_TABLE_SUBSKY_NOD);
        char *obstype = obstypet[i - 1];

        if (strcmp(obstype, MOO_OBSTYPE_OBSPNODSTARE) == 0) {
            indexrbn = pindexrbn;
            pnbskyt[i - 1] = 1;
            pmaxdistslitt[i - 1] = 0;
            pmaxdistskyt[i - 1] = 0;
        }
        else {
            nbskyt[i - 1] = 1;
            maxdistslitt[i - 1] = 0;
            maxdistskyt[i - 1] = 0;
        }
        cpl_table_select_all(nod_fibret);
        cpl_table_and_selected_int(nod_fibret, MOO_FIBRES_TABLE_INDEXRBN,
                                   CPL_EQUAL_TO, indexrbn);
        cpl_array *sel = cpl_table_where_selected(nod_fibret);
        int ft_idx = cpl_array_get_cplsize(sel, 0, NULL);
        if (strcmp(obstype, MOO_OBSTYPE_OBSPNODSTARE) == 0) {
            moo_target_table_set_ft_psnr(target_table, i - 1, nod_fibret,
                                         ft_idx);
        }
        else {
            moo_target_table_set_snr(target_table, i - 1, nod_fibret, ft_idx);
        }
        cpl_array_delete(sel);
        if (strcmp(obstype, MOO_OBSTYPE_OBSSWITCH) == 0) {
            cpl_table_select_all(nod_fibret);
            cpl_table_and_selected_int(nod_fibret, MOO_FIBRES_TABLE_INDEXRBN,
                                       CPL_EQUAL_TO, pindexrbn);
            cpl_array *sel2 = cpl_table_where_selected(nod_fibret);
            ft_idx = cpl_array_get_cplsize(sel2, 0, NULL);
            cpl_array_delete(sel2);
            moo_target_table_set_ft_psnr(target_table, i - 1, nod_fibret,
                                         ft_idx);
            cpl_table_set_int(target_table->table, MOO_TARGET_TABLE_PNBSKY,
                              i - 1, 1);
            cpl_table_set_int(target_table->table,
                              MOO_TARGET_TABLE_PMAX_DIST_SLIT, i - 1, 0);
            cpl_table_set_float(target_table->table,
                                MOO_TARGET_TABLE_PMAX_DIST_SKY, i - 1, 0);
        }
    }

/* The following addresses an issue with the gcc9 compiler series prior to
 * gcc 9.3. These compiler versions require that the variable '__func__'
 * appears in the data sharing clause if default(none) is used. However
 * adding it to the data sharing clause breaks older compiler versions
 * where these variables are pre-determined as shared.
 * This was fixed in gcc 9.3 and OpenMP 5.1.
 */
#ifdef _OPENMP
#if (__GNUC__ == 9) && (__GNUC_MINOR__ < 3)
#pragma omp parallel shared(nb_target, indexrbnt, pindexrbnt, transt, ptranst, \
                                obstypet, target_table, nod_rbn_single,        \
                                objb_rbn_img, obja_rbn_img, sci_single,        \
                                nod_rbn_qual_img, sci_qual_img, nod_rbn_img,   \
                                sci_img, nod_rbn_err_img, sci_err_img, nx)
#else
#pragma omp parallel default(none)                                       \
    shared(nb_target, indexrbnt, pindexrbnt, transt, ptranst, obstypet,  \
               target_table, nod_rbn_single, objb_rbn_img, obja_rbn_img, \
               sci_single, nod_rbn_qual_img, sci_qual_img, nod_rbn_img,  \
               sci_img, nod_rbn_err_img, sci_err_img, nx)
#endif
    {
#pragma omp for
#endif
        for (int i = 1; i <= nb_target; i++) {
            int indexrbn = indexrbnt[i - 1];
            int pindexrbn = pindexrbnt[i - 1];
            char *obstype = obstypet[i - 1];
            float trans = transt[i - 1];
            float ptrans = ptranst[i - 1];

            if (strcmp(obstype, MOO_OBSTYPE_OBSPNOD) == 0 ||
                strcmp(obstype, MOO_OBSTYPE_OBSPNODSTARE) == 0) {
                if (strcmp(obstype, MOO_OBSTYPE_OBSPNODSTARE) == 0) {
                    indexrbn = pindexrbn;
                }
                for (int j = 1; j <= nx; j++) {
                    int rej;
                    double sky_val =
                        cpl_image_get(obja_rbn_img, j, indexrbn, &rej);
                    cpl_image_set(sci_single->sky, j, i, sky_val);
                    double rbn_img_val =
                        cpl_image_get(nod_rbn_img, j, indexrbn, &rej);
                    if (rej) {
                        cpl_image_set(sci_img, j, i, NAN);
                        cpl_image_set(sci_err_img, j, i, NAN);
                        cpl_image_reject(sci_img, j, i);
                    }
                    else {
                        double sci_flux = -rbn_img_val;
                        cpl_image_set(sci_img, j, i, sci_flux);
                        double rbn_err_val =
                            cpl_image_get(nod_rbn_err_img, j, indexrbn, &rej);
                        double err = rbn_err_val;
                        cpl_image_set(sci_err_img, j, i, err);
                    }
                    double rbn_qual_val =
                        cpl_image_get(nod_rbn_qual_img, j, indexrbn, &rej);
                    cpl_image_set(sci_qual_img, j, i, rbn_qual_val);
                }
            }
            else if (strcmp(obstype, MOO_OBSTYPE_OBSSWITCH) == 0) {
                double transsum = trans + ptrans;
                for (int j = 1; j <= nx; j++) {
                    int rej, prej, rerr;
                    double sci_flux = NAN;
                    double sci_err = NAN;

                    double obja_sky_val =
                        cpl_image_get(obja_rbn_img, j, pindexrbn, &rej);
                    double objb_sky_val =
                        cpl_image_get(objb_rbn_img, j, indexrbn, &prej);
                    double sky_val =
                        (ptrans * obja_sky_val + trans * objb_sky_val) /
                        transsum * 2;
                    cpl_image_set(sci_single->sky, j, i, sky_val);

                    double nod_rbn_img_val =
                        cpl_image_get(nod_rbn_img, j, indexrbn, &rej);
                    double pnod_rbn_img_val =
                        cpl_image_get(nod_rbn_img, j, pindexrbn, &prej);

                    if (rej && prej) {
                        cpl_image_reject(sci_img, j, i);

                        int nod_rbn_qual_val =
                            (int)cpl_image_get(nod_rbn_qual_img, j, indexrbn,
                                               &rerr);
                        int pnod_rbn_qual_val =
                            (int)cpl_image_get(nod_rbn_qual_img, j, pindexrbn,
                                               &rerr);
                        int sci_qual_val = nod_rbn_qual_val | pnod_rbn_qual_val;
                        cpl_image_set(sci_qual_img, j, i, (double)sci_qual_val);
                    }
                    else if (rej) {
                        sci_flux = -2 * pnod_rbn_img_val;
                        double pnod_rbn_err_val =
                            cpl_image_get(nod_rbn_err_img, j, pindexrbn, &rerr);
                        sci_err = 2 * pnod_rbn_err_val;

                        int sci_qual_val =
                            (int)cpl_image_get(nod_rbn_qual_img, j, pindexrbn,
                                               &rerr);
                        cpl_image_set(sci_qual_img, j, i, (double)sci_qual_val);
                    }
                    else if (prej) {
                        sci_flux = 2 * nod_rbn_img_val;
                        double nod_rbn_err_val =
                            cpl_image_get(nod_rbn_err_img, j, indexrbn, &rerr);
                        sci_err = 2 * nod_rbn_err_val;

                        int sci_qual_val =
                            (int)cpl_image_get(nod_rbn_qual_img, j, indexrbn,
                                               &rerr);
                        cpl_image_set(sci_qual_img, j, i, (double)sci_qual_val);
                    }
                    else {
                        sci_flux = (nod_rbn_img_val * trans -
                                    pnod_rbn_img_val * ptrans) /
                                   transsum * 2;
                        double nod_rbn_err_val =
                            cpl_image_get(nod_rbn_err_img, j, indexrbn, &rerr);
                        double pnod_rbn_err_val =
                            cpl_image_get(nod_rbn_err_img, j, pindexrbn, &rerr);
                        sci_err = sqrt((trans * trans * nod_rbn_err_val *
                                            nod_rbn_err_val +
                                        ptrans * ptrans * pnod_rbn_err_val *
                                            pnod_rbn_err_val)) *
                                  2 / transsum;

                        int nod_rbn_qual_val =
                            (int)cpl_image_get(nod_rbn_qual_img, j, indexrbn,
                                               &rerr);
                        int pnod_rbn_qual_val =
                            (int)cpl_image_get(nod_rbn_qual_img, j, pindexrbn,
                                               &rerr);
                        int sci_qual_val = nod_rbn_qual_val | pnod_rbn_qual_val;
                        cpl_image_set(sci_qual_img, j, i, (double)sci_qual_val);
                    }
                    cpl_image_set(sci_img, j, i, sci_flux);
                    cpl_image_set(sci_err_img, j, i, sci_err);
                }
            }
            else {
                for (int j = 1; j <= nx; j++) {
                    int rej;
                    double sky_val =
                        cpl_image_get(objb_rbn_img, j, indexrbn, &rej);
                    cpl_image_set(sci_single->sky, j, i, sky_val);
                    double rbn_img_val =
                        cpl_image_get(nod_rbn_img, j, indexrbn, &rej);
                    if (rej) {
                        cpl_image_set(sci_img, j, i, NAN);
                        cpl_image_set(sci_err_img, j, i, NAN);
                        cpl_image_reject(sci_img, j, i);
                    }
                    else {
                        double sci_flux = rbn_img_val;
                        cpl_image_set(sci_img, j, i, sci_flux);
                        double rbn_err_val =
                            cpl_image_get(nod_rbn_err_img, j, indexrbn, &rej);
                        double err = rbn_err_val;
                        cpl_image_set(sci_err_img, j, i, err);
                    }
                    double rbn_qual_val =
                        cpl_image_get(nod_rbn_qual_img, j, indexrbn, &rej);
                    cpl_image_set(sci_qual_img, j, i, rbn_qual_val);
                }
            }
        }
#ifdef _OPENMP
    }
#endif
moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_error_set_message(__func__, CPL_ERROR_ILLEGAL_OUTPUT,
                              "Error in combine pair %s",
                              nod_rbn_single->extname);
        moo_sci_single_delete(sci_single);
        sci_single = NULL;
    }
    return sci_single;
}

static cpl_propertylist *
_moo_create_primary_header(cpl_propertylist *rbn_primary_header)
{
    cpl_propertylist *primary_header = NULL;

    primary_header = cpl_propertylist_new();
    cpl_propertylist_copy_property(primary_header, rbn_primary_header,
                                   MOO_PFITS_EXPTIME);
    cpl_propertylist_copy_property(primary_header, rbn_primary_header,
                                   MOO_PFITS_MJDOBS);
    cpl_propertylist_copy_property_regexp(primary_header, rbn_primary_header,
                                          "TM*", 0);
    cpl_propertylist_copy_property_regexp(primary_header, rbn_primary_header,
                                          "ESO QC IS*", 0);
    cpl_propertylist_copy_property(primary_header, rbn_primary_header,
                                   MOO_PFITS_FLUXCAL);

    return primary_header;
}

static moo_sci *
_moo_combine_pair_2d(moo_rbn *nod_rbn,
                     moo_rbn *objb_rbn,
                     moo_rbn *obja_rbn,
                     moo_target_table *target_table,
                     const char *filename)
{
    moo_sci *sci = NULL;

    cpl_ensure(nod_rbn != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(objb_rbn != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(target_table != NULL, CPL_ERROR_NULL_INPUT, NULL);

    moo_target_table *sci_target_table = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    unsigned int badpix_level =
        MOO_BADPIX_OUTSIDE_DATA_RANGE | MOO_BADPIX_CALIB_DEFECT;

    moo_fits_create(filename);
    sci = moo_sci_new();
    sci->filename = filename;
    sci->primary_header = _moo_create_primary_header(nod_rbn->primary_header);

    cpl_msg_info(__func__, "Combine pair (2D)");

    double exptimeA[3];
    double exptimeB[3];
    exptimeA[0] = moo_pfits_get_exptime(nod_rbn->primary_header);
    exptimeB[0] = moo_pfits_get_exptime(objb_rbn->primary_header);

    for (int i = 1; i < 3; i++) {
        moo_rbn_single *obja_rbn_single = moo_rbn_get_single(nod_rbn, i);
        moo_rbn_single *objb_rbn_single = moo_rbn_get_single(nod_rbn, i);
        exptimeA[i] = 0;
        exptimeB[i] = 0;
        if (obja_rbn_single != NULL) {
            cpl_propertylist *header =
                moo_rbn_single_get_header(obja_rbn_single);

            double dit = moo_pfits_get_dit(header);
            int ndit = moo_pfits_get_ndit(header);

            exptimeA[i] = ndit * dit;
        }
        if (objb_rbn_single != NULL) {
            cpl_propertylist *header =
                moo_rbn_single_get_header(objb_rbn_single);

            double dit = moo_pfits_get_dit(header);
            int ndit = moo_pfits_get_ndit(header);

            exptimeB[i] = ndit * dit;
        }
    }
    sci_target_table = moo_target_table_duplicate(target_table);
    moo_target_table_set_exptimes(sci_target_table, exptimeA, exptimeB);
    cpl_table *nod_ft = moo_rbn_get_fibre_table(nod_rbn);

    sci->target_table = sci_target_table;

    cpl_msg_indent_more();

    for (int i = 0; i < 3; i++) {
        moo_sci_single *sci_single = NULL;
        moo_rbn_single *nod_rbn_single =
            moo_rbn_load_single(nod_rbn, i, badpix_level);
        moo_rbn_single *objb_rbn_single =
            moo_rbn_load_single(objb_rbn, i, badpix_level);
        moo_rbn_single *obja_rbn_single =
            moo_rbn_load_single(obja_rbn, i, badpix_level);
        if (nod_rbn_single != NULL) {
            moo_try_check(sci_single = _moo_combine_pair_2d_single(
                              nod_rbn_single, objb_rbn_single, obja_rbn_single,
                              nod_ft, sci_target_table),
                          " ");
        }
        moo_sci_add_single(sci, i, sci_single);
    }
    moo_sci_add_target_table(sci, sci_target_table);

moo_try_cleanup:
    cpl_msg_indent_less();

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_error_set_message(__func__, CPL_ERROR_ILLEGAL_OUTPUT,
                              "Error in combine pair (2D)");
        moo_sci_delete(sci);
        sci = NULL;
    }
    return sci;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Combine the extracted spectra from a same objec
  @param    nod_rbn _RBN Wavelength calibrated science spectra from NOD
  @param    sky_rbn _RBN Wavelength calibrated science spectra from SKY
  @param    obj_rbn _RBN Wavelength calibrated science spectra from OBJECT
  @param    target_table _TARGET_TABLE_ Attached target table
  @param    f2f _F2F_ fibre to fibre table
  @param    solflux_frame the monthly averages of solar radio flux at 10.7 cm frame
  @param    airglow_group_frame the airglow line list frame (use in subsky)
  @param    airglow_var_frame the airglow scaling parameters frame (use in subsky)
  @param    combine_pair_params the combine pair parameters
  @param    sky_params the sub sky stare parameters
  @param    filename  the SCI filename
  @param    products the product structure to save file
  @param    ref_frame the reference frame to produce product KW
  @return   the _SCI_ result

 * _Flags considered as bad :
 * MOO_BADPIX_OUTSIDE_DATA_RANGE, MOO_BADPIX_CALIB_DEFECT
 *
 * _Bad pixels flags_:

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_sci *
moo_combine_pair(moo_rbn *nod_rbn,
                 moo_rbn *sky_rbn,
                 moo_rbn *obj_rbn,
                 moo_target_table *target_table,
                 moo_f2f *f2f,
                 const cpl_frame *solflux_frame,
                 const cpl_frame *airglow_group_frame,
                 const cpl_frame *airglow_var_frame,
                 moo_combine_pair_params *combine_pair_params,
                 moo_sub_sky_stare_params *sky_params,
                 const char *filename,
                 moo_products *products,
                 const cpl_frame *ref_frame)
{
    moo_sci *sci1d = NULL;
    moo_sci *psci1d = NULL;
    char *sci1d_name = NULL;
    char *psci1d_name = NULL;
    char *sci2d_name = NULL;
    char *psci2d_name = NULL;
    moo_sci *sci2d = NULL;
    moo_sci *psci2d = NULL;
    moo_sci *cpair_sci = NULL;

    cpl_ensure(nod_rbn != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(sky_rbn != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(target_table != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_msg_info(__func__, "Combine pair using optimal:%d",
                 combine_pair_params->optimal);

    if (combine_pair_params->optimal == 1) {
        cpl_msg_info(__func__, "Substract sky (1D)");
        sci1d_name = cpl_sprintf("1D_%s", filename);

        moo_try_check(sci1d =
                          moo_sub_sky_stare(obj_rbn, target_table, sky_rbn, f2f,
                                            solflux_frame, airglow_group_frame,
                                            airglow_var_frame, sky_params,
                                            sci1d_name, MOO_SCI1D_NOT_PAIRED),
                      " ");
        if (products != NULL) {
            moo_try_check(moo_products_add_sci(products, sci1d,
                                               CPL_FRAME_LEVEL_TEMPORARY, "1D",
                                               sci1d_name, ref_frame),
                          " ");
        }
#if MOO_DEBUG_COMBINEPAIR
        {
            char *testname = cpl_sprintf("CPAIR_1D_%s", filename);
            moo_sci_save(sci1d, testname);
            cpl_free(testname);
        }
#endif

        cpl_msg_info(__func__, "Substract sky paired-fibre (1D)");

        psci1d_name = cpl_sprintf("P1D_%s", filename);

        moo_try_check(psci1d =
                          moo_sub_sky_stare(obj_rbn, target_table, sky_rbn, f2f,
                                            solflux_frame, airglow_group_frame,
                                            airglow_var_frame, sky_params,
                                            psci1d_name, MOO_SCI1D_PAIRED),
                      " ");
        if (products != NULL) {
            moo_try_check(moo_products_add_sci(products, psci1d,
                                               CPL_FRAME_LEVEL_TEMPORARY, "P1D",
                                               psci1d_name, ref_frame),
                          " ");
        }
#if MOO_DEBUG_COMBINEPAIR
        {
            char *testname = cpl_sprintf("CPAIR_P1D_%s", filename);
            moo_sci_save(psci1d, testname);
            cpl_free(testname);
        }
#endif
        sci2d_name = cpl_sprintf("2D_%s", filename);
        psci2d_name = cpl_sprintf("2D_PAIRED_%s", filename);

        psci2d = moo_create_sci2d(nod_rbn, sky_rbn, obj_rbn, target_table,
                                  MOO_SCI2D_PAIRED, psci2d_name);
        sci2d = moo_create_sci2d(nod_rbn, sky_rbn, obj_rbn, target_table,
                                 MOO_SCI2D_NOT_PAIRED, sci2d_name);
        if (products != NULL) {
            moo_try_check(moo_products_add_sci(products, sci2d,
                                               CPL_FRAME_LEVEL_TEMPORARY, "2D",
                                               sci2d_name, ref_frame),
                          " ");
            moo_try_check(moo_products_add_sci(products, psci2d,
                                               CPL_FRAME_LEVEL_TEMPORARY, "2D",
                                               psci2d_name, ref_frame),
                          " ");
        }
        cpl_msg_info("test", "combine all");
        cpair_sci = moo_combine_pair_sci(target_table, sci1d, psci1d, sci2d,
                                         psci2d, filename);
    }
    else {
        moo_try_check(cpair_sci =
                          _moo_combine_pair_2d(nod_rbn, sky_rbn, obj_rbn,
                                               target_table, filename),
                      " ");
    }

moo_try_cleanup:
    cpl_free(sci1d_name);
    cpl_free(psci1d_name);
    cpl_free(sci2d_name);
    cpl_free(psci2d_name);
    moo_sci_delete(sci1d);
    moo_sci_delete(psci1d);
    moo_sci_delete(psci2d);
    moo_sci_delete(sci2d);
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_error_set_message(__func__, CPL_ERROR_ILLEGAL_OUTPUT,
                              "Error in sky subtraction");
        moo_sci_delete(cpair_sci);
        cpair_sci = NULL;
    }
    return cpair_sci;
}

moo_sci *
moo_create_sci2d(moo_rbn *nod_rbn,
                 moo_rbn *objb_rbn,
                 moo_rbn *obja_rbn,
                 moo_target_table *target_table,
                 int ispaired,
                 const char *filename)
{
    moo_sci *sci = NULL;

    cpl_ensure(nod_rbn != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(target_table != NULL, CPL_ERROR_NULL_INPUT, NULL);

    moo_target_table *sci_target_table = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();
    unsigned int badpix_level =
        MOO_BADPIX_OUTSIDE_DATA_RANGE | MOO_BADPIX_CALIB_DEFECT;

    cpl_msg_info(__func__, "Create SCI (2D) ispaired:%d", ispaired);
    cpl_table_unselect_all(target_table->table);

    if (ispaired == 0) {
        /* OBS_SWITCH OBS_NOD OBS_NODSTARE SKYRES_NOD */
        cpl_table_or_selected_string(target_table->table,
                                     MOO_TARGET_TABLE_OBSTYPE, CPL_EQUAL_TO,
                                     MOO_OBSTYPE_OBSSWITCH);
        cpl_table_and_selected_string(target_table->table,
                                      MOO_TARGET_TABLE_OBSTYPE,
                                      CPL_NOT_EQUAL_TO,
                                      MOO_OBSTYPE_OBSSWITCHSTARE);

        cpl_table_or_selected_string(target_table->table,
                                     MOO_TARGET_TABLE_OBSTYPE, CPL_EQUAL_TO,
                                     MOO_OBSTYPE_OBSNOD);
        cpl_table_or_selected_string(target_table->table,
                                     MOO_TARGET_TABLE_OBSTYPE, CPL_EQUAL_TO,
                                     MOO_OBSTYPE_SKYRESNOD);
    }
    else {
        /* OBS_SWITCH OBS_PNOD OBS_PNODSTARE */
        cpl_table_or_selected_string(target_table->table,
                                     MOO_TARGET_TABLE_OBSTYPE, CPL_EQUAL_TO,
                                     MOO_OBSTYPE_OBSSWITCH);
        cpl_table_and_selected_string(target_table->table,
                                      MOO_TARGET_TABLE_OBSTYPE,
                                      CPL_NOT_EQUAL_TO,
                                      MOO_OBSTYPE_OBSSWITCHSTARE);

        cpl_table_or_selected_string(target_table->table,
                                     MOO_TARGET_TABLE_OBSTYPE, CPL_EQUAL_TO,
                                     MOO_OBSTYPE_OBSPNOD);
    }
    cpl_table *ttable = cpl_table_extract_selected(target_table->table);
    moo_fits_create(filename);
    sci = moo_sci_new();
    sci->filename = filename;
    sci->primary_header = _moo_create_primary_header(nod_rbn->primary_header);

    sci_target_table = moo_target_table_new();
    sci_target_table->primary_header = cpl_propertylist_new();
    sci_target_table->table = ttable;
    sci->target_table = sci_target_table;
    if (ispaired) {
        moo_target_table_xswitch_paired(sci_target_table);
    }

    double exptimeA[3];
    double exptimeB[3];

    exptimeA[0] = moo_pfits_get_exptime(nod_rbn->primary_header);

    for (int i = 1; i < 3; i++) {
        moo_rbn_single *obja_rbn_single = moo_rbn_get_single(nod_rbn, i);
        moo_rbn_single *objb_rbn_single = moo_rbn_get_single(nod_rbn, i);
        exptimeA[i] = 0;
        exptimeB[i] = 0;
        if (obja_rbn_single != NULL) {
            cpl_propertylist *header =
                moo_rbn_single_get_header(obja_rbn_single);

            double dit = moo_pfits_get_dit(header);
            int ndit = moo_pfits_get_ndit(header);

            exptimeA[i] = ndit * dit;
        }
    }

    moo_target_table_set_exptimes(sci_target_table, exptimeA, exptimeB);

    cpl_table *nod_ft = moo_rbn_get_fibre_table(nod_rbn);

    cpl_msg_indent_more();

    for (int i = 0; i < 3; i++) {
        moo_sci_single *sci_single = NULL;
        moo_rbn_single *nod_rbn_single = NULL;
        moo_rbn_single *obja_rbn_single = NULL;
        moo_rbn_single *objb_rbn_single = NULL;

        nod_rbn_single = moo_rbn_load_single(nod_rbn, i, badpix_level);
        objb_rbn_single = moo_rbn_load_single(objb_rbn, i, badpix_level);

        if (nod_rbn_single != NULL) {
            if (obja_rbn != NULL) {
                obja_rbn_single =
                    moo_rbn_load_single(obja_rbn, i, badpix_level);
            }

            moo_try_check(sci_single = _moo_create_sci2d_single(
                              nod_rbn_single, objb_rbn_single, obja_rbn_single,
                              nod_ft, sci_target_table, ispaired),
                          " ");
        }
        moo_sci_add_single(sci, i, sci_single);
    }

    moo_sci_add_target_table(sci, sci_target_table);

moo_try_cleanup:
    cpl_msg_indent_less();

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_error_set_message(__func__, CPL_ERROR_ILLEGAL_OUTPUT,
                              "Error in combine pair (2D)");
        moo_sci_delete(sci);
        sci = NULL;
    }
    return sci;
}

static moo_sci_single *
_moo_combine_pair_sci_single(moo_target_table *target_table,
                             moo_sci_single *sci1d,
                             cpl_table *sci1d_table,
                             moo_sci_single *psci1d,
                             cpl_table *psci1d_table,
                             moo_sci_single *sci2d,
                             cpl_table *sci2d_table,
                             moo_sci_single *psci2d,
                             cpl_table *psci2d_table)
{
    moo_sci_single *sci_single = NULL;
    moo_sci_single *main_sci_single = NULL;
    int *sci2d_index = NULL;
    int *sci1d_index = NULL;
    int *psci2d_index = NULL;
    int *psci1d_index = NULL;

    cpl_image *sci_img = NULL;
    cpl_image *sci_err_img = NULL;
    cpl_image *sci_qual_img = NULL;
    cpl_image *sci_sky = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(target_table != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (sci1d != NULL) {
        main_sci_single = sci1d;
    }
    else if (sci2d != NULL) {
        main_sci_single = sci2d;
    }
    else if (psci1d != NULL) {
        main_sci_single = psci1d;
    }
    else if (psci2d != NULL) {
        main_sci_single = psci2d;
    }

    if (main_sci_single == NULL) {
        return NULL;
    }

    moo_try_check(sci_single = moo_sci_single_new(main_sci_single->type), " ");

    moo_pfits_append_hduclass_data(sci_single->header, sci_single->type, -1);
    cpl_propertylist_copy_property(sci_single->header, main_sci_single->header,
                                   MOO_PFITS_CRPIX1);
    cpl_propertylist_copy_property(sci_single->header, main_sci_single->header,
                                   MOO_PFITS_CRVAL1);
    cpl_propertylist_copy_property(sci_single->header, main_sci_single->header,
                                   MOO_PFITS_CD1_1);
    cpl_propertylist_copy_property(sci_single->header, main_sci_single->header,
                                   MOO_PFITS_CD1_2);
    cpl_propertylist_copy_property(sci_single->header, main_sci_single->header,
                                   MOO_PFITS_CTYPE1);
    cpl_propertylist_copy_property(sci_single->header, main_sci_single->header,
                                   MOO_PFITS_CUNIT1);
    cpl_propertylist_copy_property(sci_single->header, main_sci_single->header,
                                   MOO_PFITS_BUNIT);
    cpl_propertylist_copy_property(sci_single->header, main_sci_single->header,
                                   MOO_PFITS_CRPIX2);
    cpl_propertylist_copy_property(sci_single->header, main_sci_single->header,
                                   MOO_PFITS_CRVAL2);
    cpl_propertylist_copy_property(sci_single->header, main_sci_single->header,
                                   MOO_PFITS_CD2_1);
    cpl_propertylist_copy_property(sci_single->header, main_sci_single->header,
                                   MOO_PFITS_CD2_2);
    cpl_propertylist_copy_property(sci_single->header, main_sci_single->header,
                                   MOO_PFITS_CTYPE2);
    cpl_propertylist_copy_property(sci_single->header, main_sci_single->header,
                                   MOO_PFITS_BUNIT);
    cpl_propertylist_copy_property(sci_single->header, main_sci_single->header,
                                   MOO_PFITS_WAVELMIN);
    cpl_propertylist_copy_property(sci_single->header, main_sci_single->header,
                                   MOO_PFITS_WAVELMAX);
    cpl_propertylist_copy_property(sci_single->header, main_sci_single->header,
                                   MOO_PFITS_SPECBIN);
    int nx = hdrl_image_get_size_x(main_sci_single->image);
    int nb_target = cpl_table_get_nrow(target_table->table);

    sci_single->image = hdrl_image_new(nx, nb_target);
    sci_single->sky = cpl_image_new(nx, nb_target, CPL_TYPE_DOUBLE);
    sci_single->qual = cpl_image_new(nx, nb_target, CPL_TYPE_INT);

    sci_img = hdrl_image_get_image(sci_single->image);
    /* create the mask if empty */
    cpl_image_get_bpm(sci_img);

    sci_err_img = hdrl_image_get_error(sci_single->image);
    sci_qual_img = sci_single->qual;
    sci_sky = sci_single->sky;

    const char *transname[] = { MOO_TARGET_TABLE_TRANSRI,
                                MOO_TARGET_TABLE_TRANSYJ,
                                MOO_TARGET_TABLE_TRANSH };
    const char *ptransname[] = { MOO_TARGET_TABLE_PTRANSRI,
                                 MOO_TARGET_TABLE_PTRANSYJ,
                                 MOO_TARGET_TABLE_PTRANSH };
    const char *exptimenames[] = { MOO_TARGET_TABLE_EXPTIMERI,
                                   MOO_TARGET_TABLE_EXPTIMEYJ,
                                   MOO_TARGET_TABLE_EXPTIMEH };
    const char *exptimename = exptimenames[sci_single->type];

    float *transt = cpl_table_get_data_float(target_table->table,
                                             transname[sci_single->type]);
    float *ptranst = cpl_table_get_data_float(target_table->table,
                                              ptransname[sci_single->type]);

    int *indexrbnt =
        cpl_table_get_data_int(target_table->table, MOO_TARGET_TABLE_INDEXRBN);
    int *pindexrbnt = cpl_table_get_data_int(target_table->table,
                                             MOO_TARGET_TABLE_PAIREDINDEXRBN);
    char **obstypet = cpl_table_get_data_string(target_table->table,
                                                MOO_TARGET_TABLE_OBSTYPE);
    sci2d_index = cpl_calloc(nb_target, sizeof(int));
    sci1d_index = cpl_calloc(nb_target, sizeof(int));

    psci2d_index = cpl_calloc(nb_target, sizeof(int));
    psci1d_index = cpl_calloc(nb_target, sizeof(int));

    for (int i = 1; i <= nb_target; i++) {
        double exptime = NAN;
        int idx = i - 1;
        sci2d_index[idx] = -1;
        psci2d_index[idx] = -1;
        sci1d_index[idx] = -1;

        cpl_table_set_int(target_table->table, MOO_TARGET_TABLE_INDEXTARG, idx,
                          i);

        int indexrbn = indexrbnt[idx];
        int pindexrbn = pindexrbnt[idx];
        char *obstype = obstypet[idx];

        if (strcmp(obstype, MOO_OBSTYPE_OBSNOD) == 0 ||
            strcmp(obstype, MOO_OBSTYPE_SKYRESNOD) == 0) {
            cpl_table_select_all(sci2d_table);

            int size = cpl_table_and_selected_int(sci2d_table,
                                                  MOO_TARGET_TABLE_INDEXRBN,
                                                  CPL_EQUAL_TO, indexrbn);
            cpl_ensure(size == 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
            cpl_array *sel = cpl_table_where_selected(sci2d_table);

            int tidx = cpl_array_get_cplsize(sel, 0, NULL);

            cpl_array_delete(sel);

            exptime =
                cpl_table_get_double(sci2d_table, exptimename, tidx, NULL);
            sci2d_index[idx] = tidx;
        }
        else if (strcmp(obstype, MOO_OBSTYPE_OBSPNOD) == 0) {
            cpl_table_select_all(psci2d_table);

            int size = cpl_table_and_selected_int(psci2d_table,
                                                  MOO_TARGET_TABLE_INDEXRBN,
                                                  CPL_EQUAL_TO, indexrbn);
            cpl_ensure(size == 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
            cpl_array *sel = cpl_table_where_selected(psci2d_table);

            int tidx = cpl_array_get_cplsize(sel, 0, NULL);

            cpl_array_delete(sel);
            psci2d_index[idx] = tidx;
        }
        else if (strcmp(obstype, MOO_OBSTYPE_OBSSWITCH) == 0) {
            cpl_table_select_all(sci2d_table);

            int size = cpl_table_and_selected_int(sci2d_table,
                                                  MOO_TARGET_TABLE_INDEXRBN,
                                                  CPL_EQUAL_TO, indexrbn);
            cpl_ensure(size == 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
            cpl_array *sel = cpl_table_where_selected(sci2d_table);

            int tidx = cpl_array_get_cplsize(sel, 0, NULL);

            cpl_array_delete(sel);
            sci2d_index[idx] = tidx;

            cpl_table_select_all(psci2d_table);

            size = cpl_table_and_selected_int(psci2d_table,
                                              MOO_TARGET_TABLE_INDEXRBN,
                                              CPL_EQUAL_TO, pindexrbn);
            cpl_ensure(size == 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
            sel = cpl_table_where_selected(psci2d_table);

            tidx = cpl_array_get_cplsize(sel, 0, NULL);
            cpl_array_delete(sel);

            psci2d_index[idx] = tidx;
        }
        else if (strcmp(obstype, MOO_OBSTYPE_OBSSWITCHSTARE) == 0) {
            cpl_table_select_all(sci1d_table);

            int size = cpl_table_and_selected_int(sci1d_table,
                                                  MOO_TARGET_TABLE_INDEXRBN,
                                                  CPL_EQUAL_TO, indexrbn);
            cpl_ensure(size == 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
            cpl_array *sel = cpl_table_where_selected(sci1d_table);

            int tidx = cpl_array_get_cplsize(sel, 0, NULL);

            cpl_array_delete(sel);
            sci1d_index[idx] = tidx;

            cpl_table_select_all(psci1d_table);

            size = cpl_table_and_selected_int(psci1d_table,
                                              MOO_TARGET_TABLE_INDEXRBN,
                                              CPL_EQUAL_TO, pindexrbn);
            cpl_ensure(size == 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
            sel = cpl_table_where_selected(psci1d_table);

            tidx = cpl_array_get_cplsize(sel, 0, NULL);
            cpl_array_delete(sel);

            psci1d_index[idx] = tidx;
        }
        else if (strcmp(obstype, MOO_OBSTYPE_OBSNODSTARE) == 0) {
            cpl_table_select_all(sci2d_table);

            int size = cpl_table_and_selected_int(sci2d_table,
                                                  MOO_TARGET_TABLE_INDEXRBN,
                                                  CPL_EQUAL_TO, indexrbn);
            cpl_ensure(size == 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
            cpl_array *sel = cpl_table_where_selected(sci2d_table);

            int tidx = cpl_array_get_cplsize(sel, 0, NULL);

            cpl_array_delete(sel);
            sci2d_index[idx] = tidx;

            cpl_table_select_all(psci1d_table);

            size = cpl_table_and_selected_int(psci1d_table,
                                              MOO_TARGET_TABLE_INDEXRBN,
                                              CPL_EQUAL_TO, pindexrbn);
            cpl_ensure(size == 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
            sel = cpl_table_where_selected(psci1d_table);

            tidx = cpl_array_get_cplsize(sel, 0, NULL);
            cpl_array_delete(sel);

            psci1d_index[idx] = tidx;
        }
        else if (strcmp(obstype, MOO_OBSTYPE_OBSPNODSTARE) == 0) {
            cpl_table_select_all(psci2d_table);

            int size = cpl_table_and_selected_int(psci2d_table,
                                                  MOO_TARGET_TABLE_INDEXRBN,
                                                  CPL_EQUAL_TO, pindexrbn);
            cpl_ensure(size == 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
            cpl_array *sel = cpl_table_where_selected(psci2d_table);

            int tidx = cpl_array_get_cplsize(sel, 0, NULL);

            cpl_array_delete(sel);
            psci2d_index[idx] = tidx;

            cpl_table_select_all(sci1d_table);

            size = cpl_table_and_selected_int(sci1d_table,
                                              MOO_TARGET_TABLE_INDEXRBN,
                                              CPL_EQUAL_TO, indexrbn);
            cpl_ensure(size == 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
            sel = cpl_table_where_selected(sci1d_table);

            tidx = cpl_array_get_cplsize(sel, 0, NULL);
            cpl_array_delete(sel);

            sci1d_index[idx] = tidx;
        }
        else if (strcmp(obstype, MOO_OBSTYPE_OBSSTARE) == 0 ||
                 strcmp(obstype, MOO_OBSTYPE_SKYRESSTARE) == 0) {
            cpl_table_select_all(sci1d_table);

            int size = cpl_table_and_selected_int(sci1d_table,
                                                  MOO_TARGET_TABLE_INDEXRBN,
                                                  CPL_EQUAL_TO, indexrbn);
            cpl_ensure(size == 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
            cpl_array *sel = cpl_table_where_selected(sci1d_table);

            int tidx = cpl_array_get_cplsize(sel, 0, NULL);
            exptime =
                cpl_table_get_double(sci1d_table, exptimename, tidx, NULL);
            cpl_array_delete(sel);
            sci1d_index[idx] = tidx;
        }
        else if (strcmp(obstype, MOO_OBSTYPE_OBSPSTARE) == 0 ||
                 strcmp(obstype, MOO_OBSTYPE_SKYRESPSTARE) == 0) {
            cpl_table_select_all(psci1d_table);

            int size = cpl_table_and_selected_int(psci1d_table,
                                                  MOO_TARGET_TABLE_INDEXRBN,
                                                  CPL_EQUAL_TO, indexrbn);
            cpl_ensure(size == 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
            cpl_array *sel = cpl_table_where_selected(psci1d_table);

            int tidx = cpl_array_get_cplsize(sel, 0, NULL);

            cpl_array_delete(sel);
            psci1d_index[idx] = tidx;
        }
        cpl_table_set_double(target_table->table, exptimename, idx, exptime);
    }
/* The following addresses an issue with the gcc9 compiler series prior to
 * gcc 9.3. These compiler versions require that the variable '__func__'
 * appears in the data sharing clause if default(none) is used. However
 * adding it to the data sharing clause breaks older compiler versions
 * where these variables are pre-determined as shared.
 * This was fixed in gcc 9.3 and OpenMP 5.1.
 */
#ifdef _OPENMP
#if (__GNUC__ == 9) && (__GNUC_MINOR__ < 3)
#pragma omp parallel shared(nb_target, indexrbnt, pindexrbnt, transt, ptranst, \
                                obstypet, target_table, sci2d, psci2d, sci1d,  \
                                psci1d, sci_single, sci_qual_img, sci_img,     \
                                sci_err_img, sci_sky, nx, sci2d_index,         \
                                psci2d_index, sci1d_index, psci1d_index)
#else
#pragma omp parallel default(none)                                           \
    shared(nb_target, indexrbnt, pindexrbnt, transt, ptranst, obstypet,      \
               target_table, sci2d, psci2d, sci1d, psci1d, sci_single,       \
               sci_qual_img, sci_img, sci_err_img, sci_sky, nx, sci2d_index, \
               psci2d_index, sci1d_index, psci1d_index)
#endif
    {
#pragma omp for
#endif
        for (int i = 1; i <= nb_target; i++) {
            int idx = i - 1;
            cpl_table_set_int(target_table->table, MOO_TARGET_TABLE_INDEXTARG,
                              idx, i);

            char *obstype = obstypet[idx];
            float trans = transt[idx];
            float ptrans = ptranst[idx];
            if (strcmp(obstype, MOO_OBSTYPE_OBSNOD) == 0 ||
                strcmp(obstype, MOO_OBSTYPE_SKYRESNOD) == 0) {
                int tidx = sci2d_index[idx] + 1;
                hdrl_image *himg = moo_sci_single_get_image(sci2d);
                cpl_image *fluxes = hdrl_image_get_image(himg);
                cpl_image *skyimg = moo_sci_single_get_sky(sci2d);
                cpl_image *errs = hdrl_image_get_error(himg);
                cpl_image *qualimg = moo_sci_single_get_qual(sci2d);

                for (int j = 1; j <= nx; j++) {
                    int rej;
                    double err = cpl_image_get(errs, j, tidx, &rej);
                    double flux = cpl_image_get(fluxes, j, tidx, &rej);
                    double sky = cpl_image_get(skyimg, j, tidx, &rej);
                    double qual = cpl_image_get(qualimg, j, tidx, &rej);
                    cpl_image_set(sci_img, j, i, flux);
                    cpl_image_set(sci_err_img, j, i, err);
                    cpl_image_set(sci_sky, j, i, sky);
                    cpl_image_set(sci_qual_img, j, i, qual);
                }
            }
            else if (strcmp(obstype, MOO_OBSTYPE_OBSPNOD) == 0) {
                int tidx = psci2d_index[idx] + 1;
                hdrl_image *himg = moo_sci_single_get_image(psci2d);
                cpl_image *fluxes = hdrl_image_get_image(himg);
                cpl_image *skyimg = moo_sci_single_get_sky(psci2d);
                cpl_image *errs = hdrl_image_get_error(himg);
                cpl_image *qualimg = moo_sci_single_get_qual(psci2d);

                for (int j = 1; j <= nx; j++) {
                    int rej;
                    double err = cpl_image_get(errs, j, tidx, &rej);
                    double flux = cpl_image_get(fluxes, j, tidx, &rej);
                    double sky = cpl_image_get(skyimg, j, tidx, &rej);
                    double qual = cpl_image_get(qualimg, j, tidx, &rej);
                    cpl_image_set(sci_img, j, i, flux);
                    cpl_image_set(sci_err_img, j, i, err);
                    cpl_image_set(sci_sky, j, i, sky);
                    cpl_image_set(sci_qual_img, j, i, qual);
                }
            }
            else if (strcmp(obstype, MOO_OBSTYPE_OBSSWITCH) == 0) {
                int tidx = sci2d_index[idx] + 1;
                int ptidx = psci2d_index[idx] + 1;

                double transsum = trans + ptrans;
                hdrl_image *himg = moo_sci_single_get_image(sci2d);
                cpl_image *fluxes = hdrl_image_get_image(himg);
                cpl_image *skyimg = moo_sci_single_get_sky(sci2d);
                cpl_image *errs = hdrl_image_get_error(himg);
                cpl_image *qualimg = moo_sci_single_get_qual(sci2d);

                hdrl_image *phimg = moo_sci_single_get_image(psci2d);
                cpl_image *pfluxes = hdrl_image_get_image(phimg);
                cpl_image *pskyimg = moo_sci_single_get_sky(psci2d);
                cpl_image *perrs = hdrl_image_get_error(phimg);
                cpl_image *pqualimg = moo_sci_single_get_qual(psci2d);

                for (int j = 1; j <= nx; j++) {
                    int rej1;
                    double err1 = cpl_image_get(errs, j, tidx, &rej1);
                    double flux1 = cpl_image_get(fluxes, j, tidx, &rej1);
                    double sky1 = cpl_image_get(skyimg, j, tidx, &rej1);
                    int qual1 = (int)cpl_image_get(qualimg, j, tidx, &rej1);

                    int rej2;
                    double err2 = cpl_image_get(perrs, j, ptidx, &rej2);
                    double flux2 = cpl_image_get(pfluxes, j, ptidx, &rej2);
                    double sky2 = cpl_image_get(pskyimg, j, ptidx, &rej2);
                    int qual2 = (int)cpl_image_get(pqualimg, j, ptidx, &rej2);

                    int qual = qual1 | qual2;
                    double flux = NAN;
                    double err = NAN;
                    double sky = NAN;

                    if (!rej1 && !rej2) {
                        flux = (trans * flux1 + ptrans * flux2) / transsum * 2;

                        err =
                            sqrt((trans * err1 * err1 + ptrans * err2 * err2) /
                                 transsum) *
                            2.;
                        sky = (trans * sky1 + ptrans * sky2) / transsum * 2;
                    }

                    cpl_image_set(sci_img, j, i, flux);
                    cpl_image_set(sci_err_img, j, i, err);
                    cpl_image_set(sci_sky, j, i, sky);
                    cpl_image_set(sci_qual_img, j, i, qual);
                }
            }
            else if (strcmp(obstype, MOO_OBSTYPE_OBSNODSTARE) == 0) {
                int tidx = sci2d_index[idx] + 1;
                int ptidx = psci1d_index[idx] + 1;

                double transsum = trans + ptrans;

                hdrl_image *himg = moo_sci_single_get_image(sci2d);
                cpl_image *fluxes = hdrl_image_get_image(himg);
                cpl_image *skyimg = moo_sci_single_get_sky(sci2d);
                cpl_image *errs = hdrl_image_get_error(himg);
                cpl_image *qualimg = moo_sci_single_get_qual(sci2d);

                hdrl_image *phimg = moo_sci_single_get_image(psci1d);
                cpl_image *pfluxes = hdrl_image_get_image(phimg);
                cpl_image *pskyimg = moo_sci_single_get_sky(psci1d);
                cpl_image *perrs = hdrl_image_get_error(phimg);
                cpl_image *pqualimg = moo_sci_single_get_qual(psci1d);

                for (int j = 1; j <= nx; j++) {
                    int rej1;
                    double err1 = cpl_image_get(errs, j, tidx, &rej1);
                    double flux1 = cpl_image_get(fluxes, j, tidx, &rej1);
                    double sky1 = cpl_image_get(skyimg, j, tidx, &rej1);
                    int qual1 = (int)cpl_image_get(qualimg, j, tidx, &rej1);

                    int rej2;
                    double err2 = cpl_image_get(perrs, j, ptidx, &rej2);
                    double flux2 = cpl_image_get(pfluxes, j, ptidx, &rej2);
                    double sky2 = cpl_image_get(pskyimg, j, ptidx, &rej2);
                    int qual2 = (int)cpl_image_get(pqualimg, j, ptidx, &rej2);
                    int qual = qual1 | qual2;
                    double flux = NAN;
                    double err = NAN;
                    double sky = NAN;

                    if (rej1) {
                        if (!rej2) {
                            flux = flux2 * 2;
                            err = err2 * 2;
                            qual = qual2;
                        }
                    }
                    else {
                        if (!rej2) {
                            flux =
                                (trans * flux1 + ptrans * flux2) / transsum * 2;

                            err = sqrt((trans * err1 * err1 +
                                        ptrans * err2 * err2) /
                                       transsum) *
                                  2.;
                            sky = (trans * sky1 + ptrans * sky2) / transsum * 2;
                        }
                        else {
                            flux = flux1 * 2;
                            err = err1 * 2;
                            qual = qual1;
                        }
                    }

                    cpl_image_set(sci_img, j, i, flux);
                    cpl_image_set(sci_err_img, j, i, err);
                    cpl_image_set(sci_sky, j, i, sky);
                    cpl_image_set(sci_qual_img, j, i, qual);
                }
            }
            else if (strcmp(obstype, MOO_OBSTYPE_OBSPNODSTARE) == 0) {
                int ptidx = psci2d_index[idx] + 1;
                int tidx = sci1d_index[idx] + 1;

                double transsum = trans + ptrans;
                hdrl_image *himg = moo_sci_single_get_image(sci1d);
                cpl_image *fluxes = hdrl_image_get_image(himg);
                cpl_image *skyimg = moo_sci_single_get_sky(sci1d);
                cpl_image *errs = hdrl_image_get_error(himg);
                cpl_image *qualimg = moo_sci_single_get_qual(sci1d);

                hdrl_image *phimg = moo_sci_single_get_image(psci2d);
                cpl_image *pfluxes = hdrl_image_get_image(phimg);
                cpl_image *pskyimg = moo_sci_single_get_sky(psci2d);
                cpl_image *perrs = hdrl_image_get_error(phimg);
                cpl_image *pqualimg = moo_sci_single_get_qual(psci2d);

                for (int j = 1; j <= nx; j++) {
                    int rej1;
                    double err1 = cpl_image_get(errs, j, tidx, &rej1);
                    double flux1 = cpl_image_get(fluxes, j, tidx, &rej1);
                    double sky1 = cpl_image_get(skyimg, j, tidx, &rej1);
                    int qual1 = (int)cpl_image_get(qualimg, j, tidx, &rej1);

                    int rej2;
                    double err2 = cpl_image_get(perrs, j, ptidx, &rej2);
                    double flux2 = cpl_image_get(pfluxes, j, ptidx, &rej2);
                    double sky2 = cpl_image_get(pskyimg, j, ptidx, &rej2);
                    int qual2 = (int)cpl_image_get(pqualimg, j, ptidx, &rej2);
                    int qual = qual1 | qual2;
                    double flux = NAN;
                    double err = NAN;
                    double sky = NAN;

                    if (rej1) {
                        if (!rej2) {
                            flux = flux2 * 2;
                            err = err2 * 2;
                            qual = qual2;
                        }
                    }
                    else {
                        if (!rej2) {
                            flux =
                                (trans * flux1 + ptrans * flux2) / transsum * 2;

                            err = sqrt((trans * err1 * err1 +
                                        ptrans * err2 * err2) /
                                       transsum) *
                                  2.;
                            sky = (trans * sky1 + ptrans * sky2) / transsum * 2;
                        }
                        else {
                            flux = flux1 * 2;
                            err = err1 * 2;
                            qual = qual1;
                        }
                    }

                    cpl_image_set(sci_img, j, i, flux);
                    cpl_image_set(sci_err_img, j, i, err);
                    cpl_image_set(sci_sky, j, i, sky);
                    cpl_image_set(sci_qual_img, j, i, qual);
                }
            }
            else if (strcmp(obstype, MOO_OBSTYPE_OBSSWITCHSTARE) == 0) {
                int ptidx = psci1d_index[idx] + 1;
                int tidx = sci1d_index[idx] + 1;

                double transsum = trans + ptrans;
                hdrl_image *himg = moo_sci_single_get_image(sci1d);
                cpl_image *fluxes = hdrl_image_get_image(himg);
                cpl_image *skyimg = moo_sci_single_get_sky(sci1d);
                cpl_image *errs = hdrl_image_get_error(himg);
                cpl_image *qualimg = moo_sci_single_get_qual(sci1d);

                hdrl_image *phimg = moo_sci_single_get_image(psci1d);
                cpl_image *pfluxes = hdrl_image_get_image(phimg);
                cpl_image *pskyimg = moo_sci_single_get_sky(psci1d);
                cpl_image *perrs = hdrl_image_get_error(phimg);
                cpl_image *pqualimg = moo_sci_single_get_qual(psci1d);

                for (int j = 1; j <= nx; j++) {
                    int rej1;
                    double err1 = cpl_image_get(errs, j, tidx, &rej1);
                    double flux1 = cpl_image_get(fluxes, j, tidx, &rej1);
                    double sky1 = cpl_image_get(skyimg, j, tidx, &rej1);
                    int qual1 = (int)cpl_image_get(qualimg, j, tidx, &rej1);

                    int rej2;
                    double err2 = cpl_image_get(perrs, j, ptidx, &rej2);
                    double flux2 = cpl_image_get(pfluxes, j, ptidx, &rej2);
                    double sky2 = cpl_image_get(pskyimg, j, ptidx, &rej2);
                    int qual2 = (int)cpl_image_get(pqualimg, j, ptidx, &rej2);
                    int qual = qual1 | qual2;
                    double flux = NAN;
                    double err = NAN;
                    double sky = NAN;

                    if (rej1) {
                        if (!rej2) {
                            flux = flux2 * 2;
                            err = err2 * 2;
                            qual = qual2;
                        }
                    }
                    else {
                        if (!rej2) {
                            flux =
                                (trans * flux1 + ptrans * flux2) / transsum * 2;

                            err = sqrt((trans * err1 * err1 +
                                        ptrans * err2 * err2) /
                                       transsum) *
                                  2.;
                            sky = (trans * sky1 + ptrans * sky2) / transsum * 2;
                        }
                        else {
                            flux = flux1 * 2;
                            err = err1 * 2;
                            qual = qual1;
                        }
                    }

                    cpl_image_set(sci_img, j, i, flux);
                    cpl_image_set(sci_err_img, j, i, err);
                    cpl_image_set(sci_sky, j, i, sky);
                    cpl_image_set(sci_qual_img, j, i, qual);
                }
            }
            else if (strcmp(obstype, MOO_OBSTYPE_OBSSTARE) == 0 ||
                     strcmp(obstype, MOO_OBSTYPE_SKYRESSTARE) == 0) {
                int tidx = sci1d_index[idx] + 1;
                hdrl_image *himg = moo_sci_single_get_image(sci1d);
                cpl_image *fluxes = hdrl_image_get_image(himg);
                cpl_image *skyimg = moo_sci_single_get_sky(sci1d);
                cpl_image *errs = hdrl_image_get_error(himg);
                cpl_image *qualimg = moo_sci_single_get_qual(sci1d);

                for (int j = 1; j <= nx; j++) {
                    int rej;
                    double err = cpl_image_get(errs, j, tidx, &rej);
                    double flux = cpl_image_get(fluxes, j, tidx, &rej);
                    double sky = cpl_image_get(skyimg, j, tidx, &rej);
                    double qual = cpl_image_get(qualimg, j, tidx, &rej);
                    cpl_image_set(sci_img, j, i, flux);
                    cpl_image_set(sci_err_img, j, i, err);
                    cpl_image_set(sci_sky, j, i, sky);
                    cpl_image_set(sci_qual_img, j, i, qual);
                }
            }
            else if (strcmp(obstype, MOO_OBSTYPE_OBSPSTARE) == 0 ||
                     strcmp(obstype, MOO_OBSTYPE_SKYRESPSTARE) == 0) {
                int tidx = psci1d_index[idx] + 1;
                hdrl_image *himg = moo_sci_single_get_image(psci1d);
                cpl_image *fluxes = hdrl_image_get_image(himg);
                cpl_image *skyimg = moo_sci_single_get_sky(psci1d);
                cpl_image *errs = hdrl_image_get_error(himg);
                cpl_image *qualimg = moo_sci_single_get_qual(psci1d);

                for (int j = 1; j <= nx; j++) {
                    int rej;
                    double err = cpl_image_get(errs, j, tidx, &rej);
                    double flux = cpl_image_get(fluxes, j, tidx, &rej);
                    double sky = cpl_image_get(skyimg, j, tidx, &rej);
                    double qual = cpl_image_get(qualimg, j, tidx, &rej);
                    cpl_image_set(sci_img, j, i, flux);
                    cpl_image_set(sci_err_img, j, i, err);
                    cpl_image_set(sci_sky, j, i, sky);
                    cpl_image_set(sci_qual_img, j, i, qual);
                }
            }
        }
#ifdef _OPENMP
    }
#endif
moo_try_cleanup:
    cpl_free(sci2d_index);
    cpl_free(sci1d_index);
    cpl_free(psci2d_index);
    cpl_free(psci1d_index);
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_sci_single_delete(sci_single);
        sci_single = NULL;
    }
    return sci_single;
}
moo_sci *
moo_combine_pair_sci(moo_target_table *target_table,
                     moo_sci *sci1d,
                     moo_sci *psci1d,
                     moo_sci *sci2d,
                     moo_sci *psci2d,
                     const char *filename)
{
    moo_sci *sci = NULL;
    moo_sci *main_sci = NULL;
    cpl_table *sci1d_table = NULL;
    cpl_table *psci1d_table = NULL;
    cpl_table *sci2d_table = NULL;
    cpl_table *psci2d_table = NULL;

    cpl_ensure(target_table != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    unsigned int badpix_level =
        MOO_BADPIX_OUTSIDE_DATA_RANGE | MOO_BADPIX_CALIB_DEFECT;

    if (psci1d != NULL) {
        main_sci = psci1d;
        psci1d_table = moo_sci_get_target_table(psci1d)->table;
    }
    if (psci2d != NULL) {
        main_sci = psci2d;
        psci2d_table = moo_sci_get_target_table(psci2d)->table;
    }
    if (sci2d != NULL) {
        main_sci = sci2d;
        sci2d_table = moo_sci_get_target_table(sci2d)->table;
    }

    if (sci1d != NULL) {
        main_sci = sci1d;
        sci1d_table = moo_sci_get_target_table(sci1d)->table;
    }

    if (main_sci == NULL) {
        return NULL;
    }
    moo_fits_create(filename);
    sci = moo_sci_new();
    sci->filename = filename;
    sci->primary_header = cpl_propertylist_duplicate(main_sci->primary_header);

    cpl_msg_info(__func__, "Combine pair of SCI files");
    sci->target_table = target_table;

    cpl_msg_indent_more();

    for (int i = 0; i < 3; i++) {
        moo_sci_single *sci_single = NULL;
        moo_sci_single *sci1d_single = NULL;
        moo_sci_single *psci1d_single = NULL;
        moo_sci_single *sci2d_single = NULL;
        moo_sci_single *psci2d_single = NULL;

        if (sci1d != NULL) {
            sci1d_single = moo_sci_load_single(sci1d, i, badpix_level);
        }
        if (psci1d != NULL) {
            psci1d_single = moo_sci_load_single(psci1d, i, badpix_level);
        }
        if (sci2d != NULL) {
            sci2d_single = moo_sci_load_single(sci2d, i, badpix_level);
        }
        if (psci2d != NULL) {
            psci2d_single = moo_sci_load_single(psci2d, i, badpix_level);
        }
        moo_try_check(sci_single = _moo_combine_pair_sci_single(
                          target_table, sci1d_single, sci1d_table,
                          psci1d_single, psci1d_table, sci2d_single,
                          sci2d_table, psci2d_single, psci2d_table),
                      " ");
        moo_sci_add_single(sci, i, sci_single);
    }
    moo_sci_add_target_table(sci, moo_target_table_duplicate(target_table));

moo_try_cleanup:
    cpl_msg_indent_less();

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_error_set_message(__func__, CPL_ERROR_ILLEGAL_OUTPUT,
                              "Error in combine pair (2D)");
        moo_sci_delete(sci);
        sci = NULL;
    }
    return sci;
}

/**@}*/
