/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "moo_utils.h"
#include "moo_pfits.h"
#include "moo_detector.h"
#include "moo_badpix.h"
#include "moo_fibres_table.h"
#include "moo_fits.h"
#include "moo_pfits.h"
#include "moo_cube.h"
#include "moo_compute_linearity.h"
#include <cpl.h>
#include <hdrl.h>
#include <math.h>
#include <string.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/

static void
_moo_compute_trace_mask(cpl_mask *mask,
                        moo_loc *loc1,
                        moo_detector_type type,
                        int num)
{
    cpl_array *sel = NULL;
    int nx = cpl_mask_get_size_x(mask);
    int ny = cpl_mask_get_size_y(mask);

    if (loc1 == NULL) {
        for (int x = 1; x <= nx; x++) {
            for (int y = 1; y <= ny; y++) {
                cpl_mask_set(mask, x, y, CPL_BINARY_1);
            }
        }
    }
    else {
        int spectro_name = moo_fibres_table_get_spectro(num);
        moo_loc_single *locs1 = moo_loc_get_single(loc1, type, num);
        cpl_table *ft1 = moo_loc_get_fibre_table(loc1);
        moo_try_check(cpl_table_select_all(ft1), " ");
        moo_try_check(cpl_table_and_selected_int(ft1, MOO_FIBRES_TABLE_SPECTRO,
                                                 CPL_EQUAL_TO, spectro_name),
                      " ");
        moo_try_check(cpl_table_and_selected_int(ft1, MOO_FIBRES_TABLE_HEALTH,
                                                 CPL_EQUAL_TO, 1),
                      " ");
        sel = cpl_table_where_selected(ft1);
        cpl_size nsel = cpl_array_get_size(sel);
        cpl_image *centroid_img = moo_loc_single_get_f_centroids(locs1);
        cpl_image *wlo_img = moo_loc_single_get_f_wlo(locs1);
        cpl_image *wup_img = moo_loc_single_get_f_wup(locs1);

        for (int i = 0; i < nsel; i++) {
            int rej;
            cpl_size idx = cpl_array_get_cplsize(sel, i, NULL);
            int indexext =
                cpl_table_get_int(ft1, MOO_FIBRES_TABLE_INDEXEXT, idx, &rej);
            for (int x = 1; x <= nx; x++) {
                int rej2;
                double centroid =
                    cpl_image_get(centroid_img, x, indexext, &rej2);
                double wlo = cpl_image_get(wlo_img, x, indexext, &rej2);
                double wup = cpl_image_get(wup_img, x, indexext, &rej2);
                int yl = (int)floor(centroid - wlo);
                if (yl < 1) {
                    yl = 1;
                }
                int yu = (int)ceil(centroid + wup);
                if (yu >= ny) {
                    yu = ny;
                }
                for (int y = yl; y <= yu; y++) {
                    cpl_mask_set(mask, x, y, CPL_BINARY_1);
                }
            }
        }
    }
moo_try_cleanup:
    cpl_array_delete(sel);
}
/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/
cpl_imagelist *
_moo_compute_linearity_single(moo_detlist *detlist,
                              moo_loc *loc,
                              moo_detector_type type,
                              int num,
                              cpl_image *saturate,
                              moo_cube *cube)
{
    cpl_imagelist *result = NULL;
    cpl_imagelist *list = NULL;
    cpl_mask *mask = NULL;
    cpl_vector *exptime = NULL;
    cpl_vector *fluxes = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(detlist != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(cube != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int badpix_level = MOO_BADPIX_GOOD;
    int degree = MOO_COMPUTE_LINEARITY_DEGREE;

    moo_detlist_load_single(detlist, type, num, badpix_level);
    int size = moo_detlist_get_size(detlist);
    list = moo_detlist_get_single_data(detlist, type, num);

    if (list != NULL && cpl_imagelist_get_size(list) > 0) {
        cpl_msg_info(__func__, "Compute coefficients map for %s",
                     moo_detector_get_extname(type, num));

        int nx = cpl_image_get_size_x(cpl_imagelist_get(list, 0));
        int ny = cpl_image_get_size_y(cpl_imagelist_get(list, 0));

        result = cpl_imagelist_new();
        mask = cpl_mask_new(nx, ny);

        moo_try_check(_moo_compute_trace_mask(mask, loc, type, num), " ");
        exptime = cpl_vector_new(size);
        fluxes = cpl_vector_new(size);

        for (int i = 0; i < size; i++) {
            moo_det *det = moo_detlist_get(detlist, i);
            double t = moo_pfits_get_exptime(det->primary_header);
            if (type > 0) {
                cpl_propertylist *extheader =
                    moo_det_get_single_header(det, type, num);
                double dit = moo_pfits_get_dit(extheader);
                double ndit = moo_pfits_get_ndit(extheader);
                t = dit * ndit;
            }
            cpl_vector_set(exptime, i, t);
        }

#if CPL_COEFF_FIT
        cpl_imagelist *coefs = NULL;
        cpl_mask_not(mask);
        for (int j = 1; j <= ny; j++) {
            for (int i = 1; i <= nx; i++) {
                int rej;
                int v = cpl_image_get(saturate, i, j, &rej);
                int kdx = v + 1;
                if (kdx < size) {
                    if (kdx < 3) {
                        kdx = 0;
                    }
                    for (int k = kdx; k < size; k++) {
                        cpl_image *im = cpl_imagelist_get(list, k);
                        cpl_image_reject(im, i, j);
                    }
                }
            }
        }

        for (int i = 0; i < size; i++) {
            cpl_image *im = cpl_imagelist_get(list, i);
            cpl_mask *m = cpl_image_get_bpm(im);
            cpl_mask_or(m, mask);
        }

        cpl_image *chi2 = NULL;
        coefs = cpl_fit_imagelist_polynomial(exptime, list, 0, degree,
                                             CPL_FALSE, CPL_TYPE_DOUBLE, chi2);

#if MOO_DEBUG_COMPUTE_LINEARITY
        char *chiname =
            cpl_sprintf("CHI2_%s_%d.fits", moo_detector_get_extname(type, num),
                        degree);
        cpl_image_save(chi2, chiname, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
        cpl_free(chiname);
#endif
        cpl_image_delete(chi2);

        for (int i = 0; i <= degree; i++) {
            cpl_image *coef = cpl_imagelist_get(coefs, i);
            cpl_imagelist_set(result, cpl_image_duplicate(coef), i);
        }
        cpl_imagelist_delete(coefs);
#else
        int ntime = hdrl_imagelist_get_size(list);
        cpl_image *c0_img = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        cpl_image *c1_img = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        cpl_image *c2_img = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        cpl_imagelist_set(result, c0_img, 0);
        cpl_imagelist_set(result, c1_img, 1);
        cpl_imagelist_set(result, c2_img, 2);
        for (int y = 1; y <= ny; y++) {
            for (int x = 1; x <= nx; x++) {
                for (int t = 0; t < ntime; t++) {
                    hdrl_image *himg = hdrl_imagelist_get(list, t);
                    int rej;
                    hdrl_value val = hdrl_image_get_pixel(himg, x, y, &rej);
                    cpl_vector_set(fluxes, t, val.data);
                }
                int illuminated = cpl_mask_get(mask, x, y);
                double c0 = NAN;
                double c1 = NAN;
                double c2 = NAN;

                if (illuminated == 1) {
                    const cpl_size coef0 = 0;
                    const cpl_size coef1 = 1;
                    cpl_size coef2 = 2;
                    cpl_polynomial *poly_linfit = cpl_polynomial_new(1);
                    cpl_matrix *samppos1d =
                        cpl_matrix_wrap(1, ntime, cpl_vector_get_data(exptime));
                    cpl_vector *fitvals = fluxes;
                    const cpl_size maxdeg1d = degree;
                    cpl_polynomial_fit(poly_linfit, samppos1d, NULL, fitvals,
                                       NULL, CPL_FALSE, NULL, &maxdeg1d);
                    c0 = cpl_polynomial_get_coeff(poly_linfit, &coef0);
                    c1 = cpl_polynomial_get_coeff(poly_linfit, &coef1);
                    c2 = cpl_polynomial_get_coeff(poly_linfit, &coef2);

                    cpl_matrix_unwrap(samppos1d);
                    cpl_polynomial_delete(poly_linfit);
                }
                cpl_image_set(c0_img, x, y, c0);
                cpl_image_set(c1_img, x, y, c1);
                cpl_image_set(c2_img, x, y, c2);
            }
        }
#endif
    }
    moo_detlist_free_single(detlist, type, num);
moo_try_cleanup:
    cpl_mask_delete(mask);
    cpl_imagelist_delete(list);
    cpl_vector_delete(exptime);
    cpl_vector_delete(fluxes);
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_imagelist_delete(result);
        result = NULL;
    }
    return result;
}
/******************************************************************************/
moo_cube *
moo_compute_linearity(moo_detlist *detlist,
                      moo_loc *loc,
                      moo_saturate_map *saturate_map,
                      const char *cube_name)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    moo_cube *result = NULL;

    cpl_ensure(detlist != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(cube_name != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(saturate_map != NULL, CPL_ERROR_NULL_INPUT, NULL);

    moo_fits_create(cube_name);

    moo_try_check(result = moo_cube_new(), " ");
    result->primary_header = cpl_propertylist_new();
    moo_try_check(result->filename = cpl_strdup(cube_name), " ");

    for (int num = 1; num <= 2; num++) {
        for (int type = 0; type < 3; type++) {
            cpl_imagelist *data = NULL;
            data = _moo_compute_linearity_single(
                detlist, loc, type, num,
                saturate_map->data[type + (num - 1) * 3], result);
            moo_cube_add_data(result, data, type, num);
        }
    }
moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_cube_delete(result);
        result = NULL;
    }
    return result;
}
/******************************************************************************/
cpl_image *
_moo_compute_bpm_linear_single(cpl_imagelist *coeffs,
                               moo_loc *loc,
                               moo_detector_type type,
                               int num,
                               double kappa,
                               cpl_image *index,
                               cpl_image *flux,
                               double min_snr)
{
    cpl_image *result = NULL;
    cpl_mask *cold_mask = NULL;
    cpl_mask *hot_mask = NULL;
    cpl_mask *trace_mask = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_ensure(coeffs != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(index != NULL, CPL_ERROR_NULL_INPUT, NULL);
    int nx = cpl_image_get_size_x(index);
    int ny = cpl_image_get_size_y(index);

    result = cpl_image_new(nx, ny, CPL_TYPE_INT);

    cold_mask = cpl_mask_new(nx, ny);
    hot_mask = cpl_mask_new(nx, ny);

    for (int j = 1; j <= ny; j++) {
        for (int i = 1; i <= nx; i++) {
            int rej;
            double vi = cpl_image_get(flux, i, j, &rej);
            if (isnan(vi)) {
                cpl_image_reject(flux, i, j);
            }
        }
    }

    if (type == MOO_TYPE_RI) {
        double medflux = cpl_image_get_median(flux);
        for (int j = 1; j <= ny; j++) {
            for (int i = 1; i <= nx; i++) {
                int rej;
                int vi = cpl_image_get(index, i, j, &rej);
                if (vi == -1) {
                    double vf = cpl_image_get(flux, i, j, &rej);
                    if (vf >= medflux) {
                        cpl_mask_set(hot_mask, i, j, CPL_BINARY_1);
                    }
                    else {
                        cpl_mask_set(cold_mask, i, j, CPL_BINARY_1);
                    }
                }
            }
        }
    }
    else {
        for (int j = 1; j <= ny; j++) {
            for (int i = 1; i <= nx; i++) {
                int rej;
                int vi = cpl_image_get(index, i, j, &rej);
                if (vi == -1) {
                    double vf = cpl_image_get(flux, i, j, &rej);
                    if (fabs(vf) <= DBL_EPSILON) {
                        cpl_mask_set(hot_mask, i, j, CPL_BINARY_1);
                        cpl_image_set(result, i, j, MOO_BADPIX_HOT);
                    }
                    else {
                        cpl_mask_set(cold_mask, i, j, CPL_BINARY_1);
                        cpl_image_set(result, i, j, MOO_BADPIX_COLD);
                    }
                }
            }
        }
    }

    trace_mask = cpl_mask_new(nx, ny);
    moo_try_check(_moo_compute_trace_mask(trace_mask, loc, type, num), " ");
    cpl_mask_not(trace_mask);

    cpl_msg_info(__func__,
                 "Compute bad pixels map for %s using kappa %f and min snr %f",
                 moo_detector_get_extname(type, num), kappa, min_snr);

    int size = cpl_imagelist_get_size(coeffs);

    cpl_msg_indent_more();
    for (int k = 1; k < size; k++) {
        cpl_image *im = cpl_imagelist_get(coeffs, k);
        cpl_mask *mask = cpl_image_get_bpm(im);
        cpl_mask_or(mask, trace_mask);
        cpl_mask_or(mask, hot_mask);
        cpl_mask_or(mask, cold_mask);

        double s;

        double median = cpl_image_get_mad(im, &s);
        s *= CPL_MATH_STD_MAD;
        cpl_msg_info(__func__, "coef%d find median %f sigma %f", k, median, s);
        for (int j = 1; j <= ny; j++) {
            for (int i = 1; i <= nx; i++) {
                int rej;
                double c = cpl_image_get(im, i, j, &rej);
                if (rej == 0 && (fabs(c - median) > kappa * s)) {
                    cpl_image_set(result, i, j, MOO_BADPIX_NON_LINEAR);
                }
            }
        }
    }
    cpl_msg_indent_less();
/*
    hdrl_image* im1 = hdrl_imagelist_get(list,0);
    for(int j=1;j<=ny;j++){
      for(int i=1;i<=nx;i++){
        hdrl_value v = hdrl_image_get_pixel(im1,i,j,NULL);
        if (isnan(v.data)){
          cpl_mask_set(mask,i,j,CPL_BINARY_1);
        }
      }
    }

    for(int i=0;i<size;i++){
      hdrl_image* im = hdrl_imagelist_get(list,i);
      cpl_mask_or(hdrl_image_get_mask(im),mask);
      cpl_mask_or(hdrl_image_get_mask(im),snr_mask);
    }
    hdrl_parameter *par = hdrl_bpm_fit_parameter_create_rel_coef(
      2,kappa,kappa);
    moo_hdrl_bpm_fit_compute(par, list, exptime, &result);
    hdrl_parameter_delete(par);

    for(int j=1;j<=ny;j++){
      for(int i=1;i<=nx;i++){
          cpl_binary v= cpl_mask_get(mask,i,j);
          int rej;
          int val = cpl_image_get(result,i,j,&rej);
          if (val < 2 || v == CPL_BINARY_1){
             cpl_image_set(result,i,j,0);
          }
          else{
            cpl_image_set(result,i,j,MOO_BADPIX_NON_LINEAR);
          }
        }
      }
      cpl_mask_delete(mask);
    }
  */
moo_try_cleanup:
    cpl_mask_delete(cold_mask);
    cpl_mask_delete(hot_mask);
    cpl_mask_delete(trace_mask);

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_image_delete(result);
        result = NULL;
    }
    return result;
}

moo_bpm *
moo_compute_bpm_linearity(moo_cube *cube,
                          moo_loc *loc,
                          moo_saturate_map *saturate_map,
                          moo_linear_params *params,
                          const char *bpm_name)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    moo_bpm *result = NULL;

    cpl_ensure(cube != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(saturate_map != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(bpm_name != NULL, CPL_ERROR_NULL_INPUT, NULL);
    moo_fits_create(bpm_name);

    moo_try_check(result = moo_bpm_new(), " ");
    result->primary_header = cpl_propertylist_new();
    moo_try_check(result->filename = cpl_strdup(bpm_name), " ");

    double *ktab = params->kappa;
    double *snrtab = params->min_snr;
    for (int num = 1; num <= 2; num++) {
        for (int type = 0; type < 3; type++) {
            int idx = type + (num - 1) * 3;
            double kappa = ktab[idx];
            double min_snr = snrtab[idx];
            cpl_image *index = saturate_map->data[idx];
            cpl_image *flux = saturate_map->flux[idx];
            cpl_imagelist *cdata = cube->data[idx];
            cpl_image *resdata = NULL;
            if (cdata != NULL) {
                resdata =
                    _moo_compute_bpm_linear_single(cdata, loc, type, num, kappa,
                                                   index, flux, min_snr);
            }
            moo_bpm_add_data(result, resdata, type, num);
        }
    }
moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_bpm_delete(result);
        result = NULL;
    }
    return result;
}
/******************************************************************************/

cpl_image *
_moo_compute_saturate_pixels_single(moo_detlist *detlist,
                                    moo_loc *loc,
                                    moo_detector_type type,
                                    int num,
                                    double threshold,
                                    cpl_image **flux_img,
                                    cpl_image **err_img,
                                    cpl_image **exptime_img)
{
    cpl_image *result = NULL;
    cpl_mask *mask = NULL;
    hdrl_imagelist *list = NULL;
    cpl_image *sub = NULL;
    cpl_vector *exptime = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(detlist != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(flux_img != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(err_img != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(exptime_img != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int badpix_level = MOO_BADPIX_GOOD;
    int size = moo_detlist_get_size(detlist);

    moo_try_check(moo_detlist_load_single(detlist, type, num, badpix_level),
                  " ");
    moo_try_check(list = moo_detlist_get_image(detlist, type, num), " ");

    if (list != NULL && hdrl_imagelist_get_size(list) > 0) {
        cpl_msg_info(__func__,
                     "Compute saturate pixels map for %s using threshold %f",
                     moo_detector_get_extname(type, num), threshold);

        int nx = hdrl_imagelist_get_size_x(list);
        int ny = hdrl_imagelist_get_size_y(list);

        result = cpl_image_new(nx, ny, CPL_TYPE_INT);
        *exptime_img = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        *flux_img = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        *err_img = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        mask = cpl_mask_new(nx, ny);
        _moo_compute_trace_mask(mask, loc, type, num);
        cpl_mask_not(mask);
        for (int j = 1; j <= ny; j++) {
            for (int i = 1; i <= nx; i++) {
                cpl_binary b = cpl_mask_get(mask, i, j);
                if (b == CPL_BINARY_0) {
                    // cpl_image_set(result,i,j,0);
                    cpl_image_set(result, i, j, size - 1);
                }
                else {
                    cpl_image_set(result, i, j, NAN);
                    cpl_image_reject(result, i, j);
                }
            }
        }

        for (int k = size - 1; k >= 0; k--) {
            cpl_image *im1 = hdrl_image_get_image(hdrl_imagelist_get(list, k));
            int nb_saturate = 0;

            for (int j = 1; j <= ny; j++) {
                for (int i = 1; i <= nx; i++) {
                    int rej;
                    double v = cpl_image_get(im1, i, j, &rej);
                    cpl_binary m = cpl_mask_get(mask, i, j);
                    if (m == CPL_BINARY_0) {
                        if (v <= threshold) {
                            cpl_mask_set(mask, i, j, CPL_BINARY_1);
                            cpl_image_set(result, i, j, k);
                        }
                        else {
                            nb_saturate++;
                        }
                    }
                }
            }
            if (nb_saturate == 0) {
                break;
            }
            else {
                cpl_msg_info(__func__, "Find at %d : %d saturate pixels", k,
                             nb_saturate);
            }
        }

        exptime = cpl_vector_new(size);

        for (int i = 0; i < size; i++) {
            moo_det *det = moo_detlist_get(detlist, i);
            double t = moo_pfits_get_exptime(det->primary_header);
            if (type > 0) {
                cpl_propertylist *extheader =
                    moo_det_get_single_header(det, type, num);
                double dit = moo_pfits_get_dit(extheader);
                double ndit = moo_pfits_get_ndit(extheader);
                t = dit * ndit;
            }
            cpl_vector_set(exptime, i, t);
        }

        for (int j = 1; j <= ny; j++) {
            for (int i = 1; i <= nx; i++) {
                int rej;

                int v = cpl_image_get(result, i, j, &rej);

                if (rej == 0) {
                    if (v == size - 1) {
                        cpl_image *im1 = hdrl_image_get_image(
                            hdrl_imagelist_get(list, size - 1));
                        cpl_image *im2 = hdrl_image_get_image(
                            hdrl_imagelist_get(list, size - 2));
                        cpl_image *im3 = hdrl_image_get_image(
                            hdrl_imagelist_get(list, size - 3));
                        double v1 = cpl_image_get(im1, i, j, &rej);
                        double v2 = cpl_image_get(im2, i, j, &rej);
                        double v3 = cpl_image_get(im3, i, j, &rej);
                        double t1 = cpl_vector_get(exptime, size - 1);
                        double t2 = cpl_vector_get(exptime, size - 2);
                        double t3 = cpl_vector_get(exptime, size - 3);
                        double f1 = (v1 - v2) / (t1 - t2);
                        double f2 = (v2 - v3) / (t2 - t3);

                        if (f1 < f2 * 0.5) {
                            v = size - 2;
                            cpl_image_set(result, i, j, v);
                        }
                    }
                    else if (v == -1) {
                        v = size - 1;
                    }
                    cpl_image *im1 =
                        hdrl_image_get_image(hdrl_imagelist_get(list, v));
                    cpl_image *err1 =
                        hdrl_image_get_error(hdrl_imagelist_get(list, v));
                    double f = cpl_image_get(im1, i, j, &rej);
                    double fe = cpl_image_get(err1, i, j, &rej);
                    double e = cpl_vector_get(exptime, v);
                    cpl_image_set(*flux_img, i, j, f);
                    cpl_image_set(*err_img, i, j, fe);
                    cpl_image_set(*exptime_img, i, j, e);
                }
                else {
                    cpl_image_set(*flux_img, i, j, NAN);
                    cpl_image_set(*err_img, i, j, NAN);
                    cpl_image_set(*exptime_img, i, j, NAN);
                }
            }
        }
    }
    moo_try_check(moo_detlist_free_single(detlist, type, num), " ");

moo_try_cleanup:
    hdrl_imagelist_unwrap(list);
    cpl_mask_delete(mask);
    cpl_vector_delete(exptime);
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_image_delete(sub);
        cpl_image_delete(*flux_img);
        cpl_image_delete(*err_img);
        cpl_image_delete(*exptime_img);
        cpl_image_delete(result);
        *flux_img = NULL;
        *err_img = NULL;
        *exptime_img = NULL;
        result = NULL;
    }
    return result;
}

moo_saturate_map *
moo_compute_saturate_pixels(moo_detlist *detlist,
                            moo_loc *loc,
                            moo_linear_params *params)
{
    moo_saturate_map *result = NULL;
    cpl_image *data = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(detlist != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);

    double threshold = params->saturate_threshold;

    result = moo_saturate_map_new();
    result->primary_header = cpl_propertylist_new();

    for (int num = 1; num <= 2; num++) {
        for (int type = 0; type < 3; type++) {
            cpl_image *flux = NULL;
            cpl_image *err = NULL;
            cpl_image *exptime = NULL;
            moo_try_check(data = _moo_compute_saturate_pixels_single(
                              detlist, loc, type, num, threshold, &flux, &err,
                              &exptime),
                          " ");
            cpl_propertylist *header = cpl_propertylist_new();
            moo_saturate_map_set_data(result, type, num, data, flux, err,
                                      exptime, header);
        }
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_saturate_map_delete(result);
        result = NULL;
    }
    return result;
}
/**@}*/
