/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>

#include "moo_correct_tell.h"
#include "moo_fibres_table.h"
#include "moo_fits.h"
#include "moo_pfits.h"
#include "moo_badpix.h"
#include "moo_utils.h"
#include "moo_qc.h"
#include "mf_calctrans.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

static cpl_table *
_molecfit_data_create(double crpix1,
                      double crval1,
                      double cd1_1,
                      cpl_image *data,
                      cpl_image *data_err,
                      int id,
                      cpl_image *tell,
                      int idtell)
{
    cpl_table *result = NULL;
    int nx = cpl_image_get_size_x(data);
    int tell_nx = cpl_image_get_size_x(tell);

    cpl_ensure(nx == tell_nx, CPL_ERROR_NULL_INPUT, NULL);

    result = cpl_table_new(nx);
    cpl_table_new_column(result, MOO_CORRECT_TELL_LAMBDA, CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, MOO_CORRECT_TELL_FLUX, CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, MOO_CORRECT_TELL_DFLUX, CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, MOO_CORRECT_TELL_MASK, CPL_TYPE_INT);
    cpl_table_new_column(result, MOO_CORRECT_TELL_CFLUX, CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, MOO_CORRECT_TELL_CDFLUX, CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, MOO_CORRECT_TELL_QUAL, CPL_TYPE_INT);
    cpl_table_new_column(result, MOO_CORRECT_TELL_MTRANS, CPL_TYPE_DOUBLE);

    for (int i = 0; i < nx; i++) {
        int rej1, rej2;
        int ipix = i + 1;
        double w = (ipix - crpix1) * cd1_1 + crval1;
        double f = cpl_image_get(data, ipix, id, &rej1);

        double e = cpl_image_get(data_err, ipix, id, &rej1);
        double t = cpl_image_get(tell, ipix, idtell, &rej2);
        cpl_table_set_double(result, MOO_CORRECT_TELL_LAMBDA, i, w);
        cpl_table_set_double(result, MOO_CORRECT_TELL_FLUX, i, f);
        cpl_table_set_double(result, MOO_CORRECT_TELL_DFLUX, i, e);
        cpl_table_set_int(result, MOO_CORRECT_TELL_MASK, i, rej1);
        cpl_table_set_double(result, MOO_CORRECT_TELL_CFLUX, i, f);
        cpl_table_set_double(result, MOO_CORRECT_TELL_CDFLUX, i, e);
        cpl_table_set_int(result, MOO_CORRECT_TELL_QUAL, i, rej1);
        cpl_table_set_double(result, MOO_CORRECT_TELL_MTRANS, i, t);
    }

    return result;
}

static cpl_error_code
_moo_update_primary_header(cpl_propertylist *primary_header)
{
    cpl_error_code status = CPL_ERROR_NONE;
    status = moo_qc_set_is_tellcor(primary_header, CPL_TRUE);

    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Apply the telluric correction to 1D rebinned spectra
  @param    rbn RBN specta  
  @param    tell TELLURIC correction
  @return   Flux calibrated frame

 * _Flags considered as bad :
 *
 * _Bad pixels flags_:

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_correct_tell(moo_rbn *rbn, moo_telluric *tell)
{
    cpl_ensure_code(rbn != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(tell != NULL, CPL_ERROR_NULL_INPUT);

    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_msg_info(__func__, "Correct telluric");

    int badpix_level = MOO_BADPIX_OUTSIDE_DATA_RANGE;
    cpl_table *fibre_table = moo_rbn_get_fibre_table(rbn);
    cpl_propertylist *primary_header = moo_rbn_get_primary_header(rbn);
    _moo_update_primary_header(primary_header);
    int nfibres = cpl_table_get_nrow(fibre_table);
    int *idrbnt =
        cpl_table_get_data_int(fibre_table, MOO_FIBRES_TABLE_INDEXRBN);
    const int *spectrot =
        cpl_table_get_data_int_const(fibre_table, MOO_FIBRES_TABLE_SPECTRO);

    const char *trans_colnames[] = { MOO_FIBRES_TABLE_TRANS_RI,
                                     MOO_FIBRES_TABLE_TRANS_YJ,
                                     MOO_FIBRES_TABLE_TRANS_H };

    for (int i = 0; i < 3; i++) {
        moo_rbn_single *rbn_single = moo_rbn_load_single(rbn, i, badpix_level);

        if (rbn_single != NULL) {
            const char *trans_colname = trans_colnames[i];
            const char *band = moo_telluric_get_band(i);
            float *transt =
                cpl_table_get_data_float(fibre_table, trans_colname);
            cpl_image *tell_img = moo_telluric_get_image(tell, i);
            int tnx = cpl_image_get_size_x(tell_img);

            hdrl_image *himg = moo_rbn_single_get_image(rbn_single);
            cpl_image *img = hdrl_image_get_image(himg);
            cpl_image *err = hdrl_image_get_error(himg);
            int nx = cpl_image_get_size_x(img);
            cpl_propertylist *header = moo_rbn_single_get_header(rbn_single);
            double cd1_1 = moo_pfits_get_cd1_1(header);
            double crpix1 = moo_pfits_get_crpix1(header);
            double crval1 = moo_pfits_get_crval1(header);

            moo_try_assure(tnx == nx, CPL_ERROR_ILLEGAL_INPUT,
                           "Telluric correction file must have the rebin "
                           "size than RBN");

            for (int f = 0; f < nfibres; f++) {
                int idrbn = idrbnt[f];
                double trans = transt[f];
                int spectro = spectrot[f];

                if (!isnan(trans) && trans > 0 && idrbn > 0) {
                    int tell_index =
                        moo_telluric_get_index(tell, idrbn, spectro, band);
                    if (tell_index > 0) {
                        cpl_table *mdata =
                            _molecfit_data_create(crpix1, crval1, cd1_1, img,
                                                  err, idrbn, tell_img,
                                                  tell_index);

                        mf_calctrans_correct_spectrum(
                            mdata, MF_PARAMETERS_TRANSMISSION_TRUE);
#if MOO_DEBUG_CORRECT_TELL
                        {
                            char *name = cpl_sprintf("test_%d.fits", idrbn);
                            cpl_table_save(mdata, NULL, NULL, name,
                                           CPL_IO_CREATE);
                            cpl_free(name);
                        }
#endif

                        for (int x = 0; x < nx; x++) {
                            double flux_v =
                                cpl_table_get_double(mdata,
                                                     MOO_CORRECT_TELL_CFLUX, x,
                                                     NULL);
                            double err_v =
                                cpl_table_get_double(mdata,
                                                     MOO_CORRECT_TELL_CDFLUX, x,
                                                     NULL);
                            cpl_image_set(img, x + 1, idrbn, flux_v);
                            cpl_image_set(err, x + 1, idrbn, err_v);
                        }

                        cpl_table_delete(mdata);
                    }
                    else {
                        cpl_msg_warning(
                            __func__,
                            "No telluric reference for band %s indexrbn %d",
                            rbn_single->extname, idrbn);
                    }
                }
            }
        }
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in correct telluric");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Apply the telluric correction to SCI
  @param    sci sky Input wavelength calibrated, sky corrected frame
  @param    tell telluric corerction
  @return   the error code

 * _Flags considered as bad :
 *
 * _Bad pixels flags_:

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_correct_tell_science(moo_sci *sci, moo_telluric *tell)
{
    cpl_ensure_code(sci != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(tell != NULL, CPL_ERROR_NULL_INPUT);

    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_msg_info(__func__, "Correct telluric");

    int badpix_level = MOO_BADPIX_OUTSIDE_DATA_RANGE;
    moo_target_table *target_table = moo_sci_get_target_table(sci);

    int nbtarget = cpl_table_get_nrow(target_table->table);

    int *idrbnt =
        cpl_table_get_data_int(target_table->table, MOO_TARGET_TABLE_INDEXRBN);
    int *idtargt =
        cpl_table_get_data_int(target_table->table, MOO_TARGET_TABLE_INDEXTARG);
    const int *spectrot =
        cpl_table_get_data_int_const(target_table->table,
                                     MOO_TARGET_TABLE_SPECTRO);
    const char *trans_colnames[] = { MOO_TARGET_TABLE_TRANSRI,
                                     MOO_TARGET_TABLE_TRANSYJ,
                                     MOO_TARGET_TABLE_TRANSH };

    for (int i = 0; i < 3; i++) {
        moo_sci_single *sci_single = moo_sci_load_single(sci, i, badpix_level);

        if (sci_single != NULL) {
            const char *trans_colname = trans_colnames[i];
            const char *band = moo_telluric_get_band(i);
            float *transt =
                cpl_table_get_data_float(target_table->table, trans_colname);
            cpl_image *tell_img = moo_telluric_get_image(tell, i);

            cpl_propertylist *header = moo_sci_single_get_header(sci_single);
            double cd1_1 = moo_pfits_get_cd1_1(header);
            double crpix1 = moo_pfits_get_crpix1(header);
            double crval1 = moo_pfits_get_crval1(header);
            hdrl_image *himg = moo_sci_single_get_image(sci_single);

            cpl_image *img = hdrl_image_get_image(himg);
            cpl_image *err = hdrl_image_get_error(himg);
            int nx = cpl_image_get_size_x(img);

            for (int f = 0; f < nbtarget; f++) {
                int idrbn = idrbnt[f];
                int idtarg = idtargt[f];
                double trans = transt[f];
                int spectro = spectrot[f];

                if (!isnan(trans) && trans > 0) {
                    int tell_index =
                        moo_telluric_get_index(tell, idrbn, spectro, band);
                    if (tell_index > 0) {
                        cpl_table *mdata =
                            _molecfit_data_create(crpix1, crval1, cd1_1, img,
                                                  err, idtarg, tell_img,
                                                  tell_index);

                        mf_calctrans_correct_spectrum(
                            mdata, MF_PARAMETERS_TRANSMISSION_TRUE);
#if MOO_DEBUG_CORRECT_TELL
                        {
                            char *name = cpl_sprintf("test_%d.fits", idrbn);
                            cpl_table_save(mdata, NULL, NULL, name,
                                           CPL_IO_CREATE);
                            cpl_free(name);
                        }
#endif

                        for (int x = 0; x < nx; x++) {
                            double flux_v =
                                cpl_table_get_double(mdata,
                                                     MOO_CORRECT_TELL_CFLUX, x,
                                                     NULL);
                            double err_v =
                                cpl_table_get_double(mdata,
                                                     MOO_CORRECT_TELL_CDFLUX, x,
                                                     NULL);
                            cpl_image_set(img, x + 1, idtarg, flux_v);
                            cpl_image_set(err, x + 1, idtarg, err_v);
                        }

                        cpl_table_delete(mdata);
                    }
                    else {
                        cpl_msg_info(__func__,
                                     "no telluric correction for %s idrbn %d",
                                     band, idrbn);
                    }
                }
            }
        }
    }

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in correct telluric");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
    }
    return CPL_ERROR_NONE;
}
/**@}*/
