/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>

#include "moo_create_s1d.h"
#include "moo_fits.h"
#include "moo_pfits.h"
#include "moo_badpix.h"
#include "moo_utils.h"
#include "moo_dfs.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 */

/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

static moo_s1d *
_moo_create_s1d_single(moo_sci_single *sci_single,
                       int indextarg,
                       const char *targname,
                       double talpha,
                       double tdelta,
                       double exptime,
                       cpl_propertylist *sci_header)
{
    cpl_propertylist *pheader = NULL;
    cpl_propertylist *header = NULL;
    cpl_table *data = NULL;
    cpl_image *iflux = NULL;
    cpl_image *ierr = NULL;
    cpl_image *isky = NULL;
    moo_s1d *result = NULL;

    cpl_ensure(sci_single != NULL, CPL_ERROR_NULL_INPUT, NULL);

    result = moo_s1d_new();

    char *filename = cpl_strdup(sci_single->filename);

    char *basename = strrchr(filename, '/');
    if (basename == NULL) {
        basename = filename;
    }
    else {
        basename = basename + 1;
    }
    int len = strlen(basename);
    basename[len - 5] = '\0';
    result->filename =
        cpl_sprintf("%s_%s_%s.fits", basename, sci_single->extname, targname);
    pheader = cpl_propertylist_duplicate(sci_header);
    moo_pfits_update_ra(pheader, talpha);
    moo_pfits_update_dec(pheader, tdelta);
    cpl_propertylist_set_double(pheader, MOO_PFITS_EXPTIME, exptime);

    result->primary_header = pheader;
    header = cpl_propertylist_duplicate(sci_single->header);
    result->header = header;
    double crpix1 = moo_pfits_get_crpix1(header);
    double crval1 = moo_pfits_get_crval1(header);
    double cd1_1 = moo_pfits_get_cd1_1(header);
    int naxis1 = moo_pfits_get_naxis1(header);
    double wmin = crval1 + (1 - crpix1) * cd1_1;
    double wmax = crval1 + (naxis1 - crpix1) * cd1_1;
    cpl_propertylist_set_string(header, MOO_PFITS_EXTNAME,
                                MOO_S1D_SPETCRUM_EXTNAME);
    cpl_propertylist_append_string(header, "VOCLASS", "SPECTRUM v2.0");
    cpl_propertylist_append_double(header, "TDMIN", wmin);
    cpl_propertylist_append_double(header, "TDMAX", wmax);
    cpl_propertylist_append_string(header, "TUTYP1",
                                   "spec:Data.SpectralAxis.Value");
    cpl_propertylist_append_string(header, "TUCD1", "em.wl;obs.atmos");
    cpl_propertylist_append_string(header, "TUTYP2",
                                   "spec:Data.FluxAxis.Value");
    cpl_propertylist_append_string(header, "TUCD2",
                                   "phot.flux.density;em.wl;src.net;meta.main");
    cpl_propertylist_append_string(header, "TUTYP3",
                                   "Spectrum.Data.FluxAxis.Accuracy.StatError");
    cpl_propertylist_append_string(header, "TUCD3",
                                   "phot.flux.density;em.wl;src.net;meta.main");
    cpl_propertylist_append_string(header, "TUTYP4",
                                   "meta.code.qual;meta.main");
    cpl_propertylist_append_string(header, "TUCD4",
                                   "spec:Data.FluxAxis.Accuracy.QualityStatus");
    cpl_propertylist_append_string(header, "TUTYP5",
                                   "spec:Data.FluxAxis.Value");
    cpl_propertylist_append_string(header, "TUCD5",
                                   "phot.flux.density;em.wl;src.net;meta.main");
    data = cpl_table_new(naxis1);
    result->data = data;
    moo_try_check(cpl_table_new_column(data, MOO_S1D_WAVE, CPL_TYPE_DOUBLE),
                  " ");
    moo_try_check(cpl_table_new_column(data, MOO_S1D_FLUX, CPL_TYPE_DOUBLE),
                  " ");
    moo_try_check(cpl_table_new_column(data, MOO_S1D_ERR, CPL_TYPE_DOUBLE),
                  " ");
    moo_try_check(cpl_table_new_column(data, MOO_S1D_QUAL, CPL_TYPE_INT), " ");
    hdrl_image *himg = moo_sci_single_get_image(sci_single);
    moo_try_check(cpl_table_new_column(data, MOO_S1D_BGFLUX, CPL_TYPE_DOUBLE),
                  " ");

    iflux = hdrl_image_get_image(himg);
    ierr = hdrl_image_get_error(himg);
    isky = moo_sci_single_get_sky(sci_single);
    cpl_image *iqual = moo_sci_single_get_qual(sci_single);

    for (int i = 1; i <= naxis1; i++) {
        int rej;
        double flux = cpl_image_get(iflux, i, indextarg, &rej);
        double err = cpl_image_get(ierr, i, indextarg, &rej);
        int qual = (int)cpl_image_get(iqual, i, indextarg, &rej);
        double sky = cpl_image_get(isky, i, indextarg, &rej);
        cpl_table_set_double(data, MOO_S1D_WAVE, i - 1,
                             crval1 + (i - crpix1) * cd1_1);
        cpl_table_set_double(data, MOO_S1D_FLUX, i - 1, flux);
        cpl_table_set_double(data, MOO_S1D_ERR, i - 1, err);
        cpl_table_set_int(data, MOO_S1D_QUAL, i - 1, qual);
        cpl_table_set_double(data, MOO_S1D_BGFLUX, i - 1, sky);
    }

moo_try_cleanup:
    cpl_free(filename);
    return result;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Apply the telluric correction to SCI
  @param    sci_frame Input wavelength calibrated, sky corrected frame
  @param    params Parameters od create_s1d
  @param    tag The S1D associated tag
  @param    products the saving products structure
  @return   the error code

 * _Flags considered as bad :
 *
 * _Bad pixels flags_:

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_create_s1d(cpl_frame *sci_frame,
               moo_create_s1d_params *params,
               const char *tag,
               moo_products *products)
{
    moo_sci *sci = NULL;
    cpl_ensure_code(sci_frame != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(params != NULL, CPL_ERROR_NULL_INPUT);

    sci = moo_sci_create(sci_frame);

    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_msg_info(__func__, "Create S1D");

    if (params->do_s1d) {
        moo_target_table *ttable = moo_sci_get_target_table(sci);
        cpl_table *table = ttable->table;
        int nrow = cpl_table_get_nrow(table);

        for (int t = 0; t < nrow; t++) {
            int indextarg =
                cpl_table_get_int(table, MOO_TARGET_TABLE_INDEXTARG, t, NULL);
            const char *targname =
                cpl_table_get_string(table, MOO_TARGET_TABLE_TARGNAME, t);
            double talpha =
                cpl_table_get_double(table, MOO_TARGET_TABLE_TARGALPHA, t,
                                     NULL);
            double tdelta =
                cpl_table_get_double(table, MOO_TARGET_TABLE_TARGDELTA, t,
                                     NULL);
            for (int j = 0; j < 3; j++) {
                moo_sci_single *sci_single = moo_sci_get_single(sci, j);
                double exptime = moo_target_table_get_exptime(ttable, t, j);
                if (sci_single != NULL) {
                    moo_s1d *s1d = NULL;
                    moo_sci_load_single(sci, j, MOO_BADPIX_GOOD);

                    s1d = _moo_create_s1d_single(sci_single, indextarg,
                                                 targname, talpha, tdelta,
                                                 exptime, sci->primary_header);

                    if (products != NULL) {
                        moo_products_add_s1d(products, s1d, tag, sci_frame);
                    }
                    else {
                        moo_s1d_save(s1d);
                    }
                    moo_s1d_delete(s1d);
                }
            }
        }
    }

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in create_s1d");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
    }
    moo_sci_delete(sci);
    return CPL_ERROR_NONE;
}
/**@}*/
