/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_qc.h"
#include "moo_cube.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_cube CUBE format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_cube
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_cube.h"
 * @endcode
 */


/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_cube
  @return   1 newly allocated moo_cube or NULL in case of an error

  The returned object must be deallocated using moo_cube_delete().

 */
/*----------------------------------------------------------------------------*/
moo_cube *
moo_cube_new(void)
{
    moo_cube *res = cpl_calloc(1, sizeof(moo_cube));
    return res;
}


moo_cube *
moo_cube_load(const cpl_frame *frame)
{
    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    const char *filename = cpl_frame_get_filename(frame);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prev_state = cpl_errorstate_get();
    moo_cube *res = moo_cube_new();

    for (int i = 0; i < 2; i++) {
        for (int j = 0; j < 3; j++) {
            const char *extname = moo_detector_get_extname(j, i + 1);
            cpl_propertylist *header = NULL;
            header = moo_fits_load_extension_header(filename, NULL, extname);
            res->data_header[i * 3 + j] = header;
            if (header != NULL) {
                int naxis = moo_pfits_get_naxis(header);
                if (naxis == 3) {
                    res->data[i * 3 + j] =
                        moo_fits_load_extension_cube(filename, NULL, extname,
                                                     CPL_TYPE_FLOAT);
                }
            }
        }
    }
    if (!cpl_errorstate_is_equal(prev_state)) {
        moo_cube_delete(res);
        res = NULL;
    }

    return res;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    set cube data for relevant extension
  @param self
  @param type the detector type
  @param ntas number of TAS
  @param data the cube data
  @param header the cube header
  @return   error code

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_cube_set_data(moo_cube *self,
                  moo_detector_type type,
                  int ntas,
                  cpl_imagelist *data,
                  cpl_propertylist *header)
{
    cpl_ensure_code(ntas >= 1 && ntas <= 2, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(header != NULL, CPL_ERROR_NULL_INPUT);

    self->data[(ntas - 1) * 3 + type] = data;
    self->data_header[(ntas - 1) * 3 + type] = header;

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief Add CPL_IMAGELIST extension to CUBE filename and update moo_cube structure
  @param self moo_cube structure to update
  @param list CPL_IMAGELIST extension to store
  @param type type of detector
  @param ntas number of spectrographs

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_cube_add_data(moo_cube *self,
                  cpl_imagelist *list,
                  moo_detector_type type,
                  int ntas)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();
    const char *extname = moo_detector_get_extname(type, ntas);
    cpl_propertylist *header = cpl_propertylist_new();
    cpl_propertylist_append_string(header, MOO_PFITS_EXTNAME, extname);

    if (list != NULL) {
        cpl_imagelist_save(list, self->filename, CPL_TYPE_FLOAT, header,
                           CPL_IO_EXTEND);
    }
    else {
        cpl_propertylist_save(header, self->filename, CPL_IO_EXTEND);
    }
    moo_cube_set_data(self, type, ntas, list, header);

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_error("moo_cube", "Error for adding data to %s %s (%s)",
                      self->filename, extname,
                      cpl_error_get_message_default(cpl_error_get_code()));
        cpl_msg_info("test", "test debug cube");
        status = cpl_error_get_code();
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_cube to a FITS file
  @param    self moo_cube to write to disk or NULL
  @param    filename Name of the file to write
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_cube to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
void
moo_cube_save(moo_cube *self, const char *filename)
{
    if (self != NULL) {
        cpl_propertylist_save(self->primary_header, filename, CPL_IO_CREATE);
        for (int i = 0; i < 2; i++) {
            for (int j = 0; j < 3; j++) {
                const char *extname = moo_detector_get_extname(j, i + 1);
                moo_fits_write_extension_cube(self->data[i * 3 + j], filename,
                                              NULL, extname, CPL_TYPE_FLOAT,
                                              self->data_header[i * 3 + j]);
            }
        }
    }
}

/*----------------------------------------------------------------------------*/
static cpl_error_code
_moo_coefs_normalise(cpl_imagelist *coefs, cpl_image *flux, cpl_image *exptime)
{
    if (coefs != NULL && flux != NULL && exptime != NULL) {
        int size = cpl_imagelist_get_size(coefs);

        for (int k = 0; k < size; k++) {
            cpl_image *coef = cpl_imagelist_get(coefs, k);
            cpl_image_divide(coef, flux);
        }


        for (int k = 1; k < size; k++) {
            cpl_image *coef = cpl_imagelist_get(coefs, k);
            cpl_image *power = cpl_image_power_create(exptime, k);
            cpl_image_multiply(coef, power);
            cpl_image_delete(power);
        }
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Normalise a moo_cube using saturate map
  @param    self moo_cube to normalise
  @param    map SATURATE MAP to use
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

*/
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_cube_normalise(moo_cube *self, moo_saturate_map *map)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(map != NULL, CPL_ERROR_NULL_INPUT);
    for (int i = 0; i < 6; i++) {
        _moo_coefs_normalise(self->data[i], map->flux[i], map->exptime[i]);
    }
    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
static cpl_imagelist *
_moo_cube_merge(cpl_imagelist *coefsa,
                cpl_image *fluxa,
                cpl_imagelist *coefsb,
                cpl_image *fluxb)
{
    cpl_imagelist *result = NULL;

    if (coefsa != NULL && coefsb != NULL) {
        int size = cpl_imagelist_get_size(coefsa);
        result = cpl_imagelist_duplicate(coefsa);
        cpl_image *coefa = cpl_imagelist_get(coefsa, 0);
        cpl_image *coefb = cpl_imagelist_get(coefsb, 0);
        int nx = cpl_image_get_size_x(coefa);
        int ny = cpl_image_get_size_y(coefa);
        for (int j = 1; j <= ny; j++) {
            for (int i = 1; i <= nx; i++) {
                int reja = cpl_image_is_rejected(coefa, i, j);
                int rejb = cpl_image_is_rejected(coefb, i, j);
                if (reja) {
                    if (!rejb) {
                        for (int k = 0; k < size; k++) {
                            int rej;
                            cpl_image *ib = cpl_imagelist_get(coefsb, k);
                            cpl_image *rb = cpl_imagelist_get(result, k);
                            double v = cpl_image_get(ib, i, j, &rej);
                            cpl_image_set(rb, i, j, v);
                        }
                    }
                }
                else {
                    if (!rejb) {
                        int rej;
                        double fa = cpl_image_get(fluxa, i, j, &rej);
                        double fb = cpl_image_get(fluxb, i, j, &rej);
                        if (fa < fb) {
                            for (int k = 0; k < size; k++) {
                                cpl_image *ib = cpl_imagelist_get(coefsb, k);
                                cpl_image *rb = cpl_imagelist_get(result, k);
                                double v = cpl_image_get(ib, i, j, &rej);
                                cpl_image_set(rb, i, j, v);
                            }
                        }
                    }
                }
            }
        }
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Merging two cubes
  @param    cubea moo_cube to merge
  @param    saturatea SATURATE MAP from cubea
  @param    cubeb moo_cube to merge
  @param    saturateb SATURATE MAP from cubeb
  @return   the merging cube or NULL

*/
/*----------------------------------------------------------------------------*/
moo_cube *
moo_cube_merge(moo_cube *cubea,
               moo_saturate_map *saturatea,
               moo_cube *cubeb,
               moo_saturate_map *saturateb)
{
    moo_cube *result = NULL;
    cpl_ensure(cubea != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(saturatea != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(cubeb != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(saturateb != NULL, CPL_ERROR_NULL_INPUT, NULL);

    result = moo_cube_new();
    result->primary_header = cpl_propertylist_duplicate(cubea->primary_header);
    for (int i = 0; i < 6; i++) {
        result->data[i] = _moo_cube_merge(cubea->data[i], saturatea->flux[i],
                                          cubeb->data[i], saturateb->flux[i]);
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_cube
  @param    self    moo_cube to delete
  @return   void

  If the moo_map @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_cube_delete(moo_cube *self)
{
    if (self != NULL) {
        int i;
        if (self->primary_header != NULL) {
            cpl_propertylist_delete(self->primary_header);
        }
        for (i = 0; i < 6; i++) {
            if (self->data[i] != NULL) {
                cpl_imagelist_delete(self->data[i]);
            }
            if (self->data_header[i] != NULL) {
                cpl_propertylist_delete(self->data_header[i]);
            }
        }
        cpl_free(self->filename);
        cpl_free(self);
    }
}
/**@}*/
