/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "moo_detector.h"
#include "moo_pfits.h"
#include "moo_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_detector MOONS detector global definitions
 * @brief This module provides functions to manipulate detectors
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_detectors.h"
 * @endcode
 */

/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the ERROR extension name of a detector
  @param type the type of detector
  @param ntas number of spectrograph [1,2]

  @return name of detector

   - - -
_Error code_:
  - CPL_ERROR_ILLEGAL_INPUT if ntas >=2
 */
/*----------------------------------------------------------------------------*/
const char *
moo_detector_get_err_extname(moo_detector_type type, int ntas)
{
    cpl_ensure(ntas <= 2, CPL_ERROR_ILLEGAL_INPUT, NULL);
    if (ntas >= 1) {
        const char *res[] = { MOO_EXTNAME(MOO_ERR, MOO_DETECTOR_RI1_NAME),
                              MOO_EXTNAME(MOO_ERR, MOO_DETECTOR_RI2_NAME),
                              MOO_EXTNAME(MOO_ERR, MOO_DETECTOR_YJ1_NAME),
                              MOO_EXTNAME(MOO_ERR, MOO_DETECTOR_YJ2_NAME),
                              MOO_EXTNAME(MOO_ERR, MOO_DETECTOR_H1_NAME),
                              MOO_EXTNAME(MOO_ERR, MOO_DETECTOR_H2_NAME) };
        int idx = type * 2 + ntas - 1;
        return res[idx];
    }
    else {
        const char *res[] = { MOO_EXTNAME(MOO_ERR, MOO_CHANNEL_RI_NAME),
                              MOO_EXTNAME(MOO_ERR, MOO_CHANNEL_YJ_NAME),
                              MOO_EXTNAME(MOO_ERR, MOO_CHANNEL_H_NAME) };
        return res[type];
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the QUAL extension name of a detector
  @param type the type of detector
  @param ntas number of spectrograph [1,2]

  @return name of detector

   - - -
 _Error code_:
  - CPL_ERROR_ILLEGAL_INPUT if ntas >2
 */
/*----------------------------------------------------------------------------*/
const char *
moo_detector_get_qual_extname(moo_detector_type type, int ntas)
{
    cpl_ensure(ntas <= 2, CPL_ERROR_ILLEGAL_INPUT, NULL);

    if (ntas >= 1) {
        const char *res[] = { MOO_EXTNAME(MOO_QUAL, MOO_DETECTOR_RI1_NAME),
                              MOO_EXTNAME(MOO_QUAL, MOO_DETECTOR_RI2_NAME),
                              MOO_EXTNAME(MOO_QUAL, MOO_DETECTOR_YJ1_NAME),
                              MOO_EXTNAME(MOO_QUAL, MOO_DETECTOR_YJ2_NAME),
                              MOO_EXTNAME(MOO_QUAL, MOO_DETECTOR_H1_NAME),
                              MOO_EXTNAME(MOO_QUAL, MOO_DETECTOR_H2_NAME) };
        int idx = type * 2 + ntas - 1;
        return res[idx];
    }
    else {
        const char *res[] = { MOO_EXTNAME(MOO_QUAL, MOO_CHANNEL_RI_NAME),
                              MOO_EXTNAME(MOO_QUAL, MOO_CHANNEL_YJ_NAME),
                              MOO_EXTNAME(MOO_QUAL, MOO_CHANNEL_H_NAME) };
        return res[type];
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the extension name of a detector
  @param type the type of detector
  @param ntas number of spectrograph [1,2]

  @return name of detector

   - - -
 _Error code_:
  - CPL_ERROR_ILLEGAL_INPUT if ntas >2
 */
/*----------------------------------------------------------------------------*/
const char *
moo_detector_get_extname(moo_detector_type type, int ntas)
{
    cpl_ensure(ntas <= 2, CPL_ERROR_ILLEGAL_INPUT, NULL);
    if (ntas >= 1) {
        const char *res[] = { MOO_DETECTOR_RI1_NAME, MOO_DETECTOR_RI2_NAME,
                              MOO_DETECTOR_YJ1_NAME, MOO_DETECTOR_YJ2_NAME,
                              MOO_DETECTOR_H1_NAME,  MOO_DETECTOR_H2_NAME };
        int idx = type * 2 + ntas - 1;
        return res[idx];
    }
    else {
        return moo_detector_get_name(type);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the spctro name for a ntas
  @param ntas the ntas of detector

  @return spectro name

 - - -
 _Error code_:
  - None
 */
/*----------------------------------------------------------------------------*/
int
moo_detector_get_spectro(int ntas)
{
    return ntas;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the extension name of a detector
  @param type the type of detector

  @return type of detector

 - - -
 _Error code_:
  - None
 */
/*----------------------------------------------------------------------------*/
const char *
moo_detector_get_name(moo_detector_type type)
{
    const char *res[] = { MOO_CHANNEL_RI_NAME, MOO_CHANNEL_YJ_NAME,
                          MOO_CHANNEL_H_NAME };
    int idx = type;
    return res[idx];
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the name of a mode
  @param type the mode

  @return mode of detector

 - - -
 _Error code_:
  - None
 */
/*----------------------------------------------------------------------------*/
const char *
moo_mode_get_name(moo_mode_type type)
{
    const char *res[] = { MOO_MODE_LR_NAME, MOO_MODE_HR_NAME };
    int idx = type;
    return res[idx];
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the name of a mode from a frame
  @param frame the frame where to read the mode

  @return mode of detector

 - - -
 _Error code_:
  - None
 */
/*----------------------------------------------------------------------------*/
moo_mode_type
moo_mode_get(const cpl_frame *frame)
{
    moo_mode_type res = MOO_MODE_LR;
    cpl_propertylist *plist = NULL;

    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, res);

    const char *filename = cpl_frame_get_filename(frame);

    moo_try_check(plist = cpl_propertylist_load(filename, 0), " ");
    moo_try_check(res = moo_pfits_get_mode(plist), " ");

moo_try_cleanup:
    cpl_propertylist_delete(plist);
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the offset from a frame
  @param refframe the reference frame where to read the offset

  @return offset of detector

 - - -
 _Error code_:
  - -1 in case of error
 */
/*----------------------------------------------------------------------------*/
int
moo_offset_get(const cpl_frame *refframe)
{
    int offset = -1;
    cpl_propertylist *plist = NULL;

    cpl_ensure(refframe != NULL, CPL_ERROR_NULL_INPUT, offset);

    const char *filename = cpl_frame_get_filename(refframe);

    moo_try_check(plist = cpl_propertylist_load(filename, 0), " ");
    moo_try_check(offset = moo_pfits_get_slit_offset(plist), " ");

moo_try_cleanup:
    cpl_propertylist_delete(plist);
    return offset;
}
/**@}*/
