/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "cpl_errorstate.h"
#include "cpl_frame.h"
#include "cpl_msg.h"
#include "cpl_propertylist.h"
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "cpl_error.h"
#include "cpl_frameset.h"

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <string.h>
#include <math.h>

#include <cpl.h>

#include "moo_pfits.h"
#include "moo_dfs.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_dfs  DFS related functions
 *
 * TBD
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the EXPTIME from the frameset
  @param    set the input frameset
  @return   the extracted exptime

  - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
cpl_frameset **
moo_dfs_split_by_offset(cpl_frameset *set)
{
    cpl_frameset **res;
    cpl_ensure(set != NULL, CPL_ERROR_NULL_INPUT, NULL);

    res = cpl_calloc(sizeof(cpl_frameset *), 2);
    res[0] = cpl_frameset_new();
    res[1] = cpl_frameset_new();

    int i;

    for (i = 0; i < cpl_frameset_get_size(set); ++i) {
        const cpl_frame *frame = cpl_frameset_get_position_const(set, i);
        const char *filename = cpl_frame_get_filename(frame);
        cpl_propertylist *list = cpl_propertylist_load(filename, 0);

        double offset = moo_pfits_get_slit_offset(list);

        if (fabs(offset - MOONS_SLIT_OFFSET_POS1) <=
            MOONS_SLIT_OFFSET_EPSILON) {
            cpl_frameset_insert(res[0], cpl_frame_duplicate(frame));
        }
        else if (fabs(offset - MOONS_SLIT_OFFSET_POS2) <=
                 MOONS_SLIT_OFFSET_EPSILON) {
            cpl_frameset_insert(res[1], cpl_frame_duplicate(frame));
        }
        cpl_propertylist_delete(list);
    }

    return res;
}

cpl_error_code
moo_dfs_group_offsets(const cpl_frameset *set,
                      cpl_frameset *group1,
                      cpl_frameset *group2,
                      cpl_frameset *other)
{
    cpl_ensure(set != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
    cpl_ensure(group1 != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
    cpl_ensure(group2 != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

    cpl_errorstate status = cpl_errorstate_get();
    cpl_frameset_iterator *it = cpl_frameset_iterator_new(set);
    const cpl_frame *frame = NULL;
    while ((frame = cpl_frameset_iterator_get_const(it)) != NULL) {
        const char *filename = cpl_frame_get_filename(frame);

        // FIXME: May need error handling for the next operation too.
        const cpl_propertylist *properties = cpl_propertylist_load(filename, 0);

        // Check slit position keyword in the frame header. If the keyword does not exist, i.e. there
        // is not slit position for the current frame an error is set. Clear it and put the frame into
        // the third group.
        cpl_errorstate _status = cpl_errorstate_get();
        double offset = moo_pfits_get_slit_offset(properties);
        if (!cpl_errorstate_is_equal(_status) &&
            (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND)) {
            cpl_errorstate_set(_status);
        }

        if (fabs(offset - MOONS_SLIT_OFFSET_POS1) <=
            MOONS_SLIT_OFFSET_EPSILON) {
            cpl_frameset_insert(group1, cpl_frame_duplicate(frame));
        }
        else if (fabs(offset - MOONS_SLIT_OFFSET_POS2) <=
                 MOONS_SLIT_OFFSET_EPSILON) {
            cpl_frameset_insert(group2, cpl_frame_duplicate(frame));
        }
        else if (other != NULL) {
            cpl_frameset_insert(other, cpl_frame_duplicate(frame));
        }
        else {
            // If no frameset is provided for collecting frames without slit position
            // information, put it in both output framesets. This makes each of
            // the a complete data set for each slit position
            cpl_frameset_insert(group1, cpl_frame_duplicate(frame));
            cpl_frameset_insert(group2, cpl_frame_duplicate(frame));
        }
        cpl_propertylist_delete((cpl_propertylist *)properties);

        _status = cpl_errorstate_get();
        cpl_frameset_iterator_advance(it, 1);
        if (!cpl_errorstate_is_equal(_status) &&
            (cpl_error_get_code() == CPL_ERROR_ACCESS_OUT_OF_RANGE)) {
            cpl_errorstate_set(_status);
        }
    }
    cpl_frameset_iterator_delete(it);

    if (!cpl_errorstate_is_equal(status)) {
        return cpl_error_get_code();
    }
    return CPL_ERROR_NONE;
}

cpl_frameset *
moo_dfs_extract_tag(const cpl_frameset *set, const char *tag)
{
    cpl_ensure(set != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(tag != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frameset *selected = cpl_frameset_new();

    cpl_errorstate status = cpl_errorstate_get();
    cpl_frameset_iterator *it = cpl_frameset_iterator_new(set);
    const cpl_frame *frame = NULL;
    while ((frame = cpl_frameset_iterator_get_const(it)) != NULL) {
        if (strcmp(tag, cpl_frame_get_tag(frame)) == 0) {
            cpl_frameset_insert(selected, cpl_frame_duplicate(frame));
        }
        cpl_errorstate _status = cpl_errorstate_get();
        cpl_frameset_iterator_advance(it, 1);
        if (!cpl_errorstate_is_equal(_status) &&
            (cpl_error_get_code() == CPL_ERROR_ACCESS_OUT_OF_RANGE)) {
            cpl_errorstate_set(_status);
        }
    }
    cpl_frameset_iterator_delete(it);

    if (!cpl_errorstate_is_equal(status)) {
        cpl_frameset_delete(selected);
        return NULL;
    }
    return selected;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the group as RAW or CALIB in a frameset
  @param    set     the input frameset
  @return   error_code

  - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer
*/
/*----------------------------------------------------------------------------*/

cpl_error_code
moo_dfs_set_groups(cpl_frameset *set)
{
    cpl_errorstate prestate = cpl_errorstate_get();


    /* Loop on frames */
    cpl_frameset_iterator *it = cpl_frameset_iterator_new(set);

    int i = 0;
    cpl_frame *frame = NULL;
    while ((frame = cpl_frameset_iterator_get(it)) != NULL) {
        const char *tag = cpl_frame_get_tag(frame);

        if (tag == NULL) {
            cpl_msg_warning(cpl_func, "Frame %d has no tag", i);
        }
        else if (!strcmp(tag, MOONS_TAG_BIAS) ||
                 !strcmp(tag, MOONS_TAG_LINEARITY) ||
                 !strcmp(tag, MOONS_TAG_LINEARITY_OFF) ||
                 !strcmp(tag, MOONS_TAG_OBJECT) ||
                 !strcmp(tag, MOONS_TAG_DARK_VIS) ||
                 !strcmp(tag, MOONS_TAG_DARK_NIR) ||
                 !strcmp(tag, MOONS_TAG_FLAT) ||
                 !strcmp(tag, MOONS_TAG_FLAT_ATTACHED) ||
                 !strcmp(tag, MOONS_TAG_FLAT_OFF) ||
                 !strcmp(tag, MOONS_TAG_ARC) ||
                 !strcmp(tag, MOONS_TAG_ARC_OFF) ||
                 !strcmp(tag, MOONS_TAG_STD_FLUX) ||
                 !strcmp(tag, MOONS_TAG_STD_TELL) ||
                 !strcmp(tag, MOONS_TAG_OBJECT_STARE) ||
                 !strcmp(tag, MOONS_TAG_OBJECT_STARENOD) ||
                 !strcmp(tag, MOONS_TAG_SKY_STARENOD) ||
                 !strcmp(tag, MOONS_TAG_OBJECT_XSWITCH) ||
                 !strcmp(tag, MOONS_TAG_MOLECFIT_SCI)) {
            /* RAW frames */
            cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
        }
        else if (!strcmp(tag, MOONS_TAG_LINE_CATALOG) ||
                 !strcmp(tag, MOONS_TAG_FLUX_STD_CATALOG) ||
                 !strcmp(tag, MOONS_TAG_ATMOS_EXT) ||
                 !strcmp(tag, MOONS_TAG_ARC_LINE_LIST) ||
                 !strcmp(tag, MOONS_TAG_SKY_LINE_LIST) ||
                 !strcmp(tag, MOONS_TAG_MASTER_BIAS) ||
                 !strcmp(tag, MOONS_TAG_MASTER_DARK_VIS) ||
                 !strcmp(tag, MOONS_TAG_MASTER_DARK_NIR) ||
                 !strcmp(tag, MOONS_TAG_BP_MAP_RP) ||
                 !strcmp(tag, MOONS_TAG_BP_MAP_NL) ||
                 !strcmp(tag, MOONS_TAG_FF_TRACE_GUESS) ||
                 !strcmp(tag, MOONS_TAG_FF_TRACE) ||
                 !strcmp(tag, MOONS_TAG_P2P_MAP) ||
                 !strcmp(tag, MOONS_TAG_SPECTRAL_FORMAT) ||
                 !strcmp(tag, MOONS_TAG_LAYOUT) ||
                 !strcmp(tag, MOONS_TAG_WAVEMAP_GUESS) ||
                 !strcmp(tag, MOONS_TAG_WAVEMAP) ||
                 !strcmp(tag, MOONS_TAG_F2F_TABLE) ||
                 !strcmp(tag, MOONS_TAG_FF_EXTSPECTRA) ||
                 !strcmp(tag, MOONS_TAG_LINEARITY_COEFF_CUBE) ||
                 !strcmp(tag, MOONS_TAG_TELLURIC_CORR) ||
                 !strcmp(tag, MOONS_TAG_RESPONSE) ||
                 !strcmp(tag, MOONS_TAG_MOLECFIT_MOLECULES) ||
                 !strcmp(tag, MOONS_TAG_MOLECFIT_ATM_PARAMS) ||
                 !strcmp(tag, MOONS_TAG_MOLECFIT_BEST_FIT_PARAMS) ||
                 !strcmp(tag, MOONS_TAG_MOLECFIT_KERNEL_LIBRARY) ||
                 !strcmp(tag, MOONS_TAG_MOLECFIT_WINCLUDE)) {
            /* CALIB frames */
            cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
        }

        cpl_frameset_iterator_advance(it, 1);
        ++i;
    }
    cpl_frameset_iterator_delete(it);

    if (!cpl_errorstate_is_equal(prestate)) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "Could not identify RAW and CALIB "
                                     "frames");
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the EXPTIME from the frameset
  @param    set the input frameset
  @return   the extracted exptime
 *
 *  - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer
 */
/*----------------------------------------------------------------------------*/
double *
moo_dfs_get_exptime(cpl_frameset *set)
{
    double *res = NULL;
    cpl_ensure(set != NULL, CPL_ERROR_NULL_INPUT, NULL);
    int size = cpl_frameset_get_size(set);
    res = cpl_malloc(sizeof(double) * size);

    int i;

    for (i = 0; i < size; i++) {
        const cpl_frame *frame = cpl_frameset_get_position_const(set, i);
        const char *filename = cpl_frame_get_filename(frame);
        cpl_propertylist *list = cpl_propertylist_load(filename, 0);
        res[i] = moo_pfits_get_exptime(list);
        cpl_propertylist_delete(list);
    }

    return res;
}


/**@}*/
