/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include "moo_pfits.h"
#include "moo_ext_single.h"
#include "moo_fits.h"
#include "moo_qc.h"
#include "moo_badpix.h"
#include "moo_spectral_format.h"
#include "moo_fibres_table.h"
#include "moo_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_ext_single EXT_SINGLE format
 * @ingroup moo_ext
 * This module provides functions to create, use, and destroy a
 * @em moo_ext_single
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_ext_single.h"
 * @endcode
 */


/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_ext_single
  @return   1 newly allocated moo_ext_single or NULL in case of an error

  The returned object must be deallocated using moo_ext_single_delete().

 */
/*----------------------------------------------------------------------------*/
moo_ext_single *
moo_ext_single_new(moo_detector_type type, int ntas)
{
    moo_ext_single *res = cpl_calloc(1, sizeof(moo_ext_single));
    res->badpix_mask = MOO_BADPIX_GOOD;
    res->type = type;
    res->ntas = ntas;
    const char *extname = moo_detector_get_extname(type, ntas);
    res->extname = extname;
    res->header = cpl_propertylist_new();
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_ext_single from the given EXT filename
  @param    filename the EXT filename
  @param    type the type of detector
  @param    ntas the number id of spectropgraph
  @return   1 newly allocated moo_ext_single or NULL in case of an error

  The returned object must be deallocated using moo_ext_single_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_ext_single *
moo_ext_single_create(const char *filename, moo_detector_type type, int ntas)
{
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);

    moo_ext_single *single = NULL;
    cpl_propertylist *header = NULL;

    const char *extname = moo_detector_get_extname(type, ntas);

    header = moo_fits_load_extension_header(filename, NULL, extname);
    if (header != NULL) {
        int naxis = moo_pfits_get_naxis(header);
        if (naxis > 0) {
            single = moo_ext_single_new(type, ntas);
            single->filename = filename;
            cpl_propertylist_append(single->header, header);
        }
        cpl_propertylist_delete(header);
    }
    return single;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    load a moo_ext_single using the level for badpixel
  @param    self    the EXT_SINGLE  
  @param    level the reference badpixel to build the image mask
  @return   1 newly allocated moo_ext_single or NULL in case of an error

  The returned object must be deallocated using moo_ext_single_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_ext_single_load(moo_ext_single *self, unsigned int level)
{
    if (self != NULL) {
        if ((self->filename != NULL) && (self->extname != NULL)) {
            if (self->header == NULL) {
                cpl_size extnum =
                    cpl_fits_find_extension(self->filename, self->extname);
                if (extnum > 0) {
                    self->header =
                        cpl_propertylist_load(self->filename, extnum);
                }
            }
            if (self->qual == NULL) {
                self->qual =
                    moo_fits_load_extension_image(self->filename,
                                                  MOO_EXT_SINGLE_QUAL,
                                                  self->extname, CPL_TYPE_INT);
            }
            if (self->image == NULL) {
                cpl_image *data = NULL;
                cpl_image *err = NULL;

                data = moo_fits_load_extension_image(self->filename, NULL,
                                                     self->extname,
                                                     CPL_TYPE_DOUBLE);
                err = moo_fits_load_extension_image(self->filename,
                                                    MOO_EXT_SINGLE_ERR,
                                                    self->extname,
                                                    CPL_TYPE_DOUBLE);

                if (data != NULL && err != NULL) {
                    self->image = hdrl_image_create(data, err);
                    cpl_image_delete(data);
                    cpl_image_delete(err);
                }
            }
        }

        cpl_mask *mask = hdrl_image_get_mask(self->image);
        moo_badpix_to_mask(self->qual, mask, level);
    }
    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_ext_single
  @param    self    moo_ext_single to delete
  @return   void

  If the moo_ext_single @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_ext_single_delete(moo_ext_single *self)
{
    if (self != NULL) {
        if (self->header != NULL) {
            cpl_propertylist_delete(self->header);
        }
        if (self->image != NULL) {
            hdrl_image_delete(self->image);
        }
        if (self->qual != NULL) {
            cpl_image_delete(self->qual);
        }
        cpl_free(self);
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_ext_single to a FITS file
  @param    self          moo_ext_single to write to disk or NULL
  @param    filename      Name of the file to write
  @param    type the type of detector
  @param    ntas the number id of spectropgraph
  
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_loc_single to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
void
moo_ext_single_save(const moo_ext_single *self,
                    const char *filename,
                    moo_detector_type type,
                    int ntas)
{
    const char *extname = moo_detector_get_extname(type, ntas);
    if (self != NULL) {
        if (self->image != NULL) {
            moo_fits_write_extension_image(hdrl_image_get_image(self->image),
                                           filename, NULL, extname,
                                           CPL_TYPE_FLOAT, self->header);

            cpl_propertylist *err_header = cpl_propertylist_new();
            moo_pfits_append_hduclass_error(err_header, type, ntas,
                                            self->header);

            moo_fits_write_extension_image(hdrl_image_get_error(self->image),
                                           filename, MOO_EXT_SINGLE_ERR,
                                           extname, MOO_EXT_SINGLE_ERR_TYPE,
                                           err_header);

            cpl_propertylist_delete(err_header);
            cpl_propertylist *qual_header = cpl_propertylist_new();
            moo_pfits_append_hduclass_quality(qual_header, type, ntas,
                                              self->header, self->badpix_mask);
            moo_fits_write_extension_image(self->qual, filename,
                                           MOO_EXT_SINGLE_QUAL, extname,
                                           MOO_EXT_SINGLE_QUAL_TYPE,
                                           qual_header);
            cpl_propertylist_delete(qual_header);
        }
    }
    else {
        cpl_propertylist *h = cpl_propertylist_new();
        cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME, extname);
        cpl_propertylist_save(h, filename, CPL_IO_EXTEND);
        cpl_propertylist_delete(h);

        moo_fits_write_extension_image(NULL, filename, MOO_EXT_SINGLE_ERR,
                                       extname, MOO_EXT_SINGLE_ERR_TYPE, NULL);
        moo_fits_write_extension_image(NULL, filename, MOO_EXT_SINGLE_QUAL,
                                       extname, MOO_EXT_SINGLE_QUAL_TYPE, NULL);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_ext_single
  @brief    Dump structural information of EXT_SINGLE
  @param    self    EXT_SINGLE to dump
  @param    stream  Output stream, accepts @c stdout or @c stderr
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_ext_single_dump(const moo_ext_single *self, FILE *stream)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(stream != NULL, CPL_ERROR_NULL_INPUT);

    fprintf(stream, "---MOO_EXT_SINGLE\n");
    fprintf(stream, "filename %s extname %s\n", self->filename, self->extname);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_ext_single
  @brief   Get image of EXT_SINGLE
  @param    self    EXT_SINGLE
  @return   image or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
hdrl_image *
moo_ext_single_get_image(moo_ext_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    return self->image;
}
/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_ext_single
  @brief   Get image of data
  @param    self    EXT_SINGLE
  @return   image or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_ext_single_get_data(moo_ext_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_image *res = NULL;

    if (self->image != NULL) {
        res = hdrl_image_get_image(self->image);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_ext_single
  @brief   Get image of errs
  @param    self    EXT_SINGLE
  @return   image or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_ext_single_get_errs(moo_ext_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_image *res = NULL;

    if (self->image != NULL) {
        res = hdrl_image_get_error(self->image);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_ext_single
  @brief   Get image of qual
  @param    self    EXT_SINGLE
  @return   image or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_ext_single_get_qual(moo_ext_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->extname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (self->qual == NULL) {
        self->qual =
            moo_fits_load_extension_image(self->filename, MOO_EXT_SINGLE_QUAL,
                                          self->extname,
                                          MOO_EXT_SINGLE_QUAL_TYPE);
    }
    return self->qual;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_ext_single
  @brief   Get header of ext single
  @param    self    EXT_SINGLE
  @return   the header or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_propertylist *
moo_ext_single_get_header(moo_ext_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->extname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (self->header == NULL) {
        self->header =
            moo_fits_load_extension_header(self->filename, NULL, self->extname);
        if (self->header == NULL) {
            self->header = cpl_propertylist_new();
        }
    }
    return self->header;
}

cpl_error_code
moo_ext_single_compute_qc(moo_ext_single *self,
                          const char *colname,
                          const char *dersnr_colname,
                          cpl_table *fibres_table)
{
    cpl_array *index = NULL;
    cpl_image *snr = NULL;
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_errorstate prestate = cpl_errorstate_get();

    if (self != NULL) {
        hdrl_image *image = moo_ext_single_get_image(self);
        cpl_mask *mask = hdrl_image_get_mask(image);

        cpl_ensure_code(image != NULL, CPL_ERROR_NULL_INPUT);

        unsigned int badpix_level = MOO_BADPIX_COSMETIC |
                                    MOO_BADPIX_OUTSIDE_DATA_RANGE |
                                    MOO_BADPIX_HOT | MOO_BADPIX_CALIB_DEFECT;
        moo_try_check(moo_badpix_to_mask(self->qual, mask, badpix_level), " ");
        cpl_image *data = hdrl_image_get_image(image);
        cpl_image *err = hdrl_image_get_error(image);

        int nx = cpl_image_get_size_x(data);
        double min = cpl_image_get_min(data);
        double max = cpl_image_get_max(data);
        double rms = cpl_image_get_stdev(data);
        double median = cpl_image_get_median(data);
        double mean = cpl_image_get_mean(data);
        cpl_propertylist *header = moo_ext_single_get_header(self);

        moo_qc_set_mflat_min(header, min);
        moo_qc_set_mflat_max(header, max);

        moo_qc_set_mflat_avg(header, mean);
        moo_qc_set_mflat_med(header, median);
        moo_qc_set_mflat_rms(header, rms);
        cpl_image_reject_value(err, CPL_VALUE_ZERO);
        int nbrej = cpl_image_count_rejected(err);
        int allpix = cpl_image_get_size_x(err) * cpl_image_get_size_y(err);
        if (nbrej < allpix) {
            moo_try_check(snr = cpl_image_divide_create(data, err), " ");
            double max_snr = cpl_image_get_max(snr);
            double min_snr = cpl_image_get_min(snr);
            double median_snr = cpl_image_get_median(snr);

            moo_qc_set_mflat_sn_min(header, min_snr);
            moo_qc_set_mflat_sn_max(header, max_snr);
            moo_qc_set_mflat_sn_med(header, median_snr);

            moo_try_check(index = cpl_table_where_selected(fibres_table), " ");
            int size = cpl_array_get_size(index);

            for (int i = 0; i < size; i++) {
                int idx = cpl_array_get_cplsize(index, i, NULL);
                int h = cpl_table_get(fibres_table, MOO_FIBRES_TABLE_HEALTH,
                                      idx, NULL);
                if (h == 1) {
                    double val = NAN;
                    double dersnr = NAN;
                    cpl_errorstate fstate = cpl_errorstate_get();
                    val = cpl_image_get_median_window(snr, 1, i + 1, nx, i + 1);
                    if (cpl_errorstate_is_equal(fstate)) {
                        cpl_table_set(fibres_table, colname, idx, val);
                    }
                    else {
                        cpl_table_set(fibres_table, colname, idx, NAN);
                        cpl_errorstate_set(fstate);
                    }
                    cpl_vector *row =
                        cpl_vector_new_from_image_row(data, i + 1);
                    cpl_vector *row_filter = moo_vector_filter_nan(row);
                    dersnr = moo_vector_get_dersnr(row_filter);
                    cpl_table_set(fibres_table, dersnr_colname, idx, dersnr);
                    cpl_vector_delete(row_filter);
                    cpl_vector_delete(row);
                }
            }
        }
    }

moo_try_cleanup:
    cpl_array_delete(index);
    cpl_image_delete(snr);

    if (!cpl_errorstate_is_equal(prestate)) {
        status = cpl_error_get_code();
        cpl_msg_error("moo_ext_single", "Can't compute QC for file %s",
                      self->filename);
    }
    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Set the WCS1 of the extension
  @param    self single the extension
  @param    crpix1 the crpix1 keyword
  @param    crval1 the crval1 keyword
  @param    cd1_1 the cd1_1 keyword
  @param    ctype1 the ctype1 keyword
  @param    cunit1 the cunit1 keyword
  @return   the relevant error code

 the header of the extension must be preallocated before calling this function
 Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if a field of extension is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_ext_single_set_wcs1(moo_ext_single *self,
                        double crpix1,
                        double crval1,
                        double cd1_1,
                        const char *ctype1,
                        const char *cunit1)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->header != NULL, CPL_ERROR_ILLEGAL_INPUT);

    cpl_error_code code = CPL_ERROR_NONE;
    code =
        cpl_propertylist_append_double(self->header, MOO_PFITS_CRPIX1, crpix1);
    cpl_ensure_code(code == CPL_ERROR_NONE, code);
    code =
        cpl_propertylist_append_double(self->header, MOO_PFITS_CRVAL1, crval1);
    cpl_ensure_code(code == CPL_ERROR_NONE, code);
    code = cpl_propertylist_append_double(self->header, MOO_PFITS_CD1_1, cd1_1);
    cpl_ensure_code(code == CPL_ERROR_NONE, code);
    code =
        cpl_propertylist_append_string(self->header, MOO_PFITS_CTYPE1, ctype1);
    cpl_ensure_code(code == CPL_ERROR_NONE, code);
    code =
        cpl_propertylist_append_string(self->header, MOO_PFITS_CUNIT1, cunit1);
    cpl_ensure_code(code == CPL_ERROR_NONE, code);
    return code;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_ext_single
  @brief    Free memory associate to this single EXT
  @param    self    moo_single where want to free memory
  @return   error_code or CPL_ERROR_NONE

  Possible _cpl_error_code_ set in this function:
    - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_ext_single_free(moo_ext_single *self)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    if (self->filename != NULL && self->extname != NULL) {
        if (self->header != NULL) {
            cpl_propertylist_delete(self->header);
            self->header = NULL;
        }
        if (self->image != NULL) {
            hdrl_image_delete(self->image);
            self->image = NULL;
        }
        if (self->qual != NULL) {
            cpl_image_delete(self->qual);
            self->qual = NULL;
        }
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_ext_single
  @brief    Add two single EXT
  @param    a    single to add
  @param    b    single which is add
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_ext_single_sum(moo_ext_single *a, moo_ext_single *b)
{
    cpl_ensure_code(a != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(b != NULL, CPL_ERROR_NULL_INPUT);

    hdrl_image *a_img = moo_ext_single_get_image(a);
    hdrl_image *b_img = moo_ext_single_get_image(b);

    hdrl_image_add_image(a_img, b_img);

    cpl_image *qual1 = moo_ext_single_get_qual(a);
    cpl_image *qual2 = moo_ext_single_get_qual(b);

    cpl_image_or(a->qual, qual1, qual2);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_ext_single
  @brief Compute SNR for a given target
  @param self  [EXT_SINGLE] ext_single structure
  @param ext_idx index in EXT
  @param wmap the WAVELENGTH map corresponding to detector
  @param sinfo the spectral information corresponding to detector
  @param skylines the sky lines corresponding to detector
  @param dersnr snr compute per der_snr
  @return snr
*/
/*----------------------------------------------------------------------------*/
double
moo_ext_single_compute_snr(moo_ext_single *self,
                           int ext_idx,
                           cpl_image *wmap,
                           moo_spectral_format_info *sinfo,
                           moo_sky_lines_list *skylines,
                           double *dersnr)
{
    double snr = 0.0;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, 0.0);
    cpl_ensure(wmap != NULL, CPL_ERROR_NULL_INPUT, 0.0);
    cpl_ensure(sinfo != NULL, CPL_ERROR_NULL_INPUT, 0.0);

    hdrl_image *himg = moo_ext_single_get_image(self);
    int nx = cpl_image_get_size_x(wmap);
    int rej;
    int start = 1;
    int direction = sinfo->direction;

    double wmin = cpl_image_get(wmap, start, ext_idx, &rej);

    while (isnan(wmin)) {
        start++;
        wmin = cpl_image_get(wmap, start, ext_idx, &rej);
    }

    int stop = nx;

    double wmax = cpl_image_get(wmap, stop, ext_idx, &rej);
    while (isnan(wmax)) {
        stop--;
        wmax = cpl_image_get(wmap, stop, ext_idx, &rej);
    }

    if (direction < 0) {
        double temp = wmin;
        int itemp = start;
        wmin = wmax;
        start = stop;
        wmax = temp;
        stop = itemp;
    }
    double *zwmin = NULL;
    double *zwmax = NULL;
    cpl_array *sel = NULL;

    moo_sky_lines_list_get_free_zones(skylines, wmin, wmax, &zwmin, &zwmax,
                                      &sel);
    int size = cpl_array_get_size(sel);
    cpl_vector *data = NULL;
    cpl_vector *data_dersnr = NULL;
    for (int i = 0; i < size; i++) {
        int idx = cpl_array_get_cplsize(sel, i, NULL);
        double cmin = zwmin[idx];
        double cmax = zwmax[idx];

        int zstart = start;
        double zmin = cpl_image_get(wmap, zstart, ext_idx, &rej);
        while (zmin < cmin) {
            zstart += direction;
            zmin = cpl_image_get(wmap, zstart, ext_idx, &rej);
        }

        int zstop = zstart;
        double zmax = cpl_image_get(wmap, zstop, ext_idx, &rej);

        while (zmax < cmax && zstop < stop) {
            zstop += direction;
            zmax = cpl_image_get(wmap, zstop, ext_idx, &rej);
        }
        if (direction < 0) {
            int temp = zstart;
            zstart = zstop;
            zstop = temp;
        }
        hdrl_image *fimg =
            hdrl_image_extract(himg, zstart, ext_idx, zstop, ext_idx);
        hdrl_image_reject_value(fimg, CPL_VALUE_NAN);
        cpl_image *img = hdrl_image_get_image(fimg);
        cpl_image *error = hdrl_image_get_error(fimg);
        cpl_image_reject_value(error, CPL_VALUE_NAN | CPL_VALUE_ZERO);
        int enx = cpl_image_get_size_x(error);
        int enbrej = cpl_image_count_rejected(error);
        if (enx > enbrej) {
            cpl_vector *imrow = cpl_vector_new_from_image_row(img, 1);

            if (data_dersnr == NULL) {
                data_dersnr = cpl_vector_duplicate(imrow);
            }
            else {
                int vsize = cpl_vector_get_size(data_dersnr);
                int row_size = cpl_vector_get_size(imrow);
                cpl_vector_set_size(data_dersnr, vsize + row_size);
                for (int j = 0; j < row_size; j++) {
                    double v = cpl_vector_get(imrow, j);
                    cpl_vector_set(data_dersnr, j + vsize, v);
                }
            }
            cpl_image_divide(img, error);

            cpl_vector *row = cpl_vector_new_from_image_row(img, 1);
            if (data == NULL) {
                data = cpl_vector_duplicate(row);
            }
            else {
                int vsize = cpl_vector_get_size(data);
                int row_size = cpl_vector_get_size(row);
                cpl_vector_set_size(data, vsize + row_size);
                for (int j = 0; j < row_size; j++) {
                    double v = cpl_vector_get(row, j);
                    cpl_vector_set(data, j + vsize, v);
                }
            }
            cpl_vector_delete(row);
            cpl_vector_delete(imrow);
        }
        hdrl_image_delete(fimg);
    }
    if (data_dersnr != NULL) {
        *dersnr = moo_vector_get_dersnr(data_dersnr);
        snr = cpl_vector_get_median(data);
    }
    else {
        *dersnr = NAN;
        snr = NAN;
    }

    cpl_vector_delete(data);
    cpl_vector_delete(data_dersnr);
    cpl_array_delete(sel);
    return snr;
}
/*----------------------------------------------------------------------------*/
/**@}*/
