/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_params.h"
#include "moo_single.h"
#include "moo_badpix.h"
#include "moo_ext.h"
#include "moo_ext_single.h"
#include "moo_extract.h"
#include "moo_fibres_table.h"
#include "moo_fits.h"
#include "moo_pfits.h"
#include "moo_qc.h"
#include "moo_utils.h"
#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/


static hdrl_value
_moo_integrate_flux(hdrl_image *image,
                    int *det_qual,
                    int x,
                    double y,
                    int *qual,
                    double frac,
                    int *rej)
{
    double y1 = round(y);

    hdrl_value v1 = hdrl_image_get_pixel(image, x, (int)y1, rej);
    int nx = hdrl_image_get_size_x(image);

    *qual = det_qual[x - 1 + nx * ((int)y1 - 1)];
    double val = v1.data * frac;
    double err = v1.error * frac;

    hdrl_value res;

    res.data = val;
    res.error = err;

    return res;
}

static cpl_error_code
_moo_extract_fibre_opt(hdrl_image *image,
                       cpl_image *model,
                       cpl_image *det_qual,
                       int fibnum,
                       double *centroids,
                       double *wlo,
                       double *wup,
                       cpl_image *fluxes,
                       cpl_image *errs,
                       cpl_image *qual,
                       double aperture)
{
    cpl_ensure_code(image != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(model != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(det_qual != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(centroids != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(wlo != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(wup != NULL, CPL_ERROR_NULL_INPUT);

    int nx = hdrl_image_get_size_x(image);
    int *det_qual_data = cpl_image_get_data_int(det_qual);
    int idfib = fibnum - 1;

    for (int i = 0; i < nx; i++) {
        double yc = centroids[idfib * nx + i];
        double yl = yc - wlo[idfib * nx + i];
        double yu = yc + wup[idfib * nx + i];

        if (aperture > 0) {
            yl = yc - aperture / 2.;
            yu = yc + aperture / 2.;
        }

        if (!isnan(yc)) {
            int rej;
            int ql = 0, qu = 0;
            int qflag = 0;
            double m1 = 0.0;
            double m2 = 0.0;
            double y1 = round(yl);
            double y2 = round(yu);

            int size = y2 - y1 + 1;
            cpl_vector *vmodel = cpl_vector_new(size);
            cpl_vector *vobs = cpl_vector_new(size);
            cpl_vector *verrobs = cpl_vector_new(size);
            int *vrej = cpl_calloc(size, sizeof(int));
            cpl_vector *vw = cpl_vector_new(size);
            cpl_vector *vq = cpl_vector_new(size);

            double frac = y1 + 0.5 - yl;
            hdrl_value vl = _moo_integrate_flux(image, det_qual_data, i + 1, yl,
                                                &ql, frac, &rej);
            vrej[0] = rej;

            m1 = cpl_image_get(model, i + 1, y1, &rej);
            m1 *= frac;
            cpl_vector_set(vobs, 0, vl.data);
            cpl_vector_set(verrobs, 0, vl.error);
            cpl_vector_set(vmodel, 0, m1);
            cpl_vector_set(vq, 0, ql);

            if (rej == 0) {
                qflag |= ql;
            }

            frac = yu - y2 + 0.5;
            hdrl_value vu = _moo_integrate_flux(image, det_qual_data, i + 1, yu,
                                                &qu, frac, &rej);
            vrej[size - 1] = rej;

            m2 = cpl_image_get(model, i + 1, y2, &rej);
            m2 *= frac;

            if (rej == 0) {
                qflag |= qu;
            }

            cpl_vector_set(vmodel, size - 1, m2);
            cpl_vector_set(vq, size - 1, qu);
            cpl_vector_set(vobs, size - 1, vu.data);
            cpl_vector_set(verrobs, size - 1, vu.error);

            int vi = 1;

            for (int j = y1 + 1; j <= y2 - 1; j++) {
                double m = cpl_image_get(model, i + 1, j, &rej);
                hdrl_value v = hdrl_image_get_pixel(image, i + 1, j, &rej);
                int qualv = det_qual_data[i + nx * (j - 1)];

                vrej[vi] = rej;
                cpl_vector_set(vobs, vi, v.data);
                cpl_vector_set(verrobs, vi, v.error);
                cpl_vector_set(vmodel, vi, m);
                cpl_vector_set(vq, vi, qualv);
                vi++;
            }

            double mmin = cpl_vector_get_min(vmodel);
            if (mmin < 0) {
                cpl_vector_subtract_scalar(vmodel, mmin);
            }
            double msum = cpl_vector_get_sum(vmodel);

            cpl_vector_divide_scalar(vmodel, msum);

            for (int j = 0; j < size; j++) {
                double w = 0.0;
                double m = cpl_vector_get(vmodel, j);
                if (vrej[j] == 0) {
                    w = sqrt(m);
                }
                cpl_vector_set(vw, j, w);
            }

            double wsum = cpl_vector_get_sum(vw);

            if (wsum == 0) {
                for (int j = 0; j < size; j++) {
                    double qv = cpl_vector_get(vq, j);
                    qflag |= (int)qv;
                }
                cpl_image_reject(fluxes, i + 1, fibnum);
                cpl_image_set(fluxes, i + 1, fibnum, NAN);
                cpl_image_set(errs, i + 1, fibnum, NAN);
                cpl_image_set(qual, i + 1, fibnum, qflag);
            }
            else {
                cpl_vector_divide_scalar(vw, wsum);
                double sum = 0, errors = 0;
                moo_fit_mul(vmodel, vw, vobs, verrobs, &sum, &errors);
                cpl_image_set(fluxes, i + 1, fibnum, sum);
                if (isnan(sum)) {
                    qflag |= MOO_BADPIX_CALIB_DEFECT;
                }
                cpl_image_set(errs, i + 1, fibnum, errors);
                cpl_image_set(qual, i + 1, fibnum, qflag);
            }

            cpl_vector_delete(vmodel);
            cpl_vector_delete(vobs);
            cpl_vector_delete(verrobs);
            cpl_vector_delete(vw);
            cpl_vector_delete(vq);
            cpl_free(vrej);
        }
        else {
            cpl_image_reject(fluxes, i + 1, fibnum);
            cpl_image_set(fluxes, i + 1, fibnum, NAN);
            cpl_image_set(errs, i + 1, fibnum, NAN);
            cpl_image_set(qual, i + 1, fibnum, MOO_BADPIX_OUTSIDE_DATA_RANGE);
        }
    }

    return cpl_error_get_code();
}

static cpl_error_code
_moo_extract_fibre_sum(hdrl_image *image,
                       cpl_image *det_qual,
                       int fibnum,
                       double *centroids,
                       double *wlo,
                       double *wup,
                       cpl_image *fluxes,
                       cpl_image *errs,
                       cpl_image *qual,
                       double aperture)
{
    cpl_ensure_code(image != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(det_qual != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(centroids != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(wlo != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(wup != NULL, CPL_ERROR_NULL_INPUT);
    cpl_errorstate prestate = cpl_errorstate_get();

    int nx = hdrl_image_get_size_x(image);
    int *det_qual_data = cpl_image_get_data_int(det_qual);
    int idfib = fibnum - 1;

    for (int i = 0; i < nx; i++) {
        double yc = centroids[idfib * nx + i];
        double yl = yc - wlo[idfib * nx + i];
        double yu = yc + wup[idfib * nx + i];
        if (aperture > 0) {
            yl = yc - aperture / 2.;
            yu = yc + aperture / 2.;
        }
        if (yl < 0.5) {
            cpl_msg_error("moo_extract",
                          "#%d Invalid localisation low limit (%f) at x %d",
                          fibnum, yl, i + 1);
            break;
        }
        if (!isnan(yc)) {
            double sum = 0, errors = 0;
            int ql = 0, qu = 0;
            int qflag = 0;
            int rej;

            double y1 = round(yl);
            double y3 = round(yc);
            double frac = y1 + 0.5 - yl;
            if (y1 == y3) {
                frac = yc - yl;
            }

            hdrl_value vl = _moo_integrate_flux(image, det_qual_data, i + 1, yl,
                                                &ql, frac, &rej);

            double y2 = round(yu);
            frac = yu - y2 + 0.5;
            if (y1 == y3) {
                frac = yu - yc;
            }
            hdrl_value vu = _moo_integrate_flux(image, det_qual_data, i + 1, yu,
                                                &qu, frac, &rej);

            sum = vl.data + vu.data;

            errors = vl.error * vl.error + vu.error * vu.error;
            qflag |= ql;
            qflag |= qu;


            for (int j = y1 + 1; j <= y2 - 1; j++) {
                hdrl_value v = hdrl_image_get_pixel(image, i + 1, j, &rej);
                qflag |= det_qual_data[i + nx * (j - 1)];
                sum += v.data;
                errors += v.error * v.error;
            }

            errors = sqrt(errors);
            cpl_image_set(fluxes, i + 1, fibnum, sum);
            cpl_image_set(errs, i + 1, fibnum, errors);
            cpl_image_set(qual, i + 1, fibnum, qflag);
        }
        else {
            cpl_image_set(fluxes, i + 1, fibnum, NAN);
            cpl_image_set(errs, i + 1, fibnum, NAN);
            cpl_image_set(qual, i + 1, fibnum, MOO_BADPIX_OUTSIDE_DATA_RANGE);
        }
    }

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in extraction");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return cpl_error_get_code();
}

static moo_ext_single *
_moo_extract_single(moo_single *det,
                    moo_loc_single *loc,
                    moo_psf_single *master_flat,
                    const char *filename,
                    const int *health,
                    cpl_array *indexes,
                    const int *indexext,
                    double aperture)
{
    cpl_ensure(det != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(loc != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(health != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();
    moo_ext_single *res = moo_ext_single_new(det->type, det->ntas);

    cpl_image *fcentroids = moo_loc_single_get_f_centroids(loc);
    moo_try_assure(fcentroids != NULL, CPL_ERROR_NULL_INPUT,
                   "No data in localisation %s for extension %s", loc->filename,
                   loc->extname);
    cpl_image *fwlo = moo_loc_single_get_f_wlo(loc);
    cpl_image *fwup = moo_loc_single_get_f_wup(loc);
    hdrl_image *image = moo_single_get_image(det);
    cpl_image *det_qual = moo_single_get_qual(det);

    int nb_fibres = cpl_image_get_size_y(fcentroids);
    int nx = cpl_image_get_size_x(fcentroids);

    moo_pfits_append_hduclass_data(res->header, det->type, det->ntas);
    cpl_propertylist_copy_property_regexp(res->header, det->header, "ESO DET *",
                                          0);

    cpl_propertylist_copy_property(res->header, det->header, MOO_PFITS_BUNIT);

    cpl_propertylist_copy_property(res->header, det->header, MOO_PFITS_CRPIX1);
    cpl_propertylist_copy_property(res->header, det->header, MOO_PFITS_CRVAL1);
    cpl_propertylist_copy_property(res->header, det->header, MOO_PFITS_CD1_1);
    cpl_propertylist_copy_property(res->header, det->header, MOO_PFITS_CTYPE1);
    cpl_propertylist_copy_property(res->header, det->header, MOO_PFITS_CUNIT1);

    // FIXME: Use cpl_propertylist_update family of functions, rather than
    //        cpl_propertylist_append. This avoids unintentional keyword
    //        duplication!
    cpl_propertylist_append_double(res->header, MOO_PFITS_CRPIX2,
                                   MOO_EXT_SINGLE_CRPIX2);
    cpl_propertylist_append_double(res->header, MOO_PFITS_CRVAL2,
                                   MOO_EXT_SINGLE_CRVAL2);

    cpl_propertylist_append_double(res->header, MOO_PFITS_CD1_2, 0.);
    cpl_propertylist_append_double(res->header, MOO_PFITS_CD2_1, 0.);

    cpl_propertylist_append_double(res->header, MOO_PFITS_CD2_2,
                                   MOO_EXT_SINGLE_CDELT2);
    cpl_propertylist_append_string(res->header, MOO_PFITS_CTYPE2,
                                   MOO_EXT_SINGLE_CTYPE2);

    res->filename = filename;

    hdrl_image *himage = hdrl_image_new(nx, nb_fibres);
    cpl_image *ext_fluxes = hdrl_image_get_image(himage);
    cpl_image *ext_errs = hdrl_image_get_error(himage);
    cpl_image *ext_qual = cpl_image_new(nx, nb_fibres, CPL_TYPE_INT);

    res->image = himage;
    res->qual = ext_qual;

    double *centroids = cpl_image_get_data(fcentroids);

    double *wlo = cpl_image_get_data(fwlo);
    double *wup = cpl_image_get_data(fwup);

    if (master_flat != NULL) {
        cpl_image *model = moo_psf_single_reproject_model(master_flat, det, loc,
                                                          health, indexes);
        for (int i = 1; i <= nb_fibres; i++) {
            int idx = cpl_array_get_cplsize(indexes, i - 1, NULL);
            int h = health[idx];
            if (h == 1) {
                cpl_msg_debug(__func__,
                              "extract fibres %d index %d indexext %d", i, idx,
                              indexext[idx]);
                moo_try_check(_moo_extract_fibre_opt(image, model, det_qual, i,
                                                     centroids, wlo, wup,
                                                     ext_fluxes, ext_errs,
                                                     ext_qual, aperture),
                              " ");
            }
        }
        cpl_image_delete(model);
    }
    else {
/* The following addresses an issue with the gcc9 compiler series prior to
 * gcc 9.3. These compiler versions require that the variable '__func__'
 * appears in the data sharing clause if default(none) is used. However
 * adding it to the data sharing clause breaks older compiler versions where
 * these variables are pre-determined as shared.
 * This was fixed in gcc 9.3 and OpenMP 5.1.
 */
#ifdef _OPENMP
#if (__GNUC__ == 9) && (__GNUC_MINOR__ < 3)
#pragma omp parallel shared(nb_fibres, health, ext_fluxes, ext_errs, ext_qual, \
                                wlo, wup, centroids, image, det_qual, indexes, \
                                aperture)
#else
#pragma omp parallel default(none)                                      \
    shared(nb_fibres, health, ext_fluxes, ext_errs, ext_qual, wlo, wup, \
               centroids, image, det_qual, indexes, aperture)
#endif
        {
#pragma omp for
#endif
            for (int i = 1; i <= nb_fibres; i++) {
                //for (int i = 500; i <= 500; i++) {
                int idx = cpl_array_get_cplsize(indexes, i - 1, NULL);
                int h = health[idx];
                if (h == 1) {
                    cpl_msg_debug(__func__, "extract fibres %d", i);
                    _moo_extract_fibre_sum(image, det_qual, i, centroids, wlo,
                                           wup, ext_fluxes, ext_errs, ext_qual,
                                           aperture);
                }
            }
#ifdef _OPENMP
        }
#endif
    }

moo_try_cleanup:
    // dump error from the state
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in extraction extension %s",
                      det->extname);
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);

        moo_ext_single_delete(res);
        res = NULL;
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return res;
}


static cpl_propertylist *
_moo_create_primary_header(cpl_propertylist *det_primary_header)
{
    cpl_propertylist *primary_header = NULL;

    primary_header = cpl_propertylist_new();
    cpl_propertylist_copy_property(primary_header, det_primary_header,
                                   MOO_PFITS_EXPTIME);
    cpl_propertylist_copy_property_regexp(primary_header, det_primary_header,
                                          "TM*", 0);
    cpl_propertylist_copy_property_regexp(primary_header, det_primary_header,
                                          "ESO QC IS*", 0);
    moo_qc_set_snr_range(primary_header, MOONS_QC_SNR_RANGE_FULL);

    return primary_header;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    extract the 1D spectrum of fibres
  @param    det _DET_ frame to extract
  @param    loc _LOC_ localisation of the fibres
  @param    master_flat _PSF_ the master flat or NULL
  @param   params the extraction parameters
  @param   filename the file name for the result
  @return   the _EXT_ spectra

 * This function is to extract fibre spectra into 1D with two different methods
 * defined by extract-method: “sum” or “optimal”. The latter will request the
 * MasterFlat input, in order to use the fibre profiles along the slit.

 * _Bad pixels flags_:
  - BADPIX_OUTSIDE_DATA_RANGE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the localisation fibres table is NULL
 */
/*----------------------------------------------------------------------------*/
moo_ext *
moo_extract(moo_det *det,
            moo_loc *loc,
            moo_psf *master_flat,
            moo_extract_params *params,
            const char *filename)
{
    moo_ext *result = NULL;
    int *indexext = NULL;
    int *health = NULL;
    const char **names = NULL;
    const char **loc_names = NULL;
    int *loc_indexext = NULL;
    const int *loc_health = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(det != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(loc != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_msg_info(__func__, "Extracting using method %s ", params->method);

    unsigned int badpix_level = MOO_BADPIX_GOOD;

    if (strcmp(params->method, MOO_EXTRACT_METHOD_OPTIMAL) == 0) {
        cpl_ensure(master_flat != NULL, CPL_ERROR_NULL_INPUT, NULL);
        badpix_level = MOO_BADPIX_COSMETIC | MOO_BADPIX_HOT |
                       MOO_BADPIX_CALIB_DEFECT | MOO_BADPIX_OUTSIDE_DATA_RANGE;
    }
    moo_fits_create(filename);

    result = moo_ext_new();
    result->filename = filename;
    result->primary_header = _moo_create_primary_header(det->primary_header);

    cpl_table *fibres_table = moo_det_get_fibre_table(det);
    cpl_ensure(fibres_table != NULL, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_table *loc_fibres_table = moo_loc_get_fibre_table(loc);
    cpl_ensure(loc_fibres_table != NULL, CPL_ERROR_ILLEGAL_INPUT, NULL);

    moo_fibres_table_by_index(loc_fibres_table);
    moo_fibres_table_by_index(fibres_table);

    cpl_table *ext_fibres_table = cpl_table_duplicate(fibres_table);

    result->fibre_table = ext_fibres_table;
    moo_try_check(moo_fibres_table_add_extract_cols(ext_fibres_table), " ");

    int nrow = cpl_table_get_nrow(ext_fibres_table);
    int loc_nrow = cpl_table_get_nrow(loc_fibres_table);

    //    cpl_ensure(loc_nrow==nrow, CPL_ERROR_ILLEGAL_INPUT, NULL);
    moo_try_check(indexext = cpl_table_get_data_int(ext_fibres_table,
                                                    MOO_FIBRES_TABLE_INDEXEXT),
                  " ");
    moo_try_check(health = cpl_table_get_data_int(ext_fibres_table,
                                                  MOO_FIBRES_TABLE_HEALTH),
                  " ");
    moo_try_check(names =
                      cpl_table_get_data_string_const(ext_fibres_table,
                                                      MOO_FIBRES_TABLE_FIBRE),
                  " ");
    moo_try_check(loc_names =
                      cpl_table_get_data_string_const(loc_fibres_table,
                                                      MOO_FIBRES_TABLE_FIBRE),
                  " ");
    moo_try_check(loc_indexext =
                      cpl_table_get_data_int(loc_fibres_table,
                                             MOO_FIBRES_TABLE_INDEXEXT),
                  " ");
    moo_try_check(loc_health =
                      cpl_table_get_data_int_const(loc_fibres_table,
                                                   MOO_FIBRES_TABLE_HEALTH),
                  " ");

    for (int i = 0; i < nrow; i++) {
        const char *ext_name = names[i];
        for (int j = 0; j < loc_nrow; j++) {
            const char *loc_name = loc_names[j];
            if (strcmp(ext_name, loc_name) == 0) {
                indexext[i] = loc_indexext[j];
                if (health[i] != 0) {
                    health[i] = loc_health[j];
                }
            }
        }
    }
    cpl_msg_indent_more();

    const char *colnames[] = { "MEDIAN_SNR_RI_EXT", "MEDIAN_SNR_YJ_EXT",
                               "MEDIAN_SNR_H_EXT" };
    const char *dersnr_colnames[] = { MOO_FIBRES_TABLE_DERSNR_RI_EXT,
                                      MOO_FIBRES_TABLE_DERSNR_YJ_EXT,
                                      MOO_FIBRES_TABLE_DERSNR_H_EXT };
    for (int i = 1; i <= 2; i++) {
        cpl_array *indexes =
            moo_fibres_table_get_spectro_indexext(ext_fibres_table, i);

        int nb_selected = cpl_array_get_size(indexes);

        if (health != NULL) {
            for (int j = 0; j < 3; j++) {
                if (!cpl_errorstate_is_equal(prestate)) {
                    cpl_msg_error(__func__,
                                  "Error in extraction for DET %s LOC %s",
                                  det->filename, loc->filename);
                    cpl_errorstate_dump(prestate, CPL_FALSE,
                                        cpl_errorstate_dump_one);
                    // Recover from the error(s) (Reset to prestate))
                    cpl_errorstate_set(prestate);
                }
                moo_single *det_single =
                    moo_det_load_single(det, j, i, badpix_level);
                moo_loc_single *loc_single = moo_loc_get_single(loc, j, i);
                moo_psf_single *psf_single = NULL;
                if (strcmp(params->method, MOO_EXTRACT_METHOD_OPTIMAL) == 0) {
                    psf_single = moo_psf_get_single(master_flat, j, i);
                }
                moo_ext_single *ext_single = NULL;
                if (det_single != NULL && loc_single != NULL) {
                    double aperture = params->aperture[j + (i - 1) * 3];
                    if (aperture > 0) {
                        cpl_msg_info(
                            __func__,
                            "Extracting %d fibres in extension %s using "
                            "aperture %f",
                            nb_selected, moo_detector_get_extname(j, i),
                            aperture);
                    }
                    else {
                        cpl_msg_info(
                            __func__,
                            "Extracting %d fibres in extension %s using "
                            "aperture LOCALISATION edges",
                            nb_selected, moo_detector_get_extname(j, i));
                    }
                    ext_single =
                        _moo_extract_single(det_single, loc_single, psf_single,
                                            filename, health, indexes, indexext,
                                            aperture);
                    moo_try_check(moo_ext_single_compute_qc(ext_single,
                                                            colnames[j],
                                                            dersnr_colnames[j],
                                                            ext_fibres_table),
                                  " ");
                }
                moo_ext_add_single(result, ext_single, j, i);
            }
        }
        cpl_array_delete(indexes);
    }

    moo_ext_add_fibre_table(result, ext_fibres_table);

moo_try_cleanup:
    cpl_msg_indent_less();

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in extraction for DET %s LOC %s",
                      det->filename, loc->filename);
        moo_ext_delete(result);
        result = NULL;
    }
    return result;
}
/**@}*/
