/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_qc.h"
#include "moo_f2f.h"
#include "moo_fibres_table.h"
#include "moo_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_f2f F2F format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_f2f
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_f2f.h"
 * @endcode
 */

/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_f2f
  @return   1 newly allocated moo_f2f or NULL in case of an error

  The returned object must be deallocated using moo_f2f_delete().

 */
/*----------------------------------------------------------------------------*/
moo_f2f *
moo_f2f_new(void)
{
    moo_f2f *res = cpl_calloc(1, sizeof(moo_f2f));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief  Create a new moo_f2f and an empty structure in memory
  @param  nbrows Number of rows in F2F
  @param  fibre_table the Fibre table
  @return 1 newly allocated moo_f2f or NULL in case of an error

The returned object must be deallocated using moo_f2f_delete().

 */
/*----------------------------------------------------------------------------*/
moo_f2f *
moo_f2f_create(int nbrows, cpl_table *fibre_table)
{
    moo_f2f *res = moo_f2f_new();

    res->table = cpl_table_new(nbrows);
    cpl_table_new_column(res->table, MOO_F2F_TRANS_RI, MOO_F2F_TRANS_RI_TYPE);
    cpl_table_new_column(res->table, MOO_F2F_TRANS_YJ, MOO_F2F_TRANS_YJ_TYPE);
    cpl_table_new_column(res->table, MOO_F2F_TRANS_H, MOO_F2F_TRANS_H_TYPE);

    cpl_table_duplicate_column(res->table, MOO_F2F_INDEX, fibre_table,
                               MOO_FIBRES_TABLE_INDEX);
    cpl_table_duplicate_column(res->table, MOO_FIBRES_TABLE_SPECTRO,
                               fibre_table, MOO_FIBRES_TABLE_SPECTRO);
    cpl_table_duplicate_column(res->table, MOO_FIBRES_TABLE_INDEXEXT,
                               fibre_table, MOO_FIBRES_TABLE_INDEXEXT);
    cpl_table_new_column(res->table, MOO_FIBRES_TABLE_INDEXRBN, CPL_TYPE_INT);

    cpl_table *moons1 = moo_fibres_table_get_spectro_table(fibre_table, 1);
    int moons1_nrow = cpl_table_get_nrow(moons1);
    cpl_table_delete(moons1);

    const int *indexext =
        cpl_table_get_data_int_const(res->table, MOO_FIBRES_TABLE_INDEXEXT);
    const int *spectro =
        cpl_table_get_data_int_const(res->table, MOO_FIBRES_TABLE_SPECTRO);
    int *indexrbn =
        cpl_table_get_data_int(res->table, MOO_FIBRES_TABLE_INDEXRBN);
    for (int i = 0; i < nbrows; i++) {
        indexrbn[i] = indexext[i];
        cpl_table_set_float(res->table, MOO_F2F_TRANS_RI, i, 0.);
        cpl_table_set_float(res->table, MOO_F2F_TRANS_YJ, i, 0.);
        cpl_table_set_float(res->table, MOO_F2F_TRANS_H, i, 0.);

        if (MOO_FIBRES_TABLE_SPECTRO_2 == spectro[i] && indexrbn[i] != 0) {
            indexrbn[i] += moons1_nrow;
        }
    }
    res->primary_header = cpl_propertylist_new();

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Load a F2F table from a fits file
  @param f2f_frame the fits file to load
  @return a new allocated F2F

  The returned object must be deallocated using moo_f2f_delete().

 */
/*----------------------------------------------------------------------------*/
moo_f2f *
moo_f2f_load(const cpl_frame *f2f_frame)
{
    cpl_ensure(f2f_frame, CPL_ERROR_NULL_INPUT, NULL);
    cpl_errorstate prev_state = cpl_errorstate_get();

    const char *filename = cpl_frame_get_filename(f2f_frame);
    moo_f2f *res = moo_f2f_new();
    res->filename = filename;
    moo_try_check(res->table = cpl_table_load(filename, 1, 0), " ");
    moo_try_check(res->primary_header = cpl_propertylist_load(filename, 0),
                  " ");

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prev_state)) {
        moo_f2f_delete(res);
        res = NULL;
        cpl_errorstate_set(prev_state);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Order F2F by INDEXRBN (ASC)
  @param self F2F object

  @return error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_f2f_order_by_indexrbn(moo_f2f *self)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    cpl_propertylist *order = cpl_propertylist_new();
    cpl_propertylist_append_bool(order, MOO_FIBRES_TABLE_INDEXRBN, CPL_FALSE);
    cpl_table_sort(self->table, order);
    cpl_propertylist_delete(order);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Order F2F by SPECTRO,INDEX (ASC)
  @param self F2F object

  @return error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_f2f_order_by_index(moo_f2f *self)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    cpl_propertylist *order = cpl_propertylist_new();
    cpl_propertylist_append_bool(order, MOO_FIBRES_TABLE_SPECTRO, CPL_FALSE);
    cpl_propertylist_append_bool(order, MOO_FIBRES_TABLE_INDEX, CPL_FALSE);
    cpl_table_sort(self->table, order);
    cpl_propertylist_delete(order);

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Set transmission values in table
  @param self F2F object
  @param type detector type
  @param idxtab array of indexes to fill  
  @param values vector of values to fill
  @return   relevant cpl_error or CPl_ERROR_NONE

  The returned object must be deallocated using moo_f2f_delete().

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_f2f_set_trans(moo_f2f *self,
                  moo_detector_type type,
                  cpl_array *idxtab,
                  cpl_vector *values)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(idxtab != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(values != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;

    const char *colname = NULL;
    switch (type) {
        case MOO_TYPE_RI:
            colname = MOO_F2F_TRANS_RI;
            break;
        case MOO_TYPE_YJ:
            colname = MOO_F2F_TRANS_YJ;
            break;
        case MOO_TYPE_H:
            colname = MOO_F2F_TRANS_H;
            break;
    }

    int size = cpl_array_get_size(idxtab);

    for (int i = 0; i < size; i++) {
        cpl_size idx = cpl_array_get_cplsize(idxtab, i, NULL);
        float v = (float)cpl_vector_get(values, i);
        cpl_table_set(self->table, colname, idx, v);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get transmission column name from table
  @param self the F2F
  @param type detector type
  @return the name of the transmission column name

 */
/*----------------------------------------------------------------------------*/
const char *
moo_f2f_get_trans_colname(moo_f2f *self, moo_detector_type type)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const char *colname = NULL;
    switch (type) {
        case MOO_TYPE_RI:
            colname = MOO_F2F_TRANS_RI;
            break;
        case MOO_TYPE_YJ:
            colname = MOO_F2F_TRANS_YJ;
            break;
        case MOO_TYPE_H:
            colname = MOO_F2F_TRANS_H;
            break;
    }
    return colname;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Get transmission values from table
  @param self the F2F
  @param type detector type
  @param idxtab array of indexes to get
  @return values vector of transmission values

 */
/*----------------------------------------------------------------------------*/
cpl_vector *
moo_f2f_get_trans(moo_f2f *self, moo_detector_type type, cpl_array *idxtab)
{
    cpl_vector *res = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(idxtab != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int size = cpl_array_get_size(idxtab);
    const char *colname = moo_f2f_get_trans_colname(self, type);
    if (size > 0) {
        res = cpl_vector_new(size);
        for (int i = 0; i < size; i++) {
            cpl_size idx = cpl_array_get_cplsize(idxtab, i, NULL);
            double v = cpl_table_get(self->table, colname, idx, NULL);
            cpl_vector_set(res, i, v);
        }
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get transmission column from table
  @param self the F2F
  @param type detector type
  @return transmission column values

 */
/*----------------------------------------------------------------------------*/
float *
moo_f2f_get_trans_column(moo_f2f *self, moo_detector_type type)
{
    float *res = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const char *colname = NULL;
    switch (type) {
        case MOO_TYPE_RI:
            colname = MOO_F2F_TRANS_RI;
            break;
        case MOO_TYPE_YJ:
            colname = MOO_F2F_TRANS_YJ;
            break;
        case MOO_TYPE_H:
            colname = MOO_F2F_TRANS_H;
            break;
    }

    res = cpl_table_get_data_float(self->table, colname);

    return res;
}

cpl_error_code
moo_f2f_compute_qc(moo_f2f *self, int *ref, cpl_array *bad)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ref != NULL, CPL_ERROR_NULL_INPUT);

    cpl_table *selected = NULL;

    for (int i = 0; i < 2; i++) {
        moo_try_check(moo_qc_set_fibtrans_ref(self->primary_header, i + 1,
                                              ref[i]),
                      " ");
    }

    int size = cpl_array_get_size(bad);

    for (int i = 0; i < size; i++) {
        int idx = cpl_array_get_cplsize(bad, i, NULL);
        cpl_table_unselect_row(self->table, idx);
    }
    selected = cpl_table_extract_selected(self->table);

    const char *transname[] = { MOO_F2F_TRANS_RI, MOO_F2F_TRANS_YJ,
                                MOO_F2F_TRANS_H };

    for (int i = 0; i < 3; i++) {
        if (cpl_table_has_valid(selected, transname[i])) {
            double min = cpl_table_get_column_min(selected, transname[i]);
            double max = cpl_table_get_column_max(selected, transname[i]);
            double med = cpl_table_get_column_median(selected, transname[i]);

            moo_qc_set_fibtrans_min(self->primary_header, i, min);
            moo_qc_set_fibtrans_max(self->primary_header, i, max);
            moo_qc_set_fibtrans_med(self->primary_header, i, med);
        }
    }
moo_try_cleanup:
    cpl_table_delete(selected);

    return cpl_error_get_code();
}

cpl_error_code
moo_f2f_save(moo_f2f *self, const char *filename)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filename != NULL, CPL_ERROR_NULL_INPUT);

    cpl_propertylist *extheader = cpl_propertylist_new();
    cpl_propertylist_append_string(extheader, MOO_PFITS_EXTNAME,
                                   MOO_F2F_EXTNAME);
    self->filename = filename;
    cpl_table_save(self->table, self->primary_header, extheader, filename,
                   CPL_IO_CREATE);
    cpl_propertylist_delete(extheader);

    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_f2f
  @param    self    moo_f2f to delete
  @return   void

  If the moo_f2f @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_f2f_delete(moo_f2f *self)
{
    if (self != NULL) {
        if (self->table != NULL) {
            cpl_table_delete(self->table);
        }
        if (self->primary_header != NULL) {
            cpl_propertylist_delete(self->primary_header);
        }
        cpl_free(self);
    }
}

/**@}*/
