/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include <string.h>
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_utils.h"
#include "moo_fibres_table.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_fibres_table FIBRE TABLE format
 * @ingroup moo_data
 * This module provides functions to use FIBRE TABLE
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_map.h"
 * @endcode
 */

/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

int
moo_fibres_table_get_spectro(int num)
{
    if (num == 1) {
        return MOO_FIBRES_TABLE_SPECTRO_1;
    }
    else {
        return MOO_FIBRES_TABLE_SPECTRO_2;
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief    get the index of a spectro in the fibre table sort by index on the slit
  @param table the fibre table
  @param num the spectro number
  @return   the array index

 */
/*----------------------------------------------------------------------------*/

cpl_array *
moo_fibres_table_get_spectro_index(cpl_table *table, int num)
{
    cpl_ensure(table != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_propertylist *slitorder = cpl_propertylist_new();
    cpl_propertylist_append_bool(slitorder, MOO_FIBRES_TABLE_SPECTRO,
                                 CPL_FALSE);
    cpl_propertylist_append_bool(slitorder, MOO_FIBRES_TABLE_INDEX, CPL_FALSE);
    cpl_table_sort(table, slitorder);
    cpl_propertylist_delete(slitorder);

    const int spectro_name = moo_fibres_table_get_spectro(num);
    cpl_table_unselect_all(table);
    cpl_table_or_selected_int(table, MOO_FIBRES_TABLE_SPECTRO, CPL_EQUAL_TO,
                              spectro_name);

    cpl_array *res = cpl_table_where_selected(table);

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    get the index of a spectro in the fibre table sort by spetcro,indexext
  @param table the fibre table
  @param num the spectro number
  @return   the array index or NULL

 */
/*----------------------------------------------------------------------------*/

cpl_array *
moo_fibres_table_get_spectro_indexext(cpl_table *table, int num)
{
    cpl_array *res = NULL;
    cpl_ensure(table != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_errorstate prestate = cpl_errorstate_get();

    const int spectro_name = moo_fibres_table_get_spectro(num);

    cpl_propertylist *slitorder = NULL;

    slitorder = cpl_propertylist_new();
    cpl_propertylist_append_bool(slitorder, MOO_FIBRES_TABLE_SPECTRO,
                                 CPL_FALSE);

    moo_try_check(cpl_propertylist_append_bool(slitorder,
                                               MOO_FIBRES_TABLE_INDEXEXT,
                                               CPL_FALSE),
                  " ");
    moo_try_check(cpl_table_sort(table, slitorder), " ");

    moo_try_check(cpl_table_unselect_all(table), " ");
    moo_try_check(cpl_table_or_selected_int(table, MOO_FIBRES_TABLE_SPECTRO,
                                            CPL_EQUAL_TO, spectro_name),
                  " ");

    moo_try_check(res = cpl_table_where_selected(table), " ");

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_array_delete(res);
        res = NULL;
    }
    cpl_propertylist_delete(slitorder);
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief get the selection of a spectro in the fibre table
  @param table the fibre table
  @param num the spectro number
  @return   the table selection

 */
/*----------------------------------------------------------------------------*/

cpl_table *
moo_fibres_table_get_spectro_table(cpl_table *table, int num)
{
    cpl_ensure(table != NULL, CPL_ERROR_NULL_INPUT, NULL);
    const int spectro_name = moo_fibres_table_get_spectro(num);
    cpl_table_unselect_all(table);
    cpl_table_or_selected_int(table, MOO_FIBRES_TABLE_SPECTRO, CPL_EQUAL_TO,
                              spectro_name);
    cpl_table *selected = cpl_table_extract_selected(table);

    return selected;
}

/*----------------------------------------------------------------------------*/
/**
  @brief get the number of rows of a spectro in the fibre table
  @param table the fibre table
  @param num the spectro number
  @return   the number of rows or -1

 */
/*----------------------------------------------------------------------------*/
int
moo_fibres_table_get_spectro_nrow(cpl_table *table, int num)
{
    cpl_ensure(table != NULL, CPL_ERROR_NULL_INPUT, -1);
    const int spectro_name = moo_fibres_table_get_spectro(num);
    cpl_table_unselect_all(table);
    return cpl_table_or_selected_int(table, MOO_FIBRES_TABLE_SPECTRO,
                                     CPL_EQUAL_TO, spectro_name);
}

/*----------------------------------------------------------------------------*/
/**
  @brief add localise guess additional columns
  @param table the fibre table
  @return  error_code or CPl_ERROR_NONE

 Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_INVALID_TYPE  The specified type is not supported.
  - CPL_ERROR_ILLEGAL_OUTPUT A column with the same name already exists in table.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_add_locguess_cols(cpl_table *table)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;

    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_GOODPTSFRAC_RI,
                                  CPL_TYPE_FLOAT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_GOODPTSFRAC_YJ,
                                  CPL_TYPE_FLOAT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_GOODPTSFRAC_H,
                                  CPL_TYPE_FLOAT);
    int nrow = cpl_table_get_nrow(table);
    for (int i = 0; i < nrow; i++) {
        cpl_table_set_float(table, MOO_FIBRES_TABLE_GOODPTSFRAC_RI, i, 0.0);
        cpl_table_set_float(table, MOO_FIBRES_TABLE_GOODPTSFRAC_YJ, i, 0.0);
        cpl_table_set_float(table, MOO_FIBRES_TABLE_GOODPTSFRAC_H, i, 0.0);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief add extract additional columns
  @param table the fibre table
  @return  error_code or CPl_ERROR_NONE

 Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_INVALID_TYPE  The specified type is not supported.
  - CPL_ERROR_ILLEGAL_OUTPUT A column with the same name already exists in table.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_add_extract_cols(cpl_table *table)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;

    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MEDSNR_RI_EXT,
                                  CPL_TYPE_FLOAT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MEDSNR_YJ_EXT,
                                  CPL_TYPE_FLOAT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MEDSNR_H_EXT,
                                  CPL_TYPE_FLOAT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_DERSNR_RI_EXT,
                                  CPL_TYPE_FLOAT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_DERSNR_YJ_EXT,
                                  CPL_TYPE_FLOAT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_DERSNR_H_EXT,
                                  CPL_TYPE_FLOAT);
    status =
        cpl_table_new_column(table, MOO_FIBRES_TABLE_INDEXEXT, CPL_TYPE_INT);
    int nrow = cpl_table_get_nrow(table);
    for (int i = 0; i < nrow; i++) {
        cpl_table_set_int(table, MOO_FIBRES_TABLE_INDEXEXT, i, -1);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief add f2f additional columns
  @param table the fibre table
  @return  error_code or CPl_ERROR_NONE

 Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_INVALID_TYPE  The specified type is not supported.
  - CPL_ERROR_ILLEGAL_OUTPUT A column with the same name already exists in table.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_add_f2f_cols(cpl_table *table)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;

    status =
        cpl_table_new_column(table, MOO_FIBRES_TABLE_TRANS_RI, CPL_TYPE_FLOAT);
    status =
        cpl_table_new_column(table, MOO_FIBRES_TABLE_TRANS_YJ, CPL_TYPE_FLOAT);
    status =
        cpl_table_new_column(table, MOO_FIBRES_TABLE_TRANS_H, CPL_TYPE_FLOAT);

    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief add rebin additional columns
  @param table the fibre table
  @return  error_code or CPl_ERROR_NONE

 Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_INVALID_TYPE  The specified type is not supported.
  - CPL_ERROR_ILLEGAL_OUTPUT A column with the same name already exists in table.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_add_rebin_cols(cpl_table *table)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;

    status =
        cpl_table_new_column(table, MOO_FIBRES_TABLE_INDEXRBN, CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MEDSNR_RI_RBN,
                                  CPL_TYPE_FLOAT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MEDSNR_YJ_RBN,
                                  CPL_TYPE_FLOAT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MEDSNR_H_RBN,
                                  CPL_TYPE_FLOAT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MONOTONOUS_RI,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MONOTONOUS_YJ,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MONOTONOUS_H,
                                  CPL_TYPE_INT);
    int nrow = cpl_table_get_nrow(table);

    for (int i = 0; i < nrow; i++) {
        cpl_table_set_int(table, MOO_FIBRES_TABLE_INDEXRBN, i, 0);
        cpl_table_set_int(table, MOO_FIBRES_TABLE_MONOTONOUS_RI, i, 1);
        cpl_table_set_int(table, MOO_FIBRES_TABLE_MONOTONOUS_YJ, i, 1);
        cpl_table_set_int(table, MOO_FIBRES_TABLE_MONOTONOUS_H, i, 1);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief add nod additional columns
  @param table the fibre table
  @return  error_code or CPl_ERROR_NONE

 Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_INVALID_TYPE  The specified type is not supported.
  - CPL_ERROR_ILLEGAL_OUTPUT A column with the same name already exists in table.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_add_nod_cols(cpl_table *table)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;

    status =
        cpl_table_new_column(table, MOO_FIBRES_TABLE_TYPE_NOD, CPL_TYPE_STRING);
    int nrow = cpl_table_get_nrow(table);

    for (int i = 0; i < nrow; i++) {
        cpl_table_set_string(table, MOO_FIBRES_TABLE_TYPE_NOD, i, "");
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief erase extract additional columns
  @param table the fibre table
  @return  error_code or CPl_ERROR_NONE

 Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_INVALID_TYPE  The specified type is not supported.
  - CPL_ERROR_ILLEGAL_OUTPUT A column with the same name already exists in table.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_erase_extract_cols(cpl_table *table)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;

    status = cpl_table_erase_column(table, MOO_FIBRES_TABLE_MEDSNR_RI_EXT);
    status = cpl_table_erase_column(table, MOO_FIBRES_TABLE_MEDSNR_YJ_EXT);
    status = cpl_table_erase_column(table, MOO_FIBRES_TABLE_MEDSNR_H_EXT);
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief add wavecal additional columns
  @param table the fibre table
  @param guess the guess fibre table
  @param model the name of the guess model
  @return  error_code or CPl_ERROR_NONE

 Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_INVALID_TYPE  The specified type is not supported.
  - CPL_ERROR_ILLEGAL_OUTPUT A column with the same name already exists in table.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_add_wavecal_cols(cpl_table *table,
                                  cpl_table *guess,
                                  const char *model)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(guess != NULL, CPL_ERROR_NULL_INPUT);
    int nrow = cpl_table_get_nrow(table);
    int guess_nrow = cpl_table_get_nrow(guess);
    cpl_ensure_code(guess_nrow == nrow, CPL_ERROR_ILLEGAL_INPUT);
    cpl_error_code status = CPL_ERROR_NONE;
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MATCHLINE_RI,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MODELLINE_RI,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_DETECTLINE_RI,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_FAILEDFIT_RI,
                                  CPL_TYPE_INT);
    status =
        cpl_table_new_column(table, MOO_FIBRES_TABLE_FITLINE_RI, CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_REJECTLINE_RI,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_XDIFF_MEDIAN_RI,
                                  CPL_TYPE_DOUBLE);

    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MATCHLINE_YJ,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MODELLINE_YJ,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_DETECTLINE_YJ,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_FAILEDFIT_YJ,
                                  CPL_TYPE_INT);
    status =
        cpl_table_new_column(table, MOO_FIBRES_TABLE_FITLINE_YJ, CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_REJECTLINE_YJ,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_XDIFF_MEDIAN_YJ,
                                  CPL_TYPE_DOUBLE);

    status =
        cpl_table_new_column(table, MOO_FIBRES_TABLE_MATCHLINE_H, CPL_TYPE_INT);
    status =
        cpl_table_new_column(table, MOO_FIBRES_TABLE_MODELLINE_H, CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_DETECTLINE_H,
                                  CPL_TYPE_INT);
    status =
        cpl_table_new_column(table, MOO_FIBRES_TABLE_FAILEDFIT_H, CPL_TYPE_INT);
    status =
        cpl_table_new_column(table, MOO_FIBRES_TABLE_FITLINE_H, CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_REJECTLINE_H,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_XDIFF_MEDIAN_H,
                                  CPL_TYPE_DOUBLE);

    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MONOTONOUS_RI,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MONOTONOUS_YJ,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MONOTONOUS_H,
                                  CPL_TYPE_INT);
    if (model != NULL) {
        const char **names =
            cpl_table_get_data_string_const(table, MOO_FIBRES_TABLE_FIBRE);
        const char **guess_names =
            cpl_table_get_data_string_const(guess, MOO_FIBRES_TABLE_FIBRE);

        for (int i = 0; i < nrow; i++) {
            const char *ext_name = names[i];
            cpl_table_set_int(table, MOO_FIBRES_TABLE_MONOTONOUS_RI, i, -1);
            cpl_table_set_int(table, MOO_FIBRES_TABLE_MONOTONOUS_YJ, i, -1);
            cpl_table_set_int(table, MOO_FIBRES_TABLE_MONOTONOUS_H, i, -1);
            for (int j = 0; j < guess_nrow; j++) {
                const char *guess_name = guess_names[j];
                if (strcmp(ext_name, guess_name) == 0) {
                    for (int k = 0; k < 3; k++) {
                        int nmatched = 0;
                        int nrej = 0;
                        moo_try_check(nmatched =
                                          moo_fibres_table_get_matchline(guess,
                                                                         k, j),
                                      " ");
                        moo_try_check(nrej =
                                          moo_fibres_table_get_rejectline(guess,
                                                                          k, j),
                                      " ");
                        moo_fibres_table_set_matchline(table, k, i, nmatched);
                        moo_fibres_table_set_modelline(table, k, i,
                                                       nmatched - nrej);
                    }
                    break;
                }
            }
        }
    }
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief add wavecal additional columns
  @param table the fibre table
  @return  error_code or CPl_ERROR_NONE

 Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_INVALID_TYPE  The specified type is not supported.
  - CPL_ERROR_ILLEGAL_OUTPUT A column with the same name already exists in table.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_add_wavecalguess_cols(cpl_table *table)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;

    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_DETECTLINE_RI,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_FAILEDFIT_RI,
                                  CPL_TYPE_INT);
    status =
        cpl_table_new_column(table, MOO_FIBRES_TABLE_FITLINE_RI, CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MATCHLINE_RI,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_REJECTLINE_RI,
                                  CPL_TYPE_INT);

    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_DETECTLINE_YJ,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_FAILEDFIT_YJ,
                                  CPL_TYPE_INT);
    status =
        cpl_table_new_column(table, MOO_FIBRES_TABLE_FITLINE_YJ, CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MATCHLINE_YJ,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_REJECTLINE_YJ,
                                  CPL_TYPE_INT);

    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_DETECTLINE_H,
                                  CPL_TYPE_INT);
    status =
        cpl_table_new_column(table, MOO_FIBRES_TABLE_FAILEDFIT_H, CPL_TYPE_INT);
    status =
        cpl_table_new_column(table, MOO_FIBRES_TABLE_FITLINE_H, CPL_TYPE_INT);
    status =
        cpl_table_new_column(table, MOO_FIBRES_TABLE_MATCHLINE_H, CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_REJECTLINE_H,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MONOTONOUS_RI,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MONOTONOUS_YJ,
                                  CPL_TYPE_INT);
    status = cpl_table_new_column(table, MOO_FIBRES_TABLE_MONOTONOUS_H,
                                  CPL_TYPE_INT);

    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief get the data of median_snr for the given detector type
  @param table the fibre table
  @param type the detector type
  @return   the column data

 */
/*----------------------------------------------------------------------------*/
float *
moo_fibres_table_get_median_snr(cpl_table *table, moo_detector_type type)
{
    cpl_ensure(table != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const char *res[] = { MOO_FIBRES_TABLE_MEDSNR_RI_EXT,
                          MOO_FIBRES_TABLE_MEDSNR_YJ_EXT,
                          MOO_FIBRES_TABLE_MEDSNR_H_EXT };

    return cpl_table_get_data_float(table, res[type]);
}

/*----------------------------------------------------------------------------*/
/**
  @brief set the number od detectline for the given detector type
  @param table the fibre table
  @param type the detector type
  @param row the row
  @param value the number of lines
  @return  error code or CPL_ERROR_NONE;

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_set_detectline(cpl_table *table,
                                moo_detector_type type,
                                cpl_size row,
                                int value)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);

    const char *res[] = { MOO_FIBRES_TABLE_DETECTLINE_RI,
                          MOO_FIBRES_TABLE_DETECTLINE_YJ,
                          MOO_FIBRES_TABLE_DETECTLINE_H };

    cpl_error_code status = CPL_ERROR_NONE;
    int *data = cpl_table_get_data_int(table, res[type]);
    data[row] = value;
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief set the number od failedfit lines for the given detector type
  @param table the fibre table
  @param type the detector type
  @param row the row
  @param value the number of lines
  @return  error code or CPL_ERROR_NONE;

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_set_failedfit(cpl_table *table,
                               moo_detector_type type,
                               cpl_size row,
                               int value)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);

    const char *res[] = { MOO_FIBRES_TABLE_FAILEDFIT_RI,
                          MOO_FIBRES_TABLE_FAILEDFIT_YJ,
                          MOO_FIBRES_TABLE_FAILEDFIT_H };
    cpl_error_code status = CPL_ERROR_NONE;
    int *data = cpl_table_get_data_int(table, res[type]);
    data[row] = value;
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief set the number of fit lines for the given detector type
  @param table the fibre table
  @param type the detector type
  @param row the row
  @param value the number of lines
  @return  error code or CPL_ERROR_NONE;

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_set_fitline(cpl_table *table,
                             moo_detector_type type,
                             cpl_size row,
                             int value)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);

    const char *res[] = { MOO_FIBRES_TABLE_FITLINE_RI,
                          MOO_FIBRES_TABLE_FITLINE_YJ,
                          MOO_FIBRES_TABLE_FITLINE_H };

    cpl_error_code status = CPL_ERROR_NONE;
    int *data = cpl_table_get_data_int(table, res[type]);
    data[row] = value;
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief set the number of match lines for the given detector type
  @param table the fibre table
  @param type the detector type
  @param row the row
  @param value the number of lines
  @return  error code or CPL_ERROR_NONE;

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_set_matchline(cpl_table *table,
                               moo_detector_type type,
                               cpl_size row,
                               int value)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);

    const char *res[] = { MOO_FIBRES_TABLE_MATCHLINE_RI,
                          MOO_FIBRES_TABLE_MATCHLINE_YJ,
                          MOO_FIBRES_TABLE_MATCHLINE_H };
    cpl_error_code status = CPL_ERROR_NONE;
    status = cpl_table_set_int(table, res[type], row, value);
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief set the number of model lines for the given detector type
  @param table the fibre table
  @param type the detector type
  @param row the row
  @param value the number of lines
  @return  error code or CPL_ERROR_NONE;

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_set_modelline(cpl_table *table,
                               moo_detector_type type,
                               cpl_size row,
                               int value)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);

    const char *res[] = { MOO_FIBRES_TABLE_MODELLINE_RI,
                          MOO_FIBRES_TABLE_MODELLINE_YJ,
                          MOO_FIBRES_TABLE_MODELLINE_H };

    return cpl_table_set_int(table, res[type], row, value);
}

/*----------------------------------------------------------------------------*/
/**
  @brief set the number of rejected lines for the given detector type
  @param table the fibre table
  @param type the detector type
  @param row the row
  @param value the number of lines
  @return  error code or CPL_ERROR_NONE;

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_set_rejectline(cpl_table *table,
                                moo_detector_type type,
                                cpl_size row,
                                int value)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);

    const char *res[] = { MOO_FIBRES_TABLE_REJECTLINE_RI,
                          MOO_FIBRES_TABLE_REJECTLINE_YJ,
                          MOO_FIBRES_TABLE_REJECTLINE_H };

    cpl_error_code status = CPL_ERROR_NONE;
    status = cpl_table_set_int(table, res[type], row, value);
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief set the xdiff for the given fibre and detector type
  @param table the fibre table
  @param type the detector type
  @param row the row
  @param value the number of lines
  @return  error code or CPL_ERROR_NONE;

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_set_xdiff(cpl_table *table,
                           moo_detector_type type,
                           cpl_size row,
                           double value)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);

    const char *res[] = { MOO_FIBRES_TABLE_XDIFF_MEDIAN_RI,
                          MOO_FIBRES_TABLE_XDIFF_MEDIAN_YJ,
                          MOO_FIBRES_TABLE_XDIFF_MEDIAN_H };

    return cpl_table_set_double(table, res[type], row, value);
}

/*----------------------------------------------------------------------------*/
/**
  @brief Get the number of matched lines for the given detector type
  @param table the fibre table
  @param type the detector type
  @param row the row
  @return  number of lines

 */
/*----------------------------------------------------------------------------*/
int
moo_fibres_table_get_matchline(cpl_table *table,
                               moo_detector_type type,
                               cpl_size row)
{
    cpl_ensure(table != NULL, CPL_ERROR_NULL_INPUT, 0);
    int rej;

    const char *res[] = { MOO_FIBRES_TABLE_MATCHLINE_RI,
                          MOO_FIBRES_TABLE_MATCHLINE_YJ,
                          MOO_FIBRES_TABLE_MATCHLINE_H };

    return cpl_table_get_int(table, res[type], row, &rej);
}

/*----------------------------------------------------------------------------*/
/**
  @brief Get the number of rejected lines for the given detector type
  @param table the fibre table
  @param type the detector type
  @param row the row
  @return  number of lines

 */
/*----------------------------------------------------------------------------*/
int
moo_fibres_table_get_rejectline(cpl_table *table,
                                moo_detector_type type,
                                cpl_size row)
{
    cpl_ensure(table != NULL, CPL_ERROR_NULL_INPUT, 0);
    int rej;

    const char *res[] = { MOO_FIBRES_TABLE_REJECTLINE_RI,
                          MOO_FIBRES_TABLE_REJECTLINE_YJ,
                          MOO_FIBRES_TABLE_REJECTLINE_H };

    return cpl_table_get_int(table, res[type], row, &rej);
}

/*----------------------------------------------------------------------------*/
/**
  @brief Get xdiff for the given detector type
  @param table the fibre table
  @param type the detector type
  @param row the row
  @return the xdiff

 */
/*----------------------------------------------------------------------------*/
double
moo_fibres_table_get_xdiff(cpl_table *table,
                           moo_detector_type type,
                           cpl_size row)
{
    cpl_ensure(table != NULL, CPL_ERROR_NULL_INPUT, 0);
    int rej;

    const char *res[] = { MOO_FIBRES_TABLE_XDIFF_MEDIAN_RI,
                          MOO_FIBRES_TABLE_XDIFF_MEDIAN_YJ,
                          MOO_FIBRES_TABLE_XDIFF_MEDIAN_H };

    return cpl_table_get_double(table, res[type], row, &rej);
}

/*----------------------------------------------------------------------------*/
/**
  @brief Filter bad health data in fibre table
  @param table the fibre table
  @return   the number of good lines

 */
/*----------------------------------------------------------------------------*/
int
moo_fibres_table_filter_health(cpl_table *table)
{
    cpl_ensure(table != NULL, CPL_ERROR_NULL_INPUT, 0);
    return cpl_table_and_selected_int(table, MOO_FIBRES_TABLE_HEALTH,
                                      CPL_NOT_EQUAL_TO,
                                      MOO_FIBRES_TABLE_HEALTH_BROKEN);
}

/*----------------------------------------------------------------------------*/
/**
  @brief Load a fibre table from a file
  @param frame the frame conataining fibre table
  @return the fibre table

 */
/*----------------------------------------------------------------------------*/
cpl_table *
moo_fibre_table_load(cpl_frame *frame)
{
    cpl_table *res = NULL;
    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const char *filename = cpl_frame_get_filename(frame);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_size qnum = cpl_fits_find_extension(filename, MOO_FIBRES_TABLE_EXTNAME);
    if (qnum > 0) {
        res = cpl_table_load(filename, qnum, 0);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    get the object name of the STD type in the fibre table
  @param table the fibre table
  @return   the name or NULL if not find

 */
/*----------------------------------------------------------------------------*/

const char *
moo_fibres_table_get_std_name(cpl_table *table)
{
    const char *res = NULL;
    cpl_ensure(table != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_table_unselect_all(table);
    cpl_table_or_selected_string(table, MOO_FIBRES_TABLE_TYPE, CPL_EQUAL_TO,
                                 MOO_FIBRES_TABLE_TYPE_STD_FLUX);
    cpl_table_or_selected_string(table, MOO_FIBRES_TABLE_TYPE, CPL_EQUAL_TO,
                                 MOO_FIBRES_TABLE_TYPE_STD_TELL);

    cpl_array *indexes = cpl_table_where_selected(table);

    const char **names =
        cpl_table_get_data_string_const(table, MOO_FIBRES_TABLE_TARGNAME);
    int size = cpl_array_get_size(indexes);
    if (size == 1) {
        int idx = cpl_array_get_cplsize(indexes, 0, NULL);
        res = names[idx];
    }
    else {
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                              "Wrong number (%d) of STD found", size);
    }
    cpl_array_delete(indexes);
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    get from indexrbn the corresponding indexext and num
  @param table the fibre table
  @param indexrbn the rbn index
  @param indexext the corresponding indexext or -1
  @param num the corresponding num or -1
  @return the cpl error code

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_rbn_to_ext(cpl_table *table,
                            int indexrbn,
                            int *indexext,
                            int *num)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(indexext != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(num != NULL, CPL_ERROR_NULL_INPUT);
    cpl_table_select_all(table);
    int size = cpl_table_and_selected_int(table, MOO_FIBRES_TABLE_INDEXRBN,
                                          CPL_EQUAL_TO, indexrbn);

    if (size == 1) {
        cpl_array *sel = cpl_table_where_selected(table);
        int idx = cpl_array_get_cplsize(sel, 0, NULL);
        cpl_array_delete(sel);
        *indexext =
            cpl_table_get_int(table, MOO_FIBRES_TABLE_INDEXEXT, idx, NULL);
        const int str =
            cpl_table_get_int(table, MOO_FIBRES_TABLE_SPECTRO, idx, NULL);
        if (str == MOO_FIBRES_TABLE_SPECTRO_1) {
            *num = 1;
        }
        else {
            *num = 2;
        }
    }
    else {
        *num = -1;
        *indexext = -1;
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Order fibres table by INDEXRBN (ASC)
  @param table  fibres table

  @return error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_by_indexrbn(cpl_table *table)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_propertylist *order = cpl_propertylist_new();
    cpl_propertylist_append_bool(order, MOO_FIBRES_TABLE_INDEXRBN, CPL_FALSE);
    cpl_table_sort(table, order);
    cpl_propertylist_delete(order);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Order fibres table by INDEX (ASC)
  @param table  fibres table

  @return error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fibres_table_by_index(cpl_table *table)
{
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_propertylist *order = cpl_propertylist_new();
    cpl_propertylist_append_bool(order, MOO_FIBRES_TABLE_SPECTRO, CPL_FALSE);
    cpl_propertylist_append_bool(order, MOO_FIBRES_TABLE_INDEX, CPL_FALSE);
    cpl_table_sort(table, order);
    cpl_propertylist_delete(order);

    return CPL_ERROR_NONE;
}

/**@}*/
