/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_utils.h"
#include "moo_pfits.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_fits FITS utilities
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
/**
  @brief Write a mask as extension in FITS file
  @param mask mask to write or NULL if extension have null mask
  @param filename name of the file to write
  @param name name prefix of extension or NULL
  @param detectorname name of extension detector  
  @param header given extension header or NULL
  @return relevant error code
  
 - - -  
 _Error code_:  
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer
  
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fits_write_extension_mask(cpl_mask *mask,
                              const char *filename,
                              const char *name,
                              const char *detectorname,
                              cpl_propertylist *header)
{
    cpl_ensure_code(filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(detectorname != NULL, CPL_ERROR_NULL_INPUT);

    char *extname;
    if (name == NULL) {
        extname = cpl_sprintf("%s", detectorname);
    }
    else {
        extname = cpl_sprintf("%s_%s", name, detectorname);
    }

    cpl_propertylist *h = NULL;

    if (header != NULL) {
        h = cpl_propertylist_duplicate(header);
    }
    else {
        h = cpl_propertylist_new();
    }

    if (cpl_propertylist_has(h, MOO_PFITS_EXTNAME) == 0) {
        moo_try_check(cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                                     extname),
                      " ");
    }

    if (mask != NULL) {
        moo_try_check(cpl_mask_save(mask, filename, h, CPL_IO_EXTEND), " ");
    }
    else {
        moo_try_check(cpl_propertylist_save(h, filename, CPL_IO_EXTEND), " ");
    }
moo_try_cleanup:
    cpl_propertylist_delete(h);
    cpl_free(extname);
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief Write an image as extension in FITS file
  @param image image to write or NULL if extension have null image
  @param filename name of the file to write
  @param name name prefix of extension or NULL
  @param detectorname name of extension detector
  @param type type of image to write
  @param header given extension header or NULL
  @return relevant error code
  
 - - -  
 _Error code_:  
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer
  
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fits_write_extension_image(cpl_image *image,
                               const char *filename,
                               const char *name,
                               const char *detectorname,
                               cpl_type type,
                               cpl_propertylist *header)
{
    cpl_ensure_code(filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(detectorname != NULL, CPL_ERROR_NULL_INPUT);

    char *extname;
    if (name == NULL) {
        extname = cpl_sprintf("%s", detectorname);
    }
    else {
        extname = cpl_sprintf("%s_%s", name, detectorname);
    }
    cpl_propertylist *h = NULL;

    if (header != NULL) {
        h = cpl_propertylist_duplicate(header);
    }
    else {
        h = cpl_propertylist_new();
    }

    if (cpl_propertylist_has(h, MOO_PFITS_EXTNAME) == 0) {
        moo_try_check(cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                                     extname),
                      " ");
    }

    if (image != NULL) {
        moo_try_check(cpl_image_save(image, filename, type, h, CPL_IO_EXTEND),
                      " ");
    }
    else {
        moo_try_check(cpl_propertylist_save(h, filename, CPL_IO_EXTEND), " ");
    }

moo_try_cleanup:

    cpl_propertylist_delete(h);
    cpl_free(extname);
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief Write an image as extension in FITS file
  @param table table to write or NULL if extension have null table
  @param filename name of the file to write
  @param name name prefix of extension or NULL
  @param detectorname name of extension detector  
  @param header given extension header or NULL
  @return relevant error code
  
 - - -  
 _Error code_:  
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer
  
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fits_write_extension_table(cpl_table *table,
                               const char *filename,
                               const char *name,
                               const char *detectorname,
                               cpl_propertylist *header)
{
    cpl_ensure_code(filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(detectorname != NULL, CPL_ERROR_NULL_INPUT);

    char *extname;
    if (name == NULL) {
        extname = cpl_sprintf("%s", detectorname);
    }
    else {
        extname = cpl_sprintf("%s_%s", name, detectorname);
    }

    cpl_propertylist *h = NULL;

    if (header != NULL) {
        h = cpl_propertylist_duplicate(header);
    }
    else {
        h = cpl_propertylist_new();
    }

    if (cpl_propertylist_has(h, MOO_PFITS_EXTNAME) == 0) {
        moo_try_check(cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                                     extname),
                      " ");
    }

    if (table != NULL) {
        moo_try_check(cpl_table_save(table, NULL, h, filename, CPL_IO_EXTEND),
                      " ");
    }
    else {
        moo_try_check(cpl_propertylist_save(h, filename, CPL_IO_EXTEND), " ");
    }
moo_try_cleanup:
    cpl_propertylist_delete(h);
    cpl_free(extname);
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief Write a cube as extension in FITS file
  @param cube cube to write or NULL if extension have null image
  @param filename name of the file to write
  @param name name prefix of extension or NULL
  @param detectorname name of extension detector
  @param type type of image to write
  @param header given extension header or NULL
  @return relevant error code
  
  - - -  
 _Error code_:  
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fits_write_extension_cube(cpl_imagelist *cube,
                              const char *filename,
                              const char *name,
                              const char *detectorname,
                              cpl_type type,
                              cpl_propertylist *header)
{
    cpl_ensure_code(filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(detectorname != NULL, CPL_ERROR_NULL_INPUT);

    char *extname;
    if (name == NULL) {
        extname = cpl_sprintf("%s", detectorname);
    }
    else {
        extname = cpl_sprintf("%s_%s", name, detectorname);
    }
    cpl_propertylist *h = NULL;

    if (header != NULL) {
        h = cpl_propertylist_duplicate(header);
    }
    else {
        h = cpl_propertylist_new();
    }

    if (cpl_propertylist_has(h, MOO_PFITS_EXTNAME) == 0) {
        moo_try_check(cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                                     extname),
                      " ");
    }

    if (cube != NULL) {
        moo_try_check(cpl_imagelist_save(cube, filename, type, h,
                                         CPL_IO_EXTEND),
                      " ");
    }
    else {
        moo_try_check(cpl_propertylist_save(h, filename, CPL_IO_EXTEND), " ");
    }
moo_try_cleanup:
    cpl_propertylist_delete(h);
    cpl_free(extname);
    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
  @brief Load an image from FITS file
  @param filename given file name
  @param name name prefix of extension or NULL
  @param detectorname name of extension detector
  @param type type of image to load
  @return image or NULL
  
  - - -  
 _Error code_:  
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_fits_load_extension_image(const char *filename,
                              const char *name,
                              const char *detectorname,
                              cpl_type type)
{
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(detectorname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_image *res = NULL;

    char *extname;
    if (name == NULL) {
        extname = cpl_sprintf("%s", detectorname);
    }
    else {
        extname = cpl_sprintf("%s_%s", name, detectorname);
    }
    cpl_size extnum = cpl_fits_find_extension(filename, extname);
    if (extnum > 0) {
        moo_try_check(res = cpl_image_load(filename, type, 0, extnum),
                      "Can't load image extension %" CPL_SIZE_FORMAT
                      " in file %s",
                      extnum, filename);
    }
    else {
        cpl_msg_debug("moo_fits", "Couldn't find extension %s in %s", extname,
                      filename);
    }
moo_try_cleanup:
    cpl_free(extname);
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Load a table from FITS file
  @param filename given file name
  @param name name prefix of extension or NULL
  @param detectorname name of extension detector
  @return table or NULL
  
  - - -  
 _Error code_:  
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
cpl_table *
moo_fits_load_extension_table(const char *filename,
                              const char *name,
                              const char *detectorname)
{
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(detectorname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_table *res = NULL;

    char *extname;
    if (name == NULL) {
        extname = cpl_sprintf("%s", detectorname);
    }
    else {
        extname = cpl_sprintf("%s_%s", name, detectorname);
    }
    cpl_size extnum = cpl_fits_find_extension(filename, extname);
    if (extnum > 0) {
        moo_try_check(res = cpl_table_load(filename, extnum, 0),
                      "Can't load table extension %" CPL_SIZE_FORMAT
                      " in file %s",
                      extnum, filename);
    }
    else {
        cpl_msg_debug("moo_fits", "Couldn't find extension %s in %s", extname,
                      filename);
    }
moo_try_cleanup:
    cpl_free(extname);
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Load a mask from FITS file
  @param filename given file name
  @param name name prefix of extension or NULL
  @param detectorname name of extension detector
  @return mask or NULL
  
  - - -  
 _Error code_:  
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
cpl_mask *
moo_fits_load_extension_mask(const char *filename,
                             const char *name,
                             const char *detectorname)
{
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(detectorname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_mask *res = NULL;

    char *extname;
    if (name == NULL) {
        extname = cpl_sprintf("%s", detectorname);
    }
    else {
        extname = cpl_sprintf("%s_%s", name, detectorname);
    }
    cpl_size extnum = cpl_fits_find_extension(filename, extname);
    if (extnum > 0) {
        moo_try_check(res = cpl_mask_load(filename, 0, extnum),
                      "Can't load mask extension %" CPL_SIZE_FORMAT
                      " in file %s",
                      extnum, filename);
    }
    else {
        cpl_msg_debug("moo_fits", "Couldn't find extension %s in %s", extname,
                      filename);
    }

moo_try_cleanup:
    cpl_free(extname);
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Load a cube from FITS file
  @param filename given file name
  @param name name prefix of extension or NULL
  @param detectorname name of extension detector
  @param type type of image to load
  @return cube or NULL
  
  - - -  
 _Error code_:  
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
cpl_imagelist *
moo_fits_load_extension_cube(const char *filename,
                             const char *name,
                             const char *detectorname,
                             cpl_type type)
{
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(detectorname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_imagelist *res = NULL;

    char *extname;
    if (name == NULL) {
        extname = cpl_sprintf("%s", detectorname);
    }
    else {
        extname = cpl_sprintf("%s_%s", name, detectorname);
    }
    cpl_size extnum = cpl_fits_find_extension(filename, extname);
    if (extnum > 0) {
        moo_try_check(res = cpl_imagelist_load(filename, type, extnum),
                      "Can't load image extension %" CPL_SIZE_FORMAT
                      " in file %s",
                      extnum, filename);
    }
    else {
        cpl_msg_debug("moo_fits", "Couldn't find extension %s in %s", extname,
                      filename);
    }
moo_try_cleanup:
    cpl_free(extname);
    return res;
}
cpl_propertylist *
moo_fits_load_extension_header(const char *filename,
                               const char *name,
                               const char *detectorname)
{
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(detectorname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_propertylist *res = NULL;

    char *extname;
    if (name == NULL) {
        extname = cpl_sprintf("%s", detectorname);
    }
    else {
        extname = cpl_sprintf("%s_%s", name, detectorname);
    }
    cpl_size extnum = cpl_fits_find_extension(filename, extname);
    if (extnum > 0) {
        moo_try_check(res = cpl_propertylist_load(filename, extnum),
                      "Can't load header extension %" CPL_SIZE_FORMAT
                      " in file %s",
                      extnum, filename);
    }
    else {
        cpl_msg_debug("moo_fits", "Couldn't find extension %s in %s", extname,
                      filename);
    }
moo_try_cleanup:
    cpl_free(extname);
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Create a new fits file with empty propertylist
  @param filename given file name      
  @return error_code
  
  - - -  
 _Error code_:  
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_fits_create(const char *filename)
{
    cpl_ensure_code(filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_propertylist *h = cpl_propertylist_new();
    moo_try_check(cpl_propertylist_save(h, filename, CPL_IO_CREATE), " ");

moo_try_cleanup:
    cpl_propertylist_delete(h);
    return cpl_error_get_code();
}
/**@}*/
