/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_utils.h"
#include "moo_dfs.h"
#include "moo_pfits.h"
#include "moo_qc.h"
#include "moo_params.h"
#include "moo_badpix.h"
#include "moo_single.h"
#include "moo_det.h"
#include "moo_flat_shift_compute.h"
#include "moo_loc_single.h"
#include "moo_detector.h"
#include "moo_fibres_table.h"

#ifdef _OPENMP
#include <omp.h>
#endif

/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/


/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/


static cpl_error_code
_moo_flat_single_shift_compute(moo_single *det, moo_loc_single *loc)
{
    char *tname = NULL;
    cpl_table *grid_table = NULL;
    cpl_bivector *points = NULL;

    cpl_ensure_code(det != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(loc != NULL, CPL_ERROR_NULL_INPUT);

    cpl_image *fluxes = moo_single_get_data(det);
    int nx = cpl_image_get_size_x(fluxes);
    int xcenter = nx / 2;
    int xwidth = 100;
    int ywidth = 3;
    double ystep = 0.1;
    cpl_image *fcentroids = moo_loc_single_get_f_centroids(loc);
    int nb_fibres = cpl_image_get_size_y(fcentroids);
    int grid_size = (int)(ywidth * 2) / ystep + 1;

    grid_table = cpl_table_new(grid_size);
    moo_try_check(cpl_table_new_column(grid_table,
                                       MOO_FLAT_SHIFT_GRID_Y_COLNAME,
                                       CPL_TYPE_FLOAT),
                  " ");
    moo_try_check(cpl_table_new_column(grid_table,
                                       MOO_FLAT_SHIFT_GRID_FLUX_COLNAME,
                                       CPL_TYPE_FLOAT),
                  " ");
    moo_try_check(cpl_table_new_column(grid_table,
                                       MOO_FLAT_SHIFT_GRID_NB_COLNAME,
                                       CPL_TYPE_INT),
                  " ");
    float *grid_y =
        cpl_table_get_data_float(grid_table, MOO_FLAT_SHIFT_GRID_Y_COLNAME);
    float *grid_data =
        cpl_table_get_data_float(grid_table, MOO_FLAT_SHIFT_GRID_FLUX_COLNAME);
    int *grid_nb =
        cpl_table_get_data_int(grid_table, MOO_FLAT_SHIFT_GRID_NB_COLNAME);
    for (int i = 0; i < grid_size; i++) {
        double p = i * ystep - ywidth;
        cpl_table_set_float(grid_table, MOO_FLAT_SHIFT_GRID_Y_COLNAME, i, p);
        cpl_table_set_float(grid_table, MOO_FLAT_SHIFT_GRID_FLUX_COLNAME, i, 0);
    }

    for (int x = xcenter - xwidth; x <= xcenter + xwidth; x++) {
        for (int j = 1; j <= nb_fibres; j++) {
            int rej;
            double centroid = cpl_image_get(fcentroids, x, j, &rej);
            if (!isnan(centroid)) {
                int icentroid = (int)round(centroid);
                for (int y = icentroid - ywidth; y <= icentroid + ywidth; y++) {
                    double flux = cpl_image_get(fluxes, x, y, &rej);
                    double ypos = y - centroid;
                    int igrid = (int)((ypos + ywidth) / ystep);
                    if (!isnan(flux) && igrid >= 0 && igrid < grid_size) {
                        grid_data[igrid] += flux;
                        grid_nb[igrid]++;
                    }
                }
            }
        }
    }
    points = cpl_bivector_new(grid_size);
    double *vx_data = cpl_bivector_get_x_data(points);
    double *vy_data = cpl_bivector_get_y_data(points);
    for (int i = 0; i < grid_size; i++) {
        vx_data[i] = grid_y[i];
        vy_data[i] = grid_data[i] / (double)grid_nb[i];
    }
    double center, width, background, area;
    cpl_errorstate prev_state = cpl_errorstate_get();
    cpl_fit_mode fit_pars = CPL_FIT_CENTROID | CPL_FIT_STDEV | CPL_FIT_AREA;

    moo_gaussian_fit(points, fit_pars, &center, &width, &background, &area);
    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_errorstate_dump(prev_state, CPL_FALSE, cpl_errorstate_dump_one);
        cpl_errorstate_set(prev_state);
    }
    cpl_msg_indent_more();
    cpl_msg_info("test", "fit centroid %f stdev %f", center, width);

#if MOO_DEBUG_FLAT_SHIFT_COMPUTE
    tname = cpl_sprintf("GRID_%s.fits", det->extname);
    cpl_table_save(grid_table, NULL, NULL, tname, CPL_IO_CREATE);
#endif
    cpl_msg_indent_less();
moo_try_cleanup:
    cpl_free(tname);
    cpl_table_delete(grid_table);
    cpl_image_delete(fluxes);
    cpl_bivector_delete(points);
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Compute the shift between FF_TRACE and FLAT
  @param    det _DET_ frame to analyse
  @param    ff_trace the given FF_TRACE

_Bad pixels flags_:
  - None

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if _DET_ has not a fibre table
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_flat_shift_compute(moo_det *det, moo_loc *ff_trace)
{
    cpl_ensure_code(det != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ff_trace != NULL, CPL_ERROR_NULL_INPUT);

    int badpix_level =
        MOO_BADPIX_COSMETIC | MOO_BADPIX_HOT | MOO_BADPIX_OUTSIDE_DATA_RANGE;

    for (int i = 1; i <= 2; i++) {
        for (int j = 0; j < 3; j++) {
            moo_single *det_single = NULL;
            moo_loc_single *loc_single = NULL;
            moo_try_check(det_single =
                              moo_det_load_single(det, j, i, badpix_level),
                          " ");
            moo_try_check(loc_single = moo_loc_get_single(ff_trace, j, i), " ");
            if (det_single != NULL && loc_single != NULL) {
                cpl_msg_info(__func__, "Compute flat shift in extension %s",
                             moo_detector_get_extname(j, i));
                _moo_flat_single_shift_compute(det_single, loc_single);
            }
        }
    }
moo_try_cleanup:
    return CPL_ERROR_NONE;
}
/**@}*/
