/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include <ctype.h>
#include <string.h>
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_fibres_table.h"
#include "moo_flx.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_flx FLX format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_flx
 * 
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_flx.h"
 * @endcode
 */

 
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_flx  
  @return   1 newly allocated moo_flx or NULL in case of an error

  The returned object must be deallocated using moo_flx_delete().  
 
 */
/*----------------------------------------------------------------------------*/
moo_flx* moo_flx_new(void)
{
    moo_flx* res = cpl_calloc(1,sizeof(moo_flx));        
    return res;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Load a FLX frame and create a moo_flx
  @return   1 newly allocated moo_flx or NULL in case of an error

  The returned object must be deallocated using moo_flx_delete().  
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL  
 */
/*----------------------------------------------------------------------------*/
moo_flx* moo_flx_load(const cpl_frame* frame)
{
    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    
    const char* filename = cpl_frame_get_filename(frame);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT,
               NULL);    
    cpl_errorstate prev_state = cpl_errorstate_get();
    
    moo_flx* res = moo_flx_new();     
    res->filename = cpl_strdup(filename);
    res->primary_header = cpl_propertylist_load( filename,0);
    
    cpl_size qnum = cpl_fits_find_extension(filename, 
                MOO_FLX_CATALOG_EXTNAME);
    if ( qnum > 0){
        res->catalog = cpl_table_load(res->filename,qnum,0);
    }    
    
    if ( !cpl_errorstate_is_equal(prev_state)){
        cpl_msg_info("moo_flx","flx load %d",cpl_error_get_code());
        cpl_errorstate_set(prev_state);
    }
    return res;
}


static int _moo_strcmpi(char* s1, char* s2){
    int i;
    
    int size1 = strlen(s1);
    int size2 = strlen(s2);
    if(size1!=size2)
        return -1;
         
    for(i=0;i<size1;i++){
        if(toupper(s1[i])!=toupper(s2[i]))
            return s1[i]-s2[i];
    }
    
    return 0;
}

static void _moo_remove_spaces(const char *input, char *result)
{
  int i, j = 0;
  for (i = 0; input[i] != '\0'; i++) {
    if (!isspace((unsigned char) input[i])) {
      result[j++] = input[i];
    }
  }
  result[j] = '\0';
}

static int _moo_cmp_objname(const char* obj1, const char* obj2)
{
    int res = -1;
    
    char* str1 = cpl_strdup(obj1);
    char* str2 = cpl_strdup(obj2);
    
    _moo_remove_spaces(obj1,str1);
    _moo_remove_spaces(obj2,str2);
    
    res = _moo_strcmpi(str1,str2);
    
    cpl_free(str1);
    cpl_free(str2);
    return res;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Find the TABLE for a given object name
  @param self the FLX object
  @param objname the searched object name 
  @return the object table or NULL
  
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL  
 */
/*----------------------------------------------------------------------------*/
cpl_table* moo_flx_get_obj(moo_flx* self, const char* objname)
{
    cpl_table* res = NULL;
    cpl_ensure(objname!=NULL,CPL_ERROR_NULL_INPUT,NULL);
    int nrow = cpl_table_get_nrow(self->catalog);
    const char** names = cpl_table_get_data_string_const(
        self->catalog,MOO_FLX_CATALOG_NAME);
    
    for(int i=0; i<nrow; i++){
        const char* catname = names[i];
        if ( _moo_cmp_objname(catname,objname)==0){
            cpl_size qnum = cpl_fits_find_extension(self->filename, 
                catname);
            res = cpl_table_load(self->filename,qnum,0);
            break;
        }
    }
    
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_flx
  @param    self    moo_flx to delete
  @return   void

  If the moo_flx @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/
void moo_flx_delete(moo_flx* self)
{
    if ( self!= NULL){         
        if ( self->filename != NULL){
            cpl_free(self->filename);
        }
        if ( self->primary_header!= NULL){
            cpl_propertylist_delete(self->primary_header);
        }
        if ( self->catalog != NULL){
            cpl_table_delete(self->catalog);
        }       
        cpl_free(self);        
    }
}

/**@}*/
