/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include "moo_pfits.h"
#include "moo_utils.h"
#include "moo_loc.h"
#include "moo_loc_single.h"
#include "moo_fits.h"
#include "moo_qc.h"
#include "moo_single.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_loc_single LOC_SINGLE format
 * @ingroup moo_loc
 * This module provides functions to create, use, and destroy a
 * @em moo_loc_single
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_loc_single.h"
 * @endcode
 */


/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_loc_single
  @return   1 newly allocated moo_loc or NULL in case of an error

  The returned object must be deallocated using moo_det_delete().

 */
/*----------------------------------------------------------------------------*/
moo_loc_single *
moo_loc_single_new(void)
{
    moo_loc_single *res = cpl_calloc(1, sizeof(moo_loc_single));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_loc_single from the given LOC filename
  @param    filename the LOC filename
  @param    extname the LOC det name
  @return   1 newly allocated moo_det or NULL in case of an error

  The returned object must be deallocated using moo_loc_single_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_loc_single *
moo_loc_single_create(const char *filename, const char *extname)
{
    moo_loc_single *res = NULL;

    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(extname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    res = moo_loc_single_new();

    res->extname = extname;
    res->filename = filename;

    return res;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_loc_single
  @param    self    moo_loc to delete
  @return   void

  If the moo_loc_single @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_loc_single_delete(moo_loc_single *self)
{
    if (self != NULL) {
        if (self->f_centroids != NULL) {
            cpl_image_delete(self->f_centroids);
        }
        if (self->m_centroids != NULL) {
            cpl_image_delete(self->m_centroids);
        }
        if (self->f_wlow != NULL) {
            cpl_image_delete(self->f_wlow);
        }
        if (self->f_wup != NULL) {
            cpl_image_delete(self->f_wup);
        }
        if (self->m_wlow != NULL) {
            cpl_image_delete(self->m_wlow);
        }
        if (self->m_wup != NULL) {
            cpl_image_delete(self->m_wup);
        }
        if (self->flags != NULL) {
            cpl_image_delete(self->flags);
        }
        if (self->header != NULL) {
            cpl_propertylist_delete(self->header);
        }
        cpl_free(self);
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_loc_single to a FITS file
  @param    self          moo_loc to write to disk or NULL
  @param    filename      Name of the file to write
  @param    keep_points   1 if you want to save measured and flags
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_loc_single to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
void
moo_loc_single_save(const moo_loc_single *self,
                    const char *filename,
                    int keep_points)
{
    if (self != NULL) {
        moo_fits_write_extension_image(self->m_centroids, filename,
                                       MOO_LOC_SINGLE_MCENTROID, self->extname,
                                       CPL_TYPE_FLOAT, self->header);
        moo_fits_write_extension_image(self->f_centroids, filename,
                                       MOO_LOC_SINGLE_FCENTROID, self->extname,
                                       CPL_TYPE_FLOAT, self->header);
        moo_fits_write_extension_image(self->f_wlow, filename,
                                       MOO_LOC_SINGLE_WLO, self->extname,
                                       CPL_TYPE_FLOAT, self->header);
        moo_fits_write_extension_image(self->f_wup, filename,
                                       MOO_LOC_SINGLE_WUP, self->extname,
                                       CPL_TYPE_FLOAT, self->header);

        if (keep_points == 1) {
            moo_fits_write_extension_image(self->m_wlow, filename,
                                           MOO_LOC_SINGLE_MWLO, self->extname,
                                           CPL_TYPE_FLOAT, self->header);
            moo_fits_write_extension_image(self->m_wup, filename,
                                           MOO_LOC_SINGLE_MWUP, self->extname,
                                           CPL_TYPE_FLOAT, self->header);
            moo_fits_write_extension_image(self->flags, filename,
                                           MOO_LOC_SINGLE_FLAGS, self->extname,
                                           CPL_TYPE_INT, NULL);
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_loc_single
  @brief    Dump structural information of LOC_SINGLE
  @param    self    LOC_SINGLE to dump
  @param    stream  Output stream, accepts @c stdout or @c stderr
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_loc_single_dump(const moo_loc_single *self, FILE *stream)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(stream != NULL, CPL_ERROR_NULL_INPUT);

    fprintf(stream, "---MOO_LOC_SINGLE\n");
    fprintf(stream, "filename %s extname %s\n", self->filename, self->extname);
    fprintf(stream, "mcentroids %p\n", (void *)self->m_centroids);
    fprintf(stream, "fcentroids %p\n", (void *)self->f_centroids);
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_loc_single
  @brief   Get header of loc single
  @param    self    LOC_SINGLE
  @return   the header or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_propertylist *
moo_loc_single_get_header(moo_loc_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->extname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (self->header == NULL) {
        self->header = moo_fits_load_extension_header(self->filename,
                                                      MOO_LOC_SINGLE_MCENTROID,
                                                      self->extname);
        if (self->header == NULL) {
            self->header = cpl_propertylist_new();
        }
    }
    return self->header;
}
/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_loc_single
  @brief   Get image of fit centroids
  @param    self    LOC_SINGLE
  @return   image or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_loc_single_get_f_centroids(moo_loc_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->extname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (self->f_centroids == NULL) {
        self->f_centroids =
            moo_fits_load_extension_image(self->filename,
                                          MOO_LOC_SINGLE_FCENTROID,
                                          self->extname, CPL_TYPE_DOUBLE);
    }
    return self->f_centroids;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_loc_single
  @brief   Get the Y fit centroid for a x value
  @param    self    LOC_SINGLE
  @param    x x position
  @param    indexext index in EXT
  @param    rej rejected flag
  @return   Y fit centroid for X position

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
double
moo_loc_single_eval_f_centroids(moo_loc_single *self,
                                double x,
                                int indexext,
                                int *rej)
{
    double res;
    int rej1, rej2;
    cpl_image *fcentroids = moo_loc_single_get_f_centroids(self);
    int x1 = (int)floor(x);
    int x2 = (int)ceil(x);
    double y1 = cpl_image_get(fcentroids, x1, indexext, &rej1);
    double y2 = cpl_image_get(fcentroids, x2, indexext, &rej2);
    *rej = rej1 + rej2;
    res = y1 * (x - x1) + y2 * (x2 - x);
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_loc_single
  @brief   Get image of measured centroids
  @param    self    LOC_SINGLE
  @return   image or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_loc_single_get_m_centroids(moo_loc_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->extname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (self->m_centroids == NULL) {
        self->m_centroids =
            moo_fits_load_extension_image(self->filename,
                                          MOO_LOC_SINGLE_MCENTROID,
                                          self->extname, CPL_TYPE_DOUBLE);
    }
    return self->m_centroids;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_loc_single
  @brief   Get image of width low
  @param    self    LOC_SINGLE
  @return   image or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_loc_single_get_f_wlo(moo_loc_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->extname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (self->f_wlow == NULL) {
        self->f_wlow =
            moo_fits_load_extension_image(self->filename, MOO_LOC_SINGLE_WLO,
                                          self->extname, CPL_TYPE_DOUBLE);
    }
    return self->f_wlow;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_loc_single
  @brief   Get image of width low
  @param    self    LOC_SINGLE
  @return   image or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_loc_single_get_f_wup(moo_loc_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->extname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (self->f_wup == NULL) {
        self->f_wup =
            moo_fits_load_extension_image(self->filename, MOO_LOC_SINGLE_WUP,
                                          self->extname, CPL_TYPE_DOUBLE);
    }
    return self->f_wup;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_loc_single
  @brief   Get image of flags
  @param    self    LOC_SINGLE
  @return   image or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_loc_single_get_flags(moo_loc_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->extname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (self->flags == NULL) {
        self->flags =
            moo_fits_load_extension_image(self->filename, MOO_LOC_SINGLE_FLAGS,
                                          self->extname, CPL_TYPE_INT);
    }
    return self->flags;
}

cpl_error_code
moo_loc_single_compute_qc_guess(moo_loc_single *self, int deg_poly)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    cpl_image *mc = moo_loc_single_get_m_centroids(self);
    cpl_image *fc = moo_loc_single_get_f_centroids(self);
    cpl_image *flo = moo_loc_single_get_f_wlo(self);
    cpl_image *fup = moo_loc_single_get_f_wup(self);

    cpl_image *flags = moo_loc_single_get_flags(self);
    cpl_image *diff = cpl_image_subtract_create(mc, fc);
    cpl_image *width = cpl_image_add_create(flo, fup);

    int nx = cpl_image_get_size_x(diff);
    int ny = cpl_image_get_size_y(diff);

    cpl_binary *mask_data = cpl_mask_get_data(cpl_image_get_bpm(diff));
    int *flags_data = cpl_image_get_data(flags);
    int noutliers_pts = 0;
    int noutliers_wid = 0;
    int badpix = 0;

    for (int i = 0; i < nx * ny; i++) {
        if (flags_data[i] > 0) {
            mask_data[i] = CPL_BINARY_1;
            if (flags_data[i] == MOONS_FLAG_YDIFF_OUTLIERS) {
                noutliers_pts++;
            }
            else if (flags_data[i] == MOONS_FLAG_WDIFF_OUTLIERS) {
                noutliers_wid++;
            }
            else if (flags_data[i] == MOONS_FLAG_BADPIX) {
                badpix++;
            }
        }
    }
    cpl_image_reject_value(diff, CPL_VALUE_NAN);
    cpl_image_reject_value(width, CPL_VALUE_NAN);

    double sigma = cpl_image_get_stdev(diff);
    double median = cpl_image_get_median(diff);
    double mean = cpl_image_get_mean(diff);

    cpl_propertylist *header = moo_loc_single_get_header(self);

    moo_qc_set_residy_med(header, median);
    moo_qc_set_residy_sd(header, sigma);
    moo_qc_set_residy_avg(header, mean);
    sigma = cpl_image_get_stdev(width);
    median = cpl_image_get_median(width);
    mean = cpl_image_get_mean(width);
    moo_qc_set_widthy_med(header, median);
    moo_qc_set_widthy_sd(header, sigma);
    moo_qc_set_widthy_avg(header, mean);
    moo_qc_set_fit_deg(header, deg_poly);
    moo_qc_set_noutlier_pts(header, noutliers_pts);
    moo_qc_set_noutlier_wid(header, noutliers_wid);
    moo_qc_set_nbadpix(header, badpix);
    cpl_image_delete(diff);
    cpl_image_delete(width);

    return CPL_ERROR_NONE;
}

cpl_error_code
moo_loc_single_compute_qc_trace(moo_loc_single *self,
                                int deg_poly,
                                moo_loc_single *guess)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(guess != NULL, CPL_ERROR_NULL_INPUT);

    cpl_image *mc = moo_loc_single_get_m_centroids(self);
    cpl_image *guess_mc = moo_loc_single_get_m_centroids(guess);
    cpl_image *fc = moo_loc_single_get_f_centroids(self);
    cpl_image *guess_fc = moo_loc_single_get_f_centroids(guess);
    cpl_image *flags = moo_loc_single_get_flags(self);

    cpl_ensure_code(guess_mc != NULL, CPL_ERROR_NULL_INPUT);

    cpl_image_reject_value(guess_mc, CPL_VALUE_NAN);


    cpl_image *fdiff = cpl_image_subtract_create(fc, guess_fc);

    int nx = cpl_image_get_size_x(mc);
    int ny = cpl_image_get_size_y(mc);

    cpl_binary *mask_data = cpl_mask_get_data(cpl_image_get_bpm(mc));
    int *flags_data = cpl_image_get_data(flags);
    int nbadprofile = 0;

    for (int i = 0; i < nx * ny; i++) {
        if (flags_data[i] > 0) {
            mask_data[i] = CPL_BINARY_1;
            if (flags_data[i] == MOONS_FLAG_BADPROFILE) {
                nbadprofile++;
            }
        }
    }

    cpl_image *mdiff = cpl_image_subtract_create(mc, guess_mc);

    cpl_image_reject_value(mdiff, CPL_VALUE_NAN);
    cpl_image_reject_value(fdiff, CPL_VALUE_NAN);

    double sigma = cpl_image_get_stdev(mdiff);
    double median = cpl_image_get_median(mdiff);
    double mean = cpl_image_get_mean(mdiff);

    cpl_propertylist *header = moo_loc_single_get_header(self);

    moo_qc_set_shifty_pts_med(header, median);
    moo_qc_set_shifty_pts_sd(header, sigma);
    moo_qc_set_shifty_pts_avg(header, mean);
    moo_qc_set_fit_deg(header, deg_poly);
    moo_qc_set_nbadprofile(header, nbadprofile);

    sigma = cpl_image_get_stdev(fdiff);
    median = cpl_image_get_median(fdiff);
    mean = cpl_image_get_mean(fdiff);

    moo_qc_set_shifty_fit_med(header, median);
    moo_qc_set_shifty_fit_sd(header, sigma);
    moo_qc_set_shifty_fit_avg(header, mean);

    cpl_image_delete(fdiff);
    cpl_image_delete(mdiff);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_loc_single
  @brief   Get outside data range mask far a DET single
  @param    self    LOC_SINGLE
  @param    size_y DET size in Y
  @return   mask or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_mask *
moo_loc_single_get_ODR(moo_loc_single *self, int size_y)
{
    cpl_mask *mask = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_image *fc = moo_loc_single_get_f_centroids(self);
    cpl_image *fwlo = moo_loc_single_get_f_wlo(self);
    cpl_image *fwup = moo_loc_single_get_f_wup(self);

    int size_x = cpl_image_get_size_x(fc);
    int nb_fibres = cpl_image_get_size_y(fc);

    mask = cpl_mask_new(size_x, size_y);

    for (int j = 1; j <= nb_fibres; j++) {
        for (int i = 1; i <= size_x; i++) {
            int rej;
            double fcv = cpl_image_get(fc, i, j, &rej);
            if (!isnan(fcv)) {
                double ylo = fcv - cpl_image_get(fwlo, i, j, &rej);
                double yup = fcv + cpl_image_get(fwup, i, j, &rej);
                int iylo = floor(ylo);
                int iyup = ceil(yup);
                for (int k = iylo; k <= iyup; k++) {
                    cpl_mask_set(mask, i, k, CPL_BINARY_1);
                }
            }
        }
    }
    cpl_mask_not(mask);
    return mask;
}
/**@}*/
