/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include "moo_detector.h"
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_qc.h"
#include "moo_map.h"
#include "moo_fibres_table.h"
#include "moo_line_table.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_map MAP format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_map
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_map.h"
 * @endcode
 */

/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_map
  @return   1 newly allocated moo_ext or NULL in case of an error

  The returned object must be deallocated using moo_map_delete().

 */
/*----------------------------------------------------------------------------*/
moo_map *
moo_map_new(void)
{
    moo_map *res = cpl_calloc(1, sizeof(moo_map));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Get the LINE TABLE extension of a map
  @param self the WAVEMAP
  @return the line table

*/
/*----------------------------------------------------------------------------*/

cpl_table *
moo_map_get_line_table(moo_map *self)
{
    cpl_table *result = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (self->line_table != NULL) {
        result = self->line_table;
    }
    else {
        if (self->filename != NULL) {
            cpl_size qnum =
                cpl_fits_find_extension(self->filename, MOO_LINE_TABLE_EXTNAME);
            if (qnum > 0) {
                self->line_table = cpl_table_load(self->filename, qnum, 0);
            }
        }
        result = self->line_table;
    }
    return result;
}
/*----------------------------------------------------------------------------*/

moo_map *
moo_map_load(const cpl_frame *mapframe)
{
    cpl_ensure(mapframe != NULL, CPL_ERROR_NULL_INPUT, NULL);
    const char *filename = cpl_frame_get_filename(mapframe);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prev_state = cpl_errorstate_get();
    moo_map *res = moo_map_new();
    res->filename = cpl_strdup(filename);
    cpl_size qnum = cpl_fits_find_extension(filename, MOO_FIBRES_TABLE_EXTNAME);
    if (qnum > 0) {
        res->fibre_table = cpl_table_load(filename, qnum, 0);
    }
    cpl_errorstate_set(prev_state);

    res->primary_header = cpl_propertylist_load(filename, 0);

    for (int i = 0; i < 2; i++) {
        for (int j = 0; j < 3; j++) {
            const char *extname = moo_detector_get_extname(j, i + 1);

            cpl_propertylist *header = NULL;
            header = moo_fits_load_extension_header(filename, NULL, extname);
            res->data_header[i * 3 + j] = header;

            if (header != NULL) {
                int naxis = moo_pfits_get_naxis(header);
                if (naxis == 2) {
                    res->data[i * 3 + j] =
                        moo_fits_load_extension_image(filename, NULL, extname,
                                                      CPL_TYPE_FLOAT);
                }
            }
        }
    }
    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_errorstate_set(prev_state);
        cpl_msg_error("moo_map", "Error in loading WAVE MAP %s", filename);
        moo_map_delete(res);
        res = NULL;
    }

    return res;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    set map data for relevant extension
  @param self
  @param type the detector type
  @param ntas number of TAS
  @param data the map data
  @param header the map header
  @return   error code

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_map_set_data(moo_map *self,
                 moo_detector_type type,
                 int ntas,
                 cpl_image *data,
                 cpl_propertylist *header)
{
    cpl_ensure_code(ntas >= 1 && ntas <= 2, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(header != NULL, CPL_ERROR_NULL_INPUT);

    if (data != NULL) {
        int nx = cpl_image_get_size_x(data);
        int ny = cpl_image_get_size_y(data);
        self->data[(ntas - 1) * 3 + type] = data;
        cpl_array *wmintab = cpl_array_new(ny, CPL_TYPE_DOUBLE);
        cpl_array *wmaxtab = cpl_array_new(ny, CPL_TYPE_DOUBLE);
        for (int j = 1; j <= ny; j++) {
            cpl_image *data_copy = cpl_image_extract(data, 1, j, nx, j);
            cpl_image_reject_value(data_copy, CPL_VALUE_NAN);
            int nbrej = cpl_image_count_rejected(data_copy);
            if (nbrej < nx) {
                double lmax = cpl_image_get_max(data_copy);
                double lmin = cpl_image_get_min(data_copy);
                cpl_array_set(wmintab, j - 1, lmin);
                cpl_array_set(wmaxtab, j - 1, lmax);
            }
            cpl_image_delete(data_copy);
        }
        double wmin = cpl_array_get_median(wmintab);
        double wmax = cpl_array_get_median(wmaxtab);
        moo_qc_set_wave_min(header, wmin);
        moo_qc_set_wave_max(header, wmax);
        cpl_array_delete(wmaxtab);
        cpl_array_delete(wmintab);
    }
    self->data_header[(ntas - 1) * 3 + type] = header;
    return CPL_ERROR_NONE;
}

cpl_error_code
moo_map_set_fibre_table(moo_map *self, cpl_table *fibre_table)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(fibre_table != NULL, CPL_ERROR_NULL_INPUT);

    self->fibre_table = fibre_table;

    return CPL_ERROR_NONE;
}

cpl_error_code
moo_map_set_line_table(moo_map *self, cpl_table *line_table)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(line_table != NULL, CPL_ERROR_NULL_INPUT);

    self->line_table = line_table;

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_map
  @param    self    moo_map to delete
  @return   void

  If the moo_map @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_map_delete(moo_map *self)
{
    if (self != NULL) {
        int i;
        if (self->filename != NULL) {
            cpl_free(self->filename);
        }
        if (self->primary_header != NULL) {
            cpl_propertylist_delete(self->primary_header);
        }
        for (i = 0; i < 6; i++) {
            if (self->data[i] != NULL) {
                cpl_image_delete(self->data[i]);
            }
            if (self->data_header[i] != NULL) {
                cpl_propertylist_delete(self->data_header[i]);
            }
        }
        if (self->fibre_table != NULL) {
            cpl_table_delete(self->fibre_table);
        }
        if (self->line_table != NULL) {
            cpl_table_delete(self->line_table);
        }
        cpl_free(self);
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_map to a FITS file
  @param    self moo_map to write to disk or NULL
  @param    filename Name of the file to write
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_map to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
void
moo_map_save(moo_map *self, const char *filename)
{
    if (self != NULL) {
        cpl_propertylist_save(self->primary_header, filename, CPL_IO_CREATE);

        for (int i = 0; i < 2; i++) {
            for (int j = 0; j < 3; j++) {
                const char *extname = moo_detector_get_extname(j, i + 1);
                moo_fits_write_extension_image(self->data[i * 3 + j], filename,
                                               NULL, extname, CPL_TYPE_FLOAT,
                                               self->data_header[i * 3 + j]);
            }
        }

        if (self->fibre_table != NULL) {
            cpl_propertylist *h = cpl_propertylist_new();
            cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                           MOO_FIBRES_TABLE_EXTNAME);

            cpl_table_save(self->fibre_table, h, h, filename, CPL_IO_EXTEND);
            cpl_propertylist_delete(h);
        }
        if (self->line_table != NULL) {
            cpl_propertylist *h = cpl_propertylist_new();
            cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                           MOO_LINE_TABLE_EXTNAME);

            cpl_table_save(self->line_table, h, h, filename, CPL_IO_EXTEND);
            cpl_propertylist_delete(h);
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_map to a FITS file
  @param    self moo_map to write to disk or NULL
  @param    filename Name of the file to write
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_map to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_map_check(moo_map *self, moo_spectral_format *sformat)
{
    cpl_error_code status = CPL_ERROR_NONE;
    const char *monotonous_colnames[] = { MOO_FIBRES_TABLE_MONOTONOUS_RI,
                                          MOO_FIBRES_TABLE_MONOTONOUS_YJ,
                                          MOO_FIBRES_TABLE_MONOTONOUS_H };
    if (self != NULL) {
        cpl_table *fibre_table = self->fibre_table;
        cpl_propertylist *primary_header = self->primary_header;

        cpl_ensure_code(sformat != NULL, CPL_ERROR_NULL_INPUT);
        cpl_msg_info(__func__, "Check WAVEMAP");
        int gres = 1;
        cpl_msg_indent_more();
        for (int ntas = 1; ntas <= 2; ntas++) {
            cpl_table_select_all(fibre_table);
            int size =
                cpl_table_and_selected_int(fibre_table, MOO_FIBRES_TABLE_HEALTH,
                                           CPL_EQUAL_TO, 1);
            size = cpl_table_and_selected_int(fibre_table,
                                              MOO_FIBRES_TABLE_SPECTRO,
                                              CPL_EQUAL_TO, ntas);
            cpl_array *sel = cpl_table_where_selected(fibre_table);
            for (int type = 0; type < 3; type++) {
                cpl_image *orig_data = self->data[(ntas - 1) * 3 + type];
                cpl_propertylist *header =
                    self->data_header[(ntas - 1) * 3 + type];
                const char *monotonous_colname = monotonous_colnames[type];
                if (orig_data != NULL) {
                    cpl_image *data = cpl_image_cast(orig_data, CPL_TYPE_FLOAT);
                    int nx = cpl_image_get_size_x(data);
                    const float *tdata = cpl_image_get_data_const(data);
                    moo_spectral_format_info *info =
                        moo_spectral_format_get(sformat, type, ntas);
                    int res = 1;
                    const char *band = moo_detector_get_name(type);

                    for (int j = 0; j < size; j++) {
                        int index = cpl_array_get_cplsize(sel, j, NULL);
                        int indexext =
                            cpl_table_get_int(fibre_table,
                                              MOO_FIBRES_TABLE_INDEXEXT, index,

                                              NULL);
                        const char *fibname =
                            cpl_table_get_string(fibre_table,
                                                 MOO_FIBRES_TABLE_FIBRE, index);
                        int first = 0;
                        while (first < nx &&
                               isnan(tdata[first + (indexext - 1) * nx])) {
                            first++;
                        }
                        int fres = 1;
                        if (first < (nx - 2)) {
                            double w1 = tdata[first + (indexext - 1) * nx];
                            double w2 = tdata[first + 1 + (indexext - 1) * nx];
                            double b1 = w1 - w2;

                            for (int i = first + 1; i < nx - 1; i++) {
                                w1 = tdata[i + (indexext - 1) * nx];
                                w2 = tdata[i + 1 + (indexext - 1) * nx];
                                double b2 = w1 - w2;

                                if (b1 * b2 < 0) {
                                    cpl_msg_warning(
                                        __func__,
                                        "Not monotonous solution "
                                        "for fibre %s band %s spectro %d",
                                        fibname, band, ntas);
                                    fres = 0;
                                    res = 0;
                                    break;
                                }
                            }
                        }
                        cpl_table_set_int(fibre_table, monotonous_colname,
                                          index, fres);
                    }
                    moo_qc_set_wavecal_monotonous_solution(header, res);
                    if (res == 0) {
                        gres = res;
                    }
                    moo_spectral_format_info_delete(info);
                    cpl_image_delete(data);
                }
            }
            cpl_array_delete(sel);
        }
        moo_qc_set_wavecal_monotonous_solution(primary_header, gres);
        cpl_msg_indent_less();
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_map
  @brief    Dump structural information of MAP
  @param    self    structure to dump
  @param    stream  Output stream, accepts @c stdout or @c stderr
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_map_dump(const moo_map *self, FILE *stream)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(stream != NULL, CPL_ERROR_NULL_INPUT);

    fprintf(stream, "---MOO_MAP\n");

    int i;
    for (i = 0; i < 6; i++) {
        if (self->data[i] != NULL) {
            fprintf(stream, "data[%d] %p\n", i, self->data[i]);
            if (self->data[i] != NULL) {
                cpl_image_dump_structure(self->data[i], stream);
            }
        }
    }
    if (self->fibre_table) {
        fprintf(stream, "fibre table %p\n", self->fibre_table);
    }

    return CPL_ERROR_NONE;
}

void
moo_map_update_linetable(moo_map *self, moo_rbn *rbn)
{
    cpl_table *line_table = self->line_table;
    int nrow = cpl_table_get_nrow(line_table);
    cpl_table *fibre_table = rbn->fibre_table;

    for (int i = 0; i < nrow; i++) {
        const int spectro =
            cpl_table_get_int(line_table, MOO_LINE_TABLE_SPECTRO, i, NULL);
        int indexext =
            cpl_table_get_int(line_table, MOO_LINE_TABLE_INDEXEXT, i, NULL);
        if (indexext >= 0) {
            cpl_table_select_all(fibre_table);
            cpl_table_and_selected_int(fibre_table, MOO_FIBRES_TABLE_SPECTRO,
                                       CPL_EQUAL_TO, spectro);
            cpl_table_and_selected_int(fibre_table, MOO_FIBRES_TABLE_INDEXEXT,
                                       CPL_EQUAL_TO, indexext);
            cpl_array *sel = cpl_table_where_selected(fibre_table);
            int idx = cpl_array_get_cplsize(sel, 0, NULL);
            int indexrbn =
                cpl_table_get_int(fibre_table, MOO_FIBRES_TABLE_INDEXRBN, idx,
                                  NULL);
            cpl_table_set_int(line_table, MOO_LINE_TABLE_INDEXRBN, i, indexrbn);
            cpl_array_delete(sel);
        }
    }
}
/**@}*/
