/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include <string.h>
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_qc.h"
#include "moo_molectable.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_molectable MOLEC TABLE format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_molectable
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_molectable.h"
 * @endcode
 */


/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_molectable
  @return   1 newly allocated moo_ext or NULL in case of an error

  The returned object must be deallocated using moo_molectable_delete().

 */
/*----------------------------------------------------------------------------*/
moo_molectable *
moo_molectable_new(void)
{
    moo_molectable *res = cpl_calloc(1, sizeof(moo_molectable));
    return res;
}
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Load a moo_molectable
  @param frame the frame to load
  @param modename the detector mode LR or HR
  @return   1 newly allocated moo_molectable or NULL in case of an error

  The returned object must be deallocated using moo_molectable_delete().

 */
/*----------------------------------------------------------------------------*/
moo_molectable *
moo_molectable_load(const cpl_frame *frame, const char *modename)
{
    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    const char *filename = cpl_frame_get_filename(frame);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);

    char *extname = NULL;

    cpl_errorstate prev_state = cpl_errorstate_get();

    moo_molectable *res = moo_molectable_new();
    res->filename = cpl_strdup(filename);
    res->primary_header = cpl_propertylist_load(filename, 0);

    for (int i = 0; i < 3; i++) {
        const char *bandname = moo_detector_get_name(i);

        if (modename == NULL || strcmp(bandname, "YJ") == 0) {
            extname = cpl_sprintf("%s", bandname);
        }
        else {
            extname = cpl_sprintf("%s_%s", bandname, modename);
        }
        cpl_size qnum = cpl_fits_find_extension(filename, extname);

        if (qnum > 0) {
            cpl_propertylist *header = NULL;
            header = moo_fits_load_extension_header(filename, NULL, extname);
            res->data_header[i] = header;
            int naxis = moo_pfits_get_naxis(header);
            if (naxis > 0) {
                res->data[i] = cpl_table_load(filename, qnum, 0);
            }
        }
        cpl_free(extname);
        extname = NULL;
    }

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_free(extname);
        cpl_errorstate_set(prev_state);
        cpl_msg_error("moo_molectable", "Error in loading table %s", filename);
        moo_molectable_delete(res);
        res = NULL;
    }

    return res;
}
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief Get table from molec_table
  @param self the molec_table
  @param type the detector type 
  @return the table for the chosen type or NULL
 */
/*----------------------------------------------------------------------------*/
cpl_table *
moo_molectable_get_data(moo_molectable *self, moo_detector_type type)
{
    cpl_table *result = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    result = self->data[type];

    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Set table in molectable
  @param self the molectable
  @param type the detector type
  @param data the table to set
  @return the error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_molectable_set_data(moo_molectable *self,
                        moo_detector_type type,
                        cpl_table *data)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    if (data != NULL) {
        self->data[type] = cpl_table_duplicate(data);
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_molectable to a FITS file
  @param    self moo_molectable to write to disk or NULL
  @param    filename Name of the file to write
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_molectable to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
void
moo_molectable_save(moo_molectable *self, const char *filename)
{
    if (self != NULL) {
        cpl_propertylist_save(self->primary_header, filename, CPL_IO_CREATE);

        for (int i = 0; i < 3; i++) {
            const char *extname = moo_detector_get_name(i);

            moo_fits_write_extension_table(self->data[i], filename, NULL,
                                           extname, self->data_header[i]);
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_molectable
  @param    self    moo_molectable to delete
  @return   void

  If the moo_molectable @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_molectable_delete(moo_molectable *self)
{
    if (self != NULL) {
        int i;
        if (self->primary_header != NULL) {
            cpl_propertylist_delete(self->primary_header);
        }

        if (self->filename != NULL) {
            cpl_free(self->filename);
        }
        for (i = 0; i < 3; i++) {
            if (self->data[i] != NULL) {
                cpl_table_delete(self->data[i]);
            }
            if (self->data_header[i] != NULL) {
                cpl_propertylist_delete(self->data_header[i]);
            }
        }
        cpl_free(self);
    }
}


/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_molectable
  @brief    Dump structural information of MOLECTABLE
  @param    self    structure to dump
  @param    stream  Output stream, accepts @c stdout or @c stderr
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_molectable_dump(const moo_molectable *self, FILE *stream)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(stream != NULL, CPL_ERROR_NULL_INPUT);

    fprintf(stream, "---MOO_MAP\n");

    int i;
    for (i = 0; i < 3; i++) {
        if (self->data[i] != NULL) {
            fprintf(stream, "data[%d] %p\n", i, (void *)self->data[i]);
            if (self->data[i] != NULL) {
                cpl_table_dump_structure(self->data[i], stream);
            }
        }
    }
    return CPL_ERROR_NONE;
}
/**@}*/
