/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>
#include <mf_parameters.h>
#include "moo_utils.h"
#include "moo_params.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_params  Manage moons recipes parameters
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_params
  @param    pid the project id
  @param    recipe_id the recipe id

  @return   1 newly allocated moo_params or NULL in case of an error

  The returned object must be deallocated using moo_params_delete().
  - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_params *
moo_params_new(const char *pid, const char *recipe_id)
{
    cpl_ensure(recipe_id != NULL, CPL_ERROR_NULL_INPUT, NULL);
    moo_params *res = (moo_params *)cpl_calloc(1, sizeof(moo_params));

    if (res != NULL) {
        char *paramsid = cpl_sprintf("%s.%s", pid, recipe_id);
        res->recipe_id = paramsid;
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_params
  @param    self    moo_params to delete
  @return   void

  If the moo_params @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/
void
moo_params_delete(moo_params *self)
{
    if (self != NULL) {
        cpl_free(self->recipe_id);
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add string parameter to moons parameters list
  @param    params moons parameters
  @param    list parameters list
  @param    name of parameter
  @param    alias alias of parameter
  @param    def definition of parameter
  @param    val value of parameter
  @return error code or CPL_ERROR_NONE

- - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_string(moo_params *params,
                      cpl_parameterlist *list,
                      const char *name,
                      const char *alias,
                      const char *def,
                      const char *val)
{
    cpl_ensure_code(name != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(params != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(params->recipe_id != NULL, CPL_ERROR_NULL_INPUT);

    cpl_parameter *p = NULL;
    char *pname = cpl_sprintf("%s.%s", params->recipe_id, name);
    cpl_ensure_code(pname != NULL, CPL_ERROR_NULL_INPUT);

    moo_try_check(p = cpl_parameter_new_value(pname, CPL_TYPE_STRING, def,
                                              params->recipe_id, val),
                  " ");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, alias);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(list, p);

moo_try_cleanup:

    cpl_free(pname);
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add double parameter to moons parameters list
  @param    params moons parameters
  @param    list parameters list
  @param    name of parameter
  @param    alias alias of parameter
  @param    def definition of parameter
  @param    val value of parameter
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_double(moo_params *params,
                      cpl_parameterlist *list,
                      const char *name,
                      const char *alias,
                      const char *def,
                      double val)
{
    cpl_ensure_code(name != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(params != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(params->recipe_id != NULL, CPL_ERROR_NULL_INPUT);

    cpl_parameter *p;
    char *pname = cpl_sprintf("%s.%s", params->recipe_id, name);
    p = cpl_parameter_new_value(pname, CPL_TYPE_DOUBLE, def, params->recipe_id,
                                val);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, alias);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(list, p);

    cpl_free(pname);
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add int parameter to moons parameters list
  @param    params moons parameters
  @param    list parameters list
  @param    name of parameter
  @param    alias alias of parameter
  @param    def definition of parameter
  @param    val value of parameter
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_int(moo_params *params,
                   cpl_parameterlist *list,
                   const char *name,
                   const char *alias,
                   const char *def,
                   int val)
{
    cpl_parameter *p;

    cpl_ensure_code(name != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(params != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(params->recipe_id != NULL, CPL_ERROR_NULL_INPUT);

    char *pname = cpl_sprintf("%s.%s", params->recipe_id, name);

    p = cpl_parameter_new_value(pname, CPL_TYPE_INT, def, params->recipe_id,
                                val);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, alias);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(list, p);

    cpl_free(pname);
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add boolean parameter to moons parameters list
  @param    params moons parameters
  @param    list parameters list
  @param    name of parameter
  @param    alias alias of parameter
  @param    def definition of parameter
  @param    val value of parameter
  @return error code or CPL_ERROR_NONE
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_bool(moo_params *params,
                    cpl_parameterlist *list,
                    const char *name,
                    const char *alias,
                    const char *def,
                    int val)
{
    cpl_parameter *p;

    cpl_ensure_code(name != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(params != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(params->recipe_id != NULL, CPL_ERROR_NULL_INPUT);

    char *pname = cpl_sprintf("%s.%s", params->recipe_id, name);
    p = cpl_parameter_new_value(pname, CPL_TYPE_BOOL, def, params->recipe_id,
                                val);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, alias);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(list, p);

    cpl_free(pname);
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get double parameter from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @param    name the name of the parameter
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
double
moo_params_get_double(const moo_params *self,
                      const cpl_parameterlist *list,
                      const char *name)
{
    const cpl_parameter *p;

    char *pname = cpl_sprintf("%s.%s", self->recipe_id, name);
    p = cpl_parameterlist_find_const(list, pname);
    double res = cpl_parameter_get_double(p);
    cpl_free(pname);
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get int parameter from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @param    name the name of the parameter
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
int
moo_params_get_int(const moo_params *self,
                   const cpl_parameterlist *list,
                   const char *name)
{
    const cpl_parameter *p;

    char *pname = cpl_sprintf("%s.%s", self->recipe_id, name);
    p = cpl_parameterlist_find_const(list, pname);
    int res = cpl_parameter_get_int(p);
    cpl_free(pname);

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get boolean parameter from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @param    name the name of the parameter
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
int
moo_params_get_bool(const moo_params *self,
                    const cpl_parameterlist *list,
                    const char *name)
{
    const cpl_parameter *p;

    char *pname = cpl_sprintf("%s.%s", self->recipe_id, name);

    p = cpl_parameterlist_find_const(list, pname);
    int res = cpl_parameter_get_bool(p);

    cpl_free(pname);

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get string parameter from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @param    name the name of the parameter
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
const char *
moo_params_get_string(const moo_params *self,
                      const cpl_parameterlist *list,
                      const char *name)
{
    const cpl_parameter *p;

    char *pname = cpl_sprintf("%s.%s", self->recipe_id, name);
    p = cpl_parameterlist_find_const(list, pname);

    const char *res = cpl_parameter_get_string(p);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error("moo_params", "Can't get string parameter %s", pname);
    }
    cpl_free(pname);
    return res;
}

void
_moo_params_get_tab_double(const moo_params *self,
                           double res[],
                           const cpl_parameterlist *list,
                           const char *name)
{
    double a, b, c, d, e, f;
    const char *data = moo_params_get_string(self, list, name);
    int nb = sscanf(data, "%lf,%lf,%lf,%lf,%lf,%lf", &a, &b, &c, &d, &e, &f);
    if (nb == 1) {
        res[0] = a;
        res[1] = a;
        res[2] = a;
        res[3] = a;
        res[4] = a;
        res[5] = a;
    }
    else if (nb == 3) {
        res[0] = a;
        res[1] = b;
        res[2] = c;
        res[3] = a;
        res[4] = b;
        res[5] = c;
    }
    else if (nb == 6) {
        res[0] = a;
        res[1] = b;
        res[2] = c;
        res[3] = d;
        res[4] = e;
        res[5] = f;
    }
    else {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT,
            "Invalid parameter %s (%s) : %d values are read 1,3 or 6 expected",
            name, data, nb);
    }
}

static void
_moo_params_get_tab3_double(const moo_params *self,
                            double res[],
                            const cpl_parameterlist *list,
                            const char *name)
{
    double a, b, c;
    const char *data = moo_params_get_string(self, list, name);

    int nb = sscanf(data, "%lf,%lf,%lf", &a, &b, &c);
    if (nb == 1) {
        res[0] = a;
        res[1] = a;
        res[2] = a;
    }
    else if (nb == 3) {
        res[0] = a;
        res[1] = b;
        res[2] = c;
    }
    else {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT,
            "Invalid parameter %s (%s) : %d values are read 1,3 expected", name,
            data, nb);
    }
}

static void
_moo_params_get_tab_int(const moo_params *self,
                        int res[],
                        const cpl_parameterlist *list,
                        const char *name)
{
    int a, b, c, d, e, f;
    const char *data = moo_params_get_string(self, list, name);
    int nb = sscanf(data, "%d,%d,%d,%d,%d,%d", &a, &b, &c, &d, &e, &f);
    if (nb == 1) {
        res[0] = a;
        res[1] = a;
        res[2] = a;
        res[3] = a;
        res[4] = a;
        res[5] = a;
    }
    else if (nb == 3) {
        res[0] = a;
        res[1] = b;
        res[2] = c;
        res[3] = a;
        res[4] = b;
        res[5] = c;
    }
    else if (nb == 6) {
        res[0] = a;
        res[1] = b;
        res[2] = c;
        res[3] = d;
        res[4] = e;
        res[5] = f;
    }
    else {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT,
            "Invalid parameter %s (%s) : %d values are read 1,3 or 6 expected",
            name, data, nb);
    }
}

static void
_moo_params_get_tab3_int(const moo_params *self,
                         int res[],
                         const cpl_parameterlist *list,
                         const char *name)
{
    int a, b, c;
    const char *data = moo_params_get_string(self, list, name);
    int nb = sscanf(data, "%d,%d,%d", &a, &b, &c);
    if (nb == 1) {
        res[0] = a;
        res[1] = a;
        res[2] = a;
    }
    else if (nb == 3) {
        res[0] = a;
        res[1] = b;
        res[2] = c;
    }
    else {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT,
            "Invalid parameter %s (%s) : %d values are read 1 or 3 expected",
            name, data, nb);
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for compute noise map
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_nos(moo_params *self, cpl_parameterlist *list)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    moo_params_add_double(self, list, "nos_clip_kappa", "nos-clip-kappa",
                          "multiple of sigma in sigma clipping", 5.0);
    moo_params_add_double(self, list, "nos_clip_diff", "nos-clip-diff",
                          "minimum relative change in sigma for sigma clipping",
                          0.1);
    moo_params_add_double(self, list, "nos_clip_maxfrac", "nos-max-frac",
                          "maximum fraction of bad pixels allowed", 0.1);
    moo_params_add_int(self, list, "nos_clip_niter", "nos-clip-niter",
                       "maximum number of iterations for sigma clipping", 2);
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for compute hot map
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_hot(moo_params *self, cpl_parameterlist *list)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    moo_params_add_double(self, list, "hot_clip_kappa", "hot-clip-kappa",
                          "multiple of sigma in global sigma clipping", 3.0);
    moo_params_add_double(
        self, list, "hot_clip_diff", "hot-clip-diff",
        "minimum relative change in sigma for global sigma clipping", 0.1);
    moo_params_add_double(self, list, "hot_clip_maxfrac", "hot-max-frac",
                          "maximum fraction of bad pixels allowed", 0.1);
    moo_params_add_int(self, list, "hot_clip_niter", "hot-clip-niter",
                       "maximum number of iterations for global sigma clipping",
                       6);
    moo_params_add_int(
        self, list, "hot_local_winhsize", "hot-local-winhsize",
        "half window size in pixels for local detection of hot pixels.", 0);
    moo_params_add_double(self, list, "hot_local_kappa", "hot-local-kappa",
                          "multiple of sigma in local sigma clipping", 5.0);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new empty compute hot map set of parameters

  @return   1 newly allocated moo_nos_params or NULL in case of an error

  The returned object must be deallocated using moo_nos_params_delete().
  - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_hot_params *
moo_hot_params_new(void)
{
    moo_hot_params *res =
        (moo_hot_params *)cpl_calloc(1, sizeof(moo_hot_params));
    return res;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_hot_params
  @param    self    moo_params to delete
  @return   void

  If @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/
void
moo_hot_params_delete(moo_hot_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new empty compute noise map set of parameters

  @return   1 newly allocated moo_nos_params or NULL in case of an error

  The returned object must be deallocated using moo_nos_params_delete().
  - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_nos_params *
moo_nos_params_new(void)
{
    moo_nos_params *res =
        (moo_nos_params *)cpl_calloc(1, sizeof(moo_nos_params));
    return res;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_nos_params
  @param    self    moo_params to delete
  @return   void

  If @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/
void
moo_nos_params_delete(moo_nos_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for prepare
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if method is invalid
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_prepare(moo_params *self, cpl_parameterlist *list)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    moo_params_add_string(self, list, "prepare_ignore_detectors",
                          "prepare-ignore-detectors",
                          "ignore specified detector (0:used, 1:ignored), "
                          "ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2",
                          "0,0,0,0,0,0");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for remove crh
  @param    self    moo_params to update
  @param    list parameter list to update
  @param method the default method MOO_CRH_METHOD_MEDIAN or MOO_CRH_METHOD_SIGCLIP
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if method is invalid
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_crh(moo_params *self,
                   cpl_parameterlist *list,
                   const char *method)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(method != NULL, CPL_ERROR_NULL_INPUT);

    if (!((strcmp(method, MOO_CRH_METHOD_MEDIAN) == 0) ||
          (strcmp(method, MOO_CRH_METHOD_MEAN) == 0) ||
          (strcmp(method, MOO_CRH_METHOD_SIGCLIP) == 0))) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                     "Invalid method %s", method);
    }

    moo_params_add_double(self, list, "crh_clip_kappa", "crh-clip-kappa",
                          "multiple of sigma in sigma clipping", 5.0);
    moo_params_add_int(self, list, "crh_clip_niter", "crh-clip-niter",
                       "maximum number of iterations for sigma clipping", 5);
    moo_params_add_string(self, list, "crh_method", "crh-method",
                          "method used for removing CRH (MEDIAN|SIGCLIP|MEAN).",
                          method);
    return cpl_error_get_code();
}

moo_prepare_params *
moo_prepare_params_new(void)
{
    moo_prepare_params *res =
        (moo_prepare_params *)cpl_calloc(1, sizeof(moo_prepare_params));
    return res;
}

void
moo_prepare_params_delete(moo_prepare_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

moo_crh_params *
moo_crh_params_new(void)
{
    moo_crh_params *res =
        (moo_crh_params *)cpl_calloc(1, sizeof(moo_crh_params));
    return res;
}

void
moo_crh_params_delete(moo_crh_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for linear
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if method is invalid
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_linear(moo_params *self, cpl_parameterlist *list)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    moo_params_add_string(
        self, list, "bpmnl_clip_kappa", "bpmnl-clip-kappa",
        "multiple of sigma in sigma-clipping, to create linearity bad pixel "
        "map, for each detector, ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
        "40,70,50,40,70,50");
    moo_params_add_string(self, list, "bpmnl_min_snr", "bpmnl-min-snr",
                          "minimum SNR used in longest exposure frame for each "
                          "detector, ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
                          "10,10,10,10,10,10");
    moo_params_add_string(
        self, list, "detect_saturated_ron_kappa", "detect-saturated-ron-kappa",
        "multiple of RON to detect flux plateau in pixel growth curves, for "
        "each detector, ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
        "2,2,2,2,2,2");
    moo_params_add_double(self, list, "detect_saturated_threshold",
                          "detect-saturated-threshold",
                          "saturate detection threshold",
                          MOO_SATURATED_THRESHOLD);
    return cpl_error_get_code();
}

moo_linear_params *
moo_linear_params_new(void)
{
    moo_linear_params *res =
        (moo_linear_params *)cpl_calloc(1, sizeof(moo_linear_params));
    return res;
}

void
moo_linear_params_delete(moo_linear_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for bias
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_bias(moo_params *self, cpl_parameterlist *list)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    moo_params_add_string(self, list, "ron_estimation_method",
                          "ron-estimation-method",
                          "RON estimation method (GLOBAL|LOCAL)",
                          MOO_RON_ESTIMATION_METHOD_GLOBAL);

    return cpl_error_get_code();
}

moo_bias_params *
moo_bias_params_new(void)
{
    moo_bias_params *res =
        (moo_bias_params *)cpl_calloc(1, sizeof(moo_bias_params));
    return res;
}

void
moo_bias_params_delete(moo_bias_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for correct_bias
  @param    self    moo_params to update
  @param    list parameter list to update
  @param    method the default value to subtract_bias method
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_correct_bias(moo_params *self,
                            cpl_parameterlist *list,
                            const char *method)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    moo_params_add_string(self, list, "subtract_bias_method",
                          "subtract-bias-method",
                          "subtract bias method (MASTER|MEDIAN)", method);

    return cpl_error_get_code();
}

moo_correct_bias_params *
moo_correct_bias_params_new(void)
{
    moo_correct_bias_params *res =
        (moo_correct_bias_params *)cpl_calloc(1,
                                              sizeof(moo_correct_bias_params));
    return res;
}

void
moo_correct_bias_params_delete(moo_correct_bias_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for extraction
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_extract(moo_params *self, cpl_parameterlist *list)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    moo_params_add_string(self, list, "extract_method", "extract-method",
                          "method used for extraction (SUM|OPTIMAL)",
                          MOO_EXTRACT_METHOD_SUM);
    moo_params_add_string(
        self, list, "extract_aperture", "extract-aperture",
        "size of fibre aperture in pixels , for each detector, ordered as "
        "RI_1,YJ_1,H_1,RI_2,YJ_2,H_2. If value <=0, use "
        "localization detected edges (lo,up) for extraction.",
        "-1,-1,-1,-1,-1,-1");

    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for keep-temp
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_keep_temp(moo_params *self, cpl_parameterlist *list)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(list != NULL, CPL_ERROR_NULL_INPUT);

    moo_params_add_bool(self, list, "keep_temp", "keep-temp",
                        "keep INTERMEDIATE products", CPL_FALSE);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for quicklook
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_quicklook_fibre_list(moo_params *self, cpl_parameterlist *list)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(list != NULL, CPL_ERROR_NULL_INPUT);

    moo_params_add_string(
        self, list, "quicklook_fibre_list", "quicklook-fibre-list",
        "list of FIBRE in fibres table to convert in CHECK_OBJECT", "");

    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Dump nos params
  @param    self    single to dump
  @param    stream  Output stream, accepts @c stdout or @c stderr
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_nos_params_dump(const moo_nos_params *self, FILE *stream)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(stream != NULL, CPL_ERROR_NULL_INPUT);

    fprintf(stream, "--moo_nos_params\n");
    fprintf(stream, "clip-diff  %f\n", self->clip_diff);
    fprintf(stream, "clip-frac  %f\n", self->clip_frac);
    fprintf(stream, "clip-niter %d\n", self->clip_niter);
    fprintf(stream, "clip-kappa %f\n", self->clip_kappa);
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get compute noise parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_nos_params *
moo_params_get_nos(const moo_params *self, const cpl_parameterlist *list)
{
    moo_nos_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);

    double kappa = moo_params_get_double(self, list, "nos_clip_kappa");
    double diff = moo_params_get_double(self, list, "nos_clip_diff");
    double maxfrac = moo_params_get_double(self, list, "nos_clip_maxfrac");
    int niter = moo_params_get_int(self, list, "nos_clip_niter");
    res = moo_nos_params_new();
    res->clip_diff = diff;
    res->clip_frac = maxfrac;
    res->clip_kappa = kappa;
    res->clip_niter = niter;

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get compute hot map parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_hot_params *
moo_params_get_hot(const moo_params *self, const cpl_parameterlist *list)
{
    moo_hot_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);

    double kappa = moo_params_get_double(self, list, "hot_clip_kappa");
    double diff = moo_params_get_double(self, list, "hot_clip_diff");
    double maxfrac = moo_params_get_double(self, list, "hot_clip_maxfrac");
    int niter = moo_params_get_int(self, list, "hot_clip_niter");
    int lwinhsize = moo_params_get_int(self, list, "hot_local_winhsize");
    double lkappa = moo_params_get_double(self, list, "hot_local_kappa");
    res = moo_hot_params_new();
    res->clip_diff = diff;
    res->clip_frac = maxfrac;
    res->clip_kappa = kappa;
    res->clip_niter = niter;
    res->local_winhsize = lwinhsize;
    res->local_kappa = lkappa;
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get remove prepare parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_prepare_params *
moo_params_get_prepare(const moo_params *self, const cpl_parameterlist *list)
{
    moo_prepare_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);

    res = moo_prepare_params_new();
    _moo_params_get_tab_int(self, res->ignore_detector, list,
                            "prepare_ignore_detectors");

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get remove crh parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_crh_params *
moo_params_get_crh(const moo_params *self, const cpl_parameterlist *list)
{
    moo_crh_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);

    double kappa = moo_params_get_double(self, list, "crh_clip_kappa");
    int niter = moo_params_get_int(self, list, "crh_clip_niter");
    const char *method = moo_params_get_string(self, list, "crh_method");
    moo_try_assure(((strcmp(method, MOO_CRH_METHOD_MEDIAN) == 0) ||
                    (strcmp(method, MOO_CRH_METHOD_SIGCLIP) == 0) ||
                    (strcmp(method, MOO_CRH_METHOD_MEAN) == 0)),
                   CPL_ERROR_ILLEGAL_INPUT,
                   "Supported crh method are %s or %s (%s is given)",
                   MOO_CRH_METHOD_MEDIAN, MOO_CRH_METHOD_SIGCLIP, method);

    res = moo_crh_params_new();
    res->method = method;
    res->niter = niter;
    res->kappa = kappa;

moo_try_cleanup:
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get bias parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_bias_params *
moo_params_get_bias(const moo_params *self, const cpl_parameterlist *list)
{
    moo_bias_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);


    const char *ron_estimation_method =
        moo_params_get_string(self, list, "ron_estimation_method");

    moo_try_check(res = moo_bias_params_new(), " ");
    res->ron_estimation_method = ron_estimation_method;

moo_try_cleanup:
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get correct_bias parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_correct_bias_params *
moo_params_get_correct_bias(const moo_params *self,
                            const cpl_parameterlist *list)
{
    moo_correct_bias_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);


    const char *subtract_bias_method =
        moo_params_get_string(self, list, "subtract_bias_method");
    moo_try_check(res = moo_correct_bias_params_new(), " ");
    res->subtract_bias_method = subtract_bias_method;

moo_try_cleanup:
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get extraction parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_extract_params *
moo_params_get_extract(const moo_params *self, const cpl_parameterlist *list)
{
    moo_extract_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const char *method = NULL;
    moo_try_check(method = moo_params_get_string(self, list, "extract_method"),
                  " ");
    moo_try_assure(((strcmp(method, MOO_EXTRACT_METHOD_SUM) == 0) ||
                    (strcmp(method, MOO_EXTRACT_METHOD_OPTIMAL) == 0)),
                   CPL_ERROR_ILLEGAL_INPUT,
                   "Supported extract method are %s or %s (%s is given)",
                   MOO_EXTRACT_METHOD_SUM, MOO_EXTRACT_METHOD_OPTIMAL, method);

    res = moo_extract_params_new();
    res->method = method;
    _moo_params_get_tab_double(self, res->aperture, list, "extract_aperture");

moo_try_cleanup:
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get localisation parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_localise_params *
moo_params_get_localise(const moo_params *self, const cpl_parameterlist *list)
{
    moo_localise_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int central_pos = moo_params_get_int(self, list, "central_pos");
    int central_width = moo_params_get_int(self, list, "central_width");
    const char *method = moo_params_get_string(self, list, "loc_method");
    int keep_points = moo_params_get_bool(self, list, "keep_points");
    double rthresh = moo_params_get_double(self, list, "relative_thresh");
    double wdiff = moo_params_get_double(self, list, "wdiff_lim");
    double ydiff = moo_params_get_double(self, list, "ydiff_lim");
    int loc_xlim_hwin = moo_params_get_int(self, list, "loc_xlim_hwin");
    double loc_xlim_fracmin =
        moo_params_get_double(self, list, "loc_xlim_fracmin");
    res = moo_localise_params_new();
    res->centralpos = central_pos;
    res->centralwidth = central_width;
    res->method = method;
    res->keep_points = keep_points;
    res->relativethresh = rthresh;
    res->wdiff_lim = wdiff;
    res->ydiff_lim = ydiff;
    _moo_params_get_tab_double(self, res->ref_snr, list, "ref_snr");

    res->loc_xlim_hwin = loc_xlim_hwin;
    res->loc_xlim_fracmin = loc_xlim_fracmin;

    const char *backg_method =
        moo_params_get_string(self, list, "loc_backg_method");
    if ((strcmp(backg_method, MOO_LOCALISE_BACKG_METHOD_POLYNOMIAL) == 0) ||
        (strcmp(backg_method, MOO_LOCALISE_BACKG_METHOD_RUNNINGMIN) == 0)) {
        res->backg_method = backg_method;
    }
    else {
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                              "Invalid parameter backg-method (%s) : "
                              "POLYNOMIAL or RUNNINGMIN expected",
                              backg_method);
    }
    _moo_params_get_tab_int(self, res->backg_winhsize, list,
                            "loc_backg_winhsize");
    _moo_params_get_tab_int(self, res->backg_polydeg, list,
                            "loc_backg_polydeg");
    _moo_params_get_tab_double(self, res->backg_clip_kappalow, list,
                               "loc_backg_clip_kappalow");
    _moo_params_get_tab_double(self, res->backg_clip_kappaup, list,
                               "loc_backg_clip_kappaup");
    _moo_params_get_tab_int(self, res->backg_clip_niter, list,
                            "loc_backg_clip_niter");
    _moo_params_get_tab_int(self, res->detect_niter, list, "loc_detect_niter");
    int xgap_max = moo_params_get_int(self, list, "loc_xgap_max");
    res->xgap_max = xgap_max;
    _moo_params_get_tab_double(self, res->goodptsfrac_min, list,
                               "loc_goodptsfrac_min");
    const char *poly_deg = moo_params_get_string(self, list, "loc_poly_deg");
    int a, b, c, d, e, f;
    int nb = sscanf(poly_deg, "%d,%d,%d,%d,%d,%d", &a, &b, &c, &d, &e, &f);
    if (nb == 1) {
        for (int i = 0; i < 6; i++) {
            res->polydeg[i] = a;
        }
    }
    else if (nb == 3) {
        for (int i = 0; i < 2; i++) {
            res->polydeg[i * 3] = a;
            res->polydeg[i * 3 + 1] = b;
            res->polydeg[i * 3 + 2] = c;
        }
    }
    else if (nb == 6) {
        res->polydeg[0] = a;
        res->polydeg[1] = b;
        res->polydeg[2] = c;
        res->polydeg[3] = d;
        res->polydeg[4] = e;
        res->polydeg[5] = f;
    }
    else {
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                              "Invalid parameter poly-deg (%s) : %d values are "
                              "read and 1,3 or 6 expected",
                              poly_deg, nb);
    }

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get keep-temp parameter from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
int
moo_params_get_keep_temp(const moo_params *self, const cpl_parameterlist *list)
{
    int res = 0;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, 0);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, 0);

    res = moo_params_get_bool(self, list, "keep_temp");
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get quicklook_fibre_list parameter from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
cpl_array *
moo_params_get_quicklook_fibre_list(const moo_params *self,
                                    const cpl_parameterlist *list)
{
    cpl_array *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, 0);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, 0);

    char *string =
        cpl_strdup(moo_params_get_string(self, list, "quicklook_fibre_list"));
    char *strToken = strtok(string, ",");
    int size = 0;
    res = cpl_array_new(size, CPL_TYPE_STRING);

    while (strToken != NULL) {
        size++;
        cpl_array_set_size(res, size);
        cpl_array_set_string(res, size - 1, strToken);
        strToken = strtok(NULL, ",");
    }
    cpl_free(string);
    return res;
}

moo_localise_params *
moo_localise_params_new(void)
{
    moo_localise_params *res =
        (moo_localise_params *)cpl_calloc(1, sizeof(moo_localise_params));
    return res;
}

void
moo_localise_params_delete(moo_localise_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for localisation
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_localise(moo_params *self, cpl_parameterlist *list)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(list != NULL, CPL_ERROR_NULL_INPUT);

    moo_params_add_int(
        self, list, "central_pos", "central-pos",
        "x location of the band where initial fibre positions are determined",
        2048);
    moo_params_add_int(self, list, "central_width", "central-width",
                       "total width of the band in pixels", 25);
    moo_params_add_string(
        self, list, "loc_backg_method", "loc-backg-method",
        "method for background fitting (POLYNOMIAL|RUNNINGMIN).", "POLYNOMIAL");
    moo_params_add_string(self, list, "loc_backg_winhsize",
                          "loc-backg-winhsize",
                          "half size of window for running minimum, for each "
                          "detector, ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
                          "3,3,3,3,3,3");
    moo_params_add_string(self, list, "loc_backg_polydeg", "loc-backg-polydeg",
                          "list of background fit polynomial degree, for each "
                          "detector, ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
                          "4,4,4,4,4,4");
    moo_params_add_string(
        self, list, "loc_backg_clip_kappalow", "loc-backg-clip-kappalow",
        "list of lower multiples of sigma in background fitting, for each "
        "detector, ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
        "3,3,3,3,3,3");
    moo_params_add_string(
        self, list, "loc_backg_clip_kappaup", "loc-backg-clip-kappaup",
        "list of upper multiples of sigma in background fitting, for each "
        "detector, ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
        "1,1,1,1,1,1");
    moo_params_add_string(
        self, list, "loc_backg_clip_niter", "loc-backg-clip-niter",
        "list of numbers of iterations in background fitting, for each "
        "detector, ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
        "4,4,4,4,4,4");
    moo_params_add_string(
        self, list, "loc_detect_niter", "loc-detect-niter",
        "list of number of iterations to detect fibres, for each detector, "
        "ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
        "10,10,10,10,10,10");
    moo_params_add_int(
        self, list, "loc_xgap_max", "loc-xgap-max",
        "maximum allowed consecutive centroid detection failures before stop.",
        100);
    moo_params_add_int(
        self, list, "loc_xlim_hwin", "loc-xlim-hwin",
        "half window size in pixels for x trace limits detections.", 30);
    moo_params_add_double(
        self, list, "loc_xlim_fracmin", "loc-xlim-fracmin",
        "minimal allowed fraction of centroid determinations in the defined "
        "window, below which trace limit is set.",
        0.2);
    moo_params_add_string(
        self, list, "loc_goodptsfrac_min", "loc-goodptsfrac-min",
        "minimum fraction of good pixels to detect fibres, for each detector, "
        "ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
        "0.6,0.6,0.6,0.6,0.6,0.6");
    moo_params_add_string(self, list, "loc_poly_deg", "loc-poly-deg",
                          "list of polynomial degree for each detector, "
                          "ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
                          "9,8,10,9,8,10");
    moo_params_add_string(
        self, list, "loc_method", "loc-method",
        "method used for fitting centroid (BARYCENTER,GAUSSIAN)",
        MOO_LOCALISE_METHOD_BARYCENTER);
    moo_params_add_bool(self, list, "keep_points", "keep-points",
                        "keep measured edge positions and flag map", CPL_FALSE);
    moo_params_add_double(self, list, "relative_thresh", "relative-thresh",
                          "relative detection threshold above background", 0.1);
    moo_params_add_double(self, list, "wdiff_lim", "wdiff-lim",
                          "maximum allowed variation of localisation width", 1);
    moo_params_add_double(self, list, "ydiff_lim", "ydiff-lim",
                          "maximum allowed variation of localisation centroid",
                          0.5);
    moo_params_add_string(
        self, list, "ref_snr", "ref-snr",
        "minimum reference SNR used for localisation tracking, for each "
        "detector, ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
        "80.,80.,80.,80.,80.,80.");
    return 0;
}

moo_model_flat_params *
moo_model_flat_params_new(void)
{
    moo_model_flat_params *res =
        (moo_model_flat_params *)cpl_calloc(1, sizeof(moo_model_flat_params));
    return res;
}

void
moo_model_flat_params_delete(moo_model_flat_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for model flat
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_model_flat(moo_params *self, cpl_parameterlist *list)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(list != NULL, CPL_ERROR_NULL_INPUT);

    moo_params_add_int(self, list, "modelflat_oversample",
                       "modelflat-oversample",
                       "oversample factor in Y direction", 5);
    moo_params_add_int(self, list, "modelflat_xstep", "modelflat-xstep",
                       "step along X axis for computing parameters profile",
                       10);
    moo_params_add_int(
        self, list, "modelflat_winhsize", "modelflat-winhsize",
        "half size window along X axis for computing parameters profile", 20);
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get molecfit model parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_molecfit_model_params *
moo_params_get_molecfit_model(const moo_params *self,
                              const cpl_parameterlist *list)
{
    moo_molecfit_model_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);

    res = moo_molecfit_model_params_new();
    res->continuum_const =
        moo_params_get_double(self, list, MF_PARAMETERS_CONTINUUM_CONST);
    res->var_kern = moo_params_get_bool(self, list, MF_PARAMETERS_VAR_KERN);
    res->kern_mode = moo_params_get_bool(self, list, MF_PARAMETERS_KERN_MODE);
    res->kern_fac = moo_params_get_double(self, list, MF_PARAMETERS_KERN_FAC);

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get molecfit calctrans parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_molecfit_calctrans_params *
moo_params_get_molecfit_calctrans(const moo_params *self,
                                  const cpl_parameterlist *list)
{
    moo_molecfit_calctrans_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);

    res = moo_molecfit_calctrans_params_new();
    _moo_params_get_tab3_double(self, res->min_snr, list,
                                "molecfit_calctrans_min_snr");
    int filter_skyfibre =
        moo_params_get_bool(self, list, "molecfit_calctrans_filter_skyfibre");
    res->filter_skyfibre = filter_skyfibre;

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get create s1d parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_create_s1d_params *
moo_params_get_create_s1d(const moo_params *self, const cpl_parameterlist *list)
{
    moo_create_s1d_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);

    res = moo_create_s1d_params_new();
    int do_s1d = moo_params_get_bool(self, list, "create_s1d");
    res->do_s1d = do_s1d;

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get model flat parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_model_flat_params *
moo_params_get_model_flat(const moo_params *self, const cpl_parameterlist *list)
{
    moo_model_flat_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int oversample = moo_params_get_int(self, list, "modelflat_oversample");
    int winhsize = moo_params_get_int(self, list, "modelflat_winhsize");
    int xstep = moo_params_get_int(self, list, "modelflat_xstep");
    res = moo_model_flat_params_new();
    res->oversamp = oversample;
    res->winhsize = winhsize;
    res->xstep = xstep;
    return res;
}

moo_extract_params *
moo_extract_params_new(void)
{
    moo_extract_params *res =
        (moo_extract_params *)cpl_calloc(1, sizeof(moo_extract_params));
    return res;
}

void
moo_extract_params_delete(moo_extract_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

moo_compute_fibtrans_params *
moo_compute_fibtrans_params_new(void)
{
    moo_compute_fibtrans_params *res = (moo_compute_fibtrans_params *)
        cpl_calloc(1, sizeof(moo_compute_fibtrans_params));
    return res;
}

void
moo_compute_fibtrans_params_delete(moo_compute_fibtrans_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

moo_compute_resp_params *
moo_compute_resp_params_new(void)
{
    moo_compute_resp_params *res =
        (moo_compute_resp_params *)cpl_calloc(1,
                                              sizeof(moo_compute_resp_params));
    return res;
}

void
moo_compute_resp_params_delete(moo_compute_resp_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

moo_wavesol_params *
moo_wavesol_params_new(void)
{
    moo_wavesol_params *res =
        (moo_wavesol_params *)cpl_calloc(1, sizeof(moo_wavesol_params));

    return res;
}

void
moo_wavesol_params_delete(moo_wavesol_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

moo_rebin_params *
moo_rebin_params_new(void)
{
    moo_rebin_params *res =
        (moo_rebin_params *)cpl_calloc(1, sizeof(moo_rebin_params));
    return res;
}

void
moo_rebin_params_delete(moo_rebin_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

moo_sub_sky_stare_params *
moo_sub_sky_stare_params_new(void)
{
    moo_sub_sky_stare_params *res = (moo_sub_sky_stare_params *)
        cpl_calloc(1, sizeof(moo_sub_sky_stare_params));
    return res;
}

void
moo_sub_sky_stare_params_delete(moo_sub_sky_stare_params *self)
{
    if (self != NULL) {
        moo_skycorr_params_delete(self->sk);
        cpl_free(self);
    }
}

moo_skycorr_params *
moo_skycorr_params_new(void)
{
    moo_skycorr_params *res =
        (moo_skycorr_params *)cpl_calloc(1, sizeof(moo_skycorr_params));
    return res;
}

void
moo_skycorr_params_delete(moo_skycorr_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

moo_combine_pair_params *
moo_combine_pair_params_new(void)
{
    moo_combine_pair_params *res =
        (moo_combine_pair_params *)cpl_calloc(1,
                                              sizeof(moo_combine_pair_params));
    return res;
}

void
moo_combine_pair_params_delete(moo_combine_pair_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

moo_target_table_params *
moo_target_table_params_new(void)
{
    moo_target_table_params *res =
        (moo_target_table_params *)cpl_calloc(1,
                                              sizeof(moo_target_table_params));
    return res;
}

void
moo_target_table_params_delete(moo_target_table_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

moo_compute_snr_params *
moo_compute_snr_params_new(void)
{
    moo_compute_snr_params *res =
        (moo_compute_snr_params *)cpl_calloc(1, sizeof(moo_compute_snr_params));
    return res;
}

void
moo_compute_snr_params_delete(moo_compute_snr_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

moo_compute_slitoffset_params *
moo_compute_slitoffset_params_new(void)
{
    moo_compute_slitoffset_params *res = (moo_compute_slitoffset_params *)
        cpl_calloc(1, sizeof(moo_compute_slitoffset_params));
    return res;
}

void
moo_compute_slitoffset_params_delete(moo_compute_slitoffset_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

moo_coadd_params *
moo_coadd_params_new(void)
{
    moo_coadd_params *res =
        (moo_coadd_params *)cpl_calloc(1, sizeof(moo_coadd_params));
    return res;
}

void
moo_coadd_params_delete(moo_coadd_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for compute_fibtrans
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_compute_fibtrans(moo_params *self, cpl_parameterlist *list)
{
    moo_params_add_string(self, list, "ref_fibres", "ref-fibres",
                          "couple of fibre ids for reference transmission, for "
                          "each spectrograph, ordered as TAS1,TAS2.",
                          "506,506");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for compute_resp
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_compute_resp(moo_params *self, cpl_parameterlist *list)
{
    moo_params_add_string(
        self, list, "computeresp_filter_winhsize",
        "computeresp-filter-winhsize",
        "median filter half window size, for each band, ordered as RI,YJ,H.",
        "25,25,25");
    moo_params_add_string(self, list, "computeresp_degree",
                          "computeresp-degree",
                          "polynomial degree to fit the response, for each "
                          "band, ordered as RI,YJ,H.",
                          "4,4,4");
    moo_params_add_double(self, list, "computeresp_kappa_lo",
                          "computeresp-kappa-lo",
                          "lower multiple of sigma in response fitting.", 1);
    moo_params_add_double(self, list, "computeresp_kappa_up",
                          "computeresp-kappa-up",
                          "upper multiple of sigma in response fitting.", 4);
    moo_params_add_double(self, list, "computeresp_frac", "computeresp-frac",
                          "maximum rejected fraction to fit the response", 0.7);
    moo_params_add_int(self, list, "computeresp_niter", "computeresp-niter",
                       "number of iterations to fit the response", 5);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get compute fibtrans parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_compute_fibtrans_params *
moo_params_get_compute_fibtrans(const moo_params *self,
                                const cpl_parameterlist *list)
{
    moo_compute_fibtrans_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);


    res = moo_compute_fibtrans_params_new();


    const char *ref = moo_params_get_string(self, list, "ref_fibres");
    int a, b;
    int nb = sscanf(ref, "%d,%d", &a, &b);
    if (nb == 2) {
        res->fibref[0] = a;
        res->fibref[1] = b;
    }
    else {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT,
            "Invalid parameter ref-fibres (%s) : %d values are read 2 expected",
            ref, nb);
    }

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get compute response parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_compute_resp_params *
moo_params_get_compute_resp(const moo_params *self,
                            const cpl_parameterlist *list)
{
    moo_compute_resp_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);


    res = moo_compute_resp_params_new();
    _moo_params_get_tab3_int(self, res->filter_winhsize, list,
                             "computeresp_filter_winhsize");
    _moo_params_get_tab3_int(self, res->degree, list, "computeresp_degree");
    res->frac = moo_params_get_double(self, list, "computeresp_frac");
    res->kappa_lo = moo_params_get_double(self, list, "computeresp_kappa_lo");
    res->kappa_up = moo_params_get_double(self, list, "computeresp_kappa_up");
    res->niter = moo_params_get_int(self, list, "computeresp_niter");
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for moo_wavesol
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_wavesol(moo_params *self, cpl_parameterlist *list)
{
    moo_params_add_string(
        self, list, "wavesol_linedetect_nlines", "wavesol-linedetect-nlines",
        "list of number of expected lines for peak detection for each "
        "detector, ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
        "100,100,100,100,100,100");
    moo_params_add_string(
        self, list, "wavesol_linedetect_winhsize",
        "wavesol-linedetect-winhsize",
        "list of half size window in pixel for gaussian fit for peak detection "
        "using for pattern matching for each detector, ordered as "
        "RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
        "2,2,2,2,2,2");
    moo_params_add_string(self, list, "wavesol_ppm_tolerance",
                          "wavesol-ppm-tolerance",
                          "list of pattern points matching tolerance for each "
                          "detector, ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
                          "0.007,0.009,0.007,0.007,0.009,0.007");
    moo_params_add_string(self, list, "wavesol_ppm_wavemap_deg_x",
                          "wavesol-ppm-wavemap-deg-x",
                          "pattern matching polynomial wavelength map degree "
                          "in x for each band, ordered as RI,YJ,H",
                          "3,3,3");
    moo_params_add_int(self, list, "wavesol_ppm_wavemap_deg_y",
                       "wavesol-ppm-wavemap-deg-y",
                       "pattern matching polynomial wavelength map degree in y",
                       1);

    moo_params_add_string(self, list, "wavesol_min_fwhm", "wavesol-min-fwhm",
                          "minimum FWHM to filter lines for each detector, "
                          "ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
                          "1.9,1.2,1.2,1.9,1.2,1.2");
    moo_params_add_string(self, list, "wavesol_max_fwhm", "wavesol-max-fwhm",
                          "maximum FWHM to filter lines for each detector, "
                          "ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
                          "5,5,5,5,5,5");
    moo_params_add_string(self, list, "wavesol_min_sn", "wavesol-min-sn",
                          "minimum SNR to filter lunes for each detector, "
                          "ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
                          "4,4,4,4,4,4");
    moo_params_add_string(
        self, list, "wavesol_linefit_method", "wavesol-linefit-method",
        "method of centroid peak detection (GAUSSIAN|BARYCENTER)",
        MOO_WAVESOL_LINEFIT_GAUSSIAN);
    moo_params_add_string(
        self, list, "wavesol_linefit_winhsize", "wavesol-linefit-winhsize",
        "list of half size window in pixel for gaussian fit for peak detection "
        "for each detector, ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
        "2,2,2,2,2,2");
    moo_params_add_string(
        self, list, "wavesol_linefit_recentre", "wavesol-linefit-recentre",
        "list of flags (0:FALSE, 1:TRUE) to use recentre mode for each "
        "detector, ordered as RI_1,YJ_1,H_1,RI_2,YJ_2,H_2.",
        "0,0,0,0,0,0");
    moo_params_add_string(self, list, "wavesol_model", "wavesol-model",
                          "model of the polynomial wavelength solution (1D|2D)",
                          MOO_WAVESOL_MODEL_1D);
    moo_params_add_string(self, list, "wavesol_wavemap_deg_x",
                          "wavesol-wavemap-deg-x",
                          "polynomial wavelength map degree in x for each "
                          "band, ordered as RI,YJ,H.",
                          "3,3,3");

    moo_params_add_int(self, list, "wavesol_wavemap_deg_y",
                       "wavesol-wavemap-deg-y",
                       "polynomial wavelength map degree in y", 8);
    moo_params_add_double(self, list, "wavesol_clip_kappa",
                          "wavesol-clip-kappa",
                          "multiple of sigma in sigma clipping", 2.0);
    moo_params_add_double(self, list, "wavesol_clip_frac", "wavesol-clip-frac",
                          "maximal fractions of rejected point allowed", 0.7);
    moo_params_add_int(self, list, "wavesol_clip_niter", "wavesol-clip-niter",
                       "maximum number of iterations for sigma clipping", 10);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default science parameters for moo_wavesol
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_science_wavesol(moo_params *self, cpl_parameterlist *list)
{
    moo_params_add_string(self, list, "wavesol_control", "wavesol-control",
                          "choose to create the wavelength solution using sky "
                          "lines (NONE or CHECK or UPDATE)",
                          "NONE");
    moo_params_add_string(self, list, "wavesol_min_fwhm", "wavesol-min-fwhm",
                          "minimum FWHM to filter lines for each detectors",
                          "1.9,1.2,1.2,1.9,1.2,1.2");
    moo_params_add_string(self, list, "wavesol_max_fwhm", "wavesol-max-fwhm",
                          "maximum FWHM to filter lines for each detectors",
                          "5,5,5,5,5,5");
    moo_params_add_string(
        self, list, "wavesol_linefit_method", "wavesol-linefit-method",
        "method of centroid peak detection (GAUSSIAN|BARYCENTER)",
        MOO_WAVESOL_LINEFIT_GAUSSIAN);
    moo_params_add_string(self, list, "wavesol_linefit_winhsize",
                          "wavesol-linefit-winhsize",
                          "list of half size window in pixel for gaussian fit "
                          "for peak detection for each detectors",
                          "2,2,2,2,2,2");
    moo_params_add_string(self, list, "wavesol_linefit_recentre",
                          "wavesol-linefit-recentre",
                          "list of flags (0:FALSE 1:TRUE) to use recentre mode "
                          "for each detectors",
                          "0,0,0,0,0,0");
    moo_params_add_string(self, list, "wavesol_model", "wavesol-model",
                          "model of the polynomial wavelength solution (1D|2D)",
                          MOO_WAVESOL_MODEL_1D);
    moo_params_add_string(self, list, "wavesol_wavemap_deg_x",
                          "wavesol-wavemap-deg-x",
                          "polynomial wavelength map degree in x for each band",
                          "3,3,3");

    moo_params_add_int(self, list, "wavesol_wavemap_deg_y",
                       "wavesol-wavemap-deg-y",
                       "polynomial wavelength map degree in y", 8);
    moo_params_add_double(self, list, "wavesol_clip_kappa",
                          "wavesol-clip-kappa",
                          "multiple of sigma in sigma clipping", 2.0);
    moo_params_add_double(self, list, "wavesol_clip_frac", "wavesol-clip-frac",
                          "maximal fractions of rejected point allowed", 0.7);
    moo_params_add_int(self, list, "wavesol_clip_niter", "wavesol-clip-niter",
                       "maximum number of iterations for sigma clipping", 10);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for moo_rebin
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_rebin(moo_params *self, cpl_parameterlist *list)
{
    moo_params_add_string(
        self, list, "rebin_step", "rebin-step",
        "list of bin step in nm for each band, ordered as RI,YJ,H.",
        "0.05,0.05,0.05");
    moo_params_add_string(self, list, "rebin_method", "rebin-method",
                          "method of rebinning (INTEGRATE|INTERPOLATE)",
                          MOO_REBIN_METHOD_INTEGRATE);
    moo_params_add_bool(self, list, "rebin_flux_conserv", "rebin-flux-conserv",
                        "conserve the flux.", CPL_FALSE);
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for moo_sub_sky_stare
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_sub_sky_stare(moo_params *self, cpl_parameterlist *list)
{
    moo_params_add_string(self, list, "subskystare_method",
                          "subskystare-method",
                          "method for sky subtraction (SIMPLE|SKYCORR)",
                          MOO_SUB_SKY_STARE_METHOD_SIMPLE);
    moo_params_add_double(
        self, list, "subskystare_radius_sky", "subskystare-radius-sky",
        "maximum distance in arcmin between sky and reference position", 1.5);
    moo_params_add_double(
        self, list, "subskystare_radius_stepr", "subskystare-radius-stepr",
        "increase step in arcmin for the radius around reference position",
        0.15);
    moo_params_add_int(
        self, list, "subskystare_min_sky", "subskystare-min-sky",
        "minimum number of sky to median in the region defined by radius-sky",
        5);
    moo_params_add_int(self, list, "subskystare_maxdistslit",
                       "subskystare-maxdistslit",
                       "maximum distance of sky fibres on slit", 20);
    moo_params_add_double(self, list, "subskystare_mintrans",
                          "subskystare-mintrans",
                          "minimum transmission of sky fibres on slit", 0);
    moo_params_add_skycorr(self, list);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for moo_sub_sky_stare
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_sub_sky_stare_wnone(moo_params *self, cpl_parameterlist *list)
{
    moo_params_add_string(self, list, "subskystare_method",
                          "subskystare-method",
                          "method for sky subtraction (SIMPLE|SKYCORR|NONE)",
                          MOO_SUB_SKY_STARE_METHOD_SIMPLE);
    moo_params_add_double(
        self, list, "subskystare_radius_sky", "subskystare-radius-sky",
        "maximum distance in arcmin between sky and reference position", 1.5);
    moo_params_add_double(
        self, list, "subskystare_radius_stepr", "subskystare-radius-stepr",
        "increase step in arcmin for the radius around reference position",
        0.15);
    moo_params_add_int(
        self, list, "subskystare_min_sky", "subskystare-min-sky",
        "minimum number of sky to median in the region defined by radius-sky",
        5);
    moo_params_add_int(self, list, "subskystare_maxdistslit",
                       "subskystare-maxdistslit",
                       "maximum distance of sky fibres on slit", 20);
    moo_params_add_double(self, list, "subskystare_mintrans",
                          "subskystare-mintrans",
                          "minimum transmission of sky fibres on slit", 0);
    moo_params_add_skycorr(self, list);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new empty molecfit_model set of parameters

  @return   1 newly allocated moo_nos_params or NULL in case of an error

  The returned object must be deallocated using moo_nos_params_delete().
  - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_molecfit_model_params *
moo_molecfit_model_params_new(void)
{
    moo_molecfit_model_params *res = (moo_molecfit_model_params *)
        cpl_calloc(1, sizeof(moo_molecfit_model_params));
    return res;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_molecfit_model_params
  @param    self    moo_params to delete
  @return   void

  If @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/
void
moo_molecfit_model_params_delete(moo_molecfit_model_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new empty molecfit_calctrans set of parameters

  @return   1 newly allocated moo_nos_params or NULL in case of an error

  The returned object must be deallocated using 
  moo_molecfit_calctrans_params_delete().
  - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_molecfit_calctrans_params *
moo_molecfit_calctrans_params_new(void)
{
    moo_molecfit_calctrans_params *res = (moo_molecfit_calctrans_params *)
        cpl_calloc(1, sizeof(moo_molecfit_calctrans_params));
    return res;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_molecfit_calctrans_params
  @param    self    moo_params to delete
  @return   void

  If @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/
void
moo_molecfit_calctrans_params_delete(moo_molecfit_calctrans_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new empty create_s1d set of parameters

  @return   1 newly allocated moo_create_s1d_params or NULL in case of an error

  The returned object must be deallocated using 
  moo_create_s1d_params_delete().
  - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_create_s1d_params *
moo_create_s1d_params_new(void)
{
    moo_create_s1d_params *res =
        (moo_create_s1d_params *)cpl_calloc(1, sizeof(moo_create_s1d_params));
    return res;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_create_s1d_params
  @param    self    moo_params to delete
  @return   void

  If @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/
void
moo_create_s1d_params_delete(moo_create_s1d_params *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for molecfit_model
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if method is invalid
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_molecfit_model(moo_params *self, cpl_parameterlist *list)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);


    moo_params_add_double(self, list, MF_PARAMETERS_CONTINUUM_CONST,
                          MF_PARAMETERS_CONTINUUM_CONST,
                          MF_PARAMETERS_CONTINUUM_CONST_DESC,
                          MF_PARAMETERS_CONTINUUM_CONST_INIT);
    moo_params_add_bool(self, list, MF_PARAMETERS_VAR_KERN,
                        MF_PARAMETERS_VAR_KERN, MF_PARAMETERS_VAR_KERN_DESC,
                        MF_PARAMETERS_VAR_KERN_INIT);
    moo_params_add_bool(self, list, MF_PARAMETERS_KERN_MODE,
                        MF_PARAMETERS_KERN_MODE, MF_PARAMETERS_KERN_MODE_DESC,
                        MF_PARAMETERS_KERN_MODE_INIT);
    moo_params_add_double(self, list, MF_PARAMETERS_KERN_FAC,
                          MF_PARAMETERS_KERN_FAC, MF_PARAMETERS_KERN_FAC_DESC,
                          MF_PARAMETERS_KERN_FAC_INIT);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for molecfit_calctrans
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if method is invalid
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_molecfit_calctrans(moo_params *self, cpl_parameterlist *list)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    moo_params_add_string(
        self, list, "molecfit_calctrans_min_snr", "molecfit-calctrans-min-snr",
        "list of minimum SNR (MEDIAN_SNR_{}_EXT) for each detectors",
        "15,15,15");
    moo_params_add_bool(self, list, "molecfit_calctrans_filter_skyfibre",
                        "molecfit-calctrans-filter-skyfibre",
                        "filter the SKY fibres.", CPL_TRUE);

    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for create_s1d
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if method is invalid
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_create_s1d(moo_params *self, cpl_parameterlist *list)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    moo_params_add_bool(self, list, "create_s1d", "create-s1d",
                        "Create the S1D files from the PRODUCT SCI", CPL_FALSE);

    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for moo_sub_sky_stare
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_sub_sky_stare_simple(moo_params *self, cpl_parameterlist *list)
{
    moo_params_add_double(
        self, list, "subskystare_radius_sky", "subskystare-radius-sky",
        "maximum distance in arcmin between sky and reference position", 1.5);
    moo_params_add_double(
        self, list, "subskystare_radius_stepr", "subskystare-radius-stepr",
        "increase step in arcmin for the radius around reference position",
        0.15);
    moo_params_add_int(
        self, list, "subskystare_min_sky", "subskystare-min-sky",
        "minimum number of sky to median in the region defined by radius-sky",
        5);
    moo_params_add_int(self, list, "subskystare_maxdistslit",
                       "subskystare-maxdistslit",
                       "maximum distance of sky fibres on slit", 20);
    moo_params_add_double(self, list, "subskystare_mintrans",
                          "subskystare-mintrans",
                          "minimum transmission of sky fibres on slit", 0);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for moo_skycorr
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_skycorr(moo_params *self, cpl_parameterlist *list)
{
    moo_params_add_double(
        self, list, "skycorr_ltol", "skycorr-ltol",
        "LINE IDENTIFICATION: Relative FWHM convergence criterion ", 0.01);
    moo_params_add_double(
        self, list, "skycorr_min_line_dist_fac", "skycorr-min-line-dist-fac",
        "LINE IDENTIFICATION: Minimum distance to neighbouring lines for\
 classification as isolated line:<MIN_LINE_DIST> * <FWHM> [pixel] ",
        2.5);
    moo_params_add_double(
        self, list, "skycorr_min_line_flux_fac", "skycorr-min-line-flux-fac",
        "LINE IDENTIFICATION: Relative lower flux limit for isolated lines",
        0.);
    moo_params_add_double(
        self, list, "skycorr_fluxlim", "skycorr-fluxlim",
        "LINE IDENTIFICATION: Minimum line peak flux for consideration of\
 lines from airglow line list:<FLUXLIM> * <median flux of identified lines>\
  Automatic search -> FLUXLIM = -1",
        -1);
    moo_params_add_double(
        self, list, "skycorr_ftol", "skycorr-ftol",
        "FITTING OF SKY LINES: Relative chi^2 MPFIT convergence criterion",
        1e-3);
    moo_params_add_double(
        self, list, "skycorr_xtol", "skycorr-xtol",
        "FITTING OF SKY LINES: Relative parameter MPFIT convergence criterion",
        1e-3);
    moo_params_add_double(
        self, list, "skycorr_wtol", "skycorr-wtol",
        "FITTING OF SKY LINES: Relative chi^2 convergence criterion for iterative improvement of \
wavelength grid",
        1e-3);
    moo_params_add_int(
        self, list, "skycorr_cheby_max", "skycorr-cheby-max",
        "FITTING OF SKY LINES: maximum degree of Chebyshev polynomial for\
 wavelength grid correction:-1 = no correction,0 = linear term (coef. = 1) \
is also considered but not fitted",
        7);
    moo_params_add_int(
        self, list, "skycorr_cheby_min", "skycorr-cheby-min",
        "FITTING OF SKY LINES: Minimum degree of Chebyshev polynomial for \
wavelength grid correction. CHEBY_MIN <= CHEBY_MAX: - Iterative increase of \
polynomial degree at least until CHEBY_MIN - Procedure stops if chi^2 gets \
worse or CHEBY_MAX is reached.- Results of degree with best chi^2 are taken. \
CHEBY_MIN > CHEBY_MAX:- Iterative increase of polynomial degree until CHEBY_MAX \
is reached.- Results of degree CHEBY_MAX are taken. ",
        3);

    moo_params_add_double(
        self, list, "skycorr_cheby_const", "skycorr-cheby-const",
        "FITTING OF SKY LINES: Initial constant term for wavelength grid \
correction (shift relative to half wavelength range)",
        0);
    moo_params_add_int(self, list, "skycorr_rebintype", "skycorr-rebintype",
                       "Type of rebinning: 0 = simple rebinning \
(summation of pixel fractions) 1 = convolution with asymmetric, damped sinc kernel",
                       1);
    moo_params_add_double(
        self, list, "skycorr_weightlim", "skycorr-weightlim",
        "minimum relative weight of the strongest line group of a pixel for\
 including a pixel in the line fitting procedure",
        0.67);

    moo_params_add_double(
        self, list, "skycorr_siglim", "skycorr-siglim",
        "sigma limit for excluding outliers (e.g. object emission lines) from "
        "estimate of group flux correction factors.",
        15.);

    moo_params_add_double(
        self, list, "skycorr_fitlim", "skycorr-fitlim",
        "Lower relative uncertainty limit for the consideration of a line \
group for the fitting procedure. The value is compared to the sigma-to-mean \
ratio of the group-specific flux correction factors of the initial estimate \
(0. -> include all fittable line groups).",
        0);


    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for moo_combine_pair
  @param    self    moo_params to update
  @param    list parameter list to update
  @param    value the default value for optimal parameter
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_combine_pair(moo_params *self,
                            cpl_parameterlist *list,
                            int value)
{
    moo_params_add_bool(
        self, list, "combine_pair_optimal", "combine-pair-optimal",
        "Extract the best sky-subtracted spectra for each target", value);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for moo_create_target_table
  @param    self    moo_params to update
  @param    list parameter list to update
  @param    value the default value to ignore SKY fibre
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_target_table(moo_params *self,
                            cpl_parameterlist *list,
                            int value)
{
    moo_params_add_bool(self, list, "target_table_nosky", "target-table-nosky",
                        "ignore SKY fibres as OBJECT.", value);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for moo_sci_compute_snr
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_compute_snr(moo_params *self, cpl_parameterlist *list)
{
    moo_params_add_string(self, list, "compute_snr_wmin", "compute-snr-wmin",
                          "list of minimum wavelength in nm for each detectors",
                          "810,1000,1605");
    moo_params_add_string(self, list, "compute_snr_wmax", "compute-snr-wmax",
                          "list of maximum wavelength in nm for each detectors",
                          "830,1100,1625");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for moo_sci_compute_slitoffset
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_compute_slitoffset(moo_params *self, cpl_parameterlist *list)
{
    moo_params_add_double(self, list, "compute_slitoffset_min",
                          "compute-slitoffset-min", "min offset range", -3);
    moo_params_add_double(self, list, "compute_slitoffset_max",
                          "compute-slitoffset-max", "max offset range", 3);
    moo_params_add_double(self, list, "compute_slitoffset_step",
                          "compute-slitoffset-step", "offset step", 0.1);
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add default parameters for moo_coadd
  @param    self    moo_params to update
  @param    list parameter list to update
  @return error code or CPL_ERROR_NONE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_params_add_coadd(moo_params *self, cpl_parameterlist *list)
{
    moo_params_add_string(self, list, "coadd_method", "coadd-method",
                          "method for co-adding spectra: (MEAN|MEDIAN|SIGCLIP)",
                          MOO_COADD_METHOD_MEAN);
    moo_params_add_double(self, list, "coadd_clip_kappa", "coadd-clip-kappa",
                          "multiple of sigma in 1D spectra combination. ", 3.0);
    moo_params_add_int(self, list, "coadd_clip_niter", "coadd-clip-niter",
                       "Number of iterations for sigma clipping", 5);

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Get wavesol parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_wavesol_params *
moo_params_get_wavesol(const moo_params *self, const cpl_parameterlist *list)
{
    moo_wavesol_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);

    res = moo_wavesol_params_new();
    _moo_params_get_tab_int(self, res->linedetect_nlines, list,
                            "wavesol_linedetect_nlines");
    _moo_params_get_tab_int(self, res->linedetect_winhsize, list,
                            "wavesol_linedetect_winhsize");
    _moo_params_get_tab_double(self, res->tolerance, list,
                               "wavesol_ppm_tolerance");
    _moo_params_get_tab_int(self, res->ppm_wavemap_degx, list,
                            "wavesol_ppm_wavemap_deg_x");
    res->ppm_wavemap_degy =
        moo_params_get_int(self, list, "wavesol_ppm_wavemap_deg_y");

    double frac = moo_params_get_double(self, list, "wavesol_clip_frac");
    double kappa = moo_params_get_double(self, list, "wavesol_clip_kappa");
    int niter = moo_params_get_int(self, list, "wavesol_clip_niter");
    _moo_params_get_tab_int(self, res->wavemap_degx, list,
                            "wavesol_wavemap_deg_x");
    int degy = moo_params_get_int(self, list, "wavesol_wavemap_deg_y");
    res->clip_frac = frac;
    res->clip_niter = niter;
    res->clip_kappa = kappa;
    res->wavemap_degy = degy;
    res->control = MOO_WAVESOL_CONTROL_NONE;
    const char *linefit_method =
        moo_params_get_string(self, list, "wavesol_linefit_method");

    if ((strcmp(linefit_method, MOO_WAVESOL_LINEFIT_GAUSSIAN) == 0) ||
        (strcmp(linefit_method, MOO_WAVESOL_LINEFIT_BARYCENTER) == 0)) {
        res->linefit_method = linefit_method;
    }
    else {
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                              "Invalid parameter wavesol-linefit-method (%s) : "
                              "GAUSSIAN or BARYCENTER expected",
                              linefit_method);
    }
    _moo_params_get_tab_int(self, res->linefit_winhsize, list,
                            "wavesol_linefit_winhsize");
    _moo_params_get_tab_int(self, res->linefit_recentre, list,
                            "wavesol_linefit_recentre");
    _moo_params_get_tab_double(self, res->fwhm_min, list, "wavesol_min_fwhm");
    _moo_params_get_tab_double(self, res->fwhm_max, list, "wavesol_max_fwhm");
    _moo_params_get_tab_double(self, res->min_snr, list, "wavesol_min_sn");
    const char *model = moo_params_get_string(self, list, "wavesol_model");
    if ((strcmp(model, MOO_WAVESOL_MODEL_1D) == 0) ||
        (strcmp(model, MOO_WAVESOL_MODEL_2D) == 0)) {
        res->model = model;
    }
    else {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT,
            "Invalid parameter wavesol-model (%s) : 1D or 2D expected", model);
    }

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get science_wavesol parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_wavesol_params *
moo_params_get_science_wavesol(const moo_params *self,
                               const cpl_parameterlist *list)
{
    moo_wavesol_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);

    res = moo_wavesol_params_new();

    double frac = moo_params_get_double(self, list, "wavesol_clip_frac");
    double kappa = moo_params_get_double(self, list, "wavesol_clip_kappa");
    int niter = moo_params_get_int(self, list, "wavesol_clip_niter");
    _moo_params_get_tab_int(self, res->wavemap_degx, list,
                            "wavesol_wavemap_deg_x");
    int degy = moo_params_get_int(self, list, "wavesol_wavemap_deg_y");
    res->clip_frac = frac;
    res->clip_niter = niter;
    res->clip_kappa = kappa;
    res->wavemap_degy = degy;
    const char *control = moo_params_get_string(self, list, "wavesol_control");
    if ((strcmp(control, MOO_WAVESOL_CONTROL_NONE) == 0) ||
        (strcmp(control, MOO_WAVESOL_CONTROL_CHECK) == 0) ||
        (strcmp(control, MOO_WAVESOL_CONTROL_UPDATE) == 0)) {
        res->control = control;
    }
    else {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT,
            "Invalid parameter wavesol-control (%s) : NONE or CHECK or UPDATE",
            control);
    }
    const char *linefit_method =
        moo_params_get_string(self, list, "wavesol_linefit_method");
    if ((strcmp(linefit_method, MOO_WAVESOL_LINEFIT_GAUSSIAN) == 0) ||
        (strcmp(linefit_method, MOO_WAVESOL_LINEFIT_BARYCENTER) == 0)) {
        res->linefit_method = linefit_method;
    }
    else {
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                              "Invalid parameter wavesol-linefit-method (%s) : "
                              "GAUSSIAN or BARYCENTER expected",
                              linefit_method);
    }
    _moo_params_get_tab_int(self, res->linefit_winhsize, list,
                            "wavesol_linefit_winhsize");
    _moo_params_get_tab_int(self, res->linefit_recentre, list,
                            "wavesol_linefit_recentre");
    _moo_params_get_tab_double(self, res->fwhm_min, list, "wavesol_min_fwhm");
    _moo_params_get_tab_double(self, res->fwhm_max, list, "wavesol_max_fwhm");
    const char *model = moo_params_get_string(self, list, "wavesol_model");
    if ((strcmp(model, MOO_WAVESOL_MODEL_1D) == 0) ||
        (strcmp(model, MOO_WAVESOL_MODEL_2D) == 0)) {
        res->model = model;
    }
    else {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT,
            "Invalid parameter wavesol-model (%s) : 1D or 2D expected", model);
    }

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get rebin parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_rebin_params *
moo_params_get_rebin(const moo_params *self, const cpl_parameterlist *list)
{
    moo_rebin_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);


    res = moo_rebin_params_new();

    const char *rebin_method =
        moo_params_get_string(self, list, "rebin_method");

    if ((strcmp(rebin_method, MOO_REBIN_METHOD_INTERPOLATE) == 0) ||
        (strcmp(rebin_method, MOO_REBIN_METHOD_INTEGRATE) == 0)) {
        res->method = rebin_method;
    }
    else {
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                              "Invalid parameter rebin-method (%s) : "
                              "INTERPOLATE or INTEGRATE expected",
                              rebin_method);
        return res;
    }

    _moo_params_get_tab3_double(self, res->step, list, "rebin_step");

    res->conserv_flux = moo_params_get_bool(self, list, "rebin_flux_conserv");
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get sub sky stare parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_sub_sky_stare_params *
moo_params_get_sub_sky_stare(const moo_params *self,
                             const cpl_parameterlist *list)
{
    moo_sub_sky_stare_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);


    res = moo_sub_sky_stare_params_new();
    res->method = moo_params_get_string(self, list, "subskystare_method");

    if ((strcmp(res->method, MOO_SUB_SKY_STARE_METHOD_SIMPLE) != 0) &&
        (strcmp(res->method, MOO_SUB_SKY_STARE_METHOD_SKYCORR) != 0) &&
        (strcmp(res->method, MOO_SUB_SKY_STARE_METHOD_NONE) != 0)) {
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                              "Invalid parameter sub-sky-stare-method (%s) : "
                              "SIMPLE, SKYCORR or NONE expected",
                              res->method);
        moo_sub_sky_stare_params_delete(res);
        return NULL;
    }

    res->maxdistslit =
        moo_params_get_int(self, list, "subskystare_maxdistslit");
    res->min_trans = moo_params_get_double(self, list, "subskystare_mintrans");
    res->min_sky = moo_params_get_int(self, list, "subskystare_min_sky");
    res->radius_sky =
        moo_params_get_double(self, list, "subskystare_radius_sky");
    res->step_r = moo_params_get_double(self, list, "subskystare_radius_stepr");

    res->sk = moo_params_get_skycorr(self, list);

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get sub sky stare simple parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_sub_sky_stare_params *
moo_params_get_sub_sky_stare_simple(const moo_params *self,
                                    const cpl_parameterlist *list)
{
    moo_sub_sky_stare_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);


    res = moo_sub_sky_stare_params_new();
    res->method = MOO_SUB_SKY_STARE_METHOD_SIMPLE;

    res->maxdistslit =
        moo_params_get_int(self, list, "subskystare_maxdistslit");
    res->min_trans = moo_params_get_double(self, list, "subskystare_mintrans");
    res->min_sky = moo_params_get_int(self, list, "subskystare_min_sky");
    res->radius_sky =
        moo_params_get_double(self, list, "subskystare_radius_sky");
    res->step_r = moo_params_get_double(self, list, "subskystare_radius_stepr");

    res->sk = NULL;

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get skycorr parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_skycorr_params *
moo_params_get_skycorr(const moo_params *self, const cpl_parameterlist *list)
{
    moo_skycorr_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);


    res = moo_skycorr_params_new();

    res->ltol = moo_params_get_double(self, list, "skycorr_ltol");
    res->min_line_dist_fac =
        moo_params_get_double(self, list, "skycorr_min_line_dist_fac");
    res->min_line_flux_fac =
        moo_params_get_double(self, list, "skycorr_min_line_flux_fac");
    res->fluxlim = moo_params_get_double(self, list, "skycorr_fluxlim");
    res->ftol = moo_params_get_double(self, list, "skycorr_ftol");
    res->xtol = moo_params_get_double(self, list, "skycorr_xtol");
    res->wtol = moo_params_get_double(self, list, "skycorr_wtol");
    res->cheby_max = moo_params_get_int(self, list, "skycorr_cheby_max");
    res->cheby_min = moo_params_get_int(self, list, "skycorr_cheby_min");
    res->rebintype = moo_params_get_int(self, list, "skycorr_rebintype");
    res->cheby_const = moo_params_get_double(self, list, "skycorr_cheby_const");
    res->fluxlim = moo_params_get_double(self, list, "skycorr_weightlim");
    res->siglim = moo_params_get_double(self, list, "skycorr_siglim");
    res->fitlim = moo_params_get_double(self, list, "skycorr_fitlim");
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get combine_pair parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_combine_pair_params *
moo_params_get_combine_pair(const moo_params *self,
                            const cpl_parameterlist *list)
{
    moo_combine_pair_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);


    res = moo_combine_pair_params_new();

    res->optimal = moo_params_get_bool(self, list, "combine_pair_optimal");

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get target_table parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_target_table_params *
moo_params_get_target_table(const moo_params *self,
                            const cpl_parameterlist *list)
{
    moo_target_table_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);


    res = moo_target_table_params_new();

    res->nosky = moo_params_get_bool(self, list, "target_table_nosky");

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get compute_snr parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_compute_snr_params *
moo_params_get_compute_snr(const moo_params *self,
                           const cpl_parameterlist *list)
{
    moo_compute_snr_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);


    res = moo_compute_snr_params_new();
    _moo_params_get_tab3_double(self, res->wmin, list, "compute_snr_wmin");
    _moo_params_get_tab3_double(self, res->wmax, list, "compute_snr_wmax");

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get compute_slitoffset parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_compute_slitoffset_params *
moo_params_get_compute_slitoffset(const moo_params *self,
                                  const cpl_parameterlist *list)
{
    moo_compute_slitoffset_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);


    res = moo_compute_slitoffset_params_new();
    res->min = moo_params_get_double(self, list, "compute_slitoffset_min");
    res->max = moo_params_get_double(self, list, "compute_slitoffset_max");
    res->step = moo_params_get_double(self, list, "compute_slitoffset_step");

    return res;
}
/*------------------------------------------------------------------------*/
/**
  @brief    Get coadd parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_coadd_params *
moo_params_get_coadd(const moo_params *self, const cpl_parameterlist *list)
{
    moo_coadd_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);


    res = moo_coadd_params_new();

    const char *method = moo_params_get_string(self, list, "coadd_method");
    if ((strcmp(method, MOO_COADD_METHOD_MEAN) == 0) ||
        (strcmp(method, MOO_COADD_METHOD_MEDIAN) == 0) ||
        (strcmp(method, MOO_COADD_METHOD_SIGCLIP) == 0)) {
        res->method = method;
    }
    else {
        cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT,
            "Invalid parameter coadd-method (%s) : MEAN or MEDIAN or SIGCLIP",
            method);
    }
    res->clip_kappa = moo_params_get_double(self, list, "coadd_clip_kappa");
    res->clip_niter = moo_params_get_int(self, list, "coadd_clip_niter");
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get linear parameters from moons parameters list
  @param    self moons parameters
  @param    list the parameters list
  @return the value of the parameter
 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT The parameter list or name is a NULL pointer

 */
/*----------------------------------------------------------------------------*/
moo_linear_params *
moo_params_get_linear(const moo_params *self, const cpl_parameterlist *list)
{
    moo_linear_params *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(list != NULL, CPL_ERROR_NULL_INPUT, NULL);

    res = moo_linear_params_new();
    _moo_params_get_tab_double(self, res->kappa, list, "bpmnl_clip_kappa");
    _moo_params_get_tab_double(self, res->saturate_kappa, list,
                               "detect_saturated_ron_kappa");
    _moo_params_get_tab_double(self, res->min_snr, list, "bpmnl_min_snr");
    res->saturate_threshold =
        moo_params_get_double(self, list, "detect_saturated_threshold");

    return res;
}

/**@}*/
