/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MOO_PARAMS_H
#define MOO_PARAMS_H

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <hdrl.h>
#include "moo_region.h"
/*-----------------------------------------------------------------------------
                                   Types
 -----------------------------------------------------------------------------*/
#define MOO_CRH_METHOD_MEDIAN  "MEDIAN"
#define MOO_CRH_METHOD_MEAN    "MEAN"
#define MOO_CRH_METHOD_SIGCLIP "SIGCLIP"
typedef struct
{
    char *recipe_id;
} moo_params;


typedef struct
{
    int ignore_detector[6];
} moo_prepare_params;

typedef struct
{
    const char *method;
    double kappa;
    int niter;
} moo_crh_params;

#define MOO_SATURATED_THRESHOLD 170000
typedef struct
{
    double saturate_threshold;
    double saturate_kappa[6];
    double kappa[6];
    double min_snr[6];
} moo_linear_params;

#define MOO_RON_ESTIMATION_METHOD_GLOBAL "GLOBAL"
#define MOO_RON_ESTIMATION_METHOD_LOCAL  "LOCAL"

typedef struct
{
    const char *ron_estimation_method;
} moo_bias_params;

#define MOO_CORRECT_BIAS_METHOD_MASTER "MASTER"
#define MOO_CORRECT_BIAS_METHOD_MEDIAN "MEDIAN"

typedef struct
{
    const char *subtract_bias_method;
} moo_correct_bias_params;

typedef struct
{
    double clip_kappa;
    int clip_niter;
    double clip_diff;
    double clip_frac;
} moo_nos_params;

typedef struct
{
    double clip_kappa;
    int clip_niter;
    double clip_diff;
    double clip_frac;
    int local_winhsize;
    double local_kappa;
} moo_hot_params;

#define MOO_LOCALISE_METHOD_GAUSSIAN         "GAUSSIAN"
#define MOO_LOCALISE_METHOD_BARYCENTER       "BARYCENTER"
#define MOO_LOCALISE_BACKG_METHOD_POLYNOMIAL "POLYNOMIAL"
#define MOO_LOCALISE_BACKG_METHOD_RUNNINGMIN "RUNNINGMIN"
typedef struct
{
    int centralpos;
    int centralwidth;
    const char *backg_method;
    int backg_winhsize[6];
    int backg_polydeg[6];
    double backg_clip_kappaup[6];
    double backg_clip_kappalow[6];
    int backg_clip_niter[6];
    int detect_niter[6];
    int xgap_max;
    double goodptsfrac_min[6];
    int polydeg[6];
    const char *method;
    int keep_points;
    double relativethresh;
    double wdiff_lim;
    double ydiff_lim;
    double ref_snr[6];
    int loc_xlim_hwin;
    double loc_xlim_fracmin;
} moo_localise_params;

#define MOO_EXTRACT_METHOD_SUM     "SUM"
#define MOO_EXTRACT_METHOD_OPTIMAL "OPTIMAL"
typedef struct
{
    const char *method;
    double aperture[6];
} moo_extract_params;

typedef struct
{
    int polydeg[6];
    int oversamp;
    int xstep;
    int winhsize;
} moo_model_flat_params;

typedef struct
{
    int fibref[2];
} moo_compute_fibtrans_params;

#define MOO_WAVESOL_MODEL_1D           "1D"
#define MOO_WAVESOL_MODEL_2D           "2D"
#define MOO_WAVESOL_LINEFIT_GAUSSIAN   "GAUSSIAN"
#define MOO_WAVESOL_LINEFIT_BARYCENTER "BARYCENTER"
#define MOO_WAVESOL_CONTROL_NONE       "NONE"
#define MOO_WAVESOL_CONTROL_CHECK      "CHECK"
#define MOO_WAVESOL_CONTROL_UPDATE     "UPDATE"

typedef struct
{
    double clip_kappa;
    double clip_frac;
    int clip_niter;
    double tolerance[6];
    const char *linefit_method;
    int linefit_winhsize[6];
    int linefit_recentre[6];
    int linedetect_winhsize[6];
    int linedetect_nlines[6];
    int ppm_wavemap_degx[6];
    int ppm_wavemap_degy;

    int wavemap_degx[6];
    int wavemap_degy;

    double fwhm_min[6];
    double fwhm_max[6];
    double min_snr[6];
    const char *model;
    const char *control;
    int isrefit;
} moo_wavesol_params;

#define MOO_REBIN_METHOD_INTEGRATE   "INTEGRATE"
#define MOO_REBIN_METHOD_INTERPOLATE "INTERPOLATE"
typedef struct
{
    double step[3];
    const char *method;
    int conserv_flux;
} moo_rebin_params;

#define MOO_SUB_SKY_STARE_METHOD_SIMPLE  "SIMPLE"
#define MOO_SUB_SKY_STARE_METHOD_SKYCORR "SKYCORR"
#define MOO_SUB_SKY_STARE_METHOD_NONE    "NONE"

typedef struct
{
    double ltol;
    double min_line_dist_fac;
    double min_line_flux_fac;
    double fluxlim;
    double ftol;
    double xtol;
    double wtol;
    int cheby_max;
    int cheby_min;
    double cheby_const;
    int rebintype;
    double weightlim;
    double siglim;
    double fitlim;
} moo_skycorr_params;

typedef struct
{
    const char *method;
    double radius_sky;
    int min_sky;
    int maxdistslit;
    double step_r;
    double min_trans;
    moo_skycorr_params *sk;
} moo_sub_sky_stare_params;


#define MOO_COADD_METHOD_MEDIAN  "MEDIAN"
#define MOO_COADD_METHOD_MEAN    "MEAN"
#define MOO_COADD_METHOD_SIGCLIP "SIGCLIP"

#define MOO_COADD_WEIGHT_NONE    "NONE"
#define MOO_COADD_WEIGHT_EXPTIME "EXPTIME"

typedef struct
{
    const char *method;
    double clip_kappa;
    int clip_niter;
    const char *weight;
} moo_coadd_params;

typedef struct
{
    int nosky;
} moo_target_table_params;

typedef struct
{
    int optimal;
} moo_combine_pair_params;

typedef struct
{
    double wmin[3];
    double wmax[3];
} moo_compute_snr_params;

typedef struct
{
    double min;
    double max;
    double step;
} moo_compute_slitoffset_params;

typedef struct
{
    int filter_winhsize[3];
    double kappa_lo;
    double kappa_up;
    double frac;
    int niter;
    int degree[3];
} moo_compute_resp_params;

typedef struct
{
    double continuum_const;
    cpl_boolean kern_mode;
    double kern_fac;
    cpl_boolean var_kern;
} moo_molecfit_model_params;

typedef struct
{
    double min_snr[3];
    int filter_skyfibre;
} moo_molecfit_calctrans_params;

typedef struct
{
    int do_s1d;
} moo_create_s1d_params;

/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/
moo_params *moo_params_new(const char *pid, const char *recipe_id);
void moo_params_delete(moo_params *self);

moo_nos_params *moo_nos_params_new(void);
void moo_nos_params_delete(moo_nos_params *self);

moo_hot_params *moo_hot_params_new(void);
void moo_hot_params_delete(moo_hot_params *self);

moo_prepare_params *moo_prepare_params_new(void);
void moo_prepare_params_delete(moo_prepare_params *self);

moo_crh_params *moo_crh_params_new(void);
void moo_crh_params_delete(moo_crh_params *self);

moo_linear_params *moo_linear_params_new(void);
void moo_linear_params_delete(moo_linear_params *self);

moo_bias_params *moo_bias_params_new(void);
void moo_bias_params_delete(moo_bias_params *self);

moo_correct_bias_params *moo_correct_bias_params_new(void);
void moo_correct_bias_params_delete(moo_correct_bias_params *self);

moo_localise_params *moo_localise_params_new(void);
void moo_localise_params_delete(moo_localise_params *self);

moo_extract_params *moo_extract_params_new(void);
void moo_extract_params_delete(moo_extract_params *self);

moo_model_flat_params *moo_model_flat_params_new(void);
void moo_model_flat_params_delete(moo_model_flat_params *self);

moo_molecfit_model_params *moo_molecfit_model_params_new(void);
void moo_molecfit_model_params_delete(moo_molecfit_model_params *self);

moo_molecfit_calctrans_params *moo_molecfit_calctrans_params_new(void);
void moo_molecfit_calctrans_params_delete(moo_molecfit_calctrans_params *self);

moo_create_s1d_params *moo_create_s1d_params_new(void);
void moo_create_s1d_params_delete(moo_create_s1d_params *self);

moo_compute_fibtrans_params *moo_compute_fibtrans_params_new(void);
void moo_compute_fibtrans_params_delete(moo_compute_fibtrans_params *self);

moo_wavesol_params *moo_wavesol_params_new(void);
void moo_wavesol_params_delete(moo_wavesol_params *self);

moo_rebin_params *moo_rebin_params_new(void);
void moo_rebin_params_delete(moo_rebin_params *self);

moo_sub_sky_stare_params *moo_sub_sky_stare_params_new(void);
void moo_sub_sky_stare_params_delete(moo_sub_sky_stare_params *self);

moo_skycorr_params *moo_skycorr_params_new(void);
void moo_skycorr_params_delete(moo_skycorr_params *self);

moo_coadd_params *moo_coadd_params_new(void);
void moo_coadd_params_delete(moo_coadd_params *self);

moo_target_table_params *moo_target_table_params_new(void);
void moo_target_table_params_delete(moo_target_table_params *self);

moo_combine_pair_params *moo_combine_pair_params_new(void);
void moo_combine_pair_params_delete(moo_combine_pair_params *self);

moo_compute_snr_params *moo_compute_snr_params_new(void);
void moo_compute_snr_params_delete(moo_compute_snr_params *self);

moo_compute_slitoffset_params *moo_compute_slitoffset_params_new(void);
void moo_compute_slitoffset_params_delete(moo_compute_slitoffset_params *self);

moo_compute_resp_params *moo_compute_resp_params_new(void);
void moo_compute_resp_params_delete(moo_compute_resp_params *self);

cpl_error_code moo_nos_params_dump(const moo_nos_params *self, FILE *stream);

cpl_error_code
moo_params_add_keep_temp(moo_params *self, cpl_parameterlist *list);
cpl_error_code
moo_params_add_quicklook_fibre_list(moo_params *self, cpl_parameterlist *list);

cpl_error_code moo_params_add_nos(moo_params *self, cpl_parameterlist *list);
cpl_error_code moo_params_add_hot(moo_params *self, cpl_parameterlist *list);
cpl_error_code
moo_params_add_prepare(moo_params *self, cpl_parameterlist *list);
cpl_error_code moo_params_add_crh(moo_params *self,
                                  cpl_parameterlist *list,
                                  const char *method);
cpl_error_code moo_params_add_linear(moo_params *self, cpl_parameterlist *list);
cpl_error_code moo_params_add_bias(moo_params *self, cpl_parameterlist *list);
cpl_error_code moo_params_add_correct_bias(moo_params *self,
                                           cpl_parameterlist *list,
                                           const char *method);
cpl_error_code
moo_params_add_localise(moo_params *self, cpl_parameterlist *list);
cpl_error_code
moo_params_add_model_flat(moo_params *self, cpl_parameterlist *list);
cpl_error_code
moo_params_add_molecfit_model(moo_params *self, cpl_parameterlist *list);
cpl_error_code
moo_params_add_molecfit_calctrans(moo_params *self, cpl_parameterlist *list);
cpl_error_code
moo_params_add_create_s1d(moo_params *self, cpl_parameterlist *list);
cpl_error_code
moo_params_add_extract(moo_params *self, cpl_parameterlist *list);
cpl_error_code
moo_params_add_compute_fibtrans(moo_params *self, cpl_parameterlist *list);
cpl_error_code
moo_params_add_compute_resp(moo_params *self, cpl_parameterlist *list);
cpl_error_code
moo_params_add_wavesol(moo_params *self, cpl_parameterlist *list);
cpl_error_code
moo_params_add_science_wavesol(moo_params *self, cpl_parameterlist *list);
cpl_error_code moo_params_add_rebin(moo_params *self, cpl_parameterlist *list);
cpl_error_code
moo_params_add_sub_sky_stare(moo_params *self, cpl_parameterlist *list);
cpl_error_code
moo_params_add_sub_sky_stare_simple(moo_params *self, cpl_parameterlist *list);
cpl_error_code
moo_params_add_sub_sky_stare_wnone(moo_params *self, cpl_parameterlist *list);
cpl_error_code
moo_params_add_skycorr(moo_params *self, cpl_parameterlist *list);
cpl_error_code moo_params_add_combine_pair(moo_params *self,
                                           cpl_parameterlist *list,
                                           int value);
cpl_error_code moo_params_add_target_table(moo_params *self,
                                           cpl_parameterlist *list,
                                           int value);
cpl_error_code
moo_params_add_compute_snr(moo_params *self, cpl_parameterlist *list);
cpl_error_code
moo_params_add_compute_slitoffset(moo_params *self, cpl_parameterlist *list);
cpl_error_code moo_params_add_coadd(moo_params *self, cpl_parameterlist *list);
moo_nos_params *
moo_params_get_nos(const moo_params *self, const cpl_parameterlist *list);
moo_hot_params *
moo_params_get_hot(const moo_params *self, const cpl_parameterlist *list);
moo_linear_params *
moo_params_get_linear(const moo_params *self, const cpl_parameterlist *list);

moo_prepare_params *
moo_params_get_prepare(const moo_params *self, const cpl_parameterlist *list);
moo_crh_params *
moo_params_get_crh(const moo_params *self, const cpl_parameterlist *list);

moo_bias_params *
moo_params_get_bias(const moo_params *self, const cpl_parameterlist *list);
moo_correct_bias_params *
moo_params_get_correct_bias(const moo_params *self,
                            const cpl_parameterlist *list);
moo_localise_params *
moo_params_get_localise(const moo_params *self, const cpl_parameterlist *list);
moo_extract_params *
moo_params_get_extract(const moo_params *self, const cpl_parameterlist *list);
moo_model_flat_params *moo_params_get_model_flat(const moo_params *self,
                                                 const cpl_parameterlist *list);
moo_molecfit_model_params *
moo_params_get_molecfit_model(const moo_params *self,
                              const cpl_parameterlist *list);
moo_molecfit_calctrans_params *
moo_params_get_molecfit_calctrans(const moo_params *self,
                                  const cpl_parameterlist *list);
moo_create_s1d_params *moo_params_get_create_s1d(const moo_params *self,
                                                 const cpl_parameterlist *list);

moo_compute_fibtrans_params *
moo_params_get_compute_fibtrans(const moo_params *self,
                                const cpl_parameterlist *list);
moo_compute_resp_params *
moo_params_get_compute_resp(const moo_params *self,
                            const cpl_parameterlist *list);
moo_wavesol_params *
moo_params_get_wavesol(const moo_params *self, const cpl_parameterlist *list);
moo_wavesol_params *
moo_params_get_science_wavesol(const moo_params *self,
                               const cpl_parameterlist *list);
moo_rebin_params *
moo_params_get_rebin(const moo_params *self, const cpl_parameterlist *list);
moo_sub_sky_stare_params *
moo_params_get_sub_sky_stare(const moo_params *self,
                             const cpl_parameterlist *list);
moo_sub_sky_stare_params *
moo_params_get_sub_sky_stare_simple(const moo_params *self,
                                    const cpl_parameterlist *list);
moo_skycorr_params *
moo_params_get_skycorr(const moo_params *self, const cpl_parameterlist *list);
moo_combine_pair_params *
moo_params_get_combine_pair(const moo_params *self,
                            const cpl_parameterlist *list);
moo_target_table_params *
moo_params_get_target_table(const moo_params *self,
                            const cpl_parameterlist *list);
moo_compute_snr_params *
moo_params_get_compute_snr(const moo_params *self,
                           const cpl_parameterlist *list);
moo_compute_slitoffset_params *
moo_params_get_compute_slitoffset(const moo_params *self,
                                  const cpl_parameterlist *list);
moo_coadd_params *
moo_params_get_coadd(const moo_params *self, const cpl_parameterlist *list);
int
moo_params_get_keep_temp(const moo_params *self, const cpl_parameterlist *list);
cpl_array *moo_params_get_quicklook_fibre_list(const moo_params *self,
                                               const cpl_parameterlist *list);
#endif
