/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>
#include "moo_detector.h"
#include "moo_utils.h"
#include "moo_pfits.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_pfits     FITS header protected access
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

moo_outputs *
moo_outputs_new(int nb)
{
    moo_outputs *res = cpl_calloc(1, sizeof(moo_outputs));
    res->nb = nb;
    res->outputs = cpl_calloc(nb, sizeof(moo_output));

    return res;
}

void
moo_outputs_delete(moo_outputs *res)
{
    if (res != NULL) {
        cpl_free(res->outputs);
        cpl_free(res);
    }
}

moo_outputs *
moo_outputs_load(cpl_propertylist *header, moo_detector_type type)
{
    moo_outputs *outputs = NULL;
    cpl_ensure(header != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int nbout = 0;

    cpl_errorstate prestate = cpl_errorstate_get();
    moo_try_check(nbout = moo_pfits_get_det_outputs(header), " ");

    moo_try_check(outputs = moo_outputs_new(nbout), " ");

    outputs->minx = 1;
    outputs->miny = 1;

    for (int i = 1; i <= nbout; i++) {
        float roni = NAN;
        float gaini = NAN;
        int nx, ny, x, y, prscx, prscy, ovscx, ovscy;
        if (type == MOO_TYPE_RI) {
            roni = moo_pfits_get_det_outi_ron(header, i);
            gaini = moo_pfits_get_det_outi_gain(header, i);
            nx = moo_pfits_get_det_outi_nx(header, i);
            ny = moo_pfits_get_det_outi_ny(header, i);
            x = moo_pfits_get_det_outi_x(header, i);
            y = moo_pfits_get_det_outi_y(header, i);
            prscx = moo_pfits_get_det_outi_prscx(header, i);
            prscy = moo_pfits_get_det_outi_prscy(header, i);
            ovscx = moo_pfits_get_det_outi_ovscx(header, i);
            ovscy = moo_pfits_get_det_outi_ovscy(header, i);
        }
        else {
            roni = moo_pfits_get_det_chip_ron(header);
            gaini = 1 / moo_pfits_get_det_chip_outi_gain(header, i);
            nx = moo_pfits_get_det_chip_outi_nx(header, i);
            ny = moo_pfits_get_det_chip_outi_ny(header, i);
            x = moo_pfits_get_det_chip_outi_x(header, i);
            y = moo_pfits_get_det_chip_outi_y(header, i);
            prscx = moo_pfits_get_det_chip_outi_prscx(header, i);
            prscy = moo_pfits_get_det_chip_outi_prscy(header, i);
            ovscx = moo_pfits_get_det_chip_outi_ovscx(header, i);
            ovscy = moo_pfits_get_det_chip_outi_ovscy(header, i);
        }

        outputs->outputs[i - 1].nx = nx;
        outputs->outputs[i - 1].ny = ny;
        outputs->outputs[i - 1].x = x;
        if (outputs->minx > x) {
            outputs->minx = x;
        }
        if (outputs->miny > y) {
            outputs->miny = y;
        }

        if (i == 1) {
            if (x < nx) {
                outputs->outputs[i - 1].dx = 1;
            }
            else {
                outputs->outputs[i - 1].dx = x - (nx - 1);
            }
            outputs->outputs[i - 1].dy = 1;
        }
        else {
            int oldx = outputs->outputs[i - 2].x;
            int oldy = outputs->outputs[i - 2].y;
            int onx = outputs->outputs[i - 2].nx;
            int ony = outputs->outputs[i - 2].ny;

            if (x == oldx) {
                outputs->outputs[i - 1].dx = outputs->outputs[i - 2].dx;
            }
            else if (x > oldx) {
                outputs->outputs[i - 1].dx = outputs->outputs[i - 2].dx + onx;
            }
            else {
                outputs->outputs[i - 1].dx = outputs->outputs[i - 2].dx - onx;
            }

            if (y == oldy) {
                outputs->outputs[i - 1].dy = outputs->outputs[i - 2].dy;
            }
            else if (y > oldy) {
                outputs->outputs[i - 1].dy = outputs->outputs[i - 2].dy + ony;
            }
            else {
                outputs->outputs[i - 1].dy = outputs->outputs[i - 2].dy - ony;
            }
        }

        outputs->outputs[i - 1].y = y;
        outputs->outputs[i - 1].ron = roni;
        outputs->outputs[i - 1].gain = gaini;
        outputs->outputs[i - 1].prscx = prscx;
        outputs->outputs[i - 1].prscy = prscy;
        outputs->outputs[i - 1].ovscx = ovscx;
        outputs->outputs[i - 1].ovscy = ovscy;
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_outputs_delete(outputs);
        outputs = NULL;
    }

    return outputs;
}

cpl_error_code
moo_outputs_get_det_size(moo_outputs *self, int *nx, int *ny)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(nx != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ny != NULL, CPL_ERROR_NULL_INPUT);

    int nbout = self->nb;
    int minx = self->minx;
    int miny = self->miny;

    int sx = 0;
    int sy = 0;

    for (int i = 0; i < nbout; i++) {
        int x = self->outputs[i].x;
        int y = self->outputs[i].y;

        if (x == minx) {
            sy += self->outputs[i].ny;
        }
        if (y == miny) {
            sx += self->outputs[i].nx;
        }
    }

    *nx = sx;
    *ny = sy;

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Create the DET data from RAW header and DATA
  @param    self the OUTPUT header
  @param    raw the RAW image
  @param    nx the image size in x
  @param    ny the image size in y
  @param    dtype the detector type (RI,YJ,H)
  @return   the DET data
 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_outputs_create_det(moo_outputs *self,
                       cpl_image *raw,
                       int nx,
                       int ny,
                       moo_detector_type dtype)
{
    cpl_image *result = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(raw != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_type type = cpl_image_get_type(raw);
    int naxis1 = cpl_image_get_size_x(raw);
    int naxis2 = cpl_image_get_size_y(raw);

    result = cpl_image_new(nx, ny, type);

    for (int i = 0; i < self->nb; i++) {
        int rx = self->outputs[i].x;
        int ry = self->outputs[i].y;
        int prscx = self->outputs[i].prscx;
        int prscy = self->outputs[i].prscy;
        int rnx = self->outputs[i].nx;
        int rny = self->outputs[i].ny;

        if (((nx - rx) < rx) && (dtype == MOO_TYPE_RI)) {
            rx = naxis1 - prscx - rnx + 1;
        }
        else {
            rx = rx + prscx;
        }

        if (((ny - ry) < ry) && (dtype == MOO_TYPE_RI)) {
            ry = naxis2 - prscy - rny + 1;
        }
        else {
            ry = ry + prscy;
        }

        int x = self->outputs[i].dx;
        int y = self->outputs[i].dy;
        /*cpl_msg_info("test", "OUT%d extract[%d,%d-->%d,%d]", i, rx, ry,
                     rx + rnx - 1, ry + rny - 1);
        */
        cpl_image *extract =
            cpl_image_extract(raw, rx, ry, rx + rnx - 1, ry + rny - 1);
        /*cpl_msg_info("test", "OUT%d copy[%d,%d]", i, x, y);
        */
        cpl_image_copy(result, extract, x, y);
        cpl_image_delete(extract);
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    find out the arcfile
  @param    plist       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char *
moo_pfits_get_arcfile(const cpl_propertylist *plist)
{
    const char *value = cpl_propertylist_get_string(plist, MOO_PFITS_ARCFILE);

    cpl_ensure(value != NULL, cpl_error_get_code(), NULL);

    return value;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    find out the date
  @param    plist       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char *
moo_pfits_get_date(const cpl_propertylist *plist)
{
    const char *value = cpl_propertylist_get_string(plist, MOO_PFITS_DATE);

    cpl_ensure(value != NULL, cpl_error_get_code(), NULL);

    return value;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    find out the DATE-OBS
  @param    plist       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char *
moo_pfits_get_dateobs(const cpl_propertylist *plist)
{
    const char *value = cpl_propertylist_get_string(plist, MOO_PFITS_DATEOBS);

    cpl_ensure(value != NULL, cpl_error_get_code(), NULL);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the MJD-OBS
  @param    plist       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
double
moo_pfits_get_mjdobs(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    double value = cpl_propertylist_get_double(plist, MOO_PFITS_MJDOBS);
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the EXTNAME
  @param    plist       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char *
moo_pfits_get_extname(const cpl_propertylist *plist)
{
    const char *value = cpl_propertylist_get_string(plist, MOO_PFITS_EXTNAME);

    cpl_ensure(value != NULL, cpl_error_get_code(), NULL);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the DIT value
  @param    plist       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double
moo_pfits_get_dit(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const double value = cpl_propertylist_get_double(plist, MOO_PFITS_DET_DIT);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET NDIT value
  @param    plist       property list to read from
  @return   the requested value or 0
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_ndit(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const int value = cpl_propertylist_get_int(plist, MOO_PFITS_DET_NDIT);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET LIVE value
  @param    plist       property list to read from
  @return   the requested value or 0
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_live(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const int value = cpl_propertylist_get_bool(plist, MOO_PFITS_DET_LIVE);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET OUTPUTS value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_outputs(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const int value = cpl_propertylist_get_int(plist, MOO_PFITS_DET_OUTPUTS);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET OUTI RON value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
float
moo_pfits_get_det_outi_ron(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_OUTI_RON, i);
    const float value = cpl_propertylist_get_float(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET CHIP RON value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
float
moo_pfits_get_det_chip_ron(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const float value =
        cpl_propertylist_get_float(plist, MOO_PFITS_DET_CHIP_RON);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET OUTI GAIN value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
float
moo_pfits_get_det_outi_gain(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_OUTI_GAIN, i);
    const float value = cpl_propertylist_get_float(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET OUTI CHIP GAIN value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
float
moo_pfits_get_det_chip_outi_gain(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_CHIP_OUTI_GAIN, i);
    const float value = cpl_propertylist_get_float(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET OUTI NX value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_outi_nx(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_OUTI_NX, i);
    const int value = cpl_propertylist_get_int(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET OUTI CHIP NX value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_chip_outi_nx(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_CHIP_OUTI_NX, i);
    const int value = cpl_propertylist_get_int(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET OUTI NY value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_outi_ny(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_OUTI_NY, i);
    const int value = cpl_propertylist_get_int(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET CHIP OUTI NY value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_chip_outi_ny(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_CHIP_OUTI_NY, i);
    const int value = cpl_propertylist_get_int(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET OUTI X value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_outi_x(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_OUTI_X, i);
    const int value = cpl_propertylist_get_int(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET CHIP OUTI X value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_chip_outi_x(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_CHIP_OUTI_X, i);
    const int value = cpl_propertylist_get_int(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET OUTI Y value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_outi_y(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_OUTI_Y, i);
    const int value = cpl_propertylist_get_int(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET CHIP OUTI Y value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_chip_outi_y(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_CHIP_OUTI_Y, i);
    const int value = cpl_propertylist_get_int(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

cpl_error_code
moo_pfits_set_det_outi_x(cpl_propertylist *plist, int i, int v)
{
    char *kw = cpl_sprintf(MOO_PFITS_DET_OUTI_X, i);
    cpl_error_code code = cpl_propertylist_set_int(plist, kw, v);
    cpl_free(kw);

    return code;
}

cpl_error_code
moo_pfits_set_det_chip_outi_x(cpl_propertylist *plist, int i, int v)
{
    char *kw = cpl_sprintf(MOO_PFITS_DET_CHIP_OUTI_X, i);
    cpl_error_code code = cpl_propertylist_set_int(plist, kw, v);
    cpl_free(kw);

    return code;
}

cpl_error_code
moo_pfits_set_det_outi_y(cpl_propertylist *plist, int i, int v)
{
    char *kw = cpl_sprintf(MOO_PFITS_DET_OUTI_Y, i);
    cpl_error_code code = cpl_propertylist_set_int(plist, kw, v);
    cpl_free(kw);

    return code;
}

cpl_error_code
moo_pfits_set_det_chip_outi_y(cpl_propertylist *plist, int i, int v)
{
    char *kw = cpl_sprintf(MOO_PFITS_DET_CHIP_OUTI_Y, i);
    cpl_error_code code = cpl_propertylist_set_int(plist, kw, v);
    cpl_free(kw);

    return code;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET OUTI PRSCX value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_outi_prscx(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_OUTI_PRSCX, i);
    const int value = cpl_propertylist_get_int(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET CHIP OUTI PRSCX value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_chip_outi_prscx(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_CHIP_OUTI_PRSCX, i);
    const int value = cpl_propertylist_get_int(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET OUTI PRSCY value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_outi_prscy(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_OUTI_PRSCY, i);
    const int value = cpl_propertylist_get_int(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET CHIP OUTI PRSCY value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_chip_outi_prscy(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_CHIP_OUTI_PRSCY, i);
    const int value = cpl_propertylist_get_int(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET OUTI OVSCX value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_outi_ovscx(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_OUTI_OVSCX, i);
    const int value = cpl_propertylist_get_int(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET CHIP OUTI OVSCX value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_chip_outi_ovscx(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_CHIP_OUTI_OVSCX, i);
    const int value = cpl_propertylist_get_int(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET OUTI OVSCY value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_outi_ovscy(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_OUTI_OVSCY, i);
    const int value = cpl_propertylist_get_int(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET OUTI CHIP OVSCY value
  @param    plist property list to read from
  @param    i  output number
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_chip_outi_ovscy(const cpl_propertylist *plist, int i)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    char *kw = cpl_sprintf(MOO_PFITS_DET_CHIP_OUTI_OVSCY, i);
    const int value = cpl_propertylist_get_int(plist, kw);
    cpl_free(kw);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET CHIP NX value
  @param    plist       property list to read from
  @return   the requested value or 0
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_chip_nx(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const int value = cpl_propertylist_get_int(plist, MOO_PFITS_DET_CHIP_NX);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET CHIP NY value
  @param    plist       property list to read from
  @return   the requested value or 0
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_chip_ny(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const int value = cpl_propertylist_get_int(plist, MOO_PFITS_DET_CHIP_NY);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the ESO DET CHIP LIVE value
  @param    plist       property list to read from
  @return   the requested value or 0
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_det_chip_live(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const int value = cpl_propertylist_get_bool(plist, MOO_PFITS_DET_CHIP_LIVE);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the NAXIS value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_naxis(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const int value = cpl_propertylist_get_int(plist, MOO_PFITS_NAXIS);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the NAXIS1 value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_naxis1(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const int value = cpl_propertylist_get_int(plist, MOO_PFITS_NAXIS1);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the NAXIS2 value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_naxis2(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const int value = cpl_propertylist_get_int(plist, MOO_PFITS_NAXIS2);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    find out the EXPTIME value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double
moo_pfits_get_exptime(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const double value = cpl_propertylist_get_double(plist, MOO_PFITS_EXPTIME);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the EXPTIME value
  @param    plist property list to write to
  @param    value the value for exptime
  @return   the error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_pfits_update_exptime(cpl_propertylist *plist, double value)
{
    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_propertylist_update_double(plist, MOO_PFITS_EXPTIME, value);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the RA value
  @param    plist property list to write to
  @param    value the value for RA
  @return   the error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_pfits_update_ra(cpl_propertylist *plist, double value)
{
    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_propertylist_update_double(plist, MOO_PFITS_RA, value);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the DEC value
  @param    plist property list to write to
  @param    value the value for DEC
  @return   the error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_pfits_update_dec(cpl_propertylist *plist, double value)
{
    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_propertylist_update_double(plist, MOO_PFITS_DEC, value);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the INS SLIT OFFSET value
  @param    plist property list to read from
  @return   the requested value or 0 in case of error
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_slit_offset(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const int value =
        cpl_propertylist_get_int(plist, MOO_PFITS_INS_SLIT_OFFSET);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the INS SLIT MODE value
  @param    plist property list to read from
  @return   the requested value or LR
 */
/*----------------------------------------------------------------------------*/
moo_mode_type
moo_pfits_get_mode(const cpl_propertylist *plist)
{
    moo_mode_type result = MOO_MODE_LR;
    cpl_errorstate prestate = cpl_errorstate_get();
    const char *value = cpl_propertylist_get_string(plist, MOO_PFITS_INS_MODE);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(),
               MOO_MODE_LR);

    if (strcmp(value, MOO_MODE_HR_NAME) == 0) {
        result = MOO_MODE_HR;
    }

    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the CRPIX2 value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double
moo_pfits_get_crpix2(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const double value = cpl_propertylist_get_double(plist, MOO_PFITS_CRPIX2);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the CRVAL1 value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double
moo_pfits_get_crval1(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const double value = cpl_propertylist_get_double(plist, MOO_PFITS_CRVAL1);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the CRPIX1 value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double
moo_pfits_get_crpix1(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const double value = cpl_propertylist_get_double(plist, MOO_PFITS_CRPIX1);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the CDELT2 value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double
moo_pfits_get_cdelt2(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const double value = cpl_propertylist_get_double(plist, MOO_PFITS_CDELT2);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the CDELT1 value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double
moo_pfits_get_cdelt1(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const double value = cpl_propertylist_get_double(plist, MOO_PFITS_CDELT1);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the CD1_1 value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double
moo_pfits_get_cd1_1(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const double value = cpl_propertylist_get_double(plist, MOO_PFITS_CD1_1);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the CD2_2 value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double
moo_pfits_get_cd2_2(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const double value = cpl_propertylist_get_double(plist, MOO_PFITS_CD2_2);

    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    find out the TEL.AIRM.START value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double
moo_pfits_get_tel_airm_start(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const double value =
        cpl_propertylist_get_double(plist, MOO_PFITS_TEL_AIRM_START);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the TEL.AIRM.END value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double
moo_pfits_get_tel_airm_end(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const double value =
        cpl_propertylist_get_double(plist, MOO_PFITS_TEL_AIRM_END);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0.0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    set the ESO PRO NCOADD value
  @param    plist property list to read from
  @param    v the value
  @return   the error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_pfits_update_ncoadd(cpl_propertylist *plist, int v)
{
    cpl_error_code code =
        cpl_propertylist_update_int(plist, MOO_PFITS_PRO_NCOADD, v);
    return code;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the PRO.NOCAODD value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_pro_ncoadd(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();

    const int value = cpl_propertylist_get_int(plist, MOO_PFITS_PRO_NCOADD);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    find out the PRO.WAVESOL.MODEL value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
const char *
moo_pfits_get_pro_wavesol_model(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const char *value =
        cpl_propertylist_get_string(plist, MOO_PFITS_PRO_WAVESOL_MODEL);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), NULL);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the PRO.WAVESOL.MODEL value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
const char *
moo_pfits_get_obs_start(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const char *value = cpl_propertylist_get_string(plist, MOO_PFITS_OBS_START);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), NULL);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the PRO.REC1.ID value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
const char *
moo_pfits_get_pro_rec1_id(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const char *value = "";

    if (cpl_propertylist_has(plist, MOO_PFITS_PRO_REC1_ID)) {
        value = cpl_propertylist_get_string(plist, MOO_PFITS_PRO_REC1_ID);
    }
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the PRO.WAVESOL.DEGX value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
const char *
moo_pfits_get_pro_wavesol_degx(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    const char *value =
        cpl_propertylist_get_string(plist, MOO_PFITS_PRO_WAVESOL_DEGX);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the PRO.WAVESOL.DEGY value
  @param    plist property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int
moo_pfits_get_pro_wavesol_degy(const cpl_propertylist *plist)
{
    cpl_errorstate prestate = cpl_errorstate_get();
    int value = cpl_propertylist_get_int(plist, MOO_PFITS_PRO_WAVESOL_DEGY);
    /* Check for a change in the CPL error state */
    /* - if it did change then propagate the error and return */
    cpl_ensure(cpl_errorstate_is_equal(prestate), cpl_error_get_code(), 0);

    return value;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Set the HDUCLASS Keyword
  @param    plist property list to write to
  @return   the error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_pfits_append_hduclass(cpl_propertylist *plist)
{
    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    cpl_propertylist_append_string(plist, MOO_PFITS_HDUCLASS,
                                   MOO_PFITS_HDUCLASS_V);
    cpl_propertylist_set_comment(plist, MOO_PFITS_HDUCLASS,
                                 MOO_PFITS_HDUCLASS_C);

    cpl_propertylist_append_string(plist, MOO_PFITS_HDUDOC, MOO_PFITS_HDUDOC_V);
    cpl_propertylist_set_comment(plist, MOO_PFITS_HDUDOC, MOO_PFITS_HDUDOC_C);

    cpl_propertylist_append_string(plist, MOO_PFITS_HDUVERS,
                                   MOO_PFITS_HDUVERS_V);
    cpl_propertylist_set_comment(plist, MOO_PFITS_HDUVERS, MOO_PFITS_HDUVERS_C);

    cpl_propertylist_append_string(plist, MOO_PFITS_HDUCLAS1,
                                   MOO_PFITS_HDUCLAS1_V);
    cpl_propertylist_set_comment(plist, MOO_PFITS_HDUCLAS1,
                                 MOO_PFITS_HDUCLAS1_C);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the HDUCLASS DATA Keyword
  @param    plist property list to write to
  @param    type the detector type
  @param    ntas the number id of spectrograph
  @return   the error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_pfits_append_hduclass_data(cpl_propertylist *plist,
                               moo_detector_type type,
                               int ntas)
{
    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    moo_pfits_append_hduclass(plist);

    cpl_propertylist_append_string(plist, MOO_PFITS_HDUCLAS2,
                                   MOO_PFITS_HDUCLAS2_DATA);
    cpl_propertylist_set_comment(plist, MOO_PFITS_HDUCLAS2,
                                 MOO_PFITS_HDUCLAS2_DATA_C);
    const char *err_extname = moo_detector_get_err_extname(type, ntas);
    cpl_propertylist_append_string(plist, MOO_PFITS_ERRDATA, err_extname);
    cpl_propertylist_set_comment(plist, MOO_PFITS_ERRDATA, MOO_PFITS_ERRDATA_C);
    const char *qual_extname = moo_detector_get_qual_extname(type, ntas);
    cpl_propertylist_append_string(plist, MOO_PFITS_QUALDATA, qual_extname);
    cpl_propertylist_set_comment(plist, MOO_PFITS_QUALDATA,
                                 MOO_PFITS_QUALDATA_C);
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the HDUCLASS ERROR Keyword
  @param    plist property list to write to
  @param    type the detector type
  @param    ntas the number id of spectrograph
  @param    sci_header the SCI header
  @return   the error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_pfits_append_hduclass_error(cpl_propertylist *plist,
                                moo_detector_type type,
                                int ntas,
                                const cpl_propertylist *sci_header)
{
    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    moo_pfits_append_hduclass(plist);

    cpl_propertylist_append_string(plist, MOO_PFITS_HDUCLAS2,
                                   MOO_PFITS_HDUCLAS2_ERROR);
    cpl_propertylist_set_comment(plist, MOO_PFITS_HDUCLAS2,
                                 MOO_PFITS_HDUCLAS2_ERROR_C);
    cpl_propertylist_append_string(plist, MOO_PFITS_HDUCLAS3,
                                   MOO_PFITS_HDUCLAS3_ERROR_MSE);
    cpl_propertylist_set_comment(plist, MOO_PFITS_HDUCLAS3,
                                 MOO_PFITS_HDUCLAS3_ERROR_MSE_C);

    const char *extname = moo_detector_get_extname(type, ntas);
    cpl_propertylist_append_string(plist, MOO_PFITS_SCIDATA, extname);
    cpl_propertylist_set_comment(plist, MOO_PFITS_SCIDATA, MOO_PFITS_SCIDATA_C);

    const char *qual_extname = moo_detector_get_qual_extname(type, ntas);
    cpl_propertylist_append_string(plist, MOO_PFITS_QUALDATA, qual_extname);
    cpl_propertylist_set_comment(plist, MOO_PFITS_QUALDATA,
                                 MOO_PFITS_QUALDATA_C);
    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CRPIX1);
    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CRVAL1);
    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CTYPE1);
    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CD1_1);
    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CUNIT1);

    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CRPIX2);
    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CRVAL2);

    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CTYPE2);
    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CD2_2);

    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_BUNIT);
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the HDUCLASS QUALITY Keyword
  @param    plist property list to write to
  @param    type the detector type
  @param    ntas the number id of spectrograph
  @param    sci_header the SCI header
  @param    mask the QUALMASK value
  @return   the error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_pfits_append_hduclass_quality(cpl_propertylist *plist,
                                  moo_detector_type type,
                                  int ntas,
                                  const cpl_propertylist *sci_header,
                                  int mask)
{
    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    moo_pfits_append_hduclass(plist);

    cpl_propertylist_append_string(plist, MOO_PFITS_HDUCLAS2,
                                   MOO_PFITS_HDUCLAS2_QUALITY);
    cpl_propertylist_set_comment(plist, MOO_PFITS_HDUCLAS2,
                                 MOO_PFITS_HDUCLAS2_QUALITY_C);
    cpl_propertylist_append_string(plist, MOO_PFITS_HDUCLAS3,
                                   MOO_PFITS_HDUCLAS3_QUALITY_FLAG32);
    cpl_propertylist_set_comment(plist, MOO_PFITS_HDUCLAS3,
                                 MOO_PFITS_HDUCLAS3_QUALITY_FLAG32_C);
    const char *extname = moo_detector_get_extname(type, ntas);
    cpl_propertylist_append_string(plist, MOO_PFITS_SCIDATA, extname);
    cpl_propertylist_set_comment(plist, MOO_PFITS_SCIDATA, MOO_PFITS_SCIDATA_C);
    const char *err_extname = moo_detector_get_err_extname(type, ntas);
    cpl_propertylist_append_string(plist, MOO_PFITS_ERRDATA, err_extname);
    cpl_propertylist_set_comment(plist, MOO_PFITS_ERRDATA, MOO_PFITS_ERRDATA_C);
    cpl_propertylist_append_int(plist, MOO_PFITS_QUALMASK, mask);
    cpl_propertylist_set_comment(plist, MOO_PFITS_QUALMASK,
                                 MOO_PFITS_QUALMASK_C);
    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CRPIX1);
    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CRPIX2);
    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CRVAL1);
    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CRVAL2);
    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CTYPE1);
    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CUNIT1);

    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CTYPE2);
    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CD1_1);
    cpl_propertylist_copy_property(plist, sci_header, MOO_PFITS_CD2_2);

    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Set the FLUXCAL Keyword
  @param    plist property list to write to
  @param    val the value of fluxcal
  @return   the error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_pfits_set_fluxcal(cpl_propertylist *plist, const char *val)
{
    const char *keyname = MOO_PFITS_FLUXCAL;
    const char *keycomment = MOO_PFITS_FLUXCAL_C;
    cpl_error_code status = CPL_ERROR_NONE;
    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    if (cpl_propertylist_has(plist, keyname)) {
        status = cpl_propertylist_set_string(plist, keyname, val);
    }
    else {
        cpl_propertylist_append_string(plist, keyname, val);
        cpl_propertylist_set_comment(plist, keyname, keycomment);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the WAVELMIN Keyword
  @param    plist property list to write to
  @param    val the value of fluxcal
  @return   the error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_pfits_set_wlmin(cpl_propertylist *plist, double val)
{
    const char *keyname = MOO_PFITS_WAVELMIN;
    const char *keycomment = MOO_PFITS_WAVELMIN_C;
    cpl_error_code status = CPL_ERROR_NONE;
    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    if (cpl_propertylist_has(plist, keyname)) {
        status = cpl_propertylist_set_double(plist, keyname, val);
    }
    else {
        cpl_propertylist_append_double(plist, keyname, val);
        cpl_propertylist_set_comment(plist, keyname, keycomment);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the WAVELMAX Keyword
  @param    plist property list to write to
  @param    val the value of fluxcal
  @return   the error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_pfits_set_wlmax(cpl_propertylist *plist, double val)
{
    const char *keyname = MOO_PFITS_WAVELMAX;
    const char *keycomment = MOO_PFITS_WAVELMAX_C;
    cpl_error_code status = CPL_ERROR_NONE;
    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    if (cpl_propertylist_has(plist, keyname)) {
        status = cpl_propertylist_set_double(plist, keyname, val);
    }
    else {
        cpl_propertylist_append_double(plist, keyname, val);
        cpl_propertylist_set_comment(plist, keyname, keycomment);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the SPEC_BIN Keyword
  @param    plist property list to write to
  @param    val the value of fluxcal
  @return   the error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_pfits_set_spec_bin(cpl_propertylist *plist, double val)
{
    const char *keyname = MOO_PFITS_SPECBIN;
    const char *keycomment = MOO_PFITS_SPECBIN_C;
    cpl_error_code status = CPL_ERROR_NONE;
    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    if (cpl_propertylist_has(plist, keyname)) {
        status = cpl_propertylist_set_double(plist, keyname, val);
    }
    else {
        cpl_propertylist_append_double(plist, keyname, val);
        cpl_propertylist_set_comment(plist, keyname, keycomment);
    }
    return status;
}
/**@}*/
