/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MOO_PRODUCTS_H
#define MOO_PRODUCTS_H


/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include "moo_det.h"
#include "moo_detlist.h"
#include "moo_params.h"
#include "moo_raw.h"
#include "moo_ext.h"
#include "moo_rbn.h"
#include "moo_sci.h"
#include "moo_f2f.h"
#include "moo_psf.h"
#include "moo_loc.h"
#include "moo_map.h"
#include "moo_bpm.h"
#include "moo_cube.h"
#include "moo_resp.h"
#include "moo_telluric.h"
#include "moo_target_table.h"
#include "moo_molectable.h"
#include "moo_s1d.h"

#define DICTIONARY_ID       "ESO-VLT-DIC.PRO-1.17"
#define MOO_PRODUCTS_PIPEID "moons"
typedef struct
{
    cpl_frameset *framelist;
    cpl_frameset *temporarylist;
    const cpl_parameterlist *parlist;
    const char *recid;
    const char *pipeline_id;
    int keep_temp;
    moo_params *params;
} moo_products;

/*-----------------------------------------------------------------------------
                                   Types
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/
moo_products *moo_products_new(cpl_frameset *framelist,
                               const cpl_parameterlist *parlist,
                               const char *recid,
                               const char *pipeline_id);

void moo_products_delete(moo_products *self);
const moo_params *moo_products_get_params(const moo_products *self);
cpl_error_code
moo_products_add_frame(const moo_products *self, cpl_frame *frame);

cpl_frame *moo_products_add(moo_products *self,
                            moo_det *det,
                            cpl_frame_level level,
                            const char *tag,
                            const char *filename,
                            const cpl_frame *inherit_frame);

cpl_frame *moo_products_add_det(moo_products *self,
                                moo_det *det,
                                cpl_frame_level level,
                                const char *tag,
                                const char *filename,
                                const cpl_frame *inherit_frame,
                                const char *filter);

cpl_frame *moo_products_add_loc(moo_products *self,
                                moo_loc *loc,
                                int keep_points,
                                cpl_frame_level level,
                                const char *tag,
                                const char *filename,
                                const cpl_frame *inherit_frame);

cpl_frame *moo_products_add_ext(moo_products *self,
                                moo_ext *ext,
                                cpl_frame_level level,
                                const char *tag,
                                const char *filename,
                                const cpl_frame *inherit_frame);

cpl_frame *moo_products_add_rbn(moo_products *self,
                                moo_rbn *rbn,
                                cpl_frame_level level,
                                const char *tag,
                                const char *filename,
                                const cpl_frame *inherit_frame);

cpl_frame *moo_products_add_sci(moo_products *self,
                                moo_sci *sci,
                                cpl_frame_level level,
                                const char *tag,
                                const char *filename,
                                const cpl_frame *inherit_frame);

cpl_frame *moo_products_add_resp(moo_products *self,
                                 moo_resp *resp,
                                 cpl_frame_level level,
                                 const char *tag,
                                 const char *filename,
                                 const cpl_frame *inherit_frame);

cpl_frame *moo_products_add_telluric(moo_products *self,
                                     moo_telluric *tell,
                                     cpl_frame_level level,
                                     const char *tag,
                                     const char *filename,
                                     const cpl_frame *inherit_frame);

cpl_frame *moo_products_add_map(moo_products *self,
                                moo_map *map,
                                cpl_frame_level level,
                                const char *tag,
                                const char *filename,
                                const cpl_frame *inherit_frame,
                                moo_rbn *rbn);

cpl_frame *moo_products_add_molectable(moo_products *self,
                                       moo_molectable *mtable,
                                       cpl_frame_level level,
                                       const char *tag,
                                       const char *filename,
                                       const cpl_frame *inherit_frame);

cpl_frame *moo_products_add_bpm(moo_products *self,
                                moo_bpm *bpm,
                                cpl_frame_level level,
                                const char *tag,
                                const char *filename,
                                const cpl_frame *inherit_frame);

cpl_frame *moo_products_add_saturate_map(moo_products *self,
                                         moo_saturate_map *saturate,
                                         cpl_frame_level level,
                                         const char *tag,
                                         const char *filename,
                                         const cpl_frame *inherit_frame);

cpl_frame *moo_products_add_cube(moo_products *self,
                                 moo_cube *cube,
                                 cpl_frame_level level,
                                 const char *tag,
                                 const char *filename,
                                 const cpl_frame *inherit_frame);

cpl_frame *moo_products_add_s1d(moo_products *self,
                                moo_s1d *s1d,
                                const char *tag,
                                const cpl_frame *inherit_frame);


cpl_frame *moo_products_add_target_table(moo_products *self,
                                         moo_target_table *ttable,
                                         cpl_frame_level level,
                                         const char *tag,
                                         const char *filename,
                                         const cpl_frame *inherit_frame);

cpl_frame *moo_products_add_f2f(moo_products *self,
                                moo_f2f *f2f,
                                cpl_frame_level level,
                                const char *tag,
                                const char *filename,
                                const cpl_frame *inherit_frame);

cpl_frame *moo_products_add_psf(moo_products *self,
                                moo_psf *psf,
                                cpl_frame_level level,
                                const char *tag,
                                const char *filename,
                                const cpl_frame *inherit_frame);

cpl_frame *moo_products_add_raw(moo_products *self,
                                moo_raw *raw,
                                cpl_frame_level level,
                                const char *tag,
                                const char *filename,
                                const cpl_frame *inherit_frame);

cpl_error_code moo_masterbias(moo_det *det,
                              moo_detlist *bias_list,
                              moo_bias_params *bias_params,
                              moo_products *products);

cpl_error_code moo_masterdark(moo_det *det, moo_products *products, int mode);

#endif
