/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_psf.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_psf PSF format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_psf
 * 
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_psf.h"
 * @endcode
 */


/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_psf  
  @return   1 newly allocated moo_psf or NULL in case of an error

  The returned object must be deallocated using moo_psf_delete().  
 
 */
/*----------------------------------------------------------------------------*/
moo_psf *
moo_psf_new(void)
{
    moo_psf *res = cpl_calloc(1, sizeof(moo_psf));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new empty PSF filename
  @return   1 newly allocated moo_psf or NULL in case of an error

  The returned object must be deallocated using moo_psf_delete().  
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL  
 */
/*----------------------------------------------------------------------------*/
moo_psf *
moo_psf_create(const char *filename)
{
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_errorstate prev_state = cpl_errorstate_get();
    moo_psf *res = moo_psf_new();
    res->filename = cpl_strdup(filename);
    res->primary_header = cpl_propertylist_new();
    moo_fits_create(filename);

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_info("moo_psf", "psf load %d", cpl_error_get_code());
        cpl_errorstate_set(prev_state);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Load a PSF frame and create a moo_psf
  @return   1 newly allocated moo_psf or NULL in case of an error

  The returned object must be deallocated using moo_psf_delete().  
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL  
 */
/*----------------------------------------------------------------------------*/
moo_psf *
moo_psf_load(const cpl_frame *psfframe)
{
    cpl_ensure(psfframe != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const char *filename = cpl_frame_get_filename(psfframe);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_errorstate prev_state = cpl_errorstate_get();
    moo_psf *res = moo_psf_new();
    res->filename = cpl_strdup(filename);
    res->primary_header = cpl_propertylist_load(filename, 0);
    for (int i = 1; i <= 2; i++) {
        res->ri[i - 1] =
            moo_psf_single_create(filename,
                                  moo_detector_get_extname(MOO_TYPE_RI, i));
        res->yj[i - 1] =
            moo_psf_single_create(filename,
                                  moo_detector_get_extname(MOO_TYPE_YJ, i));
        res->h[i - 1] =
            moo_psf_single_create(filename,
                                  moo_detector_get_extname(MOO_TYPE_H, i));
    }
    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_info("moo_psf", "psf load %d", cpl_error_get_code());
        cpl_errorstate_set(prev_state);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get a PSF single from PSF
  @param self moo_psf structure to use
  @param type type of detector
  @param ntas number of spectrographs    
  @return   PSF_SINGLE

  The returned object must be deallocated using moo_psf_delete().  
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL  
 */
/*----------------------------------------------------------------------------*/
moo_psf_single *
moo_psf_get_single(moo_psf *self, moo_detector_type type, int ntas)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    switch (type) {
        case MOO_TYPE_RI:
            return self->ri[ntas - 1];
            break;
        case MOO_TYPE_YJ:
            return self->yj[ntas - 1];
            break;
        case MOO_TYPE_H:
            return self->h[ntas - 1];
            break;
        default:
            return NULL;
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief assign moo_psf_single structure in moo_psf structure
  @param self moo_psf structure to update
  @param type type of detector
  @param ntas number of spectrographs  
  @param single moo_psf_single structure

  The returned object must be deallocated using moo_psf_delete().  
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL  
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_psf_set_single(moo_psf *self,
                   moo_detector_type type,
                   int ntas,
                   moo_psf_single *single)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(single != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();

    switch (type) {
        case MOO_TYPE_RI:
            self->ri[ntas - 1] = single;
            break;
        case MOO_TYPE_YJ:
            self->yj[ntas - 1] = single;
            break;
        case MOO_TYPE_H:
            self->h[ntas - 1] = single;
            break;
    }
    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_error("moo_psf", "Error for adding ext to %s (%d)",
                      self->filename, cpl_error_get_code());
        status = cpl_error_get_code();
        cpl_errorstate_set(prev_state);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Add PSF_SINGLE extension to PSF filename and update moo_psf structure
  @param self moo_psf structure to update
  @param type type of detector
  @param ntas number of spectrographs  
  @param single PSF_SINGLE extension to store on PSF file

  The returned object must be deallocated using moo_psf_delete().  
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL  
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_psf_add_single(moo_psf *self,
                   moo_psf_single *single,
                   moo_detector_type type,
                   int ntas)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(single != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();

    moo_psf_set_single(self, type, ntas, single);
    moo_psf_single_save(single, self->filename);

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_error("moo_psf", "Error for adding single to %s (%d)",
                      self->filename, cpl_error_get_code());
        status = cpl_error_get_code();
        cpl_errorstate_set(prev_state);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_psf
  @param    self    moo_psf to delete
  @return   void

  If the moo_psf @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_psf_delete(moo_psf *self)
{
    if (self != NULL) {
        int i;
        if (self->filename != NULL) {
            cpl_free(self->filename);
        }
        if (self->primary_header != NULL) {
            cpl_propertylist_delete(self->primary_header);
        }
        for (i = 0; i < 2; i++) {
            if (self->ri[i] != NULL) {
                moo_psf_single_delete(self->ri[i]);
            }
            if (self->yj[i] != NULL) {
                moo_psf_single_delete(self->yj[i]);
            }
            if (self->h[i] != NULL) {
                moo_psf_single_delete(self->h[i]);
            }
        }
        cpl_free(self);
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_psf to a FITS file
  @param    self          moo_psf to write to disk or NULL
  @param    filename      Name of the file to write  
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_psf to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
void
moo_psf_save(moo_psf *self, const char *filename)
{
    if (self != NULL) {
        cpl_propertylist_save(self->primary_header, filename, CPL_IO_CREATE);

        int i;
        for (i = 0; i < 2; i++) {
            if (self->ri[i] != NULL) {
                moo_psf_single_save(self->ri[i], filename);
            }
            if (self->yj[i] != NULL) {
                moo_psf_single_save(self->yj[i], filename);
            }
            if (self->h[i] != NULL) {
                moo_psf_single_save(self->h[i], filename);
            }
        }
    }
}

/*----------------------------------------------------------------------------*/
/**  
  @brief    Dump structural information of PSF
  @param    self    PSF to dump
  @param    stream  Output stream, accepts @c stdout or @c stderr
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_psf_dump(const moo_psf *self, FILE *stream)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(stream != NULL, CPL_ERROR_NULL_INPUT);

    fprintf(stream, "---MOO_PSF\n");
    fprintf(stream, "filename %s\n", self->filename);

    return CPL_ERROR_NONE;
}
/**@}*/
