/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include "moo_pfits.h"
#include "moo_psf_single.h"
#include "moo_fits.h"
#include "moo_utils.h"
#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_psf_single PSF_SINGLE format
 * @ingroup moo_psf
 * This module provides functions to create, use, and destroy a 
 * @em moo_psf_single
 * 
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_psf_single.h"
 * @endcode
 */


/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_psf_single  
  @return   1 newly allocated moo_psf_single or NULL in case of an error

  The returned object must be deallocated using moo_psf_single_delete().  
 
 */
/*----------------------------------------------------------------------------*/
moo_psf_single *
moo_psf_single_new(void)
{
    moo_psf_single *res = cpl_calloc(1, sizeof(moo_psf_single));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_psf_single from the given PSF filename
  @param    filename the PSF filename
  @param    extname the PSF extension name
  @return   1 newly allocated moo_psf_single or NULL in case of an error

  The returned object must be deallocated using moo_psf_single_delete().  
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL  
 */
/*----------------------------------------------------------------------------*/
moo_psf_single *
moo_psf_single_create(const char *filename, const char *extname)
{
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(extname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    moo_psf_single *res = moo_psf_single_new();

    res->extname = extname;
    res->filename = filename;

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set cube parameters
  @param    self moo_psf_single
  @param    crpix2 reference of Y axis in cube
  @param    cd2_2 step of Y axis in cube
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  The returned object must be deallocated using moo_psf_single_delete().  
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL  
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_psf_single_set_cube_ref(moo_psf_single *self, double crpix2, double cd2_2)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    if (self->header == NULL) {
        self->header = cpl_propertylist_new();
    }
    cpl_propertylist_append_double(self->header, MOO_PFITS_CRPIX1, 1.0);
    cpl_propertylist_append_string(self->header, MOO_PFITS_CTYPE1, "LINEAR");
    cpl_propertylist_append_double(self->header, MOO_PFITS_CRPIX2, crpix2);
    cpl_propertylist_append_string(self->header, MOO_PFITS_CTYPE2, "LINEAR");
    cpl_propertylist_append_double(self->header, MOO_PFITS_CRVAL1, 1.0);
    cpl_propertylist_append_double(self->header, MOO_PFITS_CRVAL2, 0.0);
    cpl_propertylist_append_double(self->header, MOO_PFITS_CD1_1, 1.0);
    cpl_propertylist_append_double(self->header, MOO_PFITS_CD1_2, 0.0);
    cpl_propertylist_append_string(self->header, MOO_PFITS_CUNIT1,
                                   MOO_PSF_SINGLE_CUNIT1);
    cpl_propertylist_append_double(self->header, MOO_PFITS_CD2_1, 0.);
    cpl_propertylist_append_double(self->header, MOO_PFITS_CD2_2, cd2_2);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set cube parameters
  @param    self moo_psf_single
  @param    crpix2 reference of Y axis in cube
  @param    cd2_2 step of Y axis in cube
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  The returned object must be deallocated using moo_psf_single_delete().  
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL  
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_psf_single_get_cube_ref(moo_psf_single *self, double *crpix2, double *cd2_2)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(crpix2 != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cd2_2 != NULL, CPL_ERROR_NULL_INPUT);

    if (self->header == NULL) {
        self->header =
            moo_fits_load_extension_header(self->filename,
                                           MOO_PSF_SINGLE_PREFIX_EXTNAME,
                                           self->extname);
    }
    *crpix2 = moo_pfits_get_crpix2(self->header);
    *cd2_2 = moo_pfits_get_cd2_2(self->header);

    return cpl_error_get_code();
}

static void
_moo_psf_single_reproject_model_fibre(cpl_image *image,
                                      int numfib,
                                      cpl_image *centroids,
                                      cpl_image *model,
                                      double crpix2,
                                      double cd2_2,
                                      cpl_image *result,
                                      cpl_image *contrib)
{
    int nx = cpl_image_get_size_x(image);
    int ny = cpl_image_get_size_y(model);

    double *data = cpl_image_get_data(result);
    int *contrib_data = cpl_image_get_data(contrib);

    for (int x = 1; x <= nx; x++) {
        int rej;
        double centroid = cpl_image_get(centroids, x, numfib, &rej);
        if (!isnan(centroid)) {
            for (int y = 1; y <= ny; y++) {
                double ypos = centroid + (y - crpix2) * cd2_2;
                double flux = cpl_image_get(model, x, y, &rej);
                int pix = (int)floor(ypos + 0.5);
                data[(pix - 1) * nx + x - 1] += flux;
                contrib_data[(pix - 1) * nx + x - 1] += 1;
            }
        }
    }
}

cpl_image *
moo_psf_single_reproject_model(moo_psf_single *self,
                               moo_single *det,
                               moo_loc_single *loc,
                               const int *health,
                               cpl_array *indexes)
{
    cpl_image *result = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(det != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(loc != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(health != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_image *contrib = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_image *fcentroids = moo_loc_single_get_f_centroids(loc);
    hdrl_image *himage = moo_single_get_image(det);
    cpl_image *image = hdrl_image_get_image(himage);

    cpl_ensure(fcentroids != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(himage != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int nx = hdrl_image_get_size_x(himage);
    int ny = hdrl_image_get_size_y(himage);
    int nb_fibres = cpl_image_get_size_y(fcentroids);

    cpl_imagelist *cube = moo_psf_single_get_cube(self);
    double crpix2, cd2_2;
    moo_try_check(moo_psf_single_get_cube_ref(self, &crpix2, &cd2_2), " ");
    cpl_msg_debug(__func__, "Use nx %d ny %d crpix2 %f cd2_2 %f", nx, ny,
                  crpix2, cd2_2);
    result = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    contrib = cpl_image_new(nx, ny, CPL_TYPE_INT);

/* The following addresses an issue with the gcc9 compiler series prior to
 * gcc 9.3. These compiler versions require that the variable '__func__'
 * appears in the data sharing clause if default(none) is used. However
 * adding it to the data sharing clause breaks older compiler versions
 * where these variables are pre-determined as shared.
 * This was fixed in gcc 9.3 and OpenMP 5.1.
 */
#ifdef _OPENMP
#if (__GNUC__ == 9) && (__GNUC_MINOR__ < 3)
#pragma omp parallel shared(nb_fibres, health, cube, image, fcentroids, \
                                crpix2, cd2_2, result, contrib, indexes)
#else
#pragma omp parallel default(none)                                            \
    shared(nb_fibres, health, cube, image, fcentroids, crpix2, cd2_2, result, \
               contrib, indexes)
#endif
    {
#pragma omp for
#endif
        for (int i = 1; i <= nb_fibres; i++) {
            int idx = cpl_array_get_cplsize(indexes, i - 1, NULL);
            int h = health[idx];
            cpl_msg_debug(__func__, "reproject fibre %d", i);

            if (h != 0) {
                cpl_image *model = cpl_imagelist_get(cube, i - 1);
                _moo_psf_single_reproject_model_fibre(image, i, fcentroids,
                                                      model, crpix2, cd2_2,
                                                      result, contrib);
            }
        }
#ifdef _OPENMP
    }
#endif
    double *data = cpl_image_get_data(result);
    int *contrib_data = cpl_image_get_data(contrib);

    for (int i = 0; i < nx * ny; i++) {
        if (contrib_data[i] != 0) {
            data[i] /= contrib_data[i];
        }
    }
moo_try_cleanup:
    cpl_image_delete(contrib);
    // dump error from the state
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in PSF file %s ext %s", self->filename,
                      self->extname);
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        cpl_image_delete(result);
        result = NULL;
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Normalize cube model profile
  @param    self moo_psf_single    
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
  
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL  
 */
/*----------------------------------------------------------------------------*/

cpl_error_code
moo_psf_single_normalize(moo_psf_single *self)
{
    cpl_error_code status = CPL_ERROR_NONE;
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    cpl_imagelist *cube = moo_psf_single_get_cube(self);

    int nb_fibres = cpl_imagelist_get_size(cube);
    double crpix2, cd2_2;
    moo_psf_single_get_cube_ref(self, &crpix2, &cd2_2);
#ifdef _OPENMP
#pragma omp parallel default(none) shared(nb_fibres, cube, cd2_2)
    {
#pragma omp for
#endif
        for (int f = 1; f <= nb_fibres; f++) {
            cpl_image *model = cpl_imagelist_get(cube, f - 1);
            int nx = cpl_image_get_size_x(model);
            int ny = cpl_image_get_size_y(model);
            for (int i = 1; i <= nx; i++) {
                double sum = 0.0;
                int rej;
                for (int j = 1; j <= ny; j++) {
                    sum += cpl_image_get(model, i, j, &rej);
                }
                sum *= cd2_2;
                for (int j = 1; j <= ny; j++) {
                    double val = cpl_image_get(model, i, j, &rej);
                    cpl_image_set(model, i, j, val / sum);
                }
            }
        }
#ifdef _OPENMP
    }
#endif
    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_psf_single
  @param    self    moo_psf_single to delete
  @return   void

  If the moo_psf_single @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_psf_single_delete(moo_psf_single *self)
{
    if (self != NULL) {
        if (self->header != NULL) {
            cpl_propertylist_delete(self->header);
        }
        if (self->cube != NULL) {
            cpl_imagelist_delete(self->cube);
        }

        cpl_free(self);
    }
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_psf_single to a FITS file
  @param    self          moo_psf_single to write to disk or NULL
  @param    filename      Name of the file to write  
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_psf_single to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
void
moo_psf_single_save(const moo_psf_single *self, const char *filename)
{
    if (self != NULL) {
        moo_fits_write_extension_cube(self->cube, filename,
                                      MOO_PSF_SINGLE_PREFIX_EXTNAME,
                                      self->extname, CPL_TYPE_FLOAT,
                                      self->header);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_psf_single
  @brief    Dump structural information of PSF_SINGLE
  @param    self    PSF_SINGLE to dump
  @param    stream  Output stream, accepts @c stdout or @c stderr
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_psf_single_dump(const moo_psf_single *self, FILE *stream)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(stream != NULL, CPL_ERROR_NULL_INPUT);

    fprintf(stream, "---MOO_PSF_SINGLE\n");
    fprintf(stream, "filename %s extname %s\n", self->filename, self->extname);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_psf_single
  @brief   Get cube
  @param    self    PSF_SINGLE
  @return   cube or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_imagelist *
moo_psf_single_get_cube(moo_psf_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->extname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (self->cube == NULL) {
        self->cube =
            moo_fits_load_extension_cube(self->filename,
                                         MOO_PSF_SINGLE_PREFIX_EXTNAME,
                                         self->extname, CPL_TYPE_FLOAT);
    }
    return self->cube;
}
/**@}*/
