/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>
#include "moo_detector.h"
#include "moo_utils.h"
#include "moo_qc.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_qc     QC keywords
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.NCRH value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_ncrh(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_CRH_NUMBER;
    const char *comment = MOONS_QC_CRH_NUMBER_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.NCRH.TOT value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_ncrh_tot(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_CRH_TOT;
    const char *comment = MOONS_QC_CRH_TOT_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.NCRH value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_ncrh_avg(cpl_propertylist *plist, float val)
{
    const char *key = MOONS_QC_CRH_AVG;
    const char *comment = MOONS_QC_CRH_AVG_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_float(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.NCOM value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_ncom(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_NCOM;
    const char *comment = MOONS_QC_NCOM_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MBIAS.AVG value
  @param    plist property list to write to
  @param    i region number
  @param    val the keyword value
  @return   the error code

 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mbias_avg(cpl_propertylist *plist, int i, double val)
{
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    char *key = cpl_sprintf(MOONS_QC_MBIAS_AVG, i);

    if (key != NULL) {
        moo_ensure_status(cpl_propertylist_append_double(plist, key, val),
                          status);
        moo_ensure_status(cpl_propertylist_set_comment(plist, key,
                                                       MOONS_QC_MBIAS_AVG_C),
                          status);
    }
    else {
        status = cpl_error_get_code();
    }
moo_try_cleanup:
    if (key != NULL) {
        cpl_free(key);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MBIAS.MED value
  @param    plist property list to write to
  @param    i region number
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mbias_med(cpl_propertylist *plist, int i, double val)
{
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    char *key = cpl_sprintf(MOONS_QC_MBIAS_MED, i);

    if (key != NULL) {
        moo_ensure_status(cpl_propertylist_append_double(plist, key, val),
                          status);
        moo_ensure_status(cpl_propertylist_set_comment(plist, key,
                                                       MOONS_QC_MBIAS_MED_C),
                          status);
    }
    else {
        status = cpl_error_get_code();
    }
moo_try_cleanup:
    if (key != NULL) {
        cpl_free(key);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MBIAS.AVG value
  @param    plist property list to write to
  @param    i region number
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mbias_rms(cpl_propertylist *plist, int i, double val)
{
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    char *key = cpl_sprintf(MOONS_QC_MBIAS_RMS, i);

    if (key != NULL) {
        moo_ensure_status(cpl_propertylist_append_double(plist, key, val),
                          status);
        moo_ensure_status(cpl_propertylist_set_comment(plist, key,
                                                       MOONS_QC_MBIAS_RMS_C),
                          status);
    }
    else {
        status = cpl_error_get_code();
    }
moo_try_cleanup:
    if (key != NULL) {
        cpl_free(key);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MBIAS.MAD value
  @param    plist property list to write to
  @param    i region number
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mbias_mad(cpl_propertylist *plist, int i, double val)
{
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    char *key = cpl_sprintf(MOONS_QC_MBIAS_MAD, i);

    if (key != NULL) {
        moo_ensure_status(cpl_propertylist_append_double(plist, key, val),
                          status);
        moo_ensure_status(cpl_propertylist_set_comment(plist, key,
                                                       MOONS_QC_MBIAS_MAD_C),
                          status);
    }
    else {
        status = cpl_error_get_code();
    }
moo_try_cleanup:
    if (key != NULL) {
        cpl_free(key);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.RON.RAW value
  @param    plist property list to write to
  @param    i region number
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_ron_raw(cpl_propertylist *plist, int i, double val)
{
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    char *key = cpl_sprintf(MOONS_QC_RON_RAW, i);

    if (key != NULL) {
        moo_ensure_status(cpl_propertylist_append_double(plist, key, val),
                          status);
        moo_ensure_status(cpl_propertylist_set_comment(plist, key,
                                                       MOONS_QC_RON_RAW_C),
                          status);
    }
    else {
        status = cpl_error_get_code();
    }
moo_try_cleanup:
    if (key != NULL) {
        cpl_free(key);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.RON.MASTER value
  @param    plist property list to write to
  @param    i region number
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_ron_master(cpl_propertylist *plist, int i, double val)
{
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    char *key = cpl_sprintf(MOONS_QC_RON_MASTER, i);

    if (key != NULL) {
        moo_ensure_status(cpl_propertylist_append_double(plist, key, val),
                          status);
        moo_ensure_status(cpl_propertylist_set_comment(plist, key,
                                                       MOONS_QC_RON_MASTER_C),
                          status);
    }
    else {
        status = cpl_error_get_code();
    }
moo_try_cleanup:
    if (key != NULL) {
        cpl_free(key);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MDARK.NHOT value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mdark_nhot(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_MDARK_NHOT;
    const char *comment = MOONS_QC_MDARK_NHOT_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MDARK.AVG value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mdark_avg(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_MDARK_AVG;
    const char *comment = MOONS_QC_MDARK_AVG_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MDARK.NORMALISE.FACTOR value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mdark_normalise_factor(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_MDARK_NORMALISE_FACTOR;
    const char *comment = MOONS_QC_MDARK_NORMALISE_FACTOR_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MDARK.MED value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mdark_med(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_MDARK_MED;
    const char *comment = MOONS_QC_MDARK_MED_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MDARK.RMS value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mdark_rms(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_MDARK_RMS;
    const char *comment = MOONS_QC_MDARK_RMS_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MEAN.DARK.CURRENT value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mdark_current(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_MDARK_CURRENT;
    const char *comment = MOONS_QC_MDARK_CURRENT_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.RESIDY.MED value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_residy_med(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_RESIDY_MED;
    const char *comment = MOONS_QC_RESIDY_MED_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.RESIDY.STD value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_residy_sd(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_RESIDY_SD;
    const char *comment = MOONS_QC_RESIDY_SD_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.RESIDY.AVG value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_residy_avg(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_RESIDY_AVG;
    const char *comment = MOONS_QC_RESIDY_AVG_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WIDTHY.MED value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_widthy_med(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_WIDTHY_MED;
    const char *comment = MOONS_QC_WIDTHY_MED_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WIDTHY.STD value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_widthy_sd(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_WIDTHY_SD;
    const char *comment = MOONS_QC_WIDTHY_SD_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WIDTH.AVG value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_widthy_avg(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_WIDTHY_AVG;
    const char *comment = MOONS_QC_WIDTHY_AVG_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.SHIFTY.PTS.MED value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_shifty_pts_med(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_SHIFTY_PTS_MED;
    const char *comment = MOONS_QC_SHIFTY_PTS_MED_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.SHIFTY.PTS.AVG value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_shifty_pts_avg(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_SHIFTY_PTS_AVG;
    const char *comment = MOONS_QC_SHIFTY_PTS_AVG_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.SHIFTY.PTS.SD value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_shifty_pts_sd(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_SHIFTY_PTS_SD;
    const char *comment = MOONS_QC_SHIFTY_PTS_SD_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.SHIFTY.FIT.MED value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_shifty_fit_med(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_SHIFTY_FIT_MED;
    const char *comment = MOONS_QC_SHIFTY_FIT_MED_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.SHIFTY.FIT.AVG value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_shifty_fit_avg(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_SHIFTY_FIT_AVG;
    const char *comment = MOONS_QC_SHIFTY_FIT_AVG_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.SHIFTY.FIT.SD value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_shifty_fit_sd(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_SHIFTY_FIT_SD;
    const char *comment = MOONS_QC_SHIFTY_FIT_SD_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.RESIDY.AVG value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_fit_deg(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_FIT_DEG;
    const char *comment = MOONS_QC_FIT_DEG_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.NOUTLIER.PTS value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_noutlier_pts(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_NOUTLIER_PTS;
    const char *comment = MOONS_QC_NOUTLIER_PTS_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.NOUTLIER.WID value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_noutlier_wid(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_NOUTLIER_WID;
    const char *comment = MOONS_QC_NOUTLIER_WID_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.NBADPROFILE value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_nbadprofile(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_NBADPROFILE;
    const char *comment = MOONS_QC_NBADPROFILE_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.NBADPIX value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_nbadpix(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_NBADPIX;
    const char *comment = MOONS_QC_NBADPIX_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.NMISSINGFIB value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_nmissingfib(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_NMISSINGFIB;
    const char *comment = MOONS_QC_NMISSINGFIB_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MISSINGFIB value
  @param    plist property list to write to
  @param    num missing fibre number
  @param    val missing fibre name
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_missingfib(cpl_propertylist *plist, int num, const char *val)
{
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    char *key = cpl_sprintf(MOONS_QC_MISSINGFIB, num);

    if (key != NULL) {
        moo_ensure_status(cpl_propertylist_append_string(plist, key, val),
                          status);
        moo_ensure_status(cpl_propertylist_set_comment(plist, key,
                                                       MOONS_QC_MISSINGFIB_C),
                          status);
    }
    else {
        status = cpl_error_get_code();
    }
moo_try_cleanup:
    if (key != NULL) {
        cpl_free(key);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.P2P.MIN value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_p2p_min(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_P2P_MIN;
    const char *comment = MOONS_QC_P2P_MIN_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.P2P.MAX value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_p2p_max(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_P2P_MAX;
    const char *comment = MOONS_QC_P2P_MAX_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.P2P.AVG value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_p2p_avg(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_P2P_AVG;
    const char *comment = MOONS_QC_P2P_AVG_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.P2P.MED value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_p2p_med(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_P2P_MED;
    const char *comment = MOONS_QC_P2P_MED_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.P2P.MAD value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_p2p_mad(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_P2P_MAD;
    const char *comment = MOONS_QC_P2P_MAD_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.P2P.RMS value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_p2p_rms(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_P2P_RMS;
    const char *comment = MOONS_QC_P2P_RMS_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MFLAT.SN.MIN value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mflat_sn_min(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_MFLAT_SN_MIN;
    const char *comment = MOONS_QC_MFLAT_SN_MIN_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MFLAT.SN.MAX value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mflat_sn_max(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_MFLAT_SN_MAX;
    const char *comment = MOONS_QC_MFLAT_SN_MAX_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MFLAT.SN.MED value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mflat_sn_med(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_MFLAT_SN_MED;
    const char *comment = MOONS_QC_MFLAT_SN_MED_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WAVE.MIN value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_wave_min(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_WAVE_MIN;
    const char *comment = MOONS_QC_WAVE_MIN_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WAVE.MAX value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_wave_max(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_WAVE_MAX;
    const char *comment = MOONS_QC_WAVE_MAX_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.DETECT.NOISEMULT value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_qc_detect_noisemult(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_DETECT_NOISEMULT;
    const char *comment = MOONS_QC_DETECT_NOISEMULT_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MFLAT.MIN value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mflat_min(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_MFLAT_MIN;
    const char *comment = MOONS_QC_MFLAT_MIN_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MFLAT.MAX value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mflat_max(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_MFLAT_MAX;
    const char *comment = MOONS_QC_MFLAT_MAX_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MFLAT.AVG value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mflat_avg(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_MFLAT_AVG;
    const char *comment = MOONS_QC_MFLAT_AVG_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MFLAT.MED value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mflat_med(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_MFLAT_MED;
    const char *comment = MOONS_QC_MFLAT_MED_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.MFLAT.RMS value
  @param    plist property list to write to
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_mflat_rms(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_MFLAT_RMS;
    const char *comment = MOONS_QC_MFLAT_RMS_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.SPECTRO.FIBTRANS.REF.value
  @param    plist property list to write to
  @param    ntas number of TAS
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_fibtrans_ref(cpl_propertylist *plist, int ntas, int val)
{
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    char *comment = NULL;
    char *key = cpl_sprintf(MOONS_QC_FIBTRANS_REF, ntas);

    if (key != NULL) {
        moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
        comment = cpl_sprintf(MOONS_QC_FIBTRANS_REF_C, ntas);
        if (comment != NULL) {
            moo_ensure_status(cpl_propertylist_set_comment(
                                  plist, key, MOONS_QC_FIBTRANS_REF_C),
                              status);
        }
        else {
            status = cpl_error_get_code();
        }
    }
    else {
        status = cpl_error_get_code();
    }
moo_try_cleanup:
    if (key != NULL) {
        cpl_free(key);
    }
    if (comment != NULL) {
        cpl_free(comment);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.FIBTRANS.MIN value
  @param    plist property list to write to
  @param    type the detector type
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_fibtrans_min(cpl_propertylist *plist,
                        moo_detector_type type,
                        double val)
{
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    char *key = cpl_sprintf(MOONS_QC_FIBTRANS_MIN, moo_detector_get_name(type));

    if (key != NULL) {
        moo_ensure_status(cpl_propertylist_append_double(plist, key, val),
                          status);
        moo_ensure_status(cpl_propertylist_set_comment(plist, key,
                                                       MOONS_QC_FIBTRANS_MIN_C),
                          status);
    }
    else {
        status = cpl_error_get_code();
    }
moo_try_cleanup:
    if (key != NULL) {
        cpl_free(key);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.FIBTRANS.MAX value
  @param    plist property list to write to
  @param    type the detector type
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_fibtrans_max(cpl_propertylist *plist,
                        moo_detector_type type,
                        double val)
{
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    char *key = cpl_sprintf(MOONS_QC_FIBTRANS_MAX, moo_detector_get_name(type));

    if (key != NULL) {
        moo_ensure_status(cpl_propertylist_append_double(plist, key, val),
                          status);
        moo_ensure_status(cpl_propertylist_set_comment(plist, key,
                                                       MOONS_QC_FIBTRANS_MAX_C),
                          status);
    }
    else {
        status = cpl_error_get_code();
    }
moo_try_cleanup:
    if (key != NULL) {
        cpl_free(key);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.FIBTRANS.MED value
  @param    plist property list to write to
  @param    type the detector type
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_fibtrans_med(cpl_propertylist *plist,
                        moo_detector_type type,
                        double val)
{
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    char *key = cpl_sprintf(MOONS_QC_FIBTRANS_MED, moo_detector_get_name(type));

    if (key != NULL) {
        moo_ensure_status(cpl_propertylist_append_double(plist, key, val),
                          status);
        moo_ensure_status(cpl_propertylist_set_comment(plist, key,
                                                       MOONS_QC_FIBTRANS_MED_C),
                          status);
    }
    else {
        status = cpl_error_get_code();
    }
moo_try_cleanup:
    if (key != NULL) {
        cpl_free(key);
    }
    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WAVECAL.CATLINE value
  @param    plist property list to write to  
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_wavecal_catline(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_WAVECAL_CATLINE;
    const char *comment = MOONS_QC_WAVECAL_CATLINE_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WAVECAL.DETECTLINE value
  @param    plist property list to write to  
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_wavecal_detectline(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_WAVECAL_DETECTLINE;
    const char *comment = MOONS_QC_WAVECAL_DETECTLINE_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WAVECAL.FAILEDFIT value
  @param    plist property list to write to  
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_wavecal_failedfit(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_WAVECAL_FAILEDFIT;
    const char *comment = MOONS_QC_WAVECAL_FAILEDFIT_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WAVECAL.FITLINE value
  @param    plist property list to write to  
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_wavecal_fitline(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_WAVECAL_FITLINE;
    const char *comment = MOONS_QC_WAVECAL_FITLINE_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WAVECAL.MATCHLINE value
  @param    plist property list to write to  
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_wavecal_matchline(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_WAVECAL_MATCHLINE;
    const char *comment = MOONS_QC_WAVECAL_MATCHLINE_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WAVECAL.CLIPLINE value
  @param    plist property list to write to  
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_wavecal_clipline(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_WAVECAL_CLIPLINE;
    const char *comment = MOONS_QC_WAVECAL_CLIPLINE_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WAVECAL.MONOTONOUS.SOLUTION value
  @param    plist property list to write to  
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_wavecal_monotonous_solution(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_WAVECAL_MONOTONOUS_SOLUTION;
    const char *comment = MOONS_QC_WAVECAL_MONOTONOUS_SOLUTION_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    moo_ensure_status(cpl_propertylist_append_bool(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WAVECAL.VALIDFIBRE value
  @param    plist property list to write to  
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_wavecal_validfibre(cpl_propertylist *plist, int val)
{
    const char *key = MOONS_QC_WAVECAL_VALIDFIBRE;
    const char *comment = MOONS_QC_WAVECAL_VALIDFIBRE_C;

    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    moo_ensure_status(cpl_propertylist_append_int(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);

moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WAVECAL.DISP.MIN value
  @param    plist property list to write to  
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_wavecal_disp_min(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_WAVECAL_DISP_MIN;
    const char *comment = MOONS_QC_WAVECAL_DISP_MIN_C;
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);

moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WAVECAL.DISP.MAX value
  @param    plist property list to write to  
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_wavecal_disp_max(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_WAVECAL_DISP_MAX;
    const char *comment = MOONS_QC_WAVECAL_DISP_MAX_C;
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);

moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WAVECAL.RESOL.MED value
  @param    plist property list to write to  
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_wavecal_resol_med(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_WAVECAL_RESOL_MED;
    const char *comment = MOONS_QC_WAVECAL_RESOL_MED_C;
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);

moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WAVECAL.RESIDWAVE.MED value
  @param    plist property list to write to  
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_wavecal_residwave_med(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_WAVECAL_RESIDWAVE_MED;
    const char *comment = MOONS_QC_WAVECAL_RESIDWAVE_MED_C;
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);

moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.WAVECAL.RESIDWAVE.STD value
  @param    plist property list to write to  
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_wavecal_residwave_std(cpl_propertylist *plist, double val)
{
    const char *key = MOONS_QC_WAVECAL_RESIDWAVE_STD;
    const char *comment = MOONS_QC_WAVECAL_RESIDWAVE_STD_C;
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    moo_ensure_status(cpl_propertylist_append_double(plist, key, val), status);
    moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                      status);

moo_try_cleanup:
    return status;
}
/******************************************************************************/

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the QC.NCOM value
  @param    plist property list to read from
  @return   the value
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list self does not contain a property
    with the name name.
  - CPL_ERROR_TYPE_MISMATCH The sought-after property name is not of type
    CPL_TYPE_INT.
 */
/*----------------------------------------------------------------------------*/
int
moo_qc_get_ncom(cpl_propertylist *plist)
{
    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    return cpl_propertylist_get_int(plist, MOONS_QC_NCOM);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.SNR.RANGE value
  @param    plist property list to write to  
  @param    val missing fibre name
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_snr_range(cpl_propertylist *plist, const char *val)
{
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    if (cpl_propertylist_has(plist, MOONS_QC_SNR_RANGE)) {
        moo_ensure_status(cpl_propertylist_set_string(plist, MOONS_QC_SNR_RANGE,
                                                      val),
                          status);
    }
    else {
        moo_ensure_status(cpl_propertylist_append_string(plist,
                                                         MOONS_QC_SNR_RANGE,
                                                         val),
                          status);
        moo_ensure_status(cpl_propertylist_set_comment(plist,
                                                       MOONS_QC_SNR_RANGE,
                                                       MOONS_QC_SNR_RANGE_C),
                          status);
    }


moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.FRAME.RAW1 value
  @param    plist property list to write to  
  @param    val filename
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_frame_raw1(cpl_propertylist *plist, const char *val)
{
    const char *value = NULL;
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    char *name = strrchr(val, '/');
    if (name != NULL) {
        value = name + 1;
    }
    else {
        value = val;
    }
    if (cpl_propertylist_has(plist, MOONS_QC_FRAME_RAW1)) {
        moo_ensure_status(cpl_propertylist_set_string(plist,
                                                      MOONS_QC_FRAME_RAW1,
                                                      value),
                          status);
    }
    else {
        moo_ensure_status(cpl_propertylist_append_string(plist,
                                                         MOONS_QC_FRAME_RAW1,
                                                         value),
                          status);
        moo_ensure_status(cpl_propertylist_set_comment(plist,
                                                       MOONS_QC_FRAME_RAW1,
                                                       MOONS_QC_FRAME_RAW1_C),
                          status);
    }


moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.FRAME.RAW2 value
  @param    plist property list to write to  
  @param    val filename
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_frame_raw2(cpl_propertylist *plist, const char *val)
{
    const char *value = NULL;
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);

    char *name = strrchr(val, '/');
    if (name != NULL) {
        value = name + 1;
    }
    else {
        value = val;
    }
    if (cpl_propertylist_has(plist, MOONS_QC_FRAME_RAW2)) {
        moo_ensure_status(cpl_propertylist_set_string(plist,
                                                      MOONS_QC_FRAME_RAW2,
                                                      value),
                          status);
    }
    else {
        moo_ensure_status(cpl_propertylist_append_string(plist,
                                                         MOONS_QC_FRAME_RAW2,
                                                         value),
                          status);
        moo_ensure_status(cpl_propertylist_set_comment(plist,
                                                       MOONS_QC_FRAME_RAW2,
                                                       MOONS_QC_FRAME_RAW2_C),
                          status);
    }


moo_try_cleanup:
    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.IS.LINEARCOR value
  @param    plist property list to write to  
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_is_linearcor(cpl_propertylist *plist, cpl_boolean val)
{
    const char *key = MOONS_QC_IS_LINEARCOR;
    const char *comment = MOONS_QC_IS_LINEARCOR_C;
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    if (cpl_propertylist_has(plist, key)) {
        moo_ensure_status(cpl_propertylist_set_bool(plist, key, val), status);
    }
    else {
        moo_ensure_status(cpl_propertylist_append_bool(plist, key, val),
                          status);
        moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                          status);
    }
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.IS.P2PCOR value
  @param    plist property list to write to  
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_is_p2pcor(cpl_propertylist *plist, cpl_boolean val)
{
    const char *key = MOONS_QC_IS_P2PCOR;
    const char *comment = MOONS_QC_IS_P2PCOR_C;
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    if (cpl_propertylist_has(plist, key)) {
        moo_ensure_status(cpl_propertylist_set_bool(plist, key, val), status);
    }
    else {
        moo_ensure_status(cpl_propertylist_append_bool(plist, key, val),
                          status);
        moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                          status);
    }
moo_try_cleanup:
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the QC.IS.TELLCOR value
  @param    plist property list to write to  
  @param    val the keyword value
  @return   the error code
 *
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND The property list does not contain a property
    with the name name
  - CPL_ERROR_ILLEGAL_INPUT The format string has an invalid format.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_qc_set_is_tellcor(cpl_propertylist *plist, cpl_boolean val)
{
    const char *key = MOONS_QC_IS_TELLCOR;
    const char *comment = MOONS_QC_IS_TELLCOR_C;
    cpl_error_code status = CPL_ERROR_NONE;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    if (cpl_propertylist_has(plist, key)) {
        moo_ensure_status(cpl_propertylist_set_bool(plist, key, val), status);
    }
    else {
        moo_ensure_status(cpl_propertylist_append_bool(plist, key, val),
                          status);
        moo_ensure_status(cpl_propertylist_set_comment(plist, key, comment),
                          status);
    }
moo_try_cleanup:
    return status;
}
/**@}*/
