/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_badpix.h"
#include "moo_rbn.h"
#include "moo_fibres_table.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_rbn RBN format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_rbn
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_rbn.h"
 * @endcode
 */


/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_rbn
  @return   1 newly allocated moo_ext or NULL in case of an error

  The returned object must be deallocated using moo_rbn_delete().

 */
/*----------------------------------------------------------------------------*/
moo_rbn *
moo_rbn_new(void)
{
    moo_rbn *res = cpl_calloc(1, sizeof(moo_rbn));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new empty RBN filename
  @return   1 newly allocated moo_rbn or NULL in case of an error

The returned object must be deallocated using moo_rbn_delete().

Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_rbn *
moo_rbn_create(const cpl_frame *frame)
{
    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prev_state = cpl_errorstate_get();
    moo_rbn *res = moo_rbn_new();
    const char *filename = cpl_frame_get_filename(frame);

    res->filename = filename;
    res->primary_header = cpl_propertylist_load(filename, 0);

    res->ri = moo_rbn_single_create(filename, MOO_TYPE_RI);
    res->yj = moo_rbn_single_create(filename, MOO_TYPE_YJ);
    res->h = moo_rbn_single_create(filename, MOO_TYPE_H);

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_info("moo_rbn", "Error in rbn create ecode=%d filename=%s",
                     cpl_error_get_code(), filename);
        cpl_errorstate_set(prev_state);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_rbn
  @param    self    moo_rbn to delete
  @return   void

  If the moo_rbn @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_rbn_delete(moo_rbn *self)
{
    if (self != NULL) {
        if (self->primary_header != NULL) {
            cpl_propertylist_delete(self->primary_header);
        }

        if (self->ri != NULL) {
            moo_rbn_single_delete(self->ri);
        }
        if (self->yj != NULL) {
            moo_rbn_single_delete(self->yj);
        }
        if (self->h != NULL) {
            moo_rbn_single_delete(self->h);
        }

        if (self->fibre_table != NULL) {
            cpl_table_delete(self->fibre_table);
        }

        cpl_free(self);
    }
}


/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbn
  @brief    Free the given type part in RBN
  @param    self    the RBN
  @param    type    the type of extension to free
  @return   error code or CPL_ERROR_NONE

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_rbn_free_single(moo_rbn *self, moo_detector_type type)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;

    moo_rbn_single *single = moo_rbn_get_single(self, type);
    if (single != NULL) {
        moo_rbn_single_free(single);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief assign moo_rbn_single structure in moo_rbn structure
  @param self moo_rbn structure to update
  @param single moo_rbn_single structure

Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_rbn_set_single(moo_rbn *self, moo_rbn_single *single)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();

    if (single != NULL) {
        switch (single->type) {
            case MOO_TYPE_RI:
                self->ri = single;
                break;
            case MOO_TYPE_YJ:
                self->yj = single;
                break;
            case MOO_TYPE_H:
                self->h = single;
                break;
        }
    }
    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_error("moo_rbn", "Error for adding rbn to %s (%d)",
                      self->filename, cpl_error_get_code());
        status = cpl_error_get_code();
        cpl_errorstate_set(prev_state);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Add RBN_SINGLE extension to RBN filename and update moo_rbn structure
  @param self moo_rbn structure to update
  @param type the detector type (RI, YJ, H)
  @param single moo_rbn_single structure
  @return the relevant error code or CPL_ERROR_NONE

Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_rbn_add_single(moo_rbn *self,
                   moo_detector_type type,
                   moo_rbn_single *single)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();
    moo_rbn_set_single(self, single);

    moo_rbn_single_save(single, self->filename, type);

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_error("moo_rbn", "Error for adding single to %s (%s)",
                      self->filename,
                      cpl_error_get_message_default(cpl_error_get_code()));
        status = cpl_error_get_code();
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_rbn to a FITS file
  @param    self          moo_rbn to write to disk or NULL
  @param    filename      Name of the file to write
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_rbn to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
void
moo_rbn_save(moo_rbn *self, const char *filename)
{
    if (self != NULL) {
        cpl_propertylist_save(self->primary_header, filename, CPL_IO_CREATE);
        moo_rbn_single_save(self->ri, filename, MOO_TYPE_RI);
        moo_rbn_single_save(self->yj, filename, MOO_TYPE_YJ);
        moo_rbn_single_save(self->h, filename, MOO_TYPE_H);

        if (self->fibre_table != NULL) {
            cpl_propertylist *h = cpl_propertylist_new();
            cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                           MOO_FIBRES_TABLE_EXTNAME);
            cpl_table_save(self->fibre_table, h, h, filename, CPL_IO_EXTEND);
            cpl_propertylist_delete(h);
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Add fibre table to RBN filename and update moo_rbn structure
  @param self moo_rbn structure to update
  @param fibre_table fibre table to store on RBN file

Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_rbn_add_fibre_table(moo_rbn *self, cpl_table *fibre_table)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(fibre_table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();
    self->fibre_table = fibre_table;

    cpl_propertylist *h = cpl_propertylist_new();
    cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                   MOO_FIBRES_TABLE_EXTNAME);
    cpl_table_save(self->fibre_table, h, h, self->filename, CPL_IO_EXTEND);
    cpl_propertylist_delete(h);

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_error(__func__, "Error for adding fibre table to %s (%d)",
                      self->filename, cpl_error_get_code());
        status = cpl_error_get_code();
        cpl_errorstate_set(prev_state);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get a RBN single from RBN
  @param self moo_rbn structure to read
  @param type type of detector
  @return   RBN_SINGLE

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_rbn_single *
moo_rbn_get_single(moo_rbn *self, moo_detector_type type)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    switch (type) {
        case MOO_TYPE_RI:
            return self->ri;
            break;
        case MOO_TYPE_YJ:
            return self->yj;
            break;
        case MOO_TYPE_H:
            return self->h;
            break;
        default:
            return NULL;
    }
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbn
  @brief    Load the type part in RBN and return it
  @param    self    the RBN
  @param    type    the type of extension to load
  @param    level   the bad pixel level

  @return  the moo_rbn_single corresponding to the type or NULL

  Possible error code :
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 *
 */
/*----------------------------------------------------------------------------*/
moo_rbn_single *
moo_rbn_load_single(moo_rbn *self, moo_detector_type type, unsigned int level)
{
    moo_rbn_single *res = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    switch (type) {
        case MOO_TYPE_RI:
            moo_rbn_single_load(self->ri, level);
            res = self->ri;
            break;
        case MOO_TYPE_YJ:
            moo_rbn_single_load(self->yj, level);
            res = self->yj;
            break;
        case MOO_TYPE_H:
            moo_rbn_single_load(self->h, level);
            res = self->h;
            break;
        default:
            res = NULL;
            break;
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the FIBRE TABLE in RBN
  @param    self    the RBN
  @return   The FIBRE TABLE

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_table *
moo_rbn_get_fibre_table(moo_rbn *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (self->fibre_table == NULL && self->filename != NULL) {
        cpl_size qnum =
            cpl_fits_find_extension(self->filename, MOO_FIBRES_TABLE_EXTNAME);
        if (qnum > 0) {
            self->fibre_table = cpl_table_load(self->filename, qnum, 0);
        }
    }
    return self->fibre_table;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Compute SNR for targets in RBN
  @param    self [RBN] the rbn
  @param    skylines SKY_LINE_LIST
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

*/
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_rbn_compute_snr(moo_rbn *self, moo_sky_lines_list *skylines)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(skylines != NULL, CPL_ERROR_NULL_INPUT);

    cpl_table *fibre_table = moo_rbn_get_fibre_table(self);
    const char *snrs[] = { MOO_FIBRES_TABLE_MEDSNR_RI_RBN,
                           MOO_FIBRES_TABLE_MEDSNR_YJ_RBN,
                           MOO_FIBRES_TABLE_MEDSNR_H_RBN };

    int nrow = cpl_table_get_nrow(fibre_table);

    for (int t = 0; t < nrow; t++) {
        int targ_idx =
            cpl_table_get_int(fibre_table, MOO_FIBRES_TABLE_INDEXRBN, t, NULL);
        int health =
            cpl_table_get_int(fibre_table, MOO_FIBRES_TABLE_HEALTH, t, NULL);
        if (health > 0) {
            for (int i = 0; i < 3; i++) {
                moo_rbn_single *single =
                    moo_rbn_load_single(self, i, MOO_BADPIX_GOOD);
                if (single != NULL) {
                    double snr = 0.0;
                    snr =
                        moo_rbn_single_compute_snr(single, targ_idx, skylines);

                    cpl_table_set_float(fibre_table, snrs[i], t, snr);
                }
            }
        }
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbn
  @brief    Dump structural information of RBN
  @param    self    RBN to dump
  @param    stream  Output stream, accepts @c stdout or @c stderr
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_rbn_dump(const moo_rbn *self, FILE *stream)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(stream != NULL, CPL_ERROR_NULL_INPUT);

    fprintf(stream, "---MOO_RBN\n");
    fprintf(stream, "filename %s\n", self->filename);

    if (self->ri != NULL) {
        fprintf(stream, "---RI %p\n", self->ri);
        moo_rbn_single_dump(self->ri, stream);
    }

    if (self->yj != NULL) {
        fprintf(stream, "---YJ %p\n", self->yj);
        moo_rbn_single_dump(self->yj, stream);
    }

    if (self->h != NULL) {
        fprintf(stream, "---H %p\n", self->h);
        moo_rbn_single_dump(self->h, stream);
    }

    if (self->fibre_table != NULL) {
        fprintf(stream, "---FT %p\n", self->fibre_table);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbn
  @brief    Get the PRIMARY HEADER in RBN
  @param    self    the RBN
  @return   The PRIMARY_HEADER

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 *
 */
/*----------------------------------------------------------------------------*/
cpl_propertylist *
moo_rbn_get_primary_header(moo_rbn *self)
{
    if (self->primary_header == NULL && self->filename != NULL) {
        self->primary_header = cpl_propertylist_load(self->filename, 0);
    }
    return self->primary_header;
}
/**@}*/
