/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include "moo_pfits.h"
#include "moo_rbn_single.h"
#include "moo_fits.h"
#include "moo_qc.h"
#include "moo_badpix.h"
#include "moo_fibres_table.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_rbn_single RBN_SINGLE format
 * @ingroup moo_rbn
 * This module provides functions to create, use, and destroy a
 * @em moo_rbn_single
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_rbn_single.h"
 * @endcode
 */


/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_rbn_single
  @return   1 newly allocated moo_rbn_single or NULL in case of an error

  The returned object must be deallocated using moo_rbn_single_delete().

 */
/*----------------------------------------------------------------------------*/
moo_rbn_single *
moo_rbn_single_new(moo_detector_type type)
{
    moo_rbn_single *res = cpl_calloc(1, sizeof(moo_rbn_single));
    res->badpix_mask = MOO_BADPIX_GOOD;
    res->type = type;
    const char *extname = moo_detector_get_name(type);
    res->extname = extname;
    res->header = cpl_propertylist_new();
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_rbn_single
  @param    self    moo_rbn_single to delete
  @return   void

  If the moo_rbn_single @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_rbn_single_delete(moo_rbn_single *self)
{
    if (self != NULL) {
        if (self->header != NULL) {
            cpl_propertylist_delete(self->header);
        }
        if (self->image != NULL) {
            hdrl_image_delete(self->image);
        }
        if (self->qual != NULL) {
            cpl_image_delete(self->qual);
        }
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_rbn_single with the given filename and type
  @param    filename the RBN filename
  @param    type the detector type
  @return   1 newly allocated moo_rbn_single or NULL in case of an error

  The returned object must be deallocated using moo_rbn_single_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_rbn_single *
moo_rbn_single_create(const char *filename, moo_detector_type type)
{
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);

    moo_rbn_single *single = NULL;
    cpl_propertylist *header = NULL;

    const char *extname = moo_detector_get_name(type);

    header = moo_fits_load_extension_header(filename, NULL, extname);
    if (header != NULL) {
        int naxis = moo_pfits_get_naxis(header);
        if (naxis > 0) {
            single = moo_rbn_single_new(type);
            single->filename = filename;
            cpl_propertylist_append(single->header, header);
        }
        cpl_propertylist_delete(header);
    }
    return single;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_rbn_single to a FITS file
  @param    self          moo_rbn_single to write to disk or NULL
  @param    filename      Name of the file to write
  @param    type      detector type
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

*/
/*----------------------------------------------------------------------------*/
void
moo_rbn_single_save(const moo_rbn_single *self,
                    const char *filename,
                    moo_detector_type type)
{
    const char *extname = moo_detector_get_name(type);
    if (self != NULL) {
        if (self->image != NULL) {
            moo_fits_write_extension_image(hdrl_image_get_image(self->image),
                                           filename, NULL, extname,
                                           CPL_TYPE_FLOAT, self->header);

            cpl_propertylist *err_header = cpl_propertylist_new();
            moo_pfits_append_hduclass_error(err_header, type, -1, self->header);

            moo_fits_write_extension_image(hdrl_image_get_error(self->image),
                                           filename, MOO_RBN_SINGLE_ERR,
                                           extname, MOO_RBN_SINGLE_ERR_TYPE,
                                           err_header);
            cpl_propertylist_delete(err_header);
            cpl_propertylist *qual_header = cpl_propertylist_new();
            moo_pfits_append_hduclass_quality(qual_header, type, -1,
                                              self->header, self->badpix_mask);
            moo_fits_write_extension_image(self->qual, filename,
                                           MOO_RBN_SINGLE_QUAL, extname,
                                           MOO_RBN_SINGLE_QUAL_TYPE,
                                           qual_header);
            cpl_propertylist_delete(qual_header);
        }
    }
    else {
        cpl_propertylist *h = cpl_propertylist_new();
        cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME, extname);
        cpl_propertylist_save(h, filename, CPL_IO_EXTEND);
        cpl_propertylist_delete(h);

        moo_fits_write_extension_image(NULL, filename, MOO_RBN_SINGLE_ERR,
                                       extname, MOO_RBN_SINGLE_ERR_TYPE, NULL);
        moo_fits_write_extension_image(NULL, filename, MOO_RBN_SINGLE_QUAL,
                                       extname, MOO_RBN_SINGLE_QUAL_TYPE, NULL);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the WCS1 of the extension
  @param    self single the extension
  @param    crpix1 the crpix1 keyword
  @param    crval1 the crval1 keyword
  @param    cd1_1 the cd1_1 keyword
  @param    ctype1 the ctype1 keyword
  @param    cunit1 the cunit1 keyword
  @return   the relevant error code

 the header of the extension must be preallocated before calling this function
 Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if a field of extension is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_rbn_single_set_wcs1(moo_rbn_single *self,
                        double crpix1,
                        double crval1,
                        double cd1_1,
                        const char *ctype1,
                        const char *cunit1)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->header != NULL, CPL_ERROR_ILLEGAL_INPUT);

    cpl_error_code code = CPL_ERROR_NONE;
    code =
        cpl_propertylist_append_double(self->header, MOO_PFITS_CRPIX1, crpix1);
    cpl_ensure_code(code == CPL_ERROR_NONE, code);
    code =
        cpl_propertylist_append_double(self->header, MOO_PFITS_CRVAL1, crval1);
    cpl_ensure_code(code == CPL_ERROR_NONE, code);
    code = cpl_propertylist_append_double(self->header, MOO_PFITS_CD1_1, cd1_1);
    cpl_ensure_code(code == CPL_ERROR_NONE, code);
    code =
        cpl_propertylist_append_string(self->header, MOO_PFITS_CTYPE1, ctype1);
    cpl_ensure_code(code == CPL_ERROR_NONE, code);
    code =
        cpl_propertylist_append_string(self->header, MOO_PFITS_CUNIT1, cunit1);
    cpl_ensure_code(code == CPL_ERROR_NONE, code);

    return code;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbn_single
  @brief   Get image of RBN_SINGLE
  @param    self    RBN_SINGLE
  @return   image or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
hdrl_image *
moo_rbn_single_get_image(moo_rbn_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    return self->image;
}
/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbn_single
  @brief   Get image of data
  @param    self    RBN_SINGLE
  @return   image or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_rbn_single_get_data(moo_rbn_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_image *res = NULL;

    if (self->image != NULL) {
        res = hdrl_image_get_image(self->image);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbn_single
  @brief   Get image of errs
  @param    self    RBN_SINGLE
  @return   image or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_rbn_single_get_errs(moo_rbn_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_image *res = NULL;

    if (self->image != NULL) {
        res = hdrl_image_get_error(self->image);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbn_single
  @brief   Get image of qual
  @param    self    RBN_SINGLE
  @return   image or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_rbn_single_get_qual(moo_rbn_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->extname != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (self->qual == NULL) {
        self->qual =
            moo_fits_load_extension_image(self->filename, MOO_RBN_SINGLE_QUAL,
                                          self->extname,
                                          MOO_RBN_SINGLE_QUAL_TYPE);
    }
    return self->qual;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbn_single
  @brief   Get header of rbn single
  @param    self    RBN_SINGLE
  @return   the header or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_propertylist *
moo_rbn_single_get_header(moo_rbn_single *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (self->header == NULL && self->extname != NULL &&
        self->filename != NULL) {
        self->header =
            moo_fits_load_extension_header(self->filename, NULL, self->extname);
        if (self->header == NULL) {
            self->header = cpl_propertylist_new();
        }
    }
    return self->header;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    load the data of a moo_rbn_single
  @param    self
  @param    level the level of bad pixel
  @return   the error status or CPL_ERROR_NONE

  Possible _cpl_error_code_ set in this function:
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_rbn_single_load(moo_rbn_single *self, unsigned int level)
{
    if (self != NULL) {
        if ((self->filename != NULL) && (self->extname != NULL)) {
            if (self->header == NULL) {
                cpl_size extnum =
                    cpl_fits_find_extension(self->filename, self->extname);
                if (extnum > 0) {
                    self->header =
                        cpl_propertylist_load(self->filename, extnum);
                }
            }
            if (self->qual == NULL) {
                self->qual =
                    moo_fits_load_extension_image(self->filename,
                                                  MOO_RBN_SINGLE_QUAL,
                                                  self->extname, CPL_TYPE_INT);
            }
            if (self->image == NULL) {
                cpl_image *data = NULL;
                cpl_image *err = NULL;

                data = moo_fits_load_extension_image(self->filename, NULL,
                                                     self->extname,
                                                     CPL_TYPE_DOUBLE);
                err = moo_fits_load_extension_image(self->filename,
                                                    MOO_RBN_SINGLE_ERR,
                                                    self->extname,
                                                    CPL_TYPE_DOUBLE);

                if (data != NULL && err != NULL) {
                    self->image = hdrl_image_create(data, err);
                    cpl_image_delete(data);
                    cpl_image_delete(err);
                }
            }
        }

        cpl_mask *mask = hdrl_image_get_mask(self->image);
        moo_badpix_to_mask(self->qual, mask, level);
    }
    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbn_single
  @brief Compute SNR for a given target
  @param self  [RBN_SINGLE] rbn_single structure
  @param rbn_idx id of the target
  @param skylines the SKYLINES which contains OH free zones
  @return snr
*/
/*----------------------------------------------------------------------------*/
double
moo_rbn_single_compute_snr(moo_rbn_single *self,
                           int rbn_idx,
                           moo_sky_lines_list *skylines)
{
    double snr = 0.0;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, 0.0);

    hdrl_image *himg = moo_rbn_single_get_image(self);
    cpl_propertylist *header = moo_rbn_single_get_header(self);

    int nx = hdrl_image_get_size_x(himg);
    double crpix1 = moo_pfits_get_crpix1(header);
    double cd1_1 = moo_pfits_get_cd1_1(header);
    double crval1 = moo_pfits_get_crval1(header);
    double wmin = (0.5 - crpix1) * cd1_1 + crval1;
    double wmax = (nx + 0.5 - crpix1) * cd1_1 + crval1;

    double *zwmin = NULL;
    double *zwmax = NULL;
    cpl_array *sel = NULL;
    moo_sky_lines_list_get_free_zones(skylines, wmin, wmax, &zwmin, &zwmax,
                                      &sel);
    int size = cpl_array_get_size(sel);
    cpl_vector *data = NULL;
    for (int i = 0; i < size; i++) {
        int idx = cpl_array_get_cplsize(sel, i, NULL);
        double cmin = zwmin[idx];
        double cmax = zwmax[idx];
        int start = (cmin - crval1) / cd1_1 + crpix1;
        int stop = (cmax - crval1) / cd1_1 + crpix1;

        hdrl_image *fimg =
            hdrl_image_extract(himg, start, rbn_idx, stop, rbn_idx);
        hdrl_image_reject_value(fimg, CPL_VALUE_NAN);
        cpl_image *img = hdrl_image_get_image(fimg);
        cpl_image *error = hdrl_image_get_error(fimg);
        cpl_image_reject_value(error, CPL_VALUE_NAN | CPL_VALUE_ZERO);

        int enx = cpl_image_get_size_x(error);
        int enbrej = cpl_image_count_rejected(error);

        if (enx > enbrej) {
            cpl_image_divide(img, error);
            cpl_vector *row = cpl_vector_new_from_image_row(img, 1);
            if (data == NULL) {
                data = cpl_vector_duplicate(row);
            }
            else {
                int vsize = cpl_vector_get_size(data);
                int row_size = cpl_vector_get_size(row);
                cpl_vector_set_size(data, vsize + row_size);
                for (int j = 0; j < row_size; j++) {
                    double v = cpl_vector_get(row, j);
                    cpl_vector_set(data, j + vsize, v);
                }
            }
            cpl_vector_delete(row);
        }
        hdrl_image_delete(fimg);
    }
    if (data != NULL) {
        snr = cpl_vector_get_median(data);
    }
    else {
        snr = NAN;
    }

    cpl_vector_delete(data);
    cpl_array_delete(sel);

    return snr;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbn_single
  @brief    Dump structural information of a Single DET
  @param    self    rbn single to dump
  @param    stream  Output stream, accepts @c stdout or @c stderr
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_FILE_IO if a write operation fails
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_rbn_single_dump(const moo_rbn_single *self, FILE *stream)
{
    cpl_ensure_code(stream != NULL, CPL_ERROR_NULL_INPUT);

    if (self != NULL) {
        fprintf(stream, "---MOO_RBN_SINGLE\n");
        fprintf(stream, "extname %s\n", self->extname);
        fprintf(stream, "filename %s\n", self->filename);
        fprintf(stream, "image: ");
        if (self->image != NULL) {
            cpl_image_dump_structure(hdrl_image_get_image(self->image), stream);
        }
        else {
            fprintf(stream, "null\n");
        }
        fprintf(stream, "header %p\n", (void *)self->header);
        fprintf(stream, "qual: ");
        if (self->qual != NULL) {
            cpl_image_dump_structure(self->qual, stream);
        }
        else {
            fprintf(stream, "null\n");
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbn_single
  @brief    Free memory associate to this single RBN
  @param    self    moo_rbn_single where want to free memory
  @return   error_code or CPL_ERROR_NONE

  Possible _cpl_error_code_ set in this function:
    - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_rbn_single_free(moo_rbn_single *self)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    if (self->filename != NULL && self->extname != NULL) {
        if (self->header != NULL) {
            cpl_propertylist_delete(self->header);
            self->header = NULL;
        }
        if (self->image != NULL) {
            hdrl_image_delete(self->image);
            self->image = NULL;
        }
        if (self->qual != NULL) {
            cpl_image_delete(self->qual);
            self->qual = NULL;
        }
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**@}*/
