/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include <hdrl.h>
#include <assert.h>
#include "moo_rbnlist.h"
#include "moo_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_rbnlist RBN_LIST format
 * @ingroup moo_data
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_rbnlist
  @return   1 newly allocated moo_rbnlist or NULL in case of an error

  The returned object must be deallocated using moo_rbnlist_delete().

 */
moo_rbnlist *
moo_rbnlist_new(void)
{
    return (moo_rbnlist *)cpl_calloc(1, sizeof(moo_rbnlist));
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_rbnlist from the given RBN frameset
  @return   1 newly allocated moo_rbnlist or NULL in case of an error

  The returned object must be deallocated using moo_rbnlist_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
moo_rbnlist *
moo_rbnlist_create(const cpl_frameset *frameset)
{
    cpl_ensure(frameset != NULL, CPL_ERROR_NULL_INPUT, NULL);
    int i;
    moo_rbnlist *res = moo_rbnlist_new();

    for (i = 0; i < cpl_frameset_get_size(frameset); ++i) {
        const cpl_frame *current_frame =
            cpl_frameset_get_position_const(frameset, i);
        moo_rbn *rbn = moo_rbn_create(current_frame);
        moo_rbnlist_push(res, rbn);
    }

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbnlist
  @brief    Load the type part for all RBN in the rbnlist
  @param    self    the list of RBN
  @param    type    the type of extension to load
  @param    level   Mask error level
  @return   The error code or CPL_ERROR_NONE

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_rbnlist_load_single(const moo_rbnlist *self,
                        moo_detector_type type,
                        int level)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    int i;

    for (i = 0; i < self->size; i++) {
        moo_rbn *rbn = self->list[i];
        moo_rbn_load_single(rbn, type, level);
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_detlist
  @brief    Free the type part for all DET in the detlist
  @param    self    the list of DET
  @param    type    the type of extension to load
  @return   The error code or CPL_ERROR_NONE

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if num is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if num is bigger than 1
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_rbnlist_free_single(const moo_rbnlist *self, moo_detector_type type)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    int i;
    for (i = 0; i < self->size; i++) {
        moo_rbn *rbn = self->list[i];
        cpl_error_code status = moo_rbn_free_single(rbn, type);
        if (status != CPL_ERROR_NONE) {
            return status;
        }
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbnlist
  @brief    Get the number of RBN in the rbnlist
  @param    self    the list of RBN
  @return   The number of RBN or -1 on error

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_size
moo_rbnlist_get_size(const moo_rbnlist *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, -1);

    assert(self->size >= 0);

    return self->size;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbnlist
  @brief    Get the RBN at the position i in the list
  @param    self    the list of RBN
  @param    i       position in the list
  @return   RBN element or NULL

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if i <0 or i >= list size
 */
/*----------------------------------------------------------------------------*/
moo_rbn *
moo_rbnlist_get(moo_rbnlist *self, int i)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(i >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    int size = moo_rbnlist_get_size(self);
    cpl_ensure(i < size, CPL_ERROR_ILLEGAL_INPUT, NULL);

    return self->list[i];
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbnlist
  @brief    Get the all the images of the type part in the rbnlist
  @param    self    the list of RBN
  @param    type     the type of extension to load
  @return   The  image list or NULL

  TAS number is 1 or 2.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  
 */
/*----------------------------------------------------------------------------*/
hdrl_imagelist *
moo_rbnlist_get_image(const moo_rbnlist *self, moo_detector_type type)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    hdrl_imagelist *res = NULL;
    int i;

    res = hdrl_imagelist_new();

    for (i = 0; i < self->size; i++) {
        moo_rbn *rbn = self->list[i];
        moo_rbn_single *single = NULL;
        moo_try_check(single = moo_rbn_get_single(rbn, type), " ");

        if (single != NULL) {
            hdrl_image *img = moo_rbn_single_get_image(single);
            hdrl_imagelist_set(res, img, i);
        }
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in rbnlist");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        hdrl_imagelist_delete(res);
        res = NULL;
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbnlist
  @brief    Get the type data part for all RBN in the rbnlist
  @param    self    the list of RBN
  @param    type    the type of extension to load
  @return   The type data list

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  
 */
/*----------------------------------------------------------------------------*/
cpl_imagelist *
moo_rbnlist_get_single_data(const moo_rbnlist *self, moo_detector_type type)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_imagelist *res = cpl_imagelist_new();
    int i;

    for (i = 0; i < self->size; i++) {
        moo_rbn *rbn = self->list[i];
        moo_rbn_single *single = moo_rbn_get_single(rbn, type);
        if (single != NULL) {
            cpl_imagelist_set(res, moo_rbn_single_get_data(single), i);
        }
    }

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbnlist
  @brief    Get the type QUAL part for all RBN in the rbnlist
  @param    self    the list of RBN
  @param    type    the type of extension to load
  @return   The type QUAL list

  TAS number is 0 or 1.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 *
 */
/*----------------------------------------------------------------------------*/
cpl_imagelist *
moo_rbnlist_get_single_qual(const moo_rbnlist *self, moo_detector_type type)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_imagelist *res = cpl_imagelist_new();
    int i;

    for (i = 0; i < self->size; i++) {
        moo_rbn *rbn = self->list[i];
        moo_rbn_single *single = moo_rbn_get_single(rbn, type);
        if (single != NULL) {
            cpl_imagelist_set(res, moo_rbn_single_get_qual(single), i);
        }
    }

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbnlist
  @brief    Insert a RBN into an moo_rbnlist
  @param    self    The moo_rbnlist
  @param    rbn     The RBN to insert
  @param    pos     The list position (from 0 to number of RBN)
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  It is allowed to specify the position equal to the number of RBN in the
  list. This will increment the size of the rbnlist.

  No action occurs if a RBN is inserted more than once into the same
  position. It is allowed to insert the same RBN into two different
  positions in a list.

  The RBN is inserted at the position pos in the RBN list. If the RBN
  already there is only present in that one location in the list, then the
  RBN is deallocated.

  The added RBN is owned by the moo_rbnlist object, which deallocates it
  moo_rbnlist_delete is called. Other option is to use moo_rbnlist_unset to
  recover ownership of the RBN, in which case the moo_rbnlist object is
  not longer responsible for deallocating it.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if pos is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if pos is bigger than the number of
    RBN in rbnlist
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_rbnlist_set(moo_rbnlist *self, moo_rbn *rbn, cpl_size pos)
{
    cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(rbn, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pos >= 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(pos <= self->size, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Do nothing if the rbn is already there */
    if (pos < self->size && rbn == self->list[pos])
        return CPL_ERROR_NONE;

    if (pos == self->size) {
        self->size++;
        self->list =
            cpl_realloc(self->list, (size_t)self->size * sizeof(moo_rbn *));
    }
    else {
        /* Check if the rbn at the position to be overwritten
           is present in only one position */
        int i;

        for (i = 0; i < self->size; i++) {
            if (i != pos && self->list[i] == self->list[pos])
                break;
        }
        if (i == self->size) {
            /* The image at the position to be overwritten
               is present in only one position, so delete it */
            moo_rbn_delete(self->list[pos]);
        }
    }

    self->list[pos] = rbn;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbnlist
  @brief    Insert a RBN a the end of moo_rbnlist
  @param    self    The moo_rbnlist
  @param    rbn     The RBN to insert
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
  @see  moo_rbnlist_set()

  No action occurs if a RBN is inserted more than once into the same
  position. It is allowed to insert the same RBN into two different
  positions in a list.

  The RBN is inserted at the end in the RBN list. If the RBN
  already there is only present in that one location in the list, then the
  RBN is deallocated.

  The added RBN is owned by the moo_rbnlist object, which deallocates it
  moo_rbnlist_delete is called. Other option is to use moo_rbnlist_unset to
  recover ownership of the RBN, in which case the moo_rbnlist object is
  not longer responsible for deallocating it.

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_rbnlist_push(moo_rbnlist *self, moo_rbn *rbn)
{
    cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);
    return moo_rbnlist_set(self, rbn, self->size);
}
/*----------------------------------------------------------------------------*/
/**
  @ingroup  moo_rbnlist
  @brief    Remove a RBN from a RBN list
  @param    self   The moo_rbnlist
  @param    pos      The list position (from 0 to number of RBN-1)
  @return   The pointer to the removed RBN or NULL in error case

  The specified RBN is not deallocated, it is simply removed from the
  list. The pointer to the RBN is returned to let the user decide to
  deallocate it or not.
  Eventually, the RBN will have to be deallocated with moo_rbn_delete().

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if pos is negative
  - CPL_ERROR_ACCESS_OUT_OF_RANGE if pos is bigger than the number of
    images in self
 */
/*----------------------------------------------------------------------------*/
moo_rbn *
moo_rbnlist_unset(moo_rbnlist *self, cpl_size pos)
{
    moo_rbn *out;
    cpl_size i;

    cpl_ensure(self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(pos >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(pos < self->size, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);

    /* Get pointer to RBN to be removed */
    out = self->list[pos];

    /* Move the following RBN one position towards zero */
    for (i = pos + 1; i < self->size; i++) {
        self->list[i - 1] = self->list[i];
    }

    /* Decrement of the size */
    self->size--;

    return out;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbnlist
  @brief    Empty an moo_rbnlist and deallocate all its RBN
  @param    self  The RBN list or NULL
  @return   Nothing
  @see  moo_rbnlist_empty(), moo_rbnlist_delete()
  @note If @em self is @c NULL nothing is done and no error is set.

  After the call the moo_rbn list can be populated again. It must eventually
  be deallocted with a call to moo_rbnlist_delete().

 */
/*----------------------------------------------------------------------------*/
void
moo_rbnlist_empty(moo_rbnlist *self)
{
    if (self != NULL) {
        while (self->size > 0) { /* An iteration may unset more than 1 image */
            int i = self->size - 1;
            moo_rbn *del = moo_rbnlist_unset(self, i);

            moo_rbn_delete(del);

            /* If this image was inserted more than once into the list,
               the other insertions must be unset without a delete. */
            while (--i >= 0) {
                if (self->list[i] == del) {
                    /* This image was inserted more than once in the list */
                    (void)moo_rbnlist_unset(self, i);
                }
            }
        }
    }

    return;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_rbnlist
  @brief    Free memory used by a moo_rbnlist object, except the RBN
  @param    self    The RBN list or NULL
  @return   Nothing
  @see moo_rbnlist_empty()
  @note The caller must have pointers to all RBN in the list and is
        reponsible for their deallocation. If @em self is @c NULL nothing is
        done and no error is set.
 */
/*----------------------------------------------------------------------------*/
void
moo_rbnlist_unwrap(moo_rbnlist *self)
{
    if (self != NULL) {
        cpl_free(self->list);
        cpl_free(self);
    }

    return;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Free all memory used by a moo_rbnlist object including the RBN
  @param    self    The RBN list or NULL
  @return   void
  @see      moo_rbnlist_empty(), moo_rbnlist_unwrap()
  If the moo_rbnlist @em self is @c NULL, nothing is done and no error is set.

 */
void
moo_rbnlist_delete(moo_rbnlist *self)
{
    if (self != NULL) {
        moo_rbnlist_empty(self);
        moo_rbnlist_unwrap(self);
    }

    return;
}

/**@}*/
