/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_params.h"
#include "moo_single.h"
#include "moo_badpix.h"
#include "moo_ext.h"
#include "moo_ext_single.h"
#include "moo_utils.h"
#include "moo_rebin.h"
#include "moo_fits.h"
#include "moo_pfits.h"
#include "moo_qc.h"
#include "moo_fibres_table.h"
#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/
static cpl_table *
_moo_rebin_fibre_cut_edges(cpl_image *data,
                           cpl_image *wdata,
                           cpl_image *errs,
                           cpl_image *qual,
                           int numfib,
                           int direction)
{
    cpl_table *result = NULL;
    cpl_ensure(data != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(wdata != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(errs != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(qual != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_vector *all_ref_flux = cpl_vector_new_from_image_row(data, numfib);
    cpl_vector *all_ref_errs = cpl_vector_new_from_image_row(errs, numfib);
    cpl_vector *all_ref_qual = cpl_vector_new_from_image_row(qual, numfib);
    cpl_vector *all_ref_wave = cpl_vector_new_from_image_row(wdata, numfib);

    int bp = MOO_BADPIX_OUTSIDE_DATA_RANGE | MOO_BADPIX_CALIB_DEFECT;
    int nx = cpl_image_get_size_x(data);
    int ifirst = 0, ilast = nx - 1;
    for (int i = 0; i < nx; i++) {
        int q = (int)cpl_vector_get(all_ref_qual, i);
        double w = cpl_vector_get(all_ref_wave, i);
        if ((q & bp) == 0 && !isnan(w)) {
            ifirst = i;
            break;
        }
    }
    for (int i = nx - 1; i >= 0; i--) {
        int q = (int)cpl_vector_get(all_ref_qual, i);
        double w = cpl_vector_get(all_ref_wave, i);
        if ((q & bp) == 0 && !isnan(w)) {
            ilast = i;
            break;
        }
    }

    int dsize = ilast - ifirst + 1;

    result = cpl_table_new(dsize);
    cpl_table_new_column(result, MOO_REBIN_REFTABLE_WAVE, CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, MOO_REBIN_REFTABLE_X, CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, MOO_REBIN_REFTABLE_Y, CPL_TYPE_INT);
    cpl_table_new_column(result, MOO_REBIN_REFTABLE_FLUX, CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, MOO_REBIN_REFTABLE_ERR, CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, MOO_REBIN_REFTABLE_QUAL, CPL_TYPE_INT);

    for (int i = ifirst; i <= ilast; i++) {
        int it = i - ifirst;
        double q = cpl_vector_get(all_ref_qual, i);
        double f = cpl_vector_get(all_ref_flux, i);
        double w = cpl_vector_get(all_ref_wave, i);
        double e = cpl_vector_get(all_ref_errs, i);
        int x = i + 1;
        if (direction == -1) {
            x = nx - i + 1;
        }
        cpl_table_set(result, MOO_REBIN_REFTABLE_WAVE, it, w);
        cpl_table_set(result, MOO_REBIN_REFTABLE_X, it, x);
        cpl_table_set(result, MOO_REBIN_REFTABLE_Y, it, numfib);
        cpl_table_set(result, MOO_REBIN_REFTABLE_FLUX, it, f);
        cpl_table_set(result, MOO_REBIN_REFTABLE_ERR, it, e);
        cpl_table_set(result, MOO_REBIN_REFTABLE_QUAL, it, q);
    }
    cpl_propertylist *order = cpl_propertylist_new();
    cpl_propertylist_append_bool(order, MOO_REBIN_REFTABLE_WAVE, CPL_FALSE);
    cpl_table_sort(result, order);
    cpl_propertylist_delete(order);
    cpl_vector_delete(all_ref_wave);
    cpl_vector_delete(all_ref_flux);
    cpl_vector_delete(all_ref_errs);
    cpl_vector_delete(all_ref_qual);

    return result;
}

/* interpolate at the wavelength center of each bins */
static cpl_error_code
_moo_rebin_interpolate2(cpl_table *ref_table,
                        cpl_vector *res_pos,
                        cpl_vector *res_wave,
                        cpl_image *rbn_fluxes,
                        cpl_image *rbn_errs,
                        cpl_image *rbn_qual,
                        int imin,
                        int rbn_numfib)
{
    int ref_size = cpl_table_get_nrow(ref_table);
    double *ref_wave_data =
        cpl_table_get_data_double(ref_table, MOO_REBIN_REFTABLE_WAVE);
    double *ref_pos_data =
        cpl_table_get_data_double(ref_table, MOO_REBIN_REFTABLE_X);
    int *ref_qual = cpl_table_get_data_int(ref_table, MOO_REBIN_REFTABLE_QUAL);
    double *ref_flux_data =
        cpl_table_get_data_double(ref_table, MOO_REBIN_REFTABLE_FLUX);
    double *ref_errs_data =
        cpl_table_get_data_double(ref_table, MOO_REBIN_REFTABLE_ERR);

    cpl_vector *ref_wave = cpl_vector_wrap(ref_size, ref_wave_data);
    cpl_vector *ref_pos = cpl_vector_wrap(ref_size, ref_pos_data);
    cpl_vector *ref_flux = cpl_vector_wrap(ref_size, ref_flux_data);
    cpl_vector *ref_errs = cpl_vector_wrap(ref_size, ref_errs_data);

    /* second interpolation from xgrid to flux */
    int res_size = cpl_vector_get_size(res_wave);
    cpl_vector *res_flux = cpl_vector_new(res_size);
    cpl_vector *res_errs = cpl_vector_new(res_size);
    int *res_qual = cpl_calloc(res_size, sizeof(int));

    cpl_bivector *ref_flux_points =
        cpl_bivector_wrap_vectors(ref_pos, ref_flux);
    cpl_bivector *res_points = cpl_bivector_wrap_vectors(res_pos, res_flux);
    moo_interpolate_linear(res_points, res_errs, res_qual, ref_flux_points,
                           ref_errs, ref_qual);
    cpl_bivector_unwrap_vectors(ref_flux_points);
    cpl_bivector_unwrap_vectors(res_points);
    cpl_vector_unwrap(ref_wave);
    cpl_vector_unwrap(ref_pos);
    cpl_vector_unwrap(ref_flux);
    cpl_vector_unwrap(ref_errs);

    for (int i = 0; i < res_size; i++) {
        int rbn_idx = imin + 1 + i;
        int q = res_qual[i];
        double f = cpl_vector_get(res_flux, i);
        double e = cpl_vector_get(res_errs, i);

        cpl_image_set(rbn_fluxes, rbn_idx, rbn_numfib, f);
        cpl_image_set(rbn_errs, rbn_idx, rbn_numfib, e);
        cpl_image_set(rbn_qual, rbn_idx, rbn_numfib, q);
    }
#if MOO_DEBUG_REBIN
    if (rbn_numfib == 379) {
        cpl_table *t = cpl_table_new(res_size);
        cpl_table_new_column(t, "WAVE", CPL_TYPE_DOUBLE);
        cpl_table_new_column(t, "FLUX", CPL_TYPE_DOUBLE);
        cpl_table_new_column(t, "FLUX_ERR", CPL_TYPE_DOUBLE);
        cpl_table_new_column(t, "BPM", CPL_TYPE_INT);
        for (int i = 0; i < res_size; i++) {
            int rej;
            double w = cpl_vector_get(res_wave, i);
            double f = cpl_vector_get(res_flux, i);
            double e = cpl_vector_get(res_errs, i);
            int q = res_qual[i];
            cpl_table_set_double(t, "WAVE", i, w);
            cpl_table_set_double(t, "FLUX", i, f);
            cpl_table_set_double(t, "FLUX_ERR", i, e);
            cpl_table_set_int(t, "BPM", i, q);
        }
        cpl_table_save(t, NULL, NULL, "interpolate_linear_379_me.fits",
                       CPL_IO_CREATE);
        cpl_table_delete(t);
    }
#endif
    cpl_vector_delete(res_flux);
    cpl_vector_delete(res_errs);
    cpl_free(res_qual);

    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_rebin_sum(cpl_table *ref_table,
               double step,
               cpl_vector *res_pos,
               cpl_vector *res_wave,
               cpl_image *rbn_fluxes,
               cpl_image *rbn_errs,
               cpl_image *rbn_qual,
               int imin,
               int rbn_numfib)
{
    int ref_size = cpl_table_get_nrow(ref_table);
    double *ref_wave_data =
        cpl_table_get_data_double(ref_table, MOO_REBIN_REFTABLE_WAVE);
    double *ref_pos_data =
        cpl_table_get_data_double(ref_table, MOO_REBIN_REFTABLE_X);
    int *ref_qual = cpl_table_get_data_int(ref_table, MOO_REBIN_REFTABLE_QUAL);
    double *ref_flux_data =
        cpl_table_get_data_double(ref_table, MOO_REBIN_REFTABLE_FLUX);
    double *ref_errs_data =
        cpl_table_get_data_double(ref_table, MOO_REBIN_REFTABLE_ERR);

    cpl_vector *ref_wave = cpl_vector_wrap(ref_size, ref_wave_data);
    cpl_vector *ref_start = cpl_vector_new(ref_size);
    cpl_vector *ref_stop = cpl_vector_new(ref_size);
    cpl_vector *ref_step = cpl_vector_new(ref_size);

    double w1, w, w2, start, stop;
    for (int i = 1; i < ref_size - 1; i++) {
        w1 = cpl_vector_get(ref_wave, i - 1);
        w = cpl_vector_get(ref_wave, i);
        w2 = cpl_vector_get(ref_wave, i + 1);
        start = w - (w - w1) / 2.;
        stop = w + (w2 - w) / 2.;
        cpl_vector_set(ref_start, i, start);
        cpl_vector_set(ref_stop, i, stop);
        cpl_vector_set(ref_step, i, step);
    }

    w = cpl_vector_get(ref_wave, 0);
    w2 = cpl_vector_get(ref_wave, 1);
    stop = w - (w2 - w) / 2.;
    cpl_vector_set(ref_start, 0, w);
    cpl_vector_set(ref_stop, 0, stop);
    cpl_vector_set(ref_step, 0, step / 2.);

    w1 = cpl_vector_get(ref_wave, ref_size - 2);
    w = cpl_vector_get(ref_wave, ref_size - 1);
    start = w - (w - w1) / 2.;
    cpl_vector_set(ref_start, ref_size - 1, w1);
    cpl_vector_set(ref_stop, ref_size - 1, w);
    cpl_vector_set(ref_step, ref_size - 1, step / 2.);

    cpl_vector *ref_pos = cpl_vector_wrap(ref_size, ref_pos_data);
    cpl_vector *ref_flux = cpl_vector_wrap(ref_size, ref_flux_data);
    cpl_vector *ref_errs = cpl_vector_wrap(ref_size, ref_errs_data);

    int res_size = cpl_vector_get_size(res_pos);
    int first = (int)cpl_vector_get(ref_pos, 0);


    for (int i = 0; i < res_size - 1; i++) {
        double d1 = cpl_vector_get(res_pos, i);
        double d2 = cpl_vector_get(res_pos, i + 1);
        double x1 = round(d1);
        double x2 = round(d2);

        int ix1 = (int)x1 - first;
        int ix2 = (int)x2 - first;


        double flux = 0.0;
        double num = NAN;
        double den = NAN;
        double e = 0.0;
        int q = MOO_BADPIX_GOOD;

        if (ix1 == ix2) {
            start = cpl_vector_get(res_wave, i);
            stop = cpl_vector_get(res_wave, i + 1);
            den = cpl_vector_get(ref_step, ix1);
            num = (stop - start) / den;
            e = cpl_vector_get(ref_errs, ix1) * num;
            q |= ref_qual[ix1];
            flux = cpl_vector_get(ref_flux, ix1) * num;
        }
        else {
            start = cpl_vector_get(res_wave, i);
            stop = cpl_vector_get(ref_stop, ix1);
            den = cpl_vector_get(ref_step, ix1);
            num = (stop - start) / den;
            flux += cpl_vector_get(ref_flux, ix1) * num;
            e = cpl_vector_get(ref_errs, ix1) * cpl_vector_get(ref_errs, ix1) *
                num;
            q |= ref_qual[ix1];

            int min = ix1 + 1;
            int max = ix2 - 1;
            for (int j = min; j <= max; j++) {
                q |= ref_qual[j];
                start = cpl_vector_get(ref_start, j);
                stop = cpl_vector_get(ref_stop, j);
                den = cpl_vector_get(ref_step, j);
                num = (stop - start) / den;
                flux += cpl_vector_get(ref_flux, j) * num;
                e += cpl_vector_get(ref_errs, j) * cpl_vector_get(ref_errs, j) *
                     num;
                w = cpl_vector_get(ref_wave, j);
            }
            w = cpl_vector_get(ref_wave, ix2);
            start = cpl_vector_get(ref_start, ix2);
            stop = cpl_vector_get(res_wave, i + 1);
            den = cpl_vector_get(ref_step, ix2);
            num = (stop - start) / den;
            flux += cpl_vector_get(ref_flux, ix2) * num;
            e += cpl_vector_get(ref_errs, ix2) * cpl_vector_get(ref_errs, ix2) *
                 num;
            q |= ref_qual[ix2];
            e = sqrt(e);
        }
        int rbn_idx = imin + 1 + i;

        cpl_image_set(rbn_fluxes, rbn_idx, rbn_numfib, flux);
        cpl_image_set(rbn_errs, rbn_idx, rbn_numfib, e);
        cpl_image_set(rbn_qual, rbn_idx, rbn_numfib, q);
    }
#if MOO_DEBUG_REBIN
    if (rbn_numfib == 379) {
        int nx = cpl_image_get_size_x(rbn_fluxes);
        cpl_table *t = cpl_table_new(res_size - 1);
        cpl_table_new_column(t, "WAVE", CPL_TYPE_DOUBLE);
        cpl_table_new_column(t, "FLUX", CPL_TYPE_DOUBLE);
        cpl_table_new_column(t, "FLUX_ERR", CPL_TYPE_DOUBLE);
        cpl_table_new_column(t, "BPM", CPL_TYPE_INT);
        for (int i = 0; i < res_size - 1; i++) {
            int rej;
            w1 = cpl_vector_get(res_wave, i);
            w2 = cpl_vector_get(res_wave, i + 1);
            w = (w1 + w2) / 2.;
            int rbn_idx = imin + 1 + i;
            double f = cpl_image_get(rbn_fluxes, rbn_idx, rbn_numfib, &rej);
            double e = cpl_image_get(rbn_errs, rbn_idx, rbn_numfib, &rej);
            int q = cpl_image_get(rbn_qual, rbn_idx, rbn_numfib, &rej);
            cpl_table_set_double(t, "WAVE", i, w);
            cpl_table_set_double(t, "FLUX", i, f);
            cpl_table_set_double(t, "FLUX_ERR", i, e);
            cpl_table_set_int(t, "BPM", i, q);
        }
        cpl_table_save(t, NULL, NULL, "integrate_linear_379_me.fits",
                       CPL_IO_CREATE);
        cpl_table_delete(t);
    }
#endif
    cpl_vector_unwrap(ref_pos);
    cpl_vector_unwrap(ref_flux);
    cpl_vector_unwrap(ref_errs);

    cpl_vector_unwrap(ref_wave);
    cpl_vector_delete(ref_start);
    cpl_vector_delete(ref_stop);
    cpl_vector_delete(ref_step);

    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_rebin_fibre(cpl_image *data,
                 cpl_image *wmap,
                 cpl_image *errs,
                 cpl_image *qual,
                 int numfib,
                 double step,
                 double min,
                 cpl_image *rbn_fluxes,
                 cpl_image *rbn_errs,
                 cpl_image *rbn_qual,
                 int fibfirst,
                 const char *method,
                 int conserv_flux,
                 int direction)
{
    cpl_table *ref_table = NULL;
    cpl_vector *res_wave = NULL;
    cpl_vector *res_pos = NULL;
    cpl_bivector *ref_pos_points = NULL;
    cpl_bivector *res_pos_points = NULL;
    int rej;

    int rbn_numfib = numfib + fibfirst;

    ref_table =
        _moo_rebin_fibre_cut_edges(data, wmap, errs, qual, numfib, direction);
    int ref_size = cpl_table_get_nrow(ref_table);
    double ref_wmin =
        cpl_table_get_double(ref_table, MOO_REBIN_REFTABLE_WAVE, 0, &rej);
    double ref_wmax = cpl_table_get_double(ref_table, MOO_REBIN_REFTABLE_WAVE,
                                           ref_size - 1, &rej);

    int rbn_size = cpl_image_get_size_x(rbn_fluxes);
    double rbn_wmin = min;
    double rbn_wmax = min + rbn_size * step;

    if (ref_wmin < ref_wmax && ref_wmin < rbn_wmax) {
        int imin = 0;
        int imax = rbn_size;

        imin = (int)ceil((ref_wmin - min) / step);
        imax = floor((ref_wmax - min) / step);

        if (imin < 0) {
            imin = 0;
        }
        if (imin >= rbn_size) {
            imin = rbn_size - 1;
        }
        if (imax < 0) {
            imax = 0;
        }
        if (imax >= rbn_size) {
            imax = rbn_size;
        }

        for (int i = 0; i < imin; i++) {
            cpl_image_set(rbn_fluxes, i + 1, rbn_numfib, NAN);
            cpl_image_set(rbn_errs, i + 1, rbn_numfib, NAN);
            cpl_image_set(rbn_qual, i + 1, rbn_numfib,
                          MOO_BADPIX_OUTSIDE_DATA_RANGE);
        }
        for (int i = imax; i < rbn_size; i++) {
            cpl_image_set(rbn_fluxes, i + 1, rbn_numfib, NAN);
            cpl_image_set(rbn_errs, i + 1, rbn_numfib, NAN);
            cpl_image_set(rbn_qual, i + 1, rbn_numfib,
                          MOO_BADPIX_OUTSIDE_DATA_RANGE);
        }

        if (strcmp(method, MOO_REBIN_METHOD_INTERPOLATE) == 0) {
            /* first interpolate from lambda grid to xgrid */
            int res_size = (imax - imin);
            res_wave = cpl_vector_new(res_size);
            res_pos = cpl_vector_new(res_size);

            for (int i = 0; i < res_size; i++) {
                double val1 = min + (imin + i) * step + step / 2.;
                cpl_vector_set(res_wave, i, val1);
            }
        }
        else {
            /* first interpolate from lambda grid to xgrid */
            int res_size = (imax - imin) + 1;
            res_wave = cpl_vector_new(res_size);
            res_pos = cpl_vector_new(res_size);
            for (int i = 0; i < res_size; i++) {
                double val1 = min + (imin + i) * step;
                cpl_vector_set(res_wave, i, val1);
            }
        }

        cpl_errorstate prev_state = cpl_errorstate_get();

        double *ref_wave_data =
            cpl_table_get_data_double(ref_table, MOO_REBIN_REFTABLE_WAVE);
        double *ref_pos_data =
            cpl_table_get_data_double(ref_table, MOO_REBIN_REFTABLE_X);
        double *ref_flux_data =
            cpl_table_get_data_double(ref_table, MOO_REBIN_REFTABLE_FLUX);
        double *ref_errs_data =
            cpl_table_get_data_double(ref_table, MOO_REBIN_REFTABLE_ERR);

        cpl_vector *ref_wave = cpl_vector_wrap(ref_size, ref_wave_data);
        cpl_vector *ref_pos = cpl_vector_wrap(ref_size, ref_pos_data);
        cpl_vector *ref_flux = cpl_vector_wrap(ref_size, ref_flux_data);
        cpl_vector *ref_errs = cpl_vector_wrap(ref_size, ref_errs_data);
        res_pos_points = cpl_bivector_wrap_vectors(res_wave, res_pos);
        ref_pos_points = cpl_bivector_wrap_vectors(ref_wave, ref_pos);
        cpl_bivector_interpolate_linear(res_pos_points, ref_pos_points);
        cpl_bivector_unwrap_vectors(ref_pos_points);
        cpl_bivector_unwrap_vectors(res_pos_points);
        cpl_vector_unwrap(ref_wave);
        cpl_vector_unwrap(ref_pos);
        cpl_vector_unwrap(ref_flux);
        cpl_vector_unwrap(ref_errs);

        if (cpl_errorstate_is_equal(prev_state)) {
            if (strcmp(method, MOO_REBIN_METHOD_INTERPOLATE) == 0) {
                _moo_rebin_interpolate2(ref_table, res_pos, res_wave,
                                        rbn_fluxes, rbn_errs, rbn_qual, imin,
                                        rbn_numfib);
            }
            else {
                _moo_rebin_sum(ref_table, step, res_pos, res_wave, rbn_fluxes,
                               rbn_errs, rbn_qual, imin, rbn_numfib);
            }
        }
        else {
            cpl_msg_error("moo_rebin", "#%d Invalid monotonicity (%f,%f)",
                          numfib, ref_wmin, ref_wmax);
            cpl_errorstate_set(prev_state);
            for (int i = 0; i < rbn_size; i++) {
                cpl_image_set(rbn_fluxes, i + 1, rbn_numfib, NAN);
                cpl_image_set(rbn_errs, i + 1, rbn_numfib, NAN);
                cpl_image_set(rbn_qual, i + 1, rbn_numfib,
                              MOO_BADPIX_CALIB_DEFECT);
            }
        }

        if (conserv_flux) {
            /* first interpolate from lambda grid to xgrid */
            int res_size = (imax - imin) + 1;
            cpl_vector_delete(res_wave);
            cpl_vector_delete(res_pos);
            res_wave = cpl_vector_new(res_size);
            res_pos = cpl_vector_new(res_size);
            for (int i = 0; i < res_size; i++) {
                double val1 = min + (imin + i) * step;
                cpl_vector_set(res_wave, i, val1);
            }
            res_pos_points = cpl_bivector_wrap_vectors(res_wave, res_pos);
            ref_wave = cpl_vector_wrap(ref_size, ref_wave_data);
            ref_pos = cpl_vector_wrap(ref_size, ref_pos_data);
            ref_pos_points = cpl_bivector_wrap_vectors(ref_wave, ref_pos);
            cpl_bivector_interpolate_linear(res_pos_points, ref_pos_points);
            for (int i = imin; i < imax; i++) {
                int rbn_idx = i + 1;
                double pix_lo = cpl_vector_get(res_pos, i - imin);
                double pix_up = cpl_vector_get(res_pos, i - imin + 1);
                double pix_size = pix_up - pix_lo;
                double flux =
                    cpl_image_get(rbn_fluxes, rbn_idx, rbn_numfib, &rej);
                cpl_image_set(rbn_fluxes, rbn_idx, rbn_numfib, flux * pix_size);
                double err = cpl_image_get(rbn_errs, rbn_idx, rbn_numfib, &rej);
                cpl_image_set(rbn_errs, rbn_idx, rbn_numfib, err * pix_size);
            }
            cpl_bivector_unwrap_vectors(ref_pos_points);
            cpl_bivector_unwrap_vectors(res_pos_points);
            cpl_vector_unwrap(ref_wave);
            cpl_vector_unwrap(ref_pos);
        }
    }
    else {
        cpl_msg_error("moo_rebin",
                      "#%d Invalid solution (%f,%f) rbn range is '%f,%f)'",
                      numfib, ref_wmin, ref_wmax, rbn_wmin, rbn_wmax);
        for (int i = 0; i < rbn_size; i++) {
            cpl_image_set(rbn_fluxes, i + 1, rbn_numfib, NAN);
            cpl_image_set(rbn_errs, i + 1, rbn_numfib, NAN);
            cpl_image_set(rbn_qual, i + 1, rbn_numfib, MOO_BADPIX_CALIB_DEFECT);
        }
    }

    cpl_vector_delete(res_wave);
    cpl_vector_delete(res_pos);
    cpl_table_delete(ref_table);
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get if a rebin step is oversampled compared to the WAVEMAP
  @param    step rebin step
  @param    wmap _EXT_ Wave map
  @param    sformat the spectral format
  @param    params the rebin parameters
  @param    filename the file name for the result
  @return   the _RBN_ spectra

 * This function rebin the extracted spectra.

 * _Bad pixels flags_:
  - BADPIX_OUTSIDE_DATA_RANGE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
_moo_rbn_compute_wmap_step(cpl_image *wmap,
                           double *qa,
                           double *qb,
                           int direction)
{
    int rej;
    cpl_ensure_code(wmap != NULL, CPL_ERROR_NULL_INPUT);
    int nx = cpl_image_get_size_x(wmap);
    int ny = cpl_image_get_size_y(wmap);

    cpl_image *pixsize = cpl_image_new(nx - 1, ny, CPL_TYPE_DOUBLE);
    for (int j = 1; j <= ny; j++) {
        for (int i = 1; i < nx - 1; i++) {
            double diff = cpl_image_get(wmap, i + 1, j, &rej) -
                          cpl_image_get(wmap, i, j, &rej) * direction;
            cpl_image_set(pixsize, i, j, diff);
        }
    }
    moo_image_get_quartile(pixsize, qa, qb);

    cpl_image_delete(pixsize);

    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_rebin_single(moo_rbn_single *rbn,
                  moo_ext_single *ext,
                  cpl_image *wmap,
                  const char *filename,
                  const int *health,
                  double step,
                  double min,
                  int first,
                  int *indexrbn,
                  const char **fibnames,
                  int *monotonous_tab,
                  cpl_array *indexes,
                  const char *method,
                  int conserv_flux,
                  int direction)
{
    double step_min, step_max;

    cpl_ensure_code(rbn != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ext != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(wmap != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(health != NULL, CPL_ERROR_NULL_INPUT);

    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_propertylist_copy_property_regexp(rbn->header, ext->header, "ESO DET *",
                                          0);


    _moo_rbn_compute_wmap_step(wmap, &step_min, &step_max, direction);

    if ((strcmp(method, MOO_REBIN_METHOD_INTEGRATE) == 0) && step < step_min) {
        cpl_msg_warning(
            "moo_rebin",
            "Use method %s with step %f but wavemap step is in range (%f,%f)",
            method, step, step_min, step_max);
    }
    else if ((strcmp(method, MOO_REBIN_METHOD_INTERPOLATE) == 0) &&
             step > step_max) {
        cpl_msg_warning(
            "moo_rebin",
            "Use method %s with step %f but wavemap step is in range (%f,%f)",
            method, step, step_min, step_max);
    }

    cpl_image *data = moo_ext_single_get_data(ext);
    cpl_image *errs = moo_ext_single_get_errs(ext);
    cpl_image *qual = moo_ext_single_get_qual(ext);

    int nb_fibres = cpl_image_get_size_y(data);

    cpl_image *rbn_fluxes = hdrl_image_get_image(rbn->image);
    cpl_image *rbn_errs = hdrl_image_get_error(rbn->image);
    cpl_image *rbn_qual = rbn->qual;
#ifdef _OPENMP
#pragma omp parallel default(none)                                            \
    shared(nb_fibres, health, rbn_fluxes, rbn_errs, rbn_qual, step, min, ext, \
               data, wmap, errs, qual, first, indexes, method, indexrbn,      \
               fibnames, monotonous_tab, conserv_flux, direction)
    {
#pragma omp for
#endif
        for (int i = 1; i <= nb_fibres; i++) {
            int idx = cpl_array_get_cplsize(indexes, i - 1, NULL);
            int h = health[idx];
            const char *fibname = fibnames[idx];

            indexrbn[idx] = i + first;
            if (h == 1) {
                int monotonous = monotonous_tab[idx];
                if (monotonous) {
                    _moo_rebin_fibre(data, wmap, errs, qual, i, step, min,
                                     rbn_fluxes, rbn_errs, rbn_qual, first,
                                     method, conserv_flux, direction);
                }
                else {
                    cpl_msg_warning(
                        __func__,
                        "skip #%s : no monotonous solution in WAVEMAP",
                        fibname);
                }
            }
        }
#ifdef _OPENMP
    }
#endif
    // dump error from the state
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Error in rebinning EXT:%s", ext->extname);
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return CPL_ERROR_NONE;
}

static cpl_propertylist *
_moo_create_primary_header(cpl_propertylist *ext_primary_header)
{
    cpl_propertylist *primary_header = NULL;

    primary_header = cpl_propertylist_new();
    cpl_propertylist_copy_property(primary_header, ext_primary_header,
                                   MOO_PFITS_EXPTIME);
    cpl_propertylist_copy_property(primary_header, ext_primary_header,
                                   MOO_PFITS_MJDOBS);
    cpl_propertylist_copy_property_regexp(primary_header, ext_primary_header,
                                          "TM*", 0);
    cpl_propertylist_copy_property_regexp(primary_header, ext_primary_header,
                                          "ESO QC IS*", 0);
    moo_pfits_set_fluxcal(primary_header, MOO_PFITS_FLUXCAL_UNCALIBRATED);
    moo_qc_set_is_tellcor(primary_header, CPL_FALSE);
    return primary_header;
}

static cpl_error_code
_moo_update_single_header(cpl_propertylist *result_header,
                          int j,
                          double step,
                          double wlmin,
                          double wlmax)
{
    cpl_error_code status = CPL_ERROR_NONE;
    moo_pfits_append_hduclass_data(result_header, j, -1);
    cpl_propertylist_append_double(result_header, MOO_PFITS_CRPIX2,
                                   MOO_EXT_SINGLE_CRPIX2);
    cpl_propertylist_append_double(result_header, MOO_PFITS_CRVAL2,
                                   MOO_EXT_SINGLE_CRVAL2);
    cpl_propertylist_append_double(result_header, MOO_PFITS_CD1_2, 0.);
    cpl_propertylist_append_double(result_header, MOO_PFITS_CD2_1, 0.);
    cpl_propertylist_append_double(result_header, MOO_PFITS_CD2_2,
                                   MOO_EXT_SINGLE_CDELT2);
    cpl_propertylist_append_string(result_header, MOO_PFITS_CTYPE2,
                                   MOO_EXT_SINGLE_CTYPE2);
    cpl_propertylist_append_string(result_header, MOO_PFITS_BUNIT,
                                   MOO_RBN_SINGLE_BUNIT);
    moo_pfits_set_wlmin(result_header, wlmin);
    moo_pfits_set_wlmax(result_header, wlmax);
    moo_pfits_set_spec_bin(result_header, step);
    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Rebin an EXT spectra in RBN format
  @param    ext _EXT_ Extracted 1D spectra
  @param    wmap _EXT_ Wave map
  @param    sformat the spectral format
  @param    params the rebin parameters
  @param    filename the file name for the result
  @return   the _RBN_ spectra

 * This function rebin the extracted spectra.

 * _Bad pixels flags_:
  - BADPIX_OUTSIDE_DATA_RANGE

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_rbn *
moo_rebin(moo_ext *ext,
          moo_map *wmap,
          moo_spectral_format *sformat,
          moo_rebin_params *params,
          const char *filename)
{
    moo_rbn *result = NULL;
    const char *monotonous_colnames[] = { MOO_FIBRES_TABLE_MONOTONOUS_RI,
                                          MOO_FIBRES_TABLE_MONOTONOUS_YJ,
                                          MOO_FIBRES_TABLE_MONOTONOUS_H };

    cpl_ensure(ext != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(wmap != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(sformat != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);
    const char *method = params->method;
    cpl_ensure(method != NULL, CPL_ERROR_ILLEGAL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();
    unsigned int badpix_level = MOO_BADPIX_GOOD;

    cpl_table *fibres_table = moo_ext_get_fibre_table(ext);
    cpl_ensure(fibres_table != NULL, CPL_ERROR_ILLEGAL_INPUT, NULL);

    cpl_msg_info(__func__, "Rebin spectra using method %s flux conservation:%d",
                 method, params->conserv_flux);


    moo_fits_create(filename);

    result = moo_rbn_new();

    result->filename = filename;

    result->primary_header = _moo_create_primary_header(ext->primary_header);
    cpl_image **maptab = wmap->data;
    cpl_propertylist **headertab = wmap->data_header;
    cpl_table *wmap_fibre_table = wmap->fibre_table;
    cpl_table *rbn_fibre_table = cpl_table_duplicate(fibres_table);
    moo_fibres_table_add_rebin_cols(rbn_fibre_table);

    result->fibre_table = rbn_fibre_table;
    cpl_ensure(rbn_fibre_table != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_array *idx1 = NULL, *idx2 = NULL;

    moo_try_check(
        (idx1 = moo_fibres_table_get_spectro_indexext(rbn_fibre_table, 1),
         idx2 = moo_fibres_table_get_spectro_indexext(rbn_fibre_table, 2)),
        " Can't get index from INDEXEXT");


    const int *health =
        cpl_table_get_data_int_const(rbn_fibre_table, MOO_FIBRES_TABLE_HEALTH);
    int *indexrbn =
        cpl_table_get_data_int(rbn_fibre_table, MOO_FIBRES_TABLE_INDEXRBN);
    const char **fibnames =
        cpl_table_get_data_string_const(rbn_fibre_table,
                                        MOO_FIBRES_TABLE_FIBRE);

    int nrows = cpl_table_get_nrow(rbn_fibre_table);
    for (int i = 0; i < nrows; i++) {
        const char *fibname = fibnames[i];
        cpl_table_select_all(wmap_fibre_table);
        moo_table_and_selected_sequal_string(wmap_fibre_table,
                                             MOO_FIBRES_TABLE_FIBRE, fibname);
        cpl_array *sel = cpl_table_where_selected(wmap_fibre_table);

        int idx = cpl_array_get_cplsize(sel, 0, NULL);
        for (int j = 0; j < 3; j++) {
            if (cpl_table_has_column(wmap_fibre_table,
                                     monotonous_colnames[j])) {
                int res = cpl_table_get_int(wmap_fibre_table,
                                            monotonous_colnames[j], idx, NULL);
                cpl_table_set_int(rbn_fibre_table, monotonous_colnames[j], i,
                                  res);
            }
            else {
                cpl_table_set_int(rbn_fibre_table, monotonous_colnames[j], i,
                                  1);
            }
        }
        cpl_array_delete(sel);
    }

    cpl_array *idx_tab[] = { idx1, idx2 };

    int size1 = cpl_array_get_size(idx1);
    int first_tab[] = { 0, size1 };

    cpl_msg_indent_more();

    for (int j = 0; j < 3; j++) {
        double wmin, wmax;
        moo_spectral_format_get_wave_range(sformat, j, &wmin, &wmax);
        double step = params->step[j];
        moo_rbn_single *rbn_single = moo_rbn_single_new(j);
        int nx = floor((wmax - wmin) / step);
        int ny = cpl_table_get_nrow(rbn_fibre_table);
        int *monotonous_tab =
            cpl_table_get_data_int(rbn_fibre_table, monotonous_colnames[j]);
        _moo_update_single_header(rbn_single->header, j, step, wmin, wmax);
        moo_rbn_single_set_wcs1(rbn_single, 0.5, wmin, step, MOO_REBIN_CTYPE1,
                                MOO_REBIN_CUNIT1);
        for (int i = 1; i <= 2; i++) {
            cpl_array *idx = idx_tab[i - 1];
            int first = first_tab[i - 1];
            cpl_image *map = maptab[(i - 1) * 3 + j];
            cpl_propertylist *map_header = headertab[(i - 1) * 3 + j];
            moo_ext_single *ext_single =
                moo_ext_load_single(ext, j, i, badpix_level);
            if (ext_single != NULL && map != NULL) {
                if (i == 1) {
                    cpl_msg_info(
                        "moo_rebin",
                        "Rebin extension %s with step %f and range [%f,%f]",
                        moo_detector_get_name(j), step, wmin, wmax);
                }
                if (rbn_single->image == NULL) {
                    rbn_single->image = hdrl_image_new(nx, ny);
                    rbn_single->qual = cpl_image_new(nx, ny, CPL_TYPE_INT);

                    cpl_image *img = moo_rbn_single_get_data(rbn_single);
                    cpl_image *error = moo_rbn_single_get_errs(rbn_single);
                    double *idata = cpl_image_get_data(img);
                    double *edata = cpl_image_get_data(error);
                    int *qdata = cpl_image_get_data(rbn_single->qual);
                    for (int k = 0; k < nx * ny; k++) {
                        qdata[k] = MOO_BADPIX_OUTSIDE_DATA_RANGE;
                        edata[k] = NAN;
                        idata[k] = NAN;
                    }
                }
                cpl_propertylist_copy_property(rbn_single->header,
                                               ext_single->header,
                                               MOO_PFITS_BUNIT);
                moo_spectral_format_info *info =
                    moo_spectral_format_get(sformat, j, i);
                if (cpl_propertylist_has(
                        map_header, MOONS_QC_WAVECAL_MONOTONOUS_SOLUTION)) {
                    int monotonous = cpl_propertylist_get_bool(
                        map_header, MOONS_QC_WAVECAL_MONOTONOUS_SOLUTION);
                    if (!monotonous) {
                        cpl_msg_warning(__func__,
                                        "WAVEMAP %s: not monotonous solution "
                                        "for all fibres",
                                        ext_single->extname);
                    }
                }

                _moo_rebin_single(rbn_single, ext_single, map, filename, health,
                                  step, wmin, first, indexrbn, fibnames,
                                  monotonous_tab, idx, method,
                                  params->conserv_flux, info->direction);


                moo_spectral_format_info_delete(info);
            }
        }
        if (rbn_single->image == NULL) {
            moo_rbn_single_delete(rbn_single);
            rbn_single = NULL;
        }
        moo_rbn_add_single(result, j, rbn_single);
    }
    moo_rbn_add_fibre_table(result, rbn_fibre_table);

    cpl_msg_indent_less();
moo_try_cleanup:
    cpl_array_delete(idx1);
    cpl_array_delete(idx2);

    if (!cpl_errorstate_is_equal(prestate)) {
        moo_rbn_delete(result);
        result = NULL;
        cpl_msg_error(__func__, "Error in rebin for EXT %s", ext->filename);
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return result;
}
/**@}*/
