/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_detector.h"
#include "moo_utils.h"
#include "moo_fits.h"
#include "moo_region.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_region  Manage region and region list parameters
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_region 
 
  @return   1 newly allocated moo_region or NULL in case of an error

  The returned object must be deallocated using moo_region_delete().  
  - - -  
 */
/*----------------------------------------------------------------------------*/
moo_region *
moo_region_new(void)
{
    moo_region *res = (moo_region *)cpl_calloc(1, sizeof(moo_region));

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_region
  @param    self    moo_region to delete
  @return   void

  If the moo_region @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/
void
moo_region_delete(moo_region *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}


/*----------------------------------------------------------------------------*/
/**
  @brief    load a moo_region
  @param    value the string value of region to load (format is four integers 
 *  separated by comma x1,y1,x2,y2)
  @return the corresponding moo_region object or NULL in cas of error
  
 Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_region *
moo_region_load(const char *value)
{
    cpl_ensure(value != NULL, CPL_ERROR_NULL_INPUT, NULL);

    moo_region *res = NULL;
    int a, b, c, d;
    int nb = sscanf(value, "%d,%d,%d,%d", &a, &b, &c, &d);
    if (nb == 4) {
        res = moo_region_new();
        res->xb = a;
        res->yb = b;
        res->xu = c;
        res->yu = d;
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    dump a moo_region
  @param self region to dump
   
 */
/*----------------------------------------------------------------------------*/
void
moo_region_dump(moo_region *self)
{
    if (self != NULL) {
        cpl_msg_info("moo_region", "REGION %d %d %d %d", self->xb, self->yb,
                     self->xu, self->yu);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_region 
  @param size the size of the region list
  @return   1 newly allocated moo_regionlist or NULL in case of an error

  The returned object must be deallocated using moo_regionlist_delete().  
  - - -  
 * Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_ILLEGAL_INPUT if size < 0
 */
/*----------------------------------------------------------------------------*/
moo_regionlist *
moo_regionlist_new(int size)
{
    cpl_ensure(size > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    moo_regionlist *res =
        (moo_regionlist *)cpl_calloc(1, sizeof(moo_regionlist));

    res->size = size;
    res->list = (moo_region **)cpl_calloc(size, sizeof(moo_region *));

    return res;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_regionlist
  @param    self    moo_regionlist to delete

  If the moo_region @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/
void
moo_regionlist_delete(moo_regionlist *self)
{
    if (self != NULL) {
        for (int i = 0; i < self->size; i++) {
            moo_region_delete(self->list[i]);
        }
        cpl_free(self->list);
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    load a moo_regionlist
  @param    value the string value of regionlist to load (format is region string
 *  separated by : r1:r2:r3...)
  @return the corresponding moo_regionlist object or NULL in cas of error
  
 Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_regionlist *
moo_regionlist_load(const char *value)
{
    cpl_ensure(value != NULL, CPL_ERROR_NULL_INPUT, NULL);

    char *token;
    const char *delim = ":";
    /* get the first token */
    char *cvalue = cpl_strdup(value);
    token = strtok(cvalue, delim);
    moo_regionlist *res = moo_regionlist_new(100);
    int i = 0;
    /* walk through other tokens */
    while (token != NULL) {
        moo_region *reg = moo_region_load(token);
        res->list[i] = reg;
        i++;
        token = strtok(NULL, delim);
    }
    res->size = i;
    cpl_free(cvalue);

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    load a moo_regionlist
  @param    layout_frame the LAYOUT frame
  @param type the detector type
  @param ntas number of spectrograph [1,2]
  @return the corresponding moo_regionlist object or NULL in cas of error
  
 Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT in cas of invalid ntas
 */
/*----------------------------------------------------------------------------*/
moo_regionlist *
moo_regionlist_load_layout(const cpl_frame *layout_frame,
                           moo_detector_type type,
                           int ntas)
{
    cpl_ensure(layout_frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(ntas >= 1 && ntas <= 2, CPL_ERROR_ILLEGAL_INPUT, NULL);

    const char *layout_name = NULL;
    layout_name = cpl_frame_get_filename(layout_frame);

    cpl_table *layout =
        moo_fits_load_extension_table(layout_name, NULL,
                                      moo_detector_get_extname(type, ntas));

    int nrow = cpl_table_get_nrow(layout);
    moo_regionlist *res = moo_regionlist_new(nrow);

    /* walk through other tokens */
    for (int i = 0; i < nrow; i++) {
        moo_region *reg = moo_region_new();
        int llx = cpl_table_get_int(layout, MOO_REGION_LAYOUT_LLX, i, NULL);
        int lly = cpl_table_get_int(layout, MOO_REGION_LAYOUT_LLY, i, NULL);
        int urx = cpl_table_get_int(layout, MOO_REGION_LAYOUT_URX, i, NULL);
        int ury = cpl_table_get_int(layout, MOO_REGION_LAYOUT_URY, i, NULL);
        reg->xb = llx;
        reg->yb = lly;
        reg->xu = urx;
        reg->yu = ury;
        res->list[i] = reg;
    }
    res->size = nrow;
    cpl_table_delete(layout);

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    dump a moo_regionlist
  @param self regionlist to dump
   
 */
/*----------------------------------------------------------------------------*/
void
moo_regionlist_dump(moo_regionlist *self)
{
    if (self != NULL) {
        cpl_msg_info("moo_region", "REGION_LIST %d", self->size);
        for (int i = 0; i < self->size; i++) {
            moo_region_dump(self->list[i]);
        }
    }
}
/**@}*/
